/**	
 *  @file	RFPUnpack.cpp
 * 
 *  @brief	Implementation of CRFPUnpack class
 *
 *	
 */

#include	<stdio.h>
#include	<string.h>
#include	<assert.h>

#include <openssl/rsa.h>       /* SSLeay stuff */
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rc4.h>
#include <openssl/evp.h>

#include <pp/syms.h>
#include <pp/RFPUnpack.h>
#include <pp/Scramble.h>
#include <pp/SIO.h>

namespace pp
{

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

#define	RFP_BUF_SIZE	(RFP_MAX_HEADER_SIZE + RFP_MAX_SIGNATURE_SIZE)

#define	RFP_SERVER_PUBLIC_KEY_FILE	"RFPRSAKEY.PEM"
#define	RFP_SERVER_RC4_FILE			"RFPRC4KEY.TXT"

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Code
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
CRFPUnpack::CRFPUnpack(  )
{
	SetKeyFiles(NULL,NULL);
}

/*  --------------------------------------------------------------------*/
CRFPUnpack::~CRFPUnpack()
{
}

/*  --------------------------------------------------------------------*/
void CRFPUnpack::SetKeyFiles( char * pRSAKeyFile, char * pRC4KeyFile  )
{
	if (pRSAKeyFile == NULL)
		strcpy(this->rsaKeyFile,RFP_SERVER_PUBLIC_KEY_FILE);
	else
		strcpy(this->rsaKeyFile,pRSAKeyFile);

	if (pRC4KeyFile == NULL)
		strcpy(this->rc4KeyFile,RFP_SERVER_RC4_FILE);
	else
		strcpy(this->rc4KeyFile,pRC4KeyFile);
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::UnpackRFP( CSIO * pSIO )
{
	char		buffer[RFP_BUF_SIZE+1]; // +1 for null terminator
	char	*	p;
	int			error,error2;
	int			result = 0;
	int			x;
	int			count;
	int			_fileCount;
	int			bufferCount;
	int			bytesRead;
	int			signatureOffset;
	int			dataOffset;
	RFP		*	pRFP;
	RFP_FILE *	pRFP_File;
	CSIO	*	pSIOOut = NULL;
	FILE	*	fp;
	unsigned char	keyText[256];
	int				keyTextLength;
	RC4_KEY			key;

	// Read until we fill the buffer or run out of data
	// The buffer will hold the header, signature and parhaps some portion of the file data

	for ( bufferCount = 0, p = buffer; bufferCount < RFP_BUF_SIZE; )
	{
		error = RFPInput(pSIO, p, RFP_BUF_SIZE - bufferCount, &bytesRead );
		if (error != RFP_OK )
			return error;
		if (bytesRead == 0)
			break;
		bufferCount += bytesRead;
		p += bytesRead;
	}

	*p = 0; // make sure we are NULL terminated

	// Parse the Header

	error = ParseHeaderAndSignature( buffer, bufferCount, NULL, NULL, &signatureOffset, NULL, &dataOffset );
	if (error != RFP_OK)
		return error;

	// Check signature

	pRFP = GetHeader();
	if (pRFP->isSigned)
	{
		result = VerifyHeaderSignature( buffer, &buffer[signatureOffset], pRFP->rsaKeyFile[0] != 0 ? pRFP->rsaKeyFile : this->rsaKeyFile );
		if (result)
			return error;
	}
	
	// Validate the RFP Header

	error = ValidateHeader( pRFP );
	if (error != RFP_OK)
		return error;

	// Process Files

	p = &buffer[dataOffset];
	bufferCount -= dataOffset;

	for (x=0;x<GetFileCount();x++)
	{
		pRFP_File = EnumFile(x);

		// Resolve Name

		if (pRFP_File->isMetaFile)
		{
			error = ResolveMetaFile( pRFP_File->metaFileName, pRFP_File->resolvedFileName );
			if (error == RFP_SKIP_FILE)
				continue;
			if (error != RFP_OK)
				return error;
		}
		else
			strcpy(pRFP_File->resolvedFileName,pRFP_File->fileName);

		// Validate the file (version, model and etc.)

		error = ValidateFile( pRFP_File );
		if (error == RFP_SKIP_FILE)
			continue;
		if (error != RFP_OK)
			return error;

		// PreProcess the file (Get the SIO

		error = AllocateSIO( pRFP_File, &pSIOOut );
		if (error == RFP_SKIP_FILE)
			continue;
		if (error != RFP_OK)
			return error;
		if (pSIO == NULL)
			return RFP_ERROR_CANNOT_WRITE_FILE;

		// Initialize the key

		if (pRFP_File->isEncrypted)
		{
			fp = fopen(pRFP_File->rc4KeyFile[0] != 0 ? pRFP_File->rc4KeyFile : this->rc4KeyFile,"rb");
			if (fp == NULL)
				return RFP_ERROR_RC4_KEY_MISSING;
			keyTextLength = fread(keyText,1,256,fp);
			fclose( fp );
			if (keyTextLength < 1)
				return RFP_ERROR_RC4_KEY_MISSING;
			Scramble( (char *) keyText, keyTextLength );
			RC4_set_key( &key, keyTextLength, &keyText[0] );
		}

		// Write the data to the file

		error2 = RFP_OK;

		for (_fileCount = 0; _fileCount < pRFP_File->length; )
		{
			if (bufferCount > 0)
			{
				// Buffer still has data from the header read
				if (bufferCount > pRFP_File->length)
					count = pRFP_File->length;
				else
					count = bufferCount;

				if (pRFP_File->isEncrypted)
					RC4( &key, count, (unsigned char *) p, (unsigned char *) p );

				result = pSIOOut->Write( p, count);
				if (result < count)
				{
					error2 = RFP_ERROR_CANNOT_WRITE_FILE;
					break;
				}
				p += count;
				bufferCount -= count;
			}
			else
			{
				// No data left in the buffer, read more
				count = pRFP_File->length - _fileCount;
				if (count > RFP_BUF_SIZE)
					count = RFP_BUF_SIZE;

				error = RFPInput(pSIO, buffer, count, &bytesRead );
				if (error != RFP_OK)
				{
					error2 = RFP_ERROR_CANNOT_READ_FILE;
					break;
				}

				if (pRFP_File->isEncrypted)
					RC4( &key, bytesRead, (unsigned char *) buffer, (unsigned char *) buffer );

				result = pSIOOut->Write( buffer, bytesRead );
				if (result < bytesRead)
				{
					error = RFP_ERROR_CANNOT_WRITE_FILE;
					break;
				}
			}

			_fileCount += count;
		}

		// Check the signature

		if (pRFP_File->isSigned && error == RFP_OK)
			error2 = VerifySignature( pSIOOut, pRFP_File->signature, pRFP_File->rsaKeyFile[0] != 0 ? pRFP_File->rsaKeyFile : this->rsaKeyFile );

		// Post Process the file

		if (error == RFP_OK)

		error = PostProcessFile(pRFP_File,pSIOOut);
		if (error != RFP_OK)
			return error;

		FreeSIO(pRFP_File,pSIOOut);

		// Process script

		if (pRFP_File->isScript)
		{
			error = RunScript( pRFP_File->resolvedFileName, pRFP_File->script );
			if (error != RFP_OK)
				return error;
		}

		// Clean up

		error = ReleaseMetaFile( pRFP_File->metaFileName, pRFP_File->resolvedFileName );
		if (error != RFP_OK)
			return error;
	}

	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::RFPInput( CSIO * pSIO, char * pData, int dataLength, int *pBytesRead )
{
	int	error = RFP_ERROR_CANNOT_WRITE_FILE;
	*pBytesRead = 0;
	if (pSIO != NULL)
	{
		*pBytesRead = pSIO->Read(pData,dataLength);
		if (*pBytesRead >= 0)
			error = RFP_OK;
	}

	return error;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::ResolveMetaFile( char * NOTUSED(pMetaFileName), char * NOTUSED(pOutRealFileName) )
{
	return RFP_ERROR_BAD_METAFILE;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::ReleaseMetaFile( char * NOTUSED(pMetaFileName), char * NOTUSED(pResolvedFileName) )
{
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::ValidateHeader( RFP * NOTUSED(pRFP) )
{
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::ValidateFile( RFP_FILE * NOTUSED(pRFP_File) )
{
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::RunScript( char * NOTUSED(pFileName), char * NOTUSED(pScriptType) )
{
	return RFP_ERROR_BAD_SCRIPT;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::AllocateSIO( RFP_FILE * pRFP_File, CSIO ** ppSIO )
{
	if (pRFP_File->isMemoryFile == 0)
	{
		// Allocate a normal memory file
		CSIO_FP *pSIO_FP = new CSIO_FP;
		*ppSIO = pSIO_FP;

		if (pSIO_FP == NULL)
			return RFP_ERROR_MEMORY;

		int result = pSIO_FP->Open(pRFP_File->resolvedFileName,"w+");
		return result < 0 ? RFP_ERROR_CANNOT_WRITE_FILE : RFP_OK;
	}
	else
	{
		// Allocate a memory SIO
		
		*ppSIO = new CSIO_DMEM;
		if (*ppSIO == NULL)
			return RFP_ERROR_MEMORY;
		
		return RFP_OK;
	}
}

/*  --------------------------------------------------------------------*/
void CRFPUnpack::FreeSIO( RFP_FILE * NOTUSED(pRFP_File), CSIO * pSIO )
{
	delete pSIO;
}

/*  --------------------------------------------------------------------*/
int CRFPUnpack::PostProcessFile( RFP_FILE * NOTUSED(pRFP_File), CSIO * NOTUSED(pSIO) )
{
	return RFP_OK;
}

} // namespace


