/**	
 *  @file	RFPPack.h
 *  @brief	CRFPPack definition
 * 
 *  Class used to create RFP files
 *	
 */

#ifndef _RFPPack_h_
#define _RFPPack_h_
#include "pp/SIO.h"
#include "pp/RFPParser.h"

namespace pp
{

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Class
 *--------------------------------------*/

class CRFPPack : public CRFPParser
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Constructor for a CRFPPack object
	 *
     *  This object is used to create an RFP file.
	 *
	 */

	CRFPPack( );
	virtual ~CRFPPack();

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Set the default Key files
	 *
     *  @param	pRSAKeyFile	Ptr to the RSA public key File Name
     *                      (NULL = use default)
     *  @param	pRC4KeyFile	Ptr to the RC4 File Name (NULL = use
     *                      default)
	 *
	 */

	void SetKeyFiles( char * pRSAKeyFile = NULL, char * pRC4KeyFile = NULL  );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Add Header Text
     * 
     *  Use this method BEFORE PackRFP() to put additional text into the RFP
     *  header. An example string is "<HardwareVersion>1.0.0</HardwareVersion>"
     *  The text must contain 1 or more valid XML tags.
     *  This function can be called multiple times. Each call concatinates the new
     *  string to the old string.
	 *
     *  @param	pText       Text to add to the header (copied)
	 *
	 */

	void AddHeaderText( char * pText );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Creates the schema for the RFP from an RFP header
	 *
     *  @param	pSIO		Contains the XML Header
     *  @return             0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
	 *
	 */

	int CreateSchema( CSIO * pSIO );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Validates the RFP header and file(s) schema
     * 
     *  This function is called by PackRFP(), but it can be called before
     *  PackRFP() to check for just schema errors.
	 *
     *  @return             RFP_OK or RFP_ERROR_
	 *
	 */

	int ValidateSchema( );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Creates the RFP File
     * 
     *  The output of PackRFP is sent to the RFPOutput() function.
     *  The default behavior of RFPOutput() is to write the data to
     *  an SIO object. However, the user can override this method to
     *  send the information someplace else, like across a netowrk
     *  connection.
     * 
     *  NOTE: Only the RFPOutput function writes to the pSIO. If
     *  this function is overriddem, pSIO can be NULL.
	 *
     *  @param	pSIO        Where to output the RFP file.
     *  @return				0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
	 *
	 */

	int PackRFP( CSIO * pSIO = NULL );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Called by PackRFP to transmit the RFP data to it's
     *         destination.
     * 
     *  The default behavior is to write the data to the pSIO that
     *  was passed in to RFPPack().
     * 
     *  NOTE: RFPOutput is called with dataLength = 0 when the RFP
     *  file is complete.
     * 
     *  @param  pSIO        SIO object to write to (can be ignored
     *                      if overridding this function)
     *  @param  pData       Ptr to buffer containing the data
     *  @param  dataLength  # of bytes to write (1-64k)
     *  @param  totalLength The total length of the RFP. Useful for
     *                      calculatingn percentage done.
     *  @return				0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
	 *
	 */

	virtual int RFPOutput( CSIO * pSIO, char * pData, int dataLength, int totalLength );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Resolves A metafile into a realfile for metafiles
     *         where <SourceFile> is not specified in the RFP
     *         header.
     * 
     *  This function can be overridden to resolve metafiles.
     *  Te default implmentation returns an error.
     * 
     *  @param  pMetaFileName   Ptr to the Metafile to resolve
     *  @param  pOutRealFileName Buffer to return the real file name
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be resolved
	 *
	 */

	virtual int ResolveMetaFile( char *pMetaFileName, char *pOutRealFileName );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Cleans up after a Metafile.
     * 
     *  This function can be overridden to do any needed cleanup of
     *  resolved metafiles.
     *  The default implementation does nothing.
     * 
     *  @param  pMetaFileName   Ptr to the Metafile to resolve
     *  @param  pResolvedFileName Buffer containing the real file
     *                            name
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be resolved
	 *
	 */

	virtual int ReleaseMetaFile( char *pMetaFileName, char *pResolvedFileName );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Returns the state of the RFP processing
	 *
     *  @return				Returns RFP_STATE_ values
	 *
	 */

	inline int GetRFPState() {return state;}

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Returns the file index for the last error (-1 ==
     *         header)
	 *
     *  @return             error file index
	 *
	 */

	inline int GetErrorFile() {return errorFile;}

	/*  --------------------------------------------------------------------*/
	/*  --------------------------------------------------------------------*/
private:


	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Creates a file signature
     * 
     *  @param  pFileName   Ptr to the input file name
     *  @param  pKeyFileName Ptr to the RSA key file name
     *  @param  pOutBuffer  Buffer to put the signature in
     *  @param  rtnLength   Returns the length of the signature (can
     *                      be null
     *  @return				0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
     *` 
	 */

	int SignFile( char * pFileName, char *pKeyFileName, char * pOutBuffer, int *rtnLength = NULL );
	int SignBuffer( char * pInbuffer, int length, char *pKeyFileName, char * pOutBuffer, int *rtnLength = NULL );

private:
	int		state;						// Current processing state
	char	*pHeaderText;				// Additional header text
	char	rsaKeyFile[RFP_MAX_FILES];// Default RSA public key file
	char	rc4KeyFile[RFP_MAX_FILES];	// Default RC4 key file
	char * 	realFileName[RFP_MAX_FILES][RFP_MAX_FILENAME];	// Name of the REAL source file
	int		errorFile;					// Indicates which file caused the last error (-1 == header);
};

} // namespace

#endif	// _RFPPack_h_


