/*--------------------------------------------------------------------------------

	SXDB.h

	Copyright (c) 2002, Raritan Computer, Inc.

	Header file for SXDB.cpp.
	Contains CSXDB class definition.

--------------------------------------------------------------------------------*/

#ifndef _SXDB_H_
#define	_SXDB_H_


#include "pp/SXML_Errors.h"
#include "pp/SIO.h"
#include "pp/SXPath.h"


//----------------------------------------
//				Equates
//----------------------------------------

#define	SXDB_MAX_NODE_NAMES	16

	// Errors

enum
{
	SXDB_ERROR_MEMORY = -9000,			// No Memory Error
};

	// Node types

enum SXDB_TYPE
{
	SXDB_TYPE_NODE,						// Base Type
	SXDB_TYPE_DATA,						// Data Node
	SXDB_TYPE_ATTRIBUTE,				// Attribute Node
	SXDB_TYPE_ELEMENT,					// Element Node
	SXDB_TYPE_DOCUMENT					// Entire document (root of DB)
} ;

	// Access types

enum
{
	SXDB_ACCESS_READ = 0,
	SXDB_ACCESS_WRITE,
	SXDB_ACCESS_DELETE
};

//----------------------------------------
//				Data Types
//----------------------------------------

	// Get Node list array

typedef struct
{
	int		count;			// # of nodes
							// -1 == match all

	struct
	{
		const char	*pName;
		int		length;
		int		type;
	} Nodes[SXDB_MAX_NODE_NAMES];
} NODE_NAMES;

	// Get Recuresion Info

typedef struct
{
	NODE_NAMES	*	pTopNodes;			// Top level nodes we are looking for
	NODE_NAMES	*	pSubNodes;			// Sub nodes we are looking for
	CSIO		*	pOut;				// The output SIO
	int				depth;				// # of levels we will go down
	void		*	pUser;				// User account
} NODE_SEARCH_INFO;


//----------------------------------------
//				Function Prototypes
//----------------------------------------

	CSXDB_Node *						// NULL if not found, else node ptr
	SXDB_FindAnsestor
	(
		const char	* pName,			// The name we are searching for
		CSXDB_Node *	pChild,			// NULL to find the first, or the last found for next
		int				self			//  0 = Start at child->Parent()
										// !0 = Start at child
	);

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Base Node Class
//----------------------------------------

class CSXDB_Node
{
public:

	// Public Methods

	CSXDB_Node();
	virtual ~CSXDB_Node();

	
	SXDB_TYPE
	GetType
	(
	);

	void
	SetType
	(
		SXDB_TYPE	newType
	);

	virtual
	int
	Order
	(
	);

	const char *
	// returns a reference to the name
	GetName
	(
	);

	int									// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	// existing name will be freed/deleted
	// new name will be copied
	SetName
	(
		const char	*	pNewName			// The new name
	);

	virtual
	const char *
	GetData
	(
	);

	virtual
	int									// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	SetData
	(
		const char	*	pNewData			// The new data
	);

	void
	Strcpy
	(
		char	*	pString				// String to copy to
	);

	void
	Strncpy
	(
		char	*	pString,				// String to copy to
		int			maxLength				// Max # of characters to copy
	);

	int									// Same as strcmp
	Strcmp
	(
		const char	*	pString				// String to compare
	);

	int									// Same as stricmp
	Stricmp
	(
		const char	*	pString				// String to compare
	);

	virtual
	int								// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	AddData
	(
		const char	*	pData
	);

	CSXDB_Node *						// Ptr to parent
	Parent
	(
	);

	CSXDB_Node *						// Ptr to child
	Child
	(
	);

	CSXDB_Node *						// ptr to sibling
	Next
	(
	);

	CSXDB_Node *						// ptr to privious sibling
	Previous
	(
	);

	CSXDB_Node *						// NULL if not found, else node ptr
	FindChildOfType
	(
		SXDB_TYPE		type,			// The node type we are looking for
		CSXDB_Node *	pLast = NULL,	// NULL to find the first, or the last found for next
		int				ocurrance = 1	// Find the nth occurance of the element
	);

	CSXDB_Node *						// NULL if not found, else node ptr
	FindChildOfTypeByName
	(
		SXDB_TYPE		type,			// The node type we are looking for
		const char		*	pName,			// The name we are searching for
		CSXDB_Node  *	pLast = NULL,	// NULL to find the first, or the last found for next
		int				ocurrance = 1	// Find the nth occurance of the element
	);

	int									// The number of children that match type
	CountChildOfType
	(
		SXDB_TYPE		type			// The node type we are looking for
	);

	void
	AddChild
	(
		CSXDB_Node *	pNewChild		// Ptr to the new child to add
	);

	void
	InsertChild
	(
		CSXDB_Node *	pNewChild,		// Ptr to the new child to insert
		CSXDB_Node *	pBeforeChild = NULL	// NULL or the child to be before
	);

	void
	Remove
	(
	);

	// Private data Methods

private:

	CSXDB_Node		*	pParent;
	CSXDB_Node		*	pSibling;
	CSXDB_Node		*	pChild;

	SXDB_TYPE			type;			// The node type

	char		*	pName;
};

//----------------------------------------
//				Data Node Class
//----------------------------------------

class CSXDB_Data : public CSXDB_Node
{
public:

	CSXDB_Data();
	virtual ~CSXDB_Data();

	const char *
	// return a reference to the stored data
	GetData
	(
	);

	int									// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	// existing data will be freed/deleted
	// new data will be copied
	SetData
	(
		const char	*	pNewData			// The new data
	);

private:
	char			*	pData;
};

//----------------------------------------
//				Attribute Node Class
//----------------------------------------

class CSXDB_Attribute : public CSXDB_Node
{
public:

	CSXDB_Attribute();
	virtual ~CSXDB_Attribute();

	const char *					// Ptr to the data
        // returns a reference to the data
	GetData
	(
	);

	int								// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	// existing data will be freed/deleted
	// new data will be copied
	SetData
	(
		const char	*	pData
	);

};

//----------------------------------------
//				Element Node Class
//----------------------------------------

class CSXDB_Element : public CSXDB_Node
{
public:

	CSXDB_Element();
	virtual ~CSXDB_Element();


	CSXDB_Element	*	// Ptr to the new element or null if failed
	// name is copied
	AddElement
	(
		const char	*	pName
	);

	CSXDB_Attribute	*	// Ptr to the new element or null if failed
	// the name and the data are copied
	AddAttribute
	(
		const char	*	pName,
		const char	*	pData
	);

	CSXDB_Element	*	// Ptr to the new element or null if failed
	// name and data are copied
	AddElementWithData
	(
		const char	*	pName,
		const char	*	pData
	);

	int			// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	AddData
	(
		const char	*	pData
	);

	const char *
	// returns a reference to the data of the first CSXDB_Data node for this element
	GetData
	(
	);

	int			// Either 0 or SXML_ERROR_OUT_OF_MEMORY
	// existing data will be freed/deleted
	// new data will be copied
	SetData
	(
		const char	*	pData
	);

	int
	Order
	(
	);
};

//----------------------------------------
//				Document Node Class
//----------------------------------------

class CSXDB_Document : public CSXDB_Element
{
public:

	CSXDB_Document();
	virtual ~CSXDB_Document();
};

//----------------------------------------
//				Database Class
//----------------------------------------

class CSXDB
{
public:
	CSXDB();
	virtual ~CSXDB();

	CSXDB_Node *
	Root
	(
	);

	void
	SetRoot
	(
		CSXDB_Node	*pNewRoot
	);

	int									// returns error code
	Get
	(
		const char	*	pRoot,				// Ptr to the XPath statement defining the root
		const char	*	pNodes,				// Ptr to the list of top level nodes to include
		const char	*	pSubNodes,			// Ptr to the list of sub nodes to include
		int			depth,				// # of levels ot top level nodes to transvers
		CSIO	*	pOut,				// Oupput results
		void	*	pUser = NULL		// User account
	);

	CSXDB_Node *						// Ptr to the node
	GetNodeFromXPath
	(
		const char	*pXPath,				// XPath to the node to get data from
		void	*pUser = NULL,			// User account
		CSXDB_Node *pContextNode = NULL	// The context node to start searching from
	);

	CSXDB_Node *						// Ptr to the node
	GetNodeFromID
	(
		const char	*pID,					// Ptr to the unique ID
		void	*pUser = NULL,			// User account
		CSXDB_Node *pContextNode = NULL	// The context node to start searching from
	);

	const char *
	GetData
	(
		const char	*pXPath,				// XPath to the node to get data from
		void	*pUser = NULL,			// User account
		CSXDB_Node *pContextNode = NULL	// The context node to start searching from
	);

	const char *
	GetData
	(
		const char	*pXRoot,				// XPath to the root node
		const char	*pSubNode,				// Ptr t othe sub node
		void	*pUser = NULL,			// User account
		CSXDB_Node *pContextNode = NULL	// The context node to start searching from
	);

	int									// Error code or # updated
	PutData
	(
		const char	*pXPath,				// XPath to the node to get data from
		const char	*pSubNode,				// Ptr to the name of the element or attribute
		SXDB_TYPE type,					// SXDB_TYPE_ELEMENT or SXDB_TYPE_ATTRIBUTE
		const char	*pData,				// Ptr to the data to put
		void	*pUser = NULL,			// User account
		CSXDB_Node *pContextNode = NULL,// The context node to start searching from
		int		append = 0				// !0 = force append
	);

	int									// returns error code or order of the node
	Order
	(
		const char	*	pXPath,				// The XPath statement to evauluate
		void	*	pUser				// User account
	);

	int									// returns error code or # of nodes found
	Count
	(
		const char	*	pXPath,				// The XPath statement to evauluate
		void	*	pUser = NULL,		// User account
		CSXDB_Node *pContextNode = NULL	// The xpath context
	);

	int									// returns error code or # of nodes found
	Count
	(
		const char	*	pXPath,				// The XPath statement to evauluate
		const char	*	pSubNode,			// Sub node of pXPath to count
		void	*	pUser = NULL		// User account
	);

	int									// returns error code or number of top level nodes updated
	Update
	(
		const char	*	pRoot,				// Ptr to the XPath statement defining the root
		CSXDB	*	pDB,				// The new data
		int			overwrite,			// !0 = Append (on over write), 0 = Update (over write)
		void	*	pUser = NULL		// User account
	);

	int									// returns error code or number of top level nodes updated
	Update
	(
		const char	*	pRoot,				// Ptr to the XPath statement defining the root
		CSXDB_Node*	pNode,				// The new data
		int			overwrite,			// 0 = Append (on over write), !0 = Update (over write)
		void	*	pUser = NULL		// User account
	);

	int									// returns error code or # of nodes deleted (top level)
	Delete
	(
		const char	*	pRoot,				// Ptr to the XPath statement defining the nodes to delete
		void	*	pUser = NULL		// User account
	);

	virtual
	int									// !0 if access is allowed
										// 0 = Not allowed
										// -1 = unknown for this node
	CheckAccess
	(
		CSXDB_Node	*pNode,				// The node to authorize
		void		*pUser,				// User account
		int			accessType,			// SXDB_ACCESS_xxx
		int			oneLevel = 0		// !0 = only check pNode, not it's ansestors
	);

private:
	CSXDB_Node		*	pRoot;			// The root node


	int									// returns error code or number of top level nodes updated
	Update_Node
	(
		CSXDB_Node*	pNode,				// Ptr node to update
		CSXDB_Node*	pTop,				// Ptr to the new child node 
		int			overwrite,			// 0 = Append (on over write), !0 = Update (over write)
		void	*	pUser				// User account
	);

	int									// returns error code or # of nodes deleted (top level)
	ParseNodeNames
	(
		const char	*	pText,				// Text list of names
		NODE_NAMES *pNodeNames			// Ptr to the node names struct
	);

	int									// returns error code or # of nodes deleted (top level)
	SearchNodeNames
	(
		CSXDB_Node *pNode,				// The node to check
		NODE_NAMES *pNodeNames			// Ptr to the node names struct
	);

	int									// returns error code
	Get_NodeSearch
	(
		CSXDB_Node	*	pRootNode,		// Potential top level node
		int				level,			// # of levels ot top level nodes to transvers
		NODE_SEARCH_INFO * pInfo		// static info used be each level of recursion
	);

};

#endif // _SXDB_H_
