/*--------------------------------------------------------------------------------

	SXML.h

	Copyright (c) 2002, Raritan Computer, Inc.

	Header file for SXML.cpp.
	Contains CSXML class definition.
	Contains CSXML_FILE class definition

--------------------------------------------------------------------------------*/

#ifndef _SXML_H_
#define	_SXML_H_

#include "pp/SXML_Errors.h"
#include "pp/SIO.h"

//----------------------------------------
//				Equates
//----------------------------------------


	// SXML uses several internal buffers for parsing the data.
	// The size of these buffers limit how long names are
	// and what size chunks of data are passed to the event
	// handler.
	// The SXML_DEFAULT_ values define the default size
	// of these buffers. At run time, these sizes can be
	// overridden by calling SetBufferLimit(), with the
	// buffer ID (SXML_BUFFER_ID_) and the new size.
	// Calls to SetBufferLimit should be made for calling Parse()


	// Default Buffer Limits

#define	SXML_DEFAULT_NAME_SIZE			80			// # of characters allowed in a tag or attribute names
													// Names longer than this will be truncated.

#define	SXML_DEFAULT_DATA_SIZE			2048		// # of characters allowed in a data for a tag,
													// attribute data, and CDATA.
													// Data longer than this will be sent in chunks
													// of this size. See Event() for more documentations

#define	SXML_DEFAULT_NESTING			16			// Min number of nesting levels 
													// tags beyond this level will be ignored.
													// SXML allocated a buffer of SXML_DEFUALT_NESTING * SXML_DEFAULT_NAME_SIZE
													// to track nested names
													// NOTE: More nesting levels will be permitted if names 
													//       are smaller than SXML_DEFAULT_NAME_SIZE
													
#define	SXML_DEFAULT_READ_SIZE			2048		// # of bytes read from the file at one time

	// Buffer Limit IDs

enum
{
	SXML_BUFFER_ID_ALL		= 0,
	SXML_BUFFER_ID_NAME,
	SXML_BUFFER_ID_DATA,
	SXML_BUFFER_ID_NESTING,
	SXML_BUFFER_ID_READ,
};

	// Option IDs

enum
{
	SXML_OPTION_ID_ALL_DEFAULT	= 0,				// Use this ID to set all options to their default
	SXML_OPTION_ID_INCLUDE_WHITE_SPACE,				// Set this option to TRUE to include leading and trailing
													// white space in data events
													// Default is FALSE
};


	// Event Types

enum
{
	SXML_EVENT_START_DOCUMENT,
	SXML_EVENT_END_DOCUMENT,
	SXML_EVENT_START_TAG,
	SXML_EVENT_END_TAG,
	SXML_EVENT_DATA_START,
	SXML_EVENT_DATA,
	SXML_EVENT_ATTRIBUTE,
	SXML_EVENT_ATTRIBUTE_CONTINUED,
	SXML_EVENT_CDATA,

	SXML_EVENT_INIT_STATE
};

	// Parser Helper states

enum
{
	SXML_STATE_STRING	=	30000,		// Parsing a string
	SXML_STATE_STRING_ALLOC,			// Parsing a string & allocating space for it
	SXML_STATE_INT
};

	// Parsing Macros
	// These macros are designed to be used in a 'case SXML_EVENT_START_TAG'
	// If the new tag matches tagName, then action will be taken and a return
	// statement will be executed to get out event handler

#define	SXML_Tag_State( tagName, newState ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										state = newState; \
										return Event( SXML_EVENT_INIT_STATE ); \
									}

#define	SXML_Tag_State_Action( tagName, newState, action ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										state = newState; \
										action ; \
										return Event( SXML_EVENT_INIT_STATE ); \
									}

#define	SXML_Tag_Action( tagName, action) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										action ; \
										SkipTag(); \
										return 0; \
									}

#define	SXML_String( tagName, strPtr_, strLen ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										prevState = state; \
										state = SXML_STATE_STRING; \
										strPtr = (char *) strPtr_; \
										strLength = strLen; \
										return 0; \
									}

#define	SXML_String_Alloc( tagName, strPtr_, strLen ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										prevState = state; \
										state = SXML_STATE_STRING_ALLOC; \
										strPtr = (char *) strPtr_; \
										strLength = strLen; \
										return 0; \
									}

#define	SXML_String_Action( tagName, strPtr_, strLen, action ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										prevState = state; \
										state = SXML_STATE_STRING; \
										strPtr = (char *) strPtr_; \
										strLength = strLen; \
										action ; \
										return 0; \
									}

#define	SXML_Int( tagName, intPtr_ ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										prevState = state; \
										state = SXML_STATE_INT; \
										intPtr = intPtr_; \
										return 0; \
									}

#define	SXML_Int_Action( tagName, intPtr_, action ) \
									if ( strcmp( tagName, GetName() ) == 0 ) \
									{ \
										prevState = state; \
										state = SXML_STATE_INT; \
										intPtr = intPtr_; \
										action ; \
										return 0; \
									}

	// Test for valid name characters

//----------------------------------------
//				Data Types
//----------------------------------------

	// Data structure used to look up token values in FindToken()

typedef struct
{
	const char	*	name;			// Name of the token
	int			value;			// the value of the token
} SXML_TOKEN;

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Base Class
//----------------------------------------

class CSXML
{
public:

	CSXML
	(
	);

	virtual
	~CSXML
	(
	);

	int
	Parse
	(
		const char	*	pXMLData,			// Ptr to XML data or NULL to use Read()
		int			length				// Length of the XML data, if pXMLData != NULL
	);

	virtual
	int
	Event
	(
		int			event
	);

	virtual
	int									// # of bytes read or 0 for EOF
	Read
	(
		char	*	pData,				// Ptr to the buffer
		int			count				// # of bytes to read
	);

	int									// Old Limit
	SetBufferLimit
	(
		int			bufferID,			// Which buffer limit to set (SMXL_BUFFER_ID_)
		int			limit				// The new limit, or 0 for default
	);

	int									// Old value
	SetOption
	(
		int			optionID,			// Which buffer limit to set (SMXL_OPTION_ID_)
		int			value				// TRUE or FALSE
	);

//	inline
	const char *
	GetName
	(
	);

//	inline
	int
	GetNameLength
	(
	);

//	inline
	const char *
	GetData
	(
	);

//	inline
	int
	GetDataLength
	(
	);

	inline
	int
	IsMore
	(
	);

	const char *								// Returns ptr to root name
	GetRootTagName
	(
	);

	const char *								// Returns ptr to current name
	GetCurrentTagName
	(
	);

	const char *								// Returns ptr to next name or NULL if no more names
	GetNextTagName
	(
		const char	*pCurrentName			// Ptr to the current name
	);

	const char *								// Returns ptr to previous name or NULL if no more names
	GetPreviousTagName
	(
		const char	*pCurrentName			// Ptr to the current name
	);

//	inline
	void
	SkipTag
	(
	);

	inline
	void
	SkipData
	(
	);

	inline
	int
	GetLevel
	(
	);

	int
	GetFilePosition
	(
	);

	int
	GetTagPosition
	(
	);

	int
	HelperStates
	(
		int			event
	);

	int
	HelperString
	(
		int			event
	);

	int
	HelperInt
	(
		int			event
	);

#ifdef SXML_TEST
	inline
	void
	SetOutputFile
	(
		FILE	*fp						// New file to output to
	);

	inline
	FILE *
	GetOutputFile
	(
	);
#endif

	// Public data used by parse helper macros & functions

	int		state;						// Current state of the parser's FSM
	int		prevState;					// The state to return to after parsing a string or an int
	char *	strPtr;						// Where to put the parsed string
	int		strLength;					// Max length of the parsed string
	int *	intPtr;						// Where to put the parsed int

protected:

	int									// error code
	AllocBuffers
	(
		const char	*	pXMLData,			// Ptr to XML data or NULL to use Read()
		int			length				// Length of the XML data, if pXMLData != NULL
	);

	void
	FreeBuffers
	(
	);

	void
	SkipWhiteSpace
	(
	);

	int									// Error code
	ParseComment
	(
	);

	int
	ParseName
	(
	);

	int									// Error code
	ParseEntity
	(
	);

	int									// Error code
	ParseQuotedData
	(
		int		*more					// On input,  0  = First parse of string
										//            !0 = continue parsing
										// On Output, 0  = finished with data,
										//			  !0 = there is more data
	);

	int									// Error code
	ParseData
	(
	);

	int									// Error code
	ParseCData
	(
	);

	int									// Error code
	ParseAttribute
	(
	);

	int									// Error code
	ParseStartTag
	(
		int			*empty				// Return !0 if the start tag is an empty tag
										// ( <tag/> ... no end tag expected )
	);

	int									// Error code
	ParseEndTag
	(
	);

	int									// Error code
	ParseDocument
	(
	);

	int									// Error code
	ParseProlog
	(
	);

	inline
	char
	GetChar
	(
	);

	inline
	char
	NextChar
	(
	);

	const char	*							// Ptr to the look ahead data
	LookAhead
	(
		int			count				// # of bytes needed
	);

	inline
	int									// error code
	AdvancePtr
	(
		int			count				// # of bytes to advance
	);

	void
	ReadMoreData
	(
	);

	int									// Token value or -1 if not found
	FindToken
	(
		const char	*	pString,			// Ptr to the string to match
		SXML_TOKEN	*pTokenList			// Ptr to an array of SXML_TOKENs
										// List is terminated with a token that
										// has a null name ptr
	);

	// Name Buffer
	// Holds tag & attribute name

	int				nameBufferSize;		// Size of the name buffer
	char	*		pNameBuffer;		// Ptr to the name buffer
	int				nameBufferCount;	// # of bytes in the name buffer

	// Data Buffer
	// Holds chunks of tag data, attribute data and CDATA

	int				dataBufferSize;		// Size of the name buffer
	char	*		pDataBuffer;		// Ptr to the data buffer
	int				dataBufferCount;	// # of bytes in the data buffer

	// Read Buffer
	// Holds raw data from the XML file

	int				userBuffer;			// TRUE if using user's buffer
	int				readBufferSize;		// Size of the read buffer
	union {						// HACK for broken API to allow const
		char	*		pReadBuffer;	// Ptr to the read buffer
		const char *		pReadBufferConst;
	};
	int				readBufferCount;	// # of bytes in the read buffer
	const char	*		readPtr;			// Current Read position
	int				filePosition;		// The current character position in the file.

	// Nesting buffer

	int				nestBufferDepth;	// # of levels of nesting allowed
	int				nestBufferSize;		// Size of the nexting buffer
	char	*		pNestBuffer;		// Ptr to the buffer
	int				nestBufferCount;	// # of bytes in the nest buffer

	// Options

	int				includeWhiteSpace;	// If true, leading and trailing white space is included as data

	// Event processing

	int				more;				// !0 if there is more data
	int				skipData;			// !0 when we should skip remaining data
	int				skipTag;			// !0 when we are skipping all data and sub tags to the end of
	int				noSkipEndTagEvent;	// !0 when Parse End tag should NOT send an event
										// Used when nesting is too deep and tags are automatically skipped
	int				level;				// What level of nesting we are at
	int				tagStarted;			// !0 when default event handler is in a start tag
	int				tagPosition;		// The file position of the < for a start tag and the char after the > for the end tag

	// Testing

#ifdef SXML_Test
	FILE	*		fpOut;				// The output file for the default Event handler
#endif
};

//----------------------------------------
//			Standard File Varient
//				Class
//----------------------------------------

class CSXML_FILE : public CSXML
{
public:

	int
	ParseFile
	(
		const char	*	pFileName
	);

	int									// # of bytes read or 0 for EOF
	Read
	(
		char	*	pData,				// Ptr to the buffer
		int			count				// # of bytes to read
	);

private:

	FILE		*fp;					// The file to read from

};

//----------------------------------------
//			SIO Varient
//				Class
//----------------------------------------

class CSXML_SIO : public CSXML
{
public:

	int
	ParseSIO
	(
		CSIO	*	pSIO
	);

	int									// # of bytes read or 0 for EOF
	Read
	(
		char	*	pData,				// Ptr to the buffer
		int			count				// # of bytes to read
	);
private:
	CSIO		*	pSIO;				// Ptr to the SIO to read from
};

#endif	//_SXML_H_
