#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <lara.h>

#include <pp/base.h>
#include <pp_gpio.h>

#include <base_intern.h>

typedef struct {
    u_char dev_no;
    const char *device;
} gpio_info_t;

typedef struct {
    int	fd;
    gpio_info_t *info;
} gpio_dev_t;

gpio_info_t gpio_infos[] = {
    { PP_GPIO_DEV_IBM,		"/dev/gpio-ibm" },
    { PP_GPIO_DEV_FARADAY,	"/dev/gpio-faraday" },
    { PP_GPIO_DEV_OPENCORES_0,	"/dev/gpio-oc-0" },
    { PP_GPIO_DEV_OPENCORES_1,	"/dev/gpio-oc-1" },
    { PP_GPIO_DEV_XR17_0,	"/dev/gpio-xr17-0" },
    { PP_GPIO_DEV_XR17_1,	"/dev/gpio-xr17-1" },
    { 0,			NULL }
};

static pthread_mutex_t gpio_mtx = PTHREAD_MUTEX_INITIALIZER;
static int initialized = 0;
static gpio_dev_t devs[PP_GPIO_MAX_NR_DEVS];

static gpio_dev_t* gpio_check_device(u_char dev_no);

int
pp_gpio_init(void)
{
    int ret = -1;

    if (initialized) {
	goto bail;
    }

    memset(devs, 0, sizeof(devs));
    
    initialized = 1;
    ret = 0;
    
 bail:
    return ret;
}

void
pp_gpio_cleanup(void)
{
    int i;
    
    /* Actually this part cannot be partly initialized, so this check is safe */
    if (initialized) {
	MUTEX_LOCK(&gpio_mtx);
	for (i = 0; i < PP_GPIO_MAX_NR_DEVS; i++) {
	    gpio_dev_t * dev = &devs[i];
	    if (dev->info != NULL && dev->fd >= 0) {
		close(dev->fd);
		dev->fd = -1;
	    }
	}    
	MUTEX_UNLOCK(&gpio_mtx);
	initialized = 0;
    }
}

static gpio_dev_t*
gpio_check_device(u_char dev_no)
{
    gpio_dev_t *dev = NULL;

    assert(dev_no < PP_GPIO_MAX_NR_DEVS);

    MUTEX_LOCK(&gpio_mtx);
    
    dev = &devs[dev_no];
    if (dev->info == NULL) {
	gpio_info_t *info = gpio_infos;

	while (info && info->device) {
	    if (dev_no == info->dev_no) break;
	    info++;
	}

	if (!info || info->device == NULL) {
	    pp_log_err("%s(): Unsupported GPIO device %d", ___F, dev_no);
	    MUTEX_UNLOCK(&gpio_mtx);
	    goto error_out;
	}

	if ((dev->fd = open(info->device, O_RDWR)) < 0) {
	    pp_log_err("%s(): Error opening GPIO device %d ('%s')", ___F, dev_no, info->device);
	    MUTEX_UNLOCK(&gpio_mtx);
	    goto error_out;
	}

	dev->info = info;
    }

    MUTEX_UNLOCK(&gpio_mtx);
    return dev;

 error_out:
    /* this should never happen -> exit */
    pp_log_err("%s(): Don't have a GPIO device for dev nr. %d when we need it, should never happen!", ___F, dev_no);
    exit(1);    
}

u_long
pp_gpio_get(u_char dev_no)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    int err;
    u_long value;
    assert(dev);

    err = gpio_ioctl_rd_ulong(dev->fd, GPIO_IOCTL_IN, &value);
    if (err != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_rd_ulong_err for IN) on %d failed", ___F, dev_no);
	exit(1);
    }

    return value;
}

void
pp_gpio_set(u_char dev_no, u_long set, u_long clr, u_long tri)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    assert(dev);

    if (gpio_ioctl_out(dev->fd, set, clr, tri) != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_out) on %d failed", ___F, dev_no);
	exit(1);
    }
}

u_char
pp_gpio_bit_get(u_char dev_no, u_char bit)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    int value;
    assert(dev);

    if ((value = gpio_ioctl_get_bit(dev->fd, bit)) < 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_get_bit) on %d failed", ___F, dev_no);
	exit(1);
    }

    return value;
}

void
pp_gpio_bit_set(u_char dev_no, u_char bit, u_char value)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    assert(dev);

    if (gpio_ioctl_set_bit(dev->fd, bit, value) != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_set_bit) on %d failed", ___F, dev_no);
	exit(1);	
    }
}

void
pp_gpio_bit_set_enable(u_char dev_no, u_char bit, u_char value)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    assert(dev);

    if (gpio_ioctl_set_bit_enable(dev->fd, bit, value) != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_set_bit_enable) on %d failed", ___F, dev_no);
	exit(1);	
    }
}

u_long
pp_gpio_get_alternate(u_char dev_no)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    int err;
    u_long value;
    assert(dev);

    err = gpio_ioctl_rd_ulong(dev->fd, GPIO_IOCTL_GET_ALT, &value);
    if (err != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_rd_ulong_err for GET_ALT) on %d failed", ___F, dev_no);
	exit(1);
    }

    return value;
}

void
pp_gpio_set_alternate(u_char dev_no, u_long set, u_long clr)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    assert(dev);

    if (gpio_ioctl_alt_func(dev->fd, set, clr) != 0) {
	/* this should never happen -> exit */
	pp_log_err("%s(): ioctl(gpio_ioctl_set_alt) on %d failed", ___F, dev_no);
	exit(1);
    }
}

int
pp_gpio_get_fd(u_char dev_no)
{
    gpio_dev_t* dev = gpio_check_device(dev_no);
    assert(dev);

    return dev->fd;
}

