#ifndef __BASE_I2C_H
#define __BASE_I2C_H

#include <sys/ioctl.h>
#include <linux/version.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#define PP_I2C_NO_ERROR	0
#define PP_I2C_ERROR	1

#define PP_I2C_DO_LOG_ERR   1
#define PP_I2C_DONT_LOG_ERR 0

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)

static inline int32_t i2c_smbus_access(int file, char read_write, u_int8_t command, 
                                     int size, union i2c_smbus_data *data)
{
	struct i2c_smbus_ioctl_data args;

	args.read_write = read_write;
	args.command = command;
	args.size = size;
	args.data = data;
	return ioctl(file,I2C_SMBUS,&args);
}


static inline int32_t i2c_smbus_write_quick(int file, u_int8_t value)
{
	return i2c_smbus_access(file,value,0,I2C_SMBUS_QUICK,NULL);
}
	
static inline int32_t i2c_smbus_read_byte(int file)
{
	union i2c_smbus_data data;
	if (i2c_smbus_access(file,I2C_SMBUS_READ,0,I2C_SMBUS_BYTE,&data))
		return -1;
	else
		return 0x0FF & data.byte;
}

static inline int32_t i2c_smbus_write_byte(int file, u_int8_t value)
{
	return i2c_smbus_access(file,I2C_SMBUS_WRITE,value,
	                        I2C_SMBUS_BYTE,NULL);
}

static inline int32_t i2c_smbus_read_byte_data(int file, u_int8_t command)
{
	union i2c_smbus_data data;
	if (i2c_smbus_access(file,I2C_SMBUS_READ,command,
	                     I2C_SMBUS_BYTE_DATA,&data))
		return -1;
	else
		return 0x0FF & data.byte;
}

static inline int32_t i2c_smbus_write_byte_data(int file, u_int8_t command, 
                                              u_int8_t value)
{
	union i2c_smbus_data data;
	data.byte = value;
	return i2c_smbus_access(file,I2C_SMBUS_WRITE,command,
	                        I2C_SMBUS_BYTE_DATA, &data);
}

static inline int32_t i2c_smbus_read_word_data(int file, u_int8_t command)
{
	union i2c_smbus_data data;
	if (i2c_smbus_access(file,I2C_SMBUS_READ,command,
	                     I2C_SMBUS_WORD_DATA,&data))
		return -1;
	else
		return 0x0FFFF & data.word;
}

static inline int32_t i2c_smbus_write_word_data(int file, u_int8_t command, 
                                              u_int16_t value)
{
	union i2c_smbus_data data;
	data.word = value;
	return i2c_smbus_access(file,I2C_SMBUS_WRITE,command,
	                        I2C_SMBUS_WORD_DATA, &data);
}

static inline int32_t i2c_smbus_process_call(int file, u_int8_t command, u_int16_t value)
{
	union i2c_smbus_data data;
	data.word = value;
	if (i2c_smbus_access(file,I2C_SMBUS_WRITE,command,
	                     I2C_SMBUS_PROC_CALL,&data))
		return -1;
	else
		return 0x0FFFF & data.word;
}


/* Returns the number of read bytes */
static inline int32_t i2c_smbus_read_block_data(int file, u_int8_t command, 
                                              u_int8_t *values)
{
	union i2c_smbus_data data;
	int i;
	if (i2c_smbus_access(file,I2C_SMBUS_READ,command,
	                     I2C_SMBUS_BLOCK_DATA,&data))
		return -1;
	else {
		for (i = 1; i <= data.block[0]; i++)
			values[i-1] = data.block[i];
			return data.block[0];
	}
}

static inline int32_t i2c_smbus_write_block_data(int file, u_int8_t command, 
                                               u_int8_t length, u_int8_t *values)
{
	union i2c_smbus_data data;
	int i;
	if (length > 32)
		length = 32;
	for (i = 1; i <= length; i++)
		data.block[i] = values[i-1];
	data.block[0] = length;
	return i2c_smbus_access(file,I2C_SMBUS_WRITE,command,
	                        I2C_SMBUS_BLOCK_DATA, &data);
}

static inline int32_t i2c_smbus_write_i2c_block_data(int file, u_int8_t command,
                                               u_int8_t length, u_int8_t *values)
{
	union i2c_smbus_data data;
	int i;
	if (length > 32)
		length = 32;
	for (i = 1; i <= length; i++)
		data.block[i] = values[i-1];
	data.block[0] = length;
	return i2c_smbus_access(file,I2C_SMBUS_WRITE,command,
	                        I2C_SMBUS_I2C_BLOCK_DATA, &data);
}

#endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0) */

/* ---- synchronized access to /dev/i2c-0 from user space -------- */
/* Note: the '*_core' functions are _not_ synchronized             */

u_char pp_i2c_open(const char* name, int *error);
u_char pp_i2c_open_device(const char* devicename, int *error);
u_char pp_i2c_open_bus(unsigned int number, int *error);
const char* pp_i2c_dev_by_hndl(u_char handle);
void pp_i2c_close(u_char handle);

int pp_i2c_get_fd(u_char handle);
int pp_i2c_set_slave_address(u_char handle, u_char slave);
void pp_i2c_lock(u_char handle);
void pp_i2c_unlock(u_char handle);

u_char pp_i2c_rx_byte(u_char handle, u_char slave, int *error);
u_char pp_i2c_rx_byte_core(u_char handle, int do_log_err, u_char slave,
			   int *error);

void pp_i2c_tx_byte(u_char handle, u_char slave, u_char value, int *error);
void pp_i2c_tx_byte_core(u_char handle, int do_log_err, u_char slave,
			 u_char value, int *error);

u_char pp_i2c_rx_byte_data(u_char handle, u_char slave, u_char reg,
			   int *error);
u_char pp_i2c_rx_byte_data_core(u_char handle, int do_log_err, u_char slave,
				u_char reg, int *error);

void pp_i2c_tx_byte_data(u_char handle, u_char slave, u_char reg,
			 u_char value, int *error);
void pp_i2c_tx_byte_data_core(u_char handle, int do_log_err, u_char slave,
			      u_char reg, u_char value, int *error);

u_short pp_i2c_rx_word_data(u_char handle, u_char slave, u_char reg,
			    int *error);
u_short pp_i2c_rx_word_data_core(u_char handle, int do_log_err, u_char slave,
				u_char reg, int *error);

void pp_i2c_tx_word_data(u_char handle, u_char slave, u_char reg,
			 u_short value, int *error);
void pp_i2c_tx_word_data_core(u_char handle, int do_log_err, u_char slave,
			      u_char reg, u_short value, int *error);

void pp_i2c_tx_burst(u_char handle, u_char slave, const u_char* buf,
		     u_short count, int *error);

int pp_i2c_rx_burst(u_char handle, u_char slave, u_char* buf,
		    u_short buf_size, int *error);
void pp_i2c_tx_burst_core(u_char handle, u_char slave, const u_char* buf,
			  u_short count, int *error);
int pp_i2c_rx_burst_core(u_char handle, u_char slave, u_char* buf,
			 u_short buf_size, int *error);

#endif /* __BASE_I2C_H */
