#ifndef _PP_PROPCHANGE_H
#define _PP_PROPCHANGE_H

#ifdef __cplusplus
extern "C" {
#endif

/* The type of the property change listener structure. */
typedef struct _pp_propchange_listener_t pp_propchange_listener_t;

/* The type of the changed property function. */
typedef void (*changed_prop_t)(pp_propchange_listener_t * listener,
			       unsigned short prop, unsigned short prop_flags);

/*
 * The property change listener structure.
 *
 * ATTENTION: DO NOT INITIALIZE THE MEMBERS DIRECTLY IN THE SOURCE CODE!
 *
 *	      Always use the PP_DECLARE_PROPCHANGE_LISTENER() macro, the
 *	      PP_PROPCHANGE_LISTENER_INITIALIZER initializer or the
 *	      pp_propchange_init_listener() function!
 */
struct _pp_propchange_listener_t {
    /* locks the changed_prop routine (lock/unlock is done automatically) */
    pthread_mutex_t changed_prop_mtx;
    /* property change function (called when an event occurs) */
    changed_prop_t changed_prop;
};

/*
 * Use this macro to initialize a property change listener statically.
 */
#define PP_PROPCHANGE_LISTENER_INITIALIZER(changed_prop_handler)	\
    {									\
	changed_prop_mtx: PTHREAD_MUTEX_INITIALIZER,			\
	changed_prop: changed_prop_handler				\
    }

/*
 * Use this macro to declare and initialize a property change listener
 * statically.
 */
#define PP_DECLARE_PROPCHANGE_LISTENER(name, changed_prop_handler)	\
    pp_propchange_listener_t name __attribute__ ((visibility ("hidden"))) = \
	PP_PROPCHANGE_LISTENER_INITIALIZER(changed_prop_handler)

/* Use this function to initialize a property change listener dynamically. */
extern void pp_propchange_init_listener(pp_propchange_listener_t * listener,
					changed_prop_t changed_prop_handler);

/* Starts the main property change handler thread. */
extern int pp_propchange_start_main_handler(void);

/* Adds a property change listener for a specific property. */
extern void pp_propchange_add(pp_propchange_listener_t * listener, unsigned short prop);

/* Removes a property change listener for a specific property. */
extern void pp_propchange_remove(pp_propchange_listener_t * listener, unsigned short prop);

/* Removes a property change listener for all properties. */
extern void pp_propchange_remove_all(pp_propchange_listener_t * listener);

/* Enqueues a property change signal */
extern int pp_propchange_enqueue(unsigned short prop, unsigned short prop_flags);

#if defined(PP_FEAT_RPCCFG)
extern void pp_propchange_rpc_fire(unsigned short prop, unsigned short prop_flags);
#endif
    
#ifdef __cplusplus
}
#endif

#endif /* _PP_PROPCHANGE_H */
