/*
 * An abstract memory allocator.
 * Used to abstract the place where to allocate memory from
 * i.e. heap, buffer, shared memory, etc
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <stdlib.h>
#include <string.h>
#include <pp/syms.h>
#include <pp/mallocator.h>
#include <pp/win32.h>

static void* heap_malloc(pp_mallocator_t* a UNUSED, size_t s) {
    return malloc(s);
}

static void* heap_realloc(pp_mallocator_t* a UNUSED, void* m, size_t s) {
    return realloc(m, s);
}

static void heap_free(pp_mallocator_t* a UNUSED, void* m) {
    free(m);
}

/* we undefine alloc, reaclloc and free here to avoid problems
   with the dmalloc macros when the accoring build type is used,
   dmalloc is used only for heap debugging anyway */

#ifdef DMALLOC
# warning undefining alloc,realloc,free for dmalloc macros to work

# undef free
# undef alloc
# undef realloc
#endif

void* pp_malloc(pp_mallocator_t* a, size_t s) {
    return a->alloc(a, s);
}

void* pp_realloc(pp_mallocator_t* a, void* m, size_t s) {
    return a->realloc(a, m, s);
}

void pp_free(pp_mallocator_t* a, void* m) {
    a->free(a, m);
}


void  pp_mallocator_destroy(pp_mallocator_t* a) {
    if (a->destroy) {
	a->destroy(a);
    }
}

#ifndef _WIN32
void  pp_mallocator_mutex_init(pp_mallocator_t* a, pp_mutex_t* m,
			       pp_mutex_kind_t k) {
    a->mutex_init(a, m, k);
}

static void heap_mutex_init(pp_mallocator_t*a UNUSED, pp_mutex_t* m,
			    pp_mutex_kind_t k) {
    pp_mutex_init_proclocal(m, k);
}
#endif

static pp_mallocator_t heap_allocator = {
    heap_malloc,
    heap_realloc,
    heap_free,
    NULL,
    NULL,
    NULL,
#ifndef _WIN32
    heap_mutex_init,
#endif
    0
};

pp_mallocator_t* pp_mallocator_heap() {
    return &heap_allocator;
}

char* pp_strdup(pp_mallocator_t* a, const char* s) {
    return strcpy(pp_malloc(a, strlen(s) + 1), s);
}

int pp_mallocator_is_proc_shared(pp_mallocator_t* a) {
    return a->is_proc_shared;
}
