/*
 * mutex abstration
 * that may work process local (a simple pthread mutex)
 * or may work shared among many processe (a System V semaphore)
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/sem.h>
#include <pthread.h>
#include <pp/base.h>
#include <pp/shm.h>
#include <pp/mutex.h>

static struct sembuf sem01_lock_op[4] =   { { sem_num: 0,
					      sem_op: 0,
					      sem_flg: 0 },
					    { sem_num: 0,
					      sem_op: 1,
					      sem_flg: SEM_UNDO },
					    { sem_num: 1,
					      sem_op: 0,
					      sem_flg: 0 },
					    { sem_num: 1,
					      sem_op: 1,
					      sem_flg: SEM_UNDO }
};

static struct sembuf sem0_unlock_op[1] = { { sem_num: 0,
					    sem_op: -1,
					    sem_flg: (IPC_NOWAIT | SEM_UNDO) }
};

static struct sembuf sem1_unlock_op[1] = { { sem_num: 1,
					    sem_op: -1,
					    sem_flg: (IPC_NOWAIT | SEM_UNDO) }
};

#define SEM_0_LOCK_OP         &sem01_lock_op[0]
#define SEM_0_LOCK_OP_SZ      2
#define SEM_1_LOCK_OP         &sem01_lock_op[2]
#define SEM_1_LOCK_OP_SZ      2
#define SEM_01_LOCK_OP        &sem01_lock_op[0]
#define SEM_01_LOCK_OP_SZ     4
#define SEM_0_UNLOCK_OP       sem0_unlock_op
#define SEM_0_UNLOCK_OP_SZ    1
#define SEM_1_UNLOCK_OP       sem1_unlock_op
#define SEM_1_UNLOCK_OP_SZ    1

typedef struct {
    void (*lock)(pp_mutex_t* m);
    void (*unlock)(pp_mutex_t* m);
    void (*destroy)(pp_mutex_t* m);
} mutex_fkt_table_t;

static void mtx_lock(pp_mutex_t* m);
static void mtx_unlock(pp_mutex_t* m);
static void mtx_destroy(pp_mutex_t* m);
static void sem_lock(pp_mutex_t* m);
static void sem_unlock(pp_mutex_t* m);
static void sem_destroy(pp_mutex_t* m);
static void sem_lock_recursive(pp_mutex_t* m);
static void sem_unlock_recursive(pp_mutex_t* m);

static mutex_fkt_table_t mutex_fkt_table[3] = {
    {   /* 0: PP_MUTEX_PROC_LOCAL */
	mtx_lock,
	mtx_unlock,
	mtx_destroy
    },
    {   /* 1: PP_MUTEX_PROC_SHARED */
	sem_lock,
	sem_unlock,
	sem_destroy,
    },
    {   /* 2: PP_MUTEX_PROC_SHARED_RECURSIVE */
	sem_lock_recursive,
	sem_unlock_recursive,
	sem_destroy
    }
};
	
inline void pp_mutex_lock(pp_mutex_t* m) {
    mutex_fkt_table[m->type].lock(m);
}

inline void pp_mutex_unlock(pp_mutex_t* m) {
    mutex_fkt_table[m->type].unlock(m);
}

inline void pp_mutex_destroy(pp_mutex_t* m) {
    mutex_fkt_table[m->type].destroy(m);
}

static void mtx_lock(pp_mutex_t* m) {
    MUTEX_LOCK(&m->mvar.mtx);
}

static void mtx_unlock(pp_mutex_t* m) {
    MUTEX_UNLOCK(&m->mvar.mtx);
}

static void mtx_destroy(pp_mutex_t* m) {
    int r = pthread_mutex_destroy(&m->mvar.mtx);
    PP_ASSERT(r, r == 0);
}

void pp_mutex_init_proclocal(pp_mutex_t* mutex, pp_mutex_kind_t kind) {
    mutex->type = PP_MUTEX_PROC_LOCAL;
    if (kind == PP_MUTEX_RECURSIVE) {
	MUTEX_CREATE_RECURSIVE(&mutex->mvar.mtx);
    } else {
	MUTEX_CREATE_ERRORCHECKING(&mutex->mvar.mtx);
    }
}

static void sem_op(int semid, struct sembuf* ops, int nops) {
    while(0 > semop(semid, ops, nops)) {
	if (errno != EINTR) {
	    pp_log_err("%s(): sem_lock failed", ___F);
	    abort();
	}
    }
}    

int pp_sema_init(key_t key) {
    int id;
    
    if ((id = semget(key, 1, IPC_CREAT | 0666)) == -1) {
	pp_log_err("%s(): semget()", ___F);
	abort();
    }
    
    return id;
}

void pp_sema_lock(int semid) {
    sem_op(semid, SEM_0_LOCK_OP, SEM_0_LOCK_OP_SZ);
}

void pp_sema_unlock(int semid) {
    sem_op(semid, SEM_0_UNLOCK_OP, SEM_0_UNLOCK_OP_SZ);
}

void pp_sema_destroy(int semid) {
    int r = semctl(semid, 0, IPC_RMID);
    if (r != 0) pp_log_err("pp_sema_destroy(%d)", semid);
}

static void sem_lock(pp_mutex_t* m) {
    pp_sema_lock(m->mvar.sem.semid);
}

static void sem_unlock(pp_mutex_t* m) {
    pp_sema_unlock(m->mvar.sem.semid);
}

static void sem_destroy(pp_mutex_t* m) {
    int semid = m->mvar.sem.semid;
    pp_shm_remove_sem(pp_shm_get(m->mvar.sem.shmkey), semid);
    pp_sema_destroy(m->mvar.sem.semid);

}

static void sem_init_normal(pp_mutex_t* m, key_t shmkey) {
    int id;

    if ((id = semget(IPC_PRIVATE, 1, 0666)) == -1) {
	pp_log_err("%s(): semget(IPC_PRIVATE)", ___F);
	abort();
    }
    
    // init the mutex structure and inform the shared memory
    m->type = PP_MUTEX_PROC_SHARED;
    m->mvar.sem.shmkey = shmkey;
    m->mvar.sem.semid = id;
    pp_shm_add_sem(pp_shm_get(shmkey), id);
}

static void sem_lock_recursive(pp_mutex_t* m) {
    pp_sem_recursive_t* sem = &m->mvar.sem_recsv;
    int semid = sem->semid;
    pid_t pid = getpid();
    pthread_t tid = pthread_self();

    // lock sem structure 
    sem_op(semid, SEM_0_LOCK_OP, SEM_0_LOCK_OP_SZ);
    
    if (sem->pid == pid && sem->tid == tid) { // I own it, so just count up
	sem->count++;
    } else { // I don't own it, so try to get it
	if (sem->pid == 0) { // it's available, so aquire
	    sem_op(semid, SEM_1_LOCK_OP, SEM_1_LOCK_OP_SZ);
	} else do {
	    // wait for signal and lock sem 0 atomically
	    sem_op(semid, SEM_0_UNLOCK_OP, SEM_0_UNLOCK_OP_SZ);
	    sem_op(semid, SEM_01_LOCK_OP, SEM_01_LOCK_OP_SZ);
	} while (sem->pid != 0); // loop as long as it is taken by others

	// ones we come here, we acquired it, so just mark it as such
	sem->pid = pid;
	sem->tid = tid;
	sem->count = 1;
    }

    sem_op(semid, SEM_0_UNLOCK_OP, SEM_0_UNLOCK_OP_SZ);
}

static void sem_unlock_recursive(pp_mutex_t* m) {
    pp_sem_recursive_t* sem = &m->mvar.sem_recsv;
    int semid = sem->semid;
    pid_t pid = getpid();
    pthread_t tid = pthread_self();

    // lock sem structure 
    sem_op(semid, SEM_0_LOCK_OP, SEM_0_LOCK_OP_SZ);
    if (sem->pid != pid || sem->tid != tid || sem->count < 1) {
	pp_log("sem_unlock_recursive: ERROR: pid:%d=%d tid:%lu=%lu c=%d\n",
	       sem->pid, pid, sem->tid, tid, sem->count);
	abort();
    }

    if (--sem->count == 0) {
	// signal a waiting party
	sem_op(semid, SEM_1_UNLOCK_OP, SEM_1_UNLOCK_OP_SZ);
	sem->pid = 0;
	sem->tid = 0;
    }
    
    sem_op(semid, SEM_0_UNLOCK_OP, SEM_0_UNLOCK_OP_SZ);
}

static void sem_init_recursive(pp_mutex_t* m, key_t shmkey) {
    int id;

    if ((id = semget(IPC_PRIVATE, 2, 0666)) == -1) {
	pp_log_err("%s(): semget(IPC_PRIVATE)", ___F);
	abort();
    }
    
    // init the mutex structure and inform the shared memory
    m->type = PP_MUTEX_PROC_SHARED_RECURSIVE;
    m->mvar.sem_recsv.shmkey = shmkey;
    m->mvar.sem_recsv.semid = id;
    m->mvar.sem_recsv.pid = 0;
    m->mvar.sem_recsv.tid = 0;
    m->mvar.sem_recsv.count = 0;
    pp_shm_add_sem(pp_shm_get(shmkey), id);
}

void pp_mutex_init_procshared(pp_mutex_t* mutex, key_t shmkey,
			      pp_mutex_kind_t kind) {
    if (kind == PP_MUTEX_RECURSIVE) {
	sem_init_recursive(mutex, shmkey);
    } else {
	sem_init_normal(mutex, shmkey);
    }
}
