/**
 * Copyright 2001 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: distributes property change events to
 *              keyboard drivers
 */

#if defined(PP_FEAT_DEVICE) && defined(PP_FEAT_PROPCHANGE)

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <fcntl.h>
#include <semaphore.h>
#include <unistd.h>
#include <malloc.h>
#include <assert.h>
#include <pthread.h>
#include <pp/base.h>
#include <liberic_pthread.h>
#include <lara.h>

#if defined(PP_FEAT_RPCCFG)
#include <pp/rpc_ripc.h>
#endif

#define PROP_RBUF_ENTRIES 32

/* how to remove entries */
typedef enum {
    PROPCHANGE_REMOVE_ACTION_DEFAULT,	/* remove immediately if possible */
    PROPCHANGE_REMOVE_ACTION_IMMEDIATE,	/* remove immediately */
    PROPCHANGE_REMOVE_ACTION_DELAYED	/* remove later */
} propchange_remove_action_t;

/**
 * used to keep all listeners in a list
 */
typedef struct {
    struct list_head listnode;
    unsigned short prop;
    unsigned short prop_flags;
    pthread_t propchange_thread;
    pthread_mutex_t cond_mtx;
    pthread_cond_t cond;
    int thread_started;
    int thread_count;
    propchange_remove_action_t remove_action;
    pp_propchange_listener_t * listener;
} listener_node_t;

int propchange_initialized = 0;

static volatile u_int propchange_trigger_map[PP_PROP_COUNT]; /* each index represents a property to send (if value is not 0) */
static sem_t propchange_sem;
static pthread_t propchange_handler_thread;
static int propchange_handler_thread_started = 0;
static int die_propchange_handler = 0;
static pp_hash_i_t * propchange_hash = NULL;
static pthread_mutex_t propchange_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_key_t propchange_tid_key;
static pthread_once_t propchange_tid_key_once = PTHREAD_ONCE_INIT;

int propchange_init(void);
void propchange_cleanup(void);
static void propchange_listener_node_free(listener_node_t * l_node);
static void propchange_list_free(void * _list);
static listener_node_t * propchange_unlink(pp_propchange_listener_t * listener, struct list_head * list);
static void propchange_fire(unsigned short prop, unsigned short prop_flags);
static void handle_sigpropchg(int signo, siginfo_t *info, void * context);
static void * propchange_handler(void * arg);
static void * propchange_thread_func(void * arg);
static void propchange_tid_alloc(void);
static pthread_t * propchange_tid_get(void);

/* ------------------------------------------------------------------------- *
 * Init / Cleanup
 * ------------------------------------------------------------------------- */

/*
 * Initialize the propchange submodule.
 */
int
propchange_init(void)
{
    const char * fn = ___F;
    struct sigaction action;

    if (!propchange_initialized) {
	
	/* create the propchange hash */
	if ((propchange_hash = pp_hash_create_i(20)) == NULL) {
	    pp_log_err("%s(): pp_hash_create_i()", fn);
	    return -1;
	}

	/* initialize propchange_trigger_map */
	memset(&propchange_trigger_map, 0, sizeof(propchange_trigger_map));

	/* init propchange semaphore */
	sem_init(&propchange_sem, 0, 0);

	/* register for signal */
	sigemptyset(&action.sa_mask);
	action.sa_flags = SA_SIGINFO;
	action.sa_sigaction = handle_sigpropchg;
	if (sigaction(SIGPROPCHG, &action, NULL) < 0) {
	    pp_log_err("%s(): sigaction(SIGPROPCHG) failed", fn);
	    propchange_cleanup();
	    return -1;
	}

	/* switch on signals for this process in kernel module */
	if (ioctl(eric_fd, ERICIOCPROPCHANGESIGNAL, 1) < 0) {
	    pp_log_err("%s(): ioctl(ERICIOCPROPCHANGESIGNAL) failed", fn);
	    propchange_cleanup();
	    return -1;
	}

	propchange_initialized = 1;
    }
    return 0;
}

/*
 * Cleanup the propchange submodule.
 */
void
propchange_cleanup(void)
{
    const char * fn = ___F;
    pp_hash_i_t * _propchange_hash = NULL;
    struct sigaction action;

    /* switch off signals for this process in kernel module */
    if (ioctl(eric_fd, ERICIOCPROPCHANGESIGNAL, 0) < 0) {
	pp_log_err("%s(): ioctl(ERICIOCPROPCHANGESIGNAL) failed", fn);
    }

    /* unregister from signal */
    sigemptyset(&action.sa_mask);
    action.sa_flags = 0;
    action.sa_handler = SIG_IGN;
    if (sigaction(SIGPROPCHG, &action, NULL) < 0) {
	pp_log_err("%s(): sigaction(SIGPROPCHG) failed", fn);
    }

    /* soft-kill the main propchange handler thread */
    if (propchange_handler_thread_started) {
	die_propchange_handler = 1;
	sem_post(&propchange_sem);
	pthread_join(propchange_handler_thread, NULL);
	propchange_handler_thread_started = 0;
    }

    /*
     * Replace the property change hash by NULL and free the old one without
     * holding the lock. This prevents some deadlock situations.
     */
    MUTEX_LOCK(&propchange_mtx);
    if (propchange_hash) {
	_propchange_hash = propchange_hash;
	propchange_hash = NULL;
    }
    MUTEX_UNLOCK(&propchange_mtx);
    if (_propchange_hash) pp_hash_delete_i(_propchange_hash);

    propchange_initialized = 0;
}

/* ------------------------------------------------------------------------- *
 * API routines (for documentation see propchange.h)
 * ------------------------------------------------------------------------- */

void
pp_propchange_init_listener(pp_propchange_listener_t * listener,
			    changed_prop_t changed_prop_handler) {
    pthread_mutex_init(&listener->changed_prop_mtx, NULL);
    listener->changed_prop = changed_prop_handler;
}

int
pp_propchange_start_main_handler(void)
{
    assert(propchange_initialized);
    /* start the propchange handler thread */
    if (eric_pthread_create(&propchange_handler_thread, 0, 65536,
			    propchange_handler, NULL) != 0) {
	pp_log("%s(): Cannot create property change handler thread.\n", ___F);
	return -1;
    }
    propchange_handler_thread_started = 1;
    return 0;
}

void
pp_propchange_add(pp_propchange_listener_t * listener, unsigned short prop)
{
    struct list_head * list;
    listener_node_t* l_node;
    /* create and initialize the listener */
    l_node = malloc(sizeof(listener_node_t));
    l_node->listener = listener;
    l_node->thread_started = 0;
    l_node->thread_count = 0;
    l_node->remove_action = PROPCHANGE_REMOVE_ACTION_DEFAULT;
    pthread_mutex_init(&l_node->cond_mtx, NULL);
    pthread_cond_init(&l_node->cond, NULL);
    MUTEX_LOCK(&propchange_mtx);
    /*
     * propchange_hash may be NULL during cleanup!
     */
    if (propchange_hash) {
	/* look for a list where the listener should be added */
	if ((list = pp_hash_get_entry_i(propchange_hash, prop)) == NULL) {
	    /* no list found for the property -> create one */
	    list = (struct list_head *)malloc(sizeof(struct list_head));
	    INIT_LIST_HEAD(list);
	    pp_hash_set_entry_i(propchange_hash, prop, list, propchange_list_free);
	}
	/* add the listener */
	list_add(&l_node->listnode, list);
    }
    MUTEX_UNLOCK(&propchange_mtx);
}

void
pp_propchange_remove(pp_propchange_listener_t * listener, unsigned short prop)
{
    struct list_head * list;
    listener_node_t * l_node = NULL;
    
    MUTEX_LOCK(&propchange_mtx);
    /*
     * propchange_hash may be NULL during cleanup!
     */
    if (propchange_hash	&& (list = pp_hash_get_entry_i(propchange_hash, prop)) != NULL) {
	l_node = propchange_unlink(listener, list);
    }
    MUTEX_UNLOCK(&propchange_mtx);
    propchange_listener_node_free(l_node);
}

void
pp_propchange_remove_all(pp_propchange_listener_t * listener)
{
    struct list_head *list, *tmp, *tmp2;
    struct list_head del_list;
    listener_node_t *l_node = NULL;

    INIT_LIST_HEAD(&del_list);
    MUTEX_LOCK(&propchange_mtx);
    /*
     * propchange_hash may be NULL during cleanup!
     */
    if (propchange_hash) {
	/*
	 * For each list in propchange hash unlink the listener and add it to a
	 * separate delete list.
	 */
	for (list = (struct list_head *)pp_hash_get_first_entry_i(propchange_hash);
	     list != NULL;
	     list = (struct list_head *)pp_hash_get_next_entry_i(propchange_hash)) {
	    if ((l_node = propchange_unlink(listener, list)) != NULL) {
		list_add(&l_node->listnode, &del_list);
	    }
	}
    }
    MUTEX_UNLOCK(&propchange_mtx);

    /* for each listener on the delete list call its free function */
    list_for_each_safe(tmp, tmp2, &del_list) {
	l_node = list_entry(tmp, listener_node_t, listnode);
	list_del_init(&l_node->listnode);
	propchange_listener_node_free(l_node);
    }
}

int
pp_propchange_enqueue(unsigned short prop, unsigned short prop_flags)
{
    union sigval sv;
    static int eric_pid = 0;

    if (eric_pid <= 0) {
	if ((eric_pid = pp_base_get_pid_of_program_by_pidfile(ERIC_PIDFILE, ERIC_PROGRAM_PATH)) <= 0) {
	    pp_log("Cannot open %s - maybe the 'eric' process is not running.\n", ERIC_PIDFILE);
	}
    }

    if (eric_pid > 0) {
	sv.sival_int = (prop << 16) | (prop_flags & 0xffff);
	if (sigqueue(eric_pid, SIGPROPCHG, sv) == -1) {
	    pp_log_err("%s(): sigqueue()", ___F);
	    return -1;
	}
	return 0;
    }

    return -1;
}

/* ------------------------------------------------------------------------- *
 * Internal routines
 * ------------------------------------------------------------------------- */

/*
 * Free a listener node.
 *
 * NOTE: This routine may block some time!
 */
static void
propchange_listener_node_free(listener_node_t * l_node)
{
    if (l_node) {
	/*
	 * If one or more handler threads are running, wait until all have
	 * finished.
	 */
	MUTEX_LOCK(&l_node->cond_mtx);
	while (l_node->thread_count > 0) {
	    /* Check if we are running from our own propchange handler */
	    pthread_t * tid = propchange_tid_get();
	    if (tid
		&& *tid == pthread_self()
		&& l_node->remove_action != PROPCHANGE_REMOVE_ACTION_IMMEDIATE) {
		/*
		 * We are running from our own propchange handler. To prevent a
		 * deadlock we mark the node for later removal.
		 */
		l_node->remove_action = PROPCHANGE_REMOVE_ACTION_DELAYED;
		break;
	    } else {
#define HANDLER_FINISH_TIMEOUT 10
		struct timeval now;
		struct timespec timeout;
		gettimeofday(&now, NULL);
		timeout.tv_sec = now.tv_sec + HANDLER_FINISH_TIMEOUT;
		timeout.tv_nsec = now.tv_usec * 1000;
		if (pthread_cond_timedwait(&l_node->cond, &l_node->cond_mtx,
					   &timeout) == ETIMEDOUT) {
		    pp_log("%s(): Handler still busy after %d seconds of waiting. Aborting.",
			   ___F, HANDLER_FINISH_TIMEOUT);
		    abort();
		}
#undef HANDLER_FINISH_TIMEOUT
	    }
	}
	MUTEX_UNLOCK(&l_node->cond_mtx);
	if (l_node->remove_action != PROPCHANGE_REMOVE_ACTION_DELAYED) {
	    pthread_cond_destroy(&l_node->cond);
	    pthread_mutex_destroy(&l_node->cond_mtx);
	    free(l_node);
	}
    }
}

/*
 * Free a list with propchange listeners.
 *
 * NOTE: This routine may block some time!
 */
static void
propchange_list_free(void * _list)
{
    if (_list) {
	struct list_head * list = (struct list_head *)_list;
	struct list_head *tmp, *tmp2;
	list_for_each_safe(tmp, tmp2, list) {
	    listener_node_t * l_node = list_entry(tmp, listener_node_t, listnode);
	    list_del_init(&l_node->listnode);
	    propchange_listener_node_free(l_node);
	}
	free(list);
    }
}

/*
 * Unlink a specific propchange listener from a list.
 */
static listener_node_t *
propchange_unlink(pp_propchange_listener_t * listener, struct list_head * list)
{
    if (list) {
	struct list_head *tmp, *tmp2;
	list_for_each_safe(tmp, tmp2, list) {
	    listener_node_t * l_node = list_entry(tmp, listener_node_t, listnode);
	    if (l_node->listener == listener) {
		list_del_init(&l_node->listnode);
		return l_node;
	    }
	}
    }
    return NULL;
}

/*
 * Send (fire) a specific propchange signal with an optional flags parameter (bitfield!).
 */
#if defined(PP_FEAT_RPCCFG)
void
pp_propchange_rpc_fire(unsigned short prop, unsigned short prop_flags)
{
    propchange_fire(prop, prop_flags);
}
#endif /* PP_FEAT_RPCCFG */

static void
propchange_fire(unsigned short prop, unsigned short prop_flags)
{
    struct list_head * list;
    int fire_cnt = 0;

#if defined(PP_FEAT_RPCCFG)
    if (prop >= PP_PROP_COUNT) {
	prop -= PP_PROP_COUNT;
    } else if (prop == PP_PROP_CONFIGFS_CHANGED
	       || prop == PP_PROP_USB_DEV_STATE_CHANGED
	       || prop == PP_PROP_KVM_PORT_SWITCHED) {
	/* send property to master/slave */
	pp_rpc_propchange_fire(prop, prop_flags, pp_rpc_am_i_master() ? PP_RPC_TARGET_SLAVE : PP_RPC_TARGET_MASTER);
    }
#endif /* PP_FEAT_RPCCFG */

    MUTEX_LOCK(&propchange_mtx);
    /*
     * propchange_hash may be NULL during cleanup!
     */
    if (propchange_hash	&& (list = pp_hash_get_entry_i(propchange_hash, prop)) != NULL) {
	struct list_head *tmp;
	list_for_each(tmp, list) {
	    listener_node_t * l_node = list_entry(tmp, listener_node_t, listnode);
	    MUTEX_LOCK(&l_node->cond_mtx);
	    l_node->prop = prop;
	    l_node->prop_flags = prop_flags;
	    l_node->thread_started = 0;
	    if (eric_pthread_create(&l_node->propchange_thread, 1, 65536,
				    propchange_thread_func, (void*)l_node)) {
		pp_log("%s(): Thread creation failed.", ___F);
		abort();
	    }
	    ++l_node->thread_count;
	    while (!l_node->thread_started) {
#define HANDLER_START_TIMEOUT 5
		struct timeval now;
		struct timespec timeout;
		gettimeofday(&now, NULL);
		timeout.tv_sec = now.tv_sec + HANDLER_START_TIMEOUT;
		timeout.tv_nsec = now.tv_usec * 1000;
		if (pthread_cond_timedwait(&l_node->cond, &l_node->cond_mtx,
				       &timeout) == ETIMEDOUT) {
		    pp_log("%s(): Handler still not running after %d seconds of waiting. Aborting.",
			   ___F, HANDLER_START_TIMEOUT);
		    abort();
		}
#undef HANDLER_START_TIMEOUT
	    }
	    MUTEX_UNLOCK(&l_node->cond_mtx);
	    ++fire_cnt;
	}
    }
    MUTEX_UNLOCK(&propchange_mtx);

    if (fire_cnt == 0) {
	/*
	 * Keep this message here! It is useful to find code that is compiled
	 * under wrong conditions. Ideally this message should not occur.
	 */
	pp_log("%s(): There are no listeners registered for property %hu.\n", ___F, prop);
    }
}

static void
handle_sigpropchg(int signo UNUSED, siginfo_t *info, void * context UNUSED)
{
    u_short prop = info->si_int >> 16;
    u_short prop_flags = info->si_int & 0xFFFF;
    if (prop > 0 && prop < PP_PROP_COUNT) {
	/* accumulate flags (this is only safe on uniprocessor systems!) */
	/* bit 16 indicates that a property was received */
	propchange_trigger_map[prop] |= 0x10000 | prop_flags;
	sem_post(&propchange_sem);
    } else {
	pp_log("%s(): invalid property %hu received\n", ___F, prop);
    }
}

static void *
propchange_handler(void * arg UNUSED)
{
    u_int prop_map_entry;
    u_short prop;

    while (!die_propchange_handler) {
	sigset_t mysigset;
	sigemptyset(&mysigset);
	sigaddset(&mysigset, SIGPROPCHG);

	sem_wait(&propchange_sem);

	for (prop = 1; !die_propchange_handler && prop < PP_PROP_COUNT; ++prop) {
	    sigprocmask(SIG_BLOCK, &mysigset, NULL);
	    prop_map_entry = propchange_trigger_map[prop];
	    propchange_trigger_map[prop] = 0;
	    sigprocmask(SIG_UNBLOCK, &mysigset, NULL);
	    if (prop_map_entry & 0x10000) propchange_fire(prop, prop_map_entry & 0xFFFF);
	}
    }
    return NULL;
}

static void *
propchange_thread_func(void * arg)
{
    listener_node_t * l_node = (listener_node_t *)arg;
    pp_propchange_listener_t * listener = l_node->listener;
    unsigned short prop = l_node->prop;
    unsigned short prop_flags = l_node->prop_flags;
    pthread_t * tid;

    /*
     * Tell the caller that we are running and copied all necessary data by
     * setting thread_started to 1 and signaling him. The thread cound is
     * increased too.
     */
    MUTEX_LOCK(&l_node->cond_mtx);
    l_node->thread_started = 1;
    /* only one thread is waiting -> use pthread_cond_signal */
    pthread_cond_signal(&l_node->cond);
    MUTEX_UNLOCK(&l_node->cond_mtx);

    /* Mark us as running by a propchange handler by deposing our thread id */
    propchange_tid_alloc();
    if ((tid = propchange_tid_get()) != NULL) {
	*tid = pthread_self();
    } else {
	pp_log("%s(): propchange_tid_get() failed. Aborting.", ___F);
	abort();
    }

    /* call the real handler */
    MUTEX_LOCK(&l_node->listener->changed_prop_mtx);
    l_node->listener->changed_prop(listener, prop, prop_flags);
    MUTEX_UNLOCK(&l_node->listener->changed_prop_mtx);

    /*
     * Tell the caller that we are dying soon by decreasing the thread count and
     * signaling him.
     */
    MUTEX_LOCK(&l_node->cond_mtx);
    --l_node->thread_count;
    /* more than on thread may be waiting -> use pthread_cond_broadcast */
    pthread_cond_broadcast(&l_node->cond);
    MUTEX_UNLOCK(&l_node->cond_mtx);

    /* If a delayed remove action is requested remove the listener node now. */
    if (l_node->remove_action == PROPCHANGE_REMOVE_ACTION_DELAYED) {
	l_node->remove_action = PROPCHANGE_REMOVE_ACTION_IMMEDIATE;
	propchange_listener_node_free(l_node);
    }

    return NULL;
}

/* ------------------------------------------------------------------------- *
 * Thread-specific data routines
 *
 * NOTES: We use thread specific data to determine if propchange_remove*() 
 *        was called by a listener's own propchange function. In this case
 *	  we need a special handling to prevent a deadlock.
 * ------------------------------------------------------------------------- */


/* destroy the tid storage */
static void
_tid_destroy(void * propchange_tid)
{
    free(propchange_tid);
}

/* allocate the tid key */
static void
_tid_key_alloc(void)
{
    pthread_key_create(&propchange_tid_key, _tid_destroy);
}

/* allocate the tid storage */
static void
propchange_tid_alloc(void)
{
    pthread_once(&propchange_tid_key_once, _tid_key_alloc);
    pthread_setspecific(propchange_tid_key, malloc(sizeof(pthread_t)));
}

/* return a pointer to the tid storage */
static pthread_t *
propchange_tid_get(void)
{
    return (pthread_t *)pthread_getspecific(propchange_tid_key);
}

#endif /* PP_FEAT_DEVICE && PP_FEAT_PROPCHANGE */
