/*
 * rb_tree.c
 *
 * Implementation of red-black binary search tree.
 * Copyright (C) 2001 Farooq Mela.
 *
 * cf. [Cormen, Leiserson, and Rivest 1990], [Guibas and Sedgewick, 1978]
 * 
 * modified for Peppercon 2004, tbr@peppercon.de
 * for original code see: http://home.earthlink.net/~smela1/libdict.html
 *
 * Deletion of nodes might depend on function pointers which can't be used
 * in shared memory. Currently we simply assert in case somebody is trying
 * to do that, i.e. to free a node with a custom function in shared memory.
 * Later we may add a (yet unknown) magic to make that possible.
 */

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <pp/base.h>
#include <pp/rb_tree.h>

typedef struct rb_node_s rb_node;
struct rb_node_s {
    union {
	const void *key_const;
	void *key;
    };
    void		*dat;
    pp_rb_tree_del_func dat_del;
    rb_node	        *parent;
    rb_node	        *llink;
    rb_node	        *rlink;
    u_int		color:1;
};

#define RB_RED			0
#define RB_BLK			1

struct pp_rb_tree_s {
    rb_node		*root;
    rb_node              null;
    u_int		 count;
    pp_rb_tree_cmp_func	 key_cmp;
    pp_rb_tree_dup_func  key_dup;
    pp_rb_tree_del_func	 key_del;
    pp_mallocator_t     *allocator;
};

struct pp_rb_iter_s {
    const pp_rb_tree_t *tree;
    const rb_node *node;
};

static void rot_left (pp_rb_tree_t *tree, rb_node *node);
static void rot_right (pp_rb_tree_t *tree, rb_node *node);
static void insert_fixup (pp_rb_tree_t *tree, rb_node *node);
static void delete_fixup (pp_rb_tree_t *tree, rb_node *node);
static u_int node_height (const pp_rb_tree_t* tree, const rb_node *node);
static u_int node_mheight (const pp_rb_tree_t* tree, const rb_node *node);
static u_int node_pathlen (const pp_rb_tree_t* tree, const rb_node *node,
			   u_int level);
static rb_node *node_new (pp_rb_tree_t *tree, const void *key, void *dat, 
			  pp_rb_tree_del_func dat_del);
static const rb_node *node_next (const pp_rb_tree_t* tree, const rb_node *node);
static const rb_node *node_prev (const pp_rb_tree_t* tree, const rb_node *node);
static rb_node *node_max (const pp_rb_tree_t* tree, rb_node *node);
static rb_node *node_min (const pp_rb_tree_t* tree, rb_node *node);
static void node_content_swap(rb_node *n1, rb_node *n2);

#define RB_NULL_INITIALIZER { { NULL }, NULL, NULL, NULL, NULL, NULL, RB_BLK }
#define RB_NULL &tree->null

pp_rb_tree_t*
pp_rb_tree_new (pp_rb_tree_cmp_func key_cmp,
		pp_rb_tree_dup_func key_dup,
		pp_rb_tree_del_func key_del) {
    return pp_rb_tree_new_with_alloc(key_cmp, key_dup, key_del,
				     pp_mallocator_heap());
}

/* most often used case has a shortcat */
pp_rb_tree_t*
pp_rb_tree_str_new() {
    return pp_rb_tree_str_new_with_alloc(pp_mallocator_heap());
}

pp_rb_tree_t *pp_rb_tree_new_with_alloc (pp_rb_tree_cmp_func key_cmp,
					 pp_rb_tree_dup_func key_dup,
					 pp_rb_tree_del_func key_del,
					 pp_mallocator_t* a) {
    pp_rb_tree_t *tree;
    rb_node nn = RB_NULL_INITIALIZER;

    tree = pp_malloc(a, sizeof(*tree));
    tree->null      = nn;
    tree->root      = RB_NULL;
    tree->count     = 0;
    tree->key_cmp   = key_cmp;
    tree->key_dup   = key_dup;
    tree->key_del   = key_del;
    tree->allocator = a;
    return tree;
}

pp_rb_tree_t *pp_rb_tree_str_new_with_alloc(pp_mallocator_t* a) {
    return pp_rb_tree_new_with_alloc((pp_rb_tree_cmp_func)strcmp,
			  (pp_rb_tree_dup_func)pp_strdup,
			  pp_free, a);
}

/* creation shortcut for int, also used pretty often */
static int int_cmp(const void *a, const void *b) { return a - b; }

pp_rb_tree_t *pp_rb_tree_int_new() {
    return pp_rb_tree_new_with_alloc(int_cmp, NULL,NULL, pp_mallocator_heap());
}


void
pp_rb_tree_destroy (pp_rb_tree_t *tree)
{
    if (tree != NULL) {
	if (tree->root != RB_NULL) pp_rb_tree_empty(tree);
	pp_free(tree->allocator, tree);
    }
}

void*
pp_rb_tree_search (const pp_rb_tree_t *tree, const void *key)
{
    int rv;
    rb_node *node;

    assert(tree != NULL);

    node = tree->root;
    while (node != RB_NULL) {
	rv = tree->key_cmp ? tree->key_cmp(key, node->key_const)
	   : key < node->key_const ? -1 : key > node->key_const ? 1 : 0;
	if (rv == 0) return node->dat;
	node = rv < 0 ? node->llink : node->rlink;
    }

    return NULL;
}

int pp_rb_tree_insert (pp_rb_tree_t *tree, const void *key, void *dat,
		       pp_rb_tree_del_func dat_del, int overwrite)
{
    int rv = 0;
    rb_node *node, *parent = RB_NULL;

    assert(tree != NULL);

    node = tree->root;

    while (node != RB_NULL) {
	rv = tree->key_cmp ? tree->key_cmp(key, node->key_const)
	   : key < node->key_const ? -1 : key > node->key_const ? 1 : 0;
	if (rv == 0) {
	    if (overwrite == 0) return 1;
	    if (tree->key_del) {
		assert(!pp_mallocator_is_proc_shared(tree->allocator));
		tree->key_del(tree->allocator, node->key);
	    }
	    if (node->dat_del) {
		assert(!pp_mallocator_is_proc_shared(tree->allocator));
		node->dat_del(tree->allocator, node->dat);
	    }
	    if (tree->key_dup) {
		node->key = tree->key_dup(tree->allocator, key);
	    } else {
		node->key_const = key;
	    }
	    node->dat = dat;
	    node->dat_del = dat_del;
	    return 0;
	}

	parent = node;
	node = rv < 0 ? node->llink : node->rlink;
    }

    node = node_new(tree, key, dat, dat_del);

    if (parent != RB_NULL) {
	node->parent = parent;
	if (rv < 0) {
	    parent->llink = node;
	} else {
	    parent->rlink = node;
	}
	insert_fixup(tree, node);
    } else {
	tree->root = node;
	node->color = RB_BLK;
    }
    ++tree->count;
    return 0;
}

static void
insert_fixup(pp_rb_tree_t* tree, rb_node* node)
{
    rb_node *temp;

    assert(tree != NULL);
    assert(node != NULL);

    while (node != tree->root && node->parent->color == RB_RED) {
	if (node->parent == node->parent->parent->llink) {
	    temp = node->parent->parent->rlink;
	    if (temp->color == RB_RED) {
		temp->color = RB_BLK;
		node = node->parent;
		node->color = RB_BLK;
		node = node->parent;
		node->color = RB_RED;
	    } else {
		if (node == node->parent->rlink) {
		    node = node->parent;
		    rot_left(tree, node);
		}
		temp = node->parent;
		temp->color = RB_BLK;
		temp = temp->parent;
		temp->color = RB_RED;
		rot_right(tree, temp);
	    }
	} else {
	    temp = node->parent->parent->llink;
	    if (temp->color == RB_RED) {
		temp->color = RB_BLK;
		node = node->parent;
		node->color = RB_BLK;
		node = node->parent;
		node->color = RB_RED;
	    } else {
		if (node == node->parent->llink) {
		    node = node->parent;
		    rot_right(tree, node);
		}
		temp = node->parent;
		temp->color = RB_BLK;
		temp = temp->parent;
		temp->color = RB_RED;
		rot_left(tree, temp);
	    }
	}
    }

    tree->root->color = RB_BLK;
}

int 
pp_rb_tree_remove (pp_rb_tree_t *tree, const void *key, void** dat)
{
    int rv;
    rb_node *node, *temp, *out, *parent;

    assert(tree != NULL);

    node = tree->root;
    while (node != RB_NULL) {
	rv = tree->key_cmp ? tree->key_cmp(key, node->key_const)
	   : key < node->key_const ? -1 : key > node->key_const ? 1 : 0;
	if (rv == 0) break;
	node = rv < 0 ? node->llink : node->rlink;
    }
    if (node == RB_NULL) return -1;

    if (node->llink == RB_NULL || node->rlink == RB_NULL) {
	out = node;
    } else {
	for (out = node->rlink; out->llink != RB_NULL; out = out->llink) {
	    /* void */
	}
	node_content_swap(node, out);
    }

    temp = out->llink != RB_NULL ? out->llink : out->rlink;
    parent = out->parent;
    temp->parent = parent;
    if (parent != RB_NULL) {
	if (parent->llink == out) {
	    parent->llink = temp;
	} else {
	    parent->rlink = temp;
	}
    } else {
	tree->root = temp;
    }
	
    if (out->color == RB_BLK) delete_fixup(tree, temp);
    
    if (tree->key_del) {
	assert(!pp_mallocator_is_proc_shared(tree->allocator));
	tree->key_del(tree->allocator, out->key);
    }
    if (dat) {
	*dat = out->dat;
    } else {
	if (out->dat_del) {
	    assert(!pp_mallocator_is_proc_shared(tree->allocator));
	    out->dat_del(tree->allocator, out->dat);
	}
    }
    
    pp_free(tree->allocator, out);

    --tree->count;

    return 0;
}

static void
delete_fixup(pp_rb_tree_t* tree, rb_node* node)
{
    rb_node *temp;

    assert(tree != NULL);
    assert(node != NULL);

    while (node != tree->root && node->color == RB_BLK) {
	if (node->parent->llink == node) {
	    temp = node->parent->rlink;
	    if (temp->color == RB_RED) {
		temp->color = RB_BLK;
		node->parent->color = RB_RED;
		rot_left(tree, node->parent);
		temp = node->parent->rlink;
	    }
	    if (temp->llink->color == RB_BLK && temp->rlink->color == RB_BLK) {
		temp->color = RB_RED;
		node = node->parent;
	    } else {
		if (temp->rlink->color == RB_BLK) {
		    temp->llink->color = RB_BLK;
		    temp->color = RB_RED;
		    rot_right(tree, temp);
		    temp = node->parent->rlink;
		}
		temp->color = node->parent->color;
		temp->rlink->color = RB_BLK;
		node->parent->color = RB_BLK;
		rot_left(tree, node->parent);
		node = tree->root;
	    }
	} else {
	    temp = node->parent->llink;
	    if (temp->color == RB_RED) {
		temp->color = RB_BLK;
		node->parent->color = RB_RED;
		rot_right(tree, node->parent);
		temp = node->parent->llink;
	    }
	    if (temp->rlink->color == RB_BLK && temp->llink->color == RB_BLK) {
		temp->color = RB_RED;
		node = node->parent;
	    } else {
		if (temp->llink->color == RB_BLK) {
		    temp->rlink->color = RB_BLK;
		    temp->color = RB_RED;
		    rot_left(tree, temp);
		    temp = node->parent->llink;
		}
		temp->color = node->parent->color;
		node->parent->color = RB_BLK;
		temp->llink->color = RB_BLK;
		rot_right(tree, node->parent);
		node = tree->root;
	    }
	}
    }

    node->color = RB_BLK;
}

void pp_rb_tree_empty (pp_rb_tree_t *tree)
{
    rb_node *node, *parent;

    assert(tree != NULL);

    node = tree->root;
    while (node != RB_NULL) {
	parent = node->parent;
	if (node->llink != RB_NULL) {
	    node = node->llink;
	    continue;
	}
	if (node->rlink != RB_NULL) {
	    node = node->rlink;
	    continue;
	}

	if (tree->key_del) {
	    assert(!pp_mallocator_is_proc_shared(tree->allocator));
	    tree->key_del(tree->allocator, node->key);
	}
	if (node->dat_del) {
	    assert(!pp_mallocator_is_proc_shared(tree->allocator));
	    node->dat_del(tree->allocator, node->dat);
	}
	pp_free(tree->allocator, node);

	if (parent != RB_NULL) {
	    if (parent->llink == node) {
		parent->llink = RB_NULL;
	    } else {
		parent->rlink = RB_NULL;
	    }
	}
	node = parent;
    }

    tree->root = RB_NULL;
    tree->count = 0;
}

u_int
pp_rb_tree_count(const pp_rb_tree_t* tree)
{
    assert(tree != NULL);

    return tree->count;
}

u_int
pp_rb_tree_height(const pp_rb_tree_t* tree)
{
    assert(tree != NULL);

    return tree->root != RB_NULL ? node_height(tree, tree->root) : 0;
}

u_int
pp_rb_tree_mheight(const pp_rb_tree_t* tree)
{
    assert(tree != NULL);

    return tree->root != RB_NULL ? node_mheight(tree, tree->root) : 0;
}

u_int
pp_rb_tree_pathlen(const pp_rb_tree_t* tree)
{
    assert(tree != NULL);

    return tree->root != RB_NULL ? node_pathlen(tree, tree->root, 1) : 0;
}

const void *
pp_rb_tree_min(const pp_rb_tree_t* tree)
{
    const rb_node *node;

    assert(tree != NULL);

    if (tree->root == RB_NULL) return NULL;

    for (node = tree->root; node->llink != RB_NULL; node = node->llink) {
	/* void */
    }
    return node->key_const;
}

const void *
pp_rb_tree_max(const pp_rb_tree_t* tree)
{
    const rb_node *node;

    assert(tree != NULL);

    if (tree->root == RB_NULL) return NULL;

    for (node = tree->root; node->rlink != RB_NULL; node = node->rlink) {
	/* void */
    }
    return node->key_const;
}

void
pp_rb_tree_walk(const pp_rb_tree_t* tree, pp_rb_tree_vis_func visit)
{
    const rb_node *node;

    assert(tree != NULL);
    assert(visit != NULL);

    if (tree->root == RB_NULL) return;

    for (node = node_min(tree, tree->root);
	 node != RB_NULL;
	 node = node_next(tree, node)) {
	if (visit(node->key_const, node->dat) == 0) break;
    }
}

static u_int
node_height(const pp_rb_tree_t* tree, const rb_node* node)
{
    u_int l, r;

    l = node->llink != RB_NULL ? node_height(tree, node->llink) + 1 : 0;
    r = node->rlink != RB_NULL ? node_height(tree, node->rlink) + 1 : 0;
    return max(l, r);
}

static u_int
node_mheight(const pp_rb_tree_t* tree, const rb_node* node)
{
    u_int l, r;

    l = node->llink != RB_NULL ? node_mheight(tree, node->llink) + 1 : 0;
    r = node->rlink != RB_NULL ? node_mheight(tree, node->rlink) + 1 : 0;
    return min(l, r);
}

static u_int
node_pathlen(const pp_rb_tree_t* tree, const rb_node* node, u_int level)
{
    u_int n = 0;

    assert(node != RB_NULL);

    if (node->llink != RB_NULL) {
	n += level + node_pathlen(tree, node->llink, level + 1);
    }
    if (node->rlink != RB_NULL) {
	n += level + node_pathlen(tree, node->rlink, level + 1);
    }
    return n;
}

static void
rot_left(pp_rb_tree_t* tree, rb_node* node)
{
    rb_node *rlink, *parent;

    assert(tree != NULL);
    assert(node != NULL);

    rlink = node->rlink;
    node->rlink = rlink->llink;
    if (rlink->llink != RB_NULL) {
	rlink->llink->parent = node;
    }
    parent = node->parent;
    rlink->parent = parent;
    if (parent != RB_NULL) {
	if (parent->llink == node) {
	    parent->llink = rlink;
	} else {
	    parent->rlink = rlink;
	}
    } else {
	tree->root = rlink;
    }
    rlink->llink = node;
    node->parent = rlink;
}

static void
rot_right(pp_rb_tree_t* tree, rb_node* node)
{
    rb_node *llink, *parent;

    assert(tree != NULL);
    assert(node != NULL);

    llink = node->llink;
    node->llink = llink->rlink;
    if (llink->rlink != RB_NULL) {
	llink->rlink->parent = node;
    }
    parent = node->parent;
    llink->parent = parent;
    if (parent != RB_NULL) {
	if (parent->llink == node) {
	    parent->llink = llink;
	} else {
	    parent->rlink = llink;
	}
    } else {
	tree->root = llink;
    }
    llink->rlink = node;
    node->parent = llink;
}

static rb_node *
node_new(pp_rb_tree_t* tree, const void* key, void* dat, pp_rb_tree_del_func dat_del)
{
    rb_node *node;

    node = pp_malloc(tree->allocator, sizeof(*node));
    if (tree->key_dup) {
	node->key = tree->key_dup(tree->allocator, key);
    } else {
	node->key_const = key;
    }
    node->dat = dat;
    node->dat_del = dat_del;
    node->color = RB_RED;
    node->parent = RB_NULL;
    node->llink = RB_NULL;
    node->rlink = RB_NULL;

    return node;
}

static const rb_node *
node_next(const pp_rb_tree_t* tree, const rb_node* node)
{
    rb_node *temp;

    assert(node != NULL);

    if (node->rlink != RB_NULL) {
	for (node = node->rlink; node->llink != RB_NULL; node = node->llink) {
	    /* void */
	}
    } else {
	temp = node->parent;
	while (temp != RB_NULL && temp->rlink == node) {
	    node = temp;
	    temp = temp->parent;
	}
	node = temp;
    }

    return node;
}

static const rb_node *
node_prev(const pp_rb_tree_t* tree, const rb_node* node)
{
    rb_node *temp;

    assert(node != NULL);

    if (node->llink != RB_NULL) {
	for (node = node->llink; node->rlink != RB_NULL; node = node->rlink) {
	    /* void */
	}
    } else {
	temp = node->parent;
	while (temp != RB_NULL && temp->llink == node) {
	    node = temp;
	    temp = temp->parent;
	}
	node = temp;
    }

    return node;
}

static rb_node *
node_max(const pp_rb_tree_t* tree, rb_node* node)
{
    assert(node != NULL);

    while (node->rlink != RB_NULL) {
	node = node->rlink;
    }
    return node;
}

static rb_node *
node_min(const pp_rb_tree_t* tree, rb_node* node)
{
    assert(node != NULL);

    while (node->llink != RB_NULL) {
	node = node->llink;
    }
    return node;
}

static void
node_content_swap(rb_node *n1, rb_node *n2) {
    void* key;
    void* dat;
    pp_rb_tree_del_func dat_del;
    key         = n1->key;
    dat         = n1->dat;
    dat_del     = n1->dat_del;
    n1->key     = n2->key;
    n1->dat     = n2->dat;
    n1->dat_del = n2->dat_del;
    n2->key     = key;
    n2->dat     = dat;
    n2->dat_del = dat_del;
}

#undef RB_NULL
#define RB_NULL (&iter->tree->null)

pp_rb_iter_t *
pp_rb_iter_new(const pp_rb_tree_t* tree)
{
    pp_rb_iter_t *iter;

    assert(tree != NULL);

    iter = malloc(sizeof(*iter));
    iter->tree = tree;
    iter->node = RB_NULL;
    return iter;
}

#define RETVALID(iter)	return iter->node != RB_NULL

int
pp_rb_iter_valid(const pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    RETVALID(iter);
}

void
pp_rb_iter_invalidate(pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    iter->node = RB_NULL;
}

int
pp_rb_iter_next(pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    if (iter->node == RB_NULL) {
	pp_rb_iter_first(iter);
    } else {
	iter->node = node_next(iter->tree, iter->node);
    }
    RETVALID(iter);
}

int
pp_rb_iter_prev(pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    if (iter->node == RB_NULL) {
	pp_rb_iter_last(iter);
    } else {
	iter->node = node_prev(iter->tree, iter->node);
    }
    RETVALID(iter);
}

int
pp_rb_iter_nextn(pp_rb_iter_t* iter, u_int count)
{
    assert(iter != NULL);

    if (count) {
	if (iter->node == RB_NULL) {
	    pp_rb_iter_first(iter);
	    count--;
	}

	while (count-- && iter->node) {
	    iter->node = node_next(iter->tree, iter->node);
	}
    }

    RETVALID(iter);
}

int
pp_rb_iter_prevn(pp_rb_iter_t* iter, u_int count)
{
    assert(iter != NULL);

    if (count) {
	if (iter->node == RB_NULL) {
	    pp_rb_iter_last(iter);
	    count--;
	}

	while (count-- && iter->node) {
	    iter->node = node_prev(iter->tree, iter->node);
	}
    }

    RETVALID(iter);
}

int
pp_rb_iter_first(pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    if (iter->tree->root == RB_NULL) {
	iter->node = RB_NULL;
    } else {
	iter->node = node_min(iter->tree, iter->tree->root);
    }
    RETVALID(iter);
}

int
pp_rb_iter_last(pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    if (iter->tree->root == RB_NULL) {
	iter->node = RB_NULL;
    } else {
	iter->node = node_max(iter->tree, iter->tree->root);
    }
    RETVALID(iter);
}

int
pp_rb_iter_search(pp_rb_iter_t* iter, const void* key)
{
    int rv;
    rb_node *node;
    pp_rb_tree_cmp_func cmp;

    assert(iter != NULL);

    cmp = iter->tree->key_cmp;
    for (node = iter->tree->root; node != RB_NULL;) {
	rv = cmp ? cmp(key, node->key_const)
	   : key < node->key_const ? -1 : key > node->key_const ? 1 : 0;
	if (rv == 0) break;
	node = rv < 0 ? node->llink : node->rlink;
    }
    iter->node = node;
    RETVALID(iter);
}

void *
pp_rb_iter_key(const pp_rb_iter_t* iter)
{
    assert(iter != NULL);
    assert(iter->tree != NULL);
    assert(iter->tree->key_dup); /* sanity check! */

    return iter->node != RB_NULL ? iter->node->key : NULL;
}

const void *
pp_rb_iter_key_const(const pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    return iter->node != RB_NULL ? iter->node->key_const : NULL;
}

void *
pp_rb_iter_data(const pp_rb_iter_t* iter)
{
    assert(iter != NULL);

    return iter->node != RB_NULL ? iter->node->dat : NULL;
}
