/*
 * shared memory allocation and management
 *
 * This shared memory allocator is realized using 2 structures
 * 1. a global one, residing in the shared memory
 *    - shmheader_t
 *    - it contains the actual memory manager, the balloc mallocater
 *    - and a list of all semaphore keys used by objects
 *      stored in the shared memory
 *    - and a fixed length map of well known object keys pointing
 *      to the actual objects in the shared memory
 * 2. a process local one, resing on the heap of a particular process
 *    - pp_shm_t 
 *    - it contains a map of semaphare keys containing the process
 *      local semaphore ids, corresponding to the list of semaphores
 *      in shmheader_t
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <stdlib.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <pp/base.h>
#include <pp/vector.h>
#include <pp/dict.h>
#include <pp/shm.h>
#include <pp/mutex.h>
#include <pp/mallocator.h>
#include "balloc_intern.h"

typedef struct shmheader_s {
    bmallocator_t ballocator;   // memory manager
    int           shmkey;       // my own shared memory key
    vector_t      semmap;       // semaphore ids, needed for deep clean
    size_t        idxmap_size;  // size of index map
    void*         idxmap;       // start of index map of idxmap_size
} shmheader_t;

struct pp_shm_s {
    int          shmid;  // id of our shared memory
    shmheader_t* shmhp;  // pointer to shared memory header, i.e. to shm
    int          sema;   // global lock, with well known key
    int          in_init;// 
};

// currently we support a single shm only, so this is kinda singleton
static pp_shm_t* myshm = NULL;

static inline pp_shm_t* shm_get_intern(key_t shmkey UNUSED) {
    return myshm;
}

#define SHM_HEADER_SIZE(h) \
    (sizeof(shmheader_t) + ((h->idxmap_size - 1) * sizeof(void*)))

static inline void shm_lock_intern(pp_shm_t* shm) {
    if (!shm->in_init) {
	pp_sema_lock(shm->sema);
    }
}

static void shm_lock(pp_mallocator_t* a) {
    shmheader_t* shmhp = (shmheader_t*)a;
    pp_shm_t* shm = pp_shm_get(shmhp->shmkey);
    shm_lock_intern(shm);
}

static inline void shm_unlock_intern(pp_shm_t* shm) {
    if (!shm->in_init) {
	pp_sema_unlock(shm->sema);
    }
}

static void shm_unlock(pp_mallocator_t* a) {
    shmheader_t* shmhp = (shmheader_t*)a;
    pp_shm_t* shm = pp_shm_get(shmhp->shmkey);
    shm_unlock_intern(shm);
}


static void shm_mutex_init(pp_mallocator_t* a, pp_mutex_t* m,
			   pp_mutex_kind_t mtx_kind) {
    shmheader_t* shmhp = (shmheader_t*)a;
    pp_mutex_init_procshared(m, shmhp->shmkey, mtx_kind);
}

/*
 * cleans the proc local structure only!
 * call pp_shm_destroy if you wonna cleanup also the shared resources
 */
void pp_shm_cleanup(pp_shm_t* shm) {
    if (0 != shmdt(shm->shmhp))
	pp_log_err("pp_shm_cleanup: shmdt");
    free(shm);
}

/*
 * frees all resources based on the shared memory key,
 * even the shared ones, except the semapore! This function may
 * also be called in case the shared memory was not initialized
 * previously, since we wonna make sure cleanup will be performed.
 * (you shouldn't call this method under normal circumstances)
 */
void pp_shm_destroy(key_t shmkey) {
    const char* fname = ___F;
    int sema;
    pp_shm_t* shm;
    int shmid;

    // try to lock the semaphore, if existent
    if ((sema = semget(shmkey, 0, 0)) >= 0) {
	pp_sema_lock(sema);
    } else {
	pp_log_err("%s(): semget(GLOBAL)", fname);
    }

    // determine shmid
    if ((shm = shm_get_intern(shmkey)) != NULL) {
	shmid = shm->shmid;
	pp_shm_cleanup(shm);
    } else {
	if((shmid = shmget(shmkey, 0, 0)) < 0)
	    pp_log_err("%s(): shmget", fname);
    }

    // if we have found a shmid, try to attach and destroy all semas
    // managed by this shm and destroy the shm itself
    if (shmid >= 0) {
	shmheader_t* p;
	unsigned long i;
	if ((long)(p = shmat(shmid, PP_SHM_VIRTADDR, 0)) > 0) {
	    for (i = 0; i < vector_size(&p->semmap); ++i) {
		void * _semid = vector_get(&p->semmap, i);
		pp_sema_destroy((int)_semid);
	    }
	    shmdt(p);
	} else {
	    pp_log_err("%s(): shmctl(%d)", fname, shmid);
	}
	if (0 != shmctl(shmid, IPC_RMID, NULL))
	    pp_log_err("%s(): shmctl(IPC_RMID)", fname);
    }

    // unlock if possible
    if (sema >= 0)
	pp_sema_unlock(sema);
}

pp_shm_t* pp_shm_init(key_t shmkey, size_t shmsize, size_t idxmap_size, 
		      pp_shm_init_fkt init_fkt, ...) {
    const char* fname = ___F;
    int shm_created, init_err = PP_SUC;
    shmheader_t* shmhp = NULL;
    pp_shm_t* shm;
    va_list ap;

    // create the process local shm allocator structure
    shm = (pp_shm_t*)malloc(sizeof(pp_shm_t));
    shm->in_init = 0;
    
    assert(myshm == NULL);
    myshm = shm;    

    if (0 > (shm->sema = semget(shmkey, 1, IPC_CREAT | 0666))) {
	pp_log_err("%s(): semget(GLOBAL)", ___F);
	abort();
    }

    // lock shm
    pp_sema_lock(shm->sema);
    
    // get shm and find out whether its newly created or not
    if (0 > (shm->shmid = shmget(shmkey, shmsize,
				 IPC_CREAT | IPC_EXCL | 0666))) {
	if (errno != EEXIST || 0 > (shm->shmid = shmget(shmkey, 0, 0))) {
	    pp_log_err("%s shmget", fname);
	    abort();
	}
	shm_created = 0;
    } else {
	shm_created = 1;
    }
    
    if (-1 == (long)(shmhp = shmat(shm->shmid, PP_SHM_VIRTADDR, 0))) {
	pp_log_err("%s shmat", fname);
	abort();
    }
    shm->shmhp = shmhp;

    if (shm_created) { // initialice newly created shared memory header
	shm->in_init = 1;
	shmhp->shmkey = shmkey;
	shmhp->idxmap_size = idxmap_size;
	memset(&shmhp->idxmap, 0, idxmap_size * sizeof(void*));
	pp_binit(&shmhp->ballocator, (char*)shmhp + SHM_HEADER_SIZE(shmhp),
		 shmsize - SHM_HEADER_SIZE(shmhp), 0,
		 shm_lock, shm_unlock, shm_mutex_init, 1);
	vector_new_with_alloc(&shmhp->semmap, 10, NULL,
			      (pp_mallocator_t*)&shmhp->ballocator);
	va_start(ap, init_fkt);
	init_err = init_fkt(&shmhp->idxmap,
			    (pp_mallocator_t*)&shmhp->ballocator, ap);
	va_end(ap);
	shm->in_init = 0;
    }
    pp_sema_unlock(shm->sema);

    // cleanup everything in case init_fkt failed
    if (init_err == PP_ERR) {
	pp_shm_destroy(shmkey);
	shm = NULL;
    }
    return shm;
}

void* pp_shm_get_obj(pp_shm_t* shm, unsigned long idx) {
    void* p;
    pp_sema_lock(shm->sema);
    assert(idx < shm->shmhp->idxmap_size);
    p = (&shm->shmhp->idxmap)[idx];
    pp_sema_unlock(shm->sema);
    return p;
}

pp_shm_t* pp_shm_get(key_t shmkey) {
    pp_shm_t* s = shm_get_intern(shmkey);
    assert(s != NULL);
    assert(s->shmhp->shmkey == shmkey);
    return s;
}

/*
 * returns the key of this shared memory
 */
int pp_shm_get_key(pp_shm_t* shm) {
    assert(shm);
    return shm->shmhp->shmkey;
}

/*
 * adds a semophore to the list of managed semaphores by
 * this shared memory
 */
void pp_shm_add_sem(pp_shm_t* shm, int semid) {
    shmheader_t* sh = shm->shmhp;
    assert(semid >= 0);
    shm_lock_intern(shm);
    vector_add(&sh->semmap, (void*)(long)semid);
    shm_unlock_intern(shm);
}

/*
 * removes a semaphore from the list of managed semaphores
 * by this shared memory
 */
void pp_shm_remove_sem(pp_shm_t* shm, int semid) {
    unsigned int i;
    vector_t* semmap = &shm->shmhp->semmap;
    
    shm_lock_intern(shm);
    for (i = 0; i < vector_size(semmap); ++i) {
	void * _semid = vector_get(semmap, i);
	if (semid == (int)_semid) {
	    vector_remove(semmap, i);
	    pp_sema_unlock(shm->sema);
	    return;
	}
    }
    shm_unlock_intern(shm);
    assert(0);
}
