#include <pp/base.h>

/*
 * Escapes any special characters in string `s'.
 *     - plain_list - contains all special chars
 *     - esc_list   - corresponding replacement chars
 *     - esc_char   - the escape character, to be prepended to esc_list chars
 * Eg: `a' translates to `\b',
 *     - if `a' is in plain_list, 
 *     - `b' is at the same pos in esc_list,
 *     - and esc_char is `\'.
 * Returns:
 *     - the escaped string (to be free'd by caller)
 */
char *
pp_str_escape(char esc_char, const char* plain_list, const char* esc_list, const char* s)
{
    char c;
    const char *pp; /* ptr into plain_list */
    const char *pe; /* ptr into esc_list */
    pp_strstream_t stream;
    char *ret = NULL;

    if (!s) goto bail;

    pp_strstream_init(&stream);

    while ((c = *s++) != '\0') {
	if (c == esc_char) {
	    /* escape char */
	    pp_strappendchr(&stream, esc_char);
	    pp_strappendchr(&stream, esc_char);
	} else if ((pp = index(plain_list, c)) != NULL) {
	    /* special char */
	    pe = esc_list + (pp - plain_list);
	    pp_strappendchr(&stream, esc_char);
	    pp_strappendchr(&stream, *pe);
	} else {
	    /* normal char */
	    pp_strappendchr(&stream, c);
	}
    }

    ret = pp_strstream_buf_and_free(&stream);
bail:
    return ret;
}

/* Undoes pp_str_escape() when called with its result str and otherwise same params. */
char *
pp_str_unescape(char esc_char, const char* plain_list, const char* esc_list, const char* s)
{
    char c, cn;
    const char *pp; /* ptr into plain_list */
    const char *pe; /* ptr into esc_list */
    pp_strstream_t stream;
    char *ret = NULL;

    if (!s) goto bail;

    pp_strstream_init(&stream);

    while ((c = *s++) != '\0') {
	if (c == esc_char) {
	    /* escaped char */
	    cn = *s;
	    if (cn == esc_char) {
		/* escape char */
		pp_strappendchr(&stream, esc_char);
		s++;
	    } else if ((pe = index(esc_list, cn)) != NULL) {
		/* special char */
		pp = plain_list + (pe - esc_list);
		pp_strappendchr(&stream, *pp);
		s++;
	    }
	} else {
	    /* normal char */
	    pp_strappendchr(&stream, c);
	}
    }

    ret = pp_strstream_buf_and_free(&stream);
bail:
    return ret;
}
