/**
 * Copyright 2001 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: little vector implementation to store
 *              an arbitrary number of elements.
 *              the elements have to be of the same
 *              type in case the element delete function is
 *              going to be used
 */
#include <assert.h>
//#include <pp/base.h>
#include <stdlib.h>
#include <string.h>
#include <pp/vector.h>
#include <pp/win32.h>

#define PTR_TO_ELEM(vector, idx) \
    (void*)&((char*)(vector)->elements)[vector->elemsize * (idx)]

#define VEC_OWN  0x01   // vector owns its memory
#define VEC_PTR  0x02   // vector contains pointers, no structs
#define VEC_HEAP 0x04   // vector uses standard heap allocator

static void vector_adjust(vector_t* vec, unsigned long more) {
    if((vec->size + more) > vec->capacity) {
	vec->capacity += (more > VECTOR_INCREMENT ? more : VECTOR_INCREMENT);
	vec->elements = (void**)pp_realloc(vec->alloc, vec->elements,
					   vec->capacity * vec->elemsize);
    }
}

static void vector_delete_elem(const vector_t* vec, unsigned long idx) {
    if (vec->delelem) {
	void* elemptr;
	assert(!pp_mallocator_is_proc_shared(vec->alloc));
	if (vec->state & VEC_PTR) {
	    elemptr = vec->elements[idx];
	} else {
	    elemptr = PTR_TO_ELEM(vec, idx);
	}
	if (vec->state & VEC_HEAP) {
	    ((vector_elem_del_func_simple)vec->delelem)(elemptr);
	} else {
	    vec->delelem(vec->alloc, elemptr);
	}
    }
}

/**
 * initializes the vector structure
 * in case the vergiven vector_t sructure pointer is NULL
 * we are going to allocate a new one
 */
vector_t*
vector_new(vector_t* vec, size_t initcapacity, vector_elem_del_func_simple delelem)
{
    vector_t* v;
    v = vector_new_with_alloc(vec, initcapacity, (vector_elem_del_func)delelem,
			      pp_mallocator_heap());
    v->state |= VEC_HEAP;
    return v;
}

/**
 * initializes a vector with the given alloc
 */
vector_t*
vector_new_with_alloc(vector_t* vec, size_t initcapacity,
		      vector_elem_del_func delelem,
		      pp_mallocator_t* a) {
    vector_t* v = vector_new2_with_alloc(vec, initcapacity, sizeof(void*),
					 delelem, a);
    v->state |= VEC_PTR;
    return v;
}

/**
 * initializes the vector structure with the given element size
 */
vector_t*
vector_new2(vector_t* vec, size_t initcapacity, size_t elemsize,
	    vector_elem_del_func_simple delelem) {
    vector_t* v;
    v = vector_new2_with_alloc(vec, initcapacity, elemsize,
			       (vector_elem_del_func)delelem,
			       pp_mallocator_heap());
    v->state |= VEC_HEAP;
    return v;
}

/**
 * initializes the vector structure with the given element size
 */
vector_t*
vector_new2_with_alloc(vector_t* vec, size_t initcapacity, size_t elemsize,
		       vector_elem_del_func delelem,
		       pp_mallocator_t* a) {
    int state = 0;
    if (vec == NULL) {
	vec = (vector_t*)pp_malloc(a, sizeof(vector_t));
    	state = VEC_OWN;
    }
    memset(vec, 0, sizeof(vector_t));
    vec->state    = state;
    vec->alloc    = a;
    vec->capacity = initcapacity;
    vec->elemsize = elemsize;
    vec->elements = initcapacity ?
	(void**)pp_malloc(a, initcapacity * elemsize) : NULL;
    vec->delelem = delelem;
    return vec;
}

/**
 * add the element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
void
vector_add(vector_t* vec, void* element)
{
    assert(vec->state & VEC_PTR);
    assert(vec->has_const_elements == 0);
    vector_adjust(vec, 1);
    vec->elements[vec->size++] = element;
    vec->has_non_const_elements = 1;
}

/**
 * add the element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
void
vector_add_const(vector_t* vec, const void* element)
{
    assert(vec->state & VEC_PTR);
    assert(vec->delelem == NULL);
    assert(vec->has_non_const_elements == 0);
    vector_adjust(vec, 1);
    vec->elements_const[vec->size++] = element;
    vec->has_const_elements = 1;
}

/**
 * copy a structure into the vector.
 * the vector, most probably, must have been initialized with vector_new2
 */
void
vector_add2(vector_t* vec, void* elemptr)
{
    vector_adjust(vec, 1);
    memcpy(PTR_TO_ELEM(vec, vec->size++), elemptr, vec->elemsize);
}

/**
 * add all elements of the given vector to this vector.
 * Note that all the newly added elements from now on are going to
 * be deleted with the delete method of this vector
 * In case the copy function is empty, the elements
 * will be moved to our vector and deleted from the other
 */
void
vector_addvec(vector_t* vec, vector_t* addvec, void* (*copyelem)(void* elm)) {
    size_t i, addvec_sz = vector_size(addvec);
    vector_adjust(vec, addvec_sz);
    assert(vec->state & VEC_PTR && addvec->state & VEC_PTR);
    if (copyelem) {
	assert(vec->has_const_elements == 0);
	for (i = 0; i < addvec_sz; ++i) {
	    vec->elements[vec->size++] = copyelem(vector_get(addvec, i));
	}
	vec->has_non_const_elements = 1;
    } else {
	for (i = 0; i < addvec_sz; ++i) {
	    if (addvec->has_const_elements) {
		assert(vec->has_non_const_elements == 0);
		vec->elements_const[vec->size++] = addvec->elements_const[i];
		vec->has_const_elements = 1;
	    } else {
		assert(vec->has_const_elements == 0);
		vec->elements[vec->size++] = addvec->elements[i];
		vec->has_non_const_elements = 1;
	    }
	}
	addvec->size = 0;
    }
}

/*
 * In case the copy function is empty, a bit-copy will be
 * performed, elements are not deleted from the other vector
 */
void
vector_addvec2(vector_t* vec, vector_t* addvec,
	       void (*copyelem)(void* dest, void* src, size_t size)) {
    size_t i, addvec_sz = vector_size(addvec);
    vector_adjust(vec, addvec_sz);
    assert(!(vec->state & VEC_PTR) && !(addvec->state & VEC_PTR));
    assert(vec->elemsize == addvec->elemsize);
    for (i = 0; i < addvec_sz; ++i) {
	if (copyelem)
	    copyelem(PTR_TO_ELEM(vec, vec->size++),
		     PTR_TO_ELEM(addvec, i), vec->elemsize);
	else {
	    memcpy(PTR_TO_ELEM(vec, vec->size++),
		   PTR_TO_ELEM(addvec, i), vec->elemsize);
	}
    }   
}

/**
 * returns the element at the specified index
 */
void*
vector_get(const vector_t* vec, unsigned long idx)
{
    assert(vec->state & VEC_PTR);
    assert(vec->has_non_const_elements);
    assert(idx < vec->size);
    return vec->elements[idx];
}

/**
 * returns the const element at the specified index
 */
const void*
vector_get_const(const vector_t* vec, unsigned long idx)
{
    assert(vec->state & VEC_PTR);
    assert(idx < vec->size);
    return vec->elements_const[idx];
}

/**
 * returns a pointer to the element at the specified index
 */
void*
vector_get2(const vector_t* vec, unsigned long idx)
{
    assert(idx < vec->size);
    return PTR_TO_ELEM(vec, idx);
}

/**
 * sets the element at the specified index
 * ATTENTION, asserts that the index exists
 */
void
vector_set(const vector_t* vec, unsigned long idx, void* v)
{
    assert(vec->state & VEC_PTR);
    assert(vec->has_const_elements == 0);
    assert(idx < vec->size);
    vector_delete_elem(vec, idx);
    vec->elements[idx] = v;
}

/**
 * sets the const element at the specified index
 * ATTENTION, asserts that the index exists
 */
void
vector_set_const(const vector_t* vec, unsigned long idx, const void* v)
{
    assert(vec->state & VEC_PTR);
    assert(vec->has_non_const_elements == 0);
    assert(idx < vec->size);
    vector_delete_elem(vec, idx);
    vec->elements_const[idx] = v;
}

/**
 * copies a element pointed to by vptr into the vector
 */
void
vector_set2(const vector_t* vec, unsigned long idx, void* vptr)
{
    assert(idx < vec->size);
    vector_delete_elem(vec, idx);
    memcpy(PTR_TO_ELEM(vec, idx), vptr, vec->elemsize);
}

/**
 * swaps two elements in the vector
 */
void
vector_swap(const vector_t* vec, unsigned long idx1, unsigned long idx2)
{
    void * p;
    assert(idx1 < vec->size);
    assert(idx2 < vec->size);
    p = vec->elements[idx2];
    vec->elements[idx2] = vec->elements[idx1];
    vec->elements[idx1] = p;
}


/**
 * removes the last element and returns it. The element
 * is not deleted. asserts that vector_size > 0.
 */
void* vector_pop(vector_t* vec) {
    assert(vec->state & VEC_PTR);
    assert(vec->size > 0);
    return vec->elements[--vec->size];
}

/**
 * returns the number of elements currently
 * contained in the vector
 */
size_t
vector_size(const vector_t* vec) {
    assert(vec);
    return vec->size;
}

/**
 * removes element at the given index and moves
 * all other elements one index to the front
 */
void
vector_remove(vector_t* vec, unsigned long idx)
{
    unsigned long taillen;
    assert(idx < vec->size);

    vector_delete_elem(vec, idx);
    vec->size--;
    if ((taillen = vec->size - idx) > 0) {
	memmove(PTR_TO_ELEM(vec, idx), PTR_TO_ELEM(vec, idx + 1),
		taillen * vec->elemsize);
    }
}

/**
 * removes element at the given index and moves
 * all other elements one index to the front
 */
void
vector_remove_dont_delete(vector_t* vec, unsigned long idx)
{
    unsigned long taillen;
    assert(idx < vec->size);
    vec->size--;
    if ((taillen = vec->size - idx) > 0) {
	memmove(PTR_TO_ELEM(vec, idx), PTR_TO_ELEM(vec, idx + 1),
		taillen * vec->elemsize);
    }
}

/**
 * sortes elements in vector using the specified 
 * compare function
 */
void 
vector_qsort(vector_t* vec, int(*comp)(const void*, const void*)) 
{
    qsort(vec->elements, vec->size, vec->elemsize, comp);
}


/**
 * cleans up all the elements using the element
 * delete function, if any, and deletes the allocated
 * memory
 */
void
vector_delete(vector_t* vec)
{
    unsigned long i;
    if (!vec) return;
    if (vec->delelem) {
	assert(!pp_mallocator_is_proc_shared(vec->alloc));
	if (vec->state & VEC_PTR) {
	    if (vec->state & VEC_HEAP) {
		for (i=0; i<vec->size; ++i)
		    ((vector_elem_del_func_simple)vec->delelem)(vec->elements[i]);
	    } else {
		for (i=0; i<vec->size; ++i)
		    vec->delelem(vec->alloc, vec->elements[i]);
	    }
	} else {
	    if (vec->state & VEC_HEAP) {
		for (i=0; i<vec->size; ++i)
		    ((vector_elem_del_func_simple)vec->delelem)(PTR_TO_ELEM(vec, i));
	    } else {
		for (i=0; i<vec->size; ++i) 
		    vec->delelem(vec->alloc, PTR_TO_ELEM(vec, i));
	    }
	}
    }
    vector_delete_wo_elems(vec);
}

/**
 * deletes the vector but leaves the elements untouched
 * nomatter whether the element delete function is set or not
 */
void vector_delete_wo_elems(vector_t* vec) {
    if (!vec) return;
    pp_free(vec->alloc, vec->elements);
    if (vec->state & VEC_OWN) pp_free(vec->alloc, vec);    
}
