/* crypto/bio/bss_acpt.c */
/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * This package is an SSL implementation written
 * by Eric Young (eay@cryptsoft.com).
 * The implementation was written so as to conform with Netscapes SSL.
 * 
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.  The following conditions
 * apply to all code found in this distribution, be it the RC4, RSA,
 * lhash, DES, etc., code; not just the SSL code.  The SSL documentation
 * included with this distribution is covered by the same copyright terms
 * except that the holder is Tim Hudson (tjh@cryptsoft.com).
 * 
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * If this package is used in a product, Eric Young should be given attribution
 * as the author of the parts of the library used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    "This product includes cryptographic software written by
 *     Eric Young (eay@cryptsoft.com)"
 *    The word 'cryptographic' can be left out if the rouines from the library
 *    being used are not cryptographic related :-).
 * 4. If you include any Windows specific code (or a derivative thereof) from 
 *    the apps directory (application code) you must include an acknowledgement:
 *    "This product includes software written by Tim Hudson (tjh@cryptsoft.com)"
 * 
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <openssl/ssl.h>
#include <pp/base.h>
#include <pp/bio.h>
#include <pp/bio_internal.h>

typedef struct {
    int state;
    char * param_addr;
    int addr_family;
    int accept_sock;
    int accept_nbio;
    int nbio;
    int bind_mode;
    BIO * bio_chain;
} pp_accept_ctx_t;

static int accept_new(BIO * b);
static int accept_free(BIO * b);
static int accept_state(BIO * b, pp_accept_ctx_t * ctx);
static int accept_read(BIO * b, char * out, int outl);
static int accept_write(BIO * b, const char * in, int inl);
static long accept_ctrl(BIO * b, int cmd, long num, void * ptr);
static int accept_puts(BIO * b, const char * str);
static void accept_close_socket(BIO * b);
static int get_accept_socket(char * host_port, int bind_mode, int addr_family);
static int do_accept(int sock, int addr_family);

#define ACPT_S_BEFORE			1
#define ACPT_S_GET_ACCEPT_SOCKET	2
#define ACPT_S_OK			3

static BIO_METHOD methods_pp_accept = {
    BIO_TYPE_PP_ACCEPT,
    "pp socket accept",
    accept_write,
    accept_read,
    accept_puts,
    NULL, /* connect_gets, */
    accept_ctrl,
    accept_new,
    accept_free,
    NULL,
};

BIO_METHOD * BIO_pp_s_accept(void)
{
    return &methods_pp_accept;
}

static int accept_new(BIO * b)
{
    pp_accept_ctx_t * ctx;

    b->init = 0;
    b->num = -1;
    b->flags = 0;

    if (!(ctx = (pp_accept_ctx_t *)OPENSSL_malloc(sizeof(*ctx)))) {
	return 0;
    }

    memset(ctx, 0, sizeof(*ctx));
    ctx->accept_sock = -1;
    ctx->bind_mode = BIO_BIND_NORMAL;
    ctx->state = ACPT_S_BEFORE;

    b->ptr = (char *)ctx;
    b->init = 1;
    b->shutdown = 1;

    return 1;
}

static int accept_free(BIO * b)
{
    pp_accept_ctx_t * ctx;

    if (b == NULL || !b->init) return 0;

    if (b->shutdown) {
	accept_close_socket(b);
	ctx = (pp_accept_ctx_t *)b->ptr;
	if (ctx->param_addr != NULL) OPENSSL_free(ctx->param_addr);
	if (ctx->bio_chain != NULL) BIO_free_all(ctx->bio_chain);
	OPENSSL_free(ctx);
	b->ptr = NULL;
	b->init = 0;
	b->flags = 0;
    }

    return 1;
}

static int accept_state(BIO * b, pp_accept_ctx_t * ctx)
{
    const char * fn = ___F;
    BIO * new_b = NULL;
    int s = -1;
    int i;

again:
    switch (ctx->state) {
      case ACPT_S_BEFORE:
	  if (ctx->param_addr == NULL) {
	      pp_log("%s(): No accept port specified.\n", fn);
	      return -1;
	  }
	  s = get_accept_socket(ctx->param_addr, ctx->bind_mode, ctx->addr_family);
	  if (s == -1) return -1;

	  if (ctx->accept_nbio && !BIO_socket_nbio(s, 1)) {
	      close(s);
	      pp_log("%s(): Couldn't set non-blocking mode on accept socket.\n", fn);
	      return -1;
	  }
	  ctx->accept_sock = s;
	  b->num = s;
	  ctx->state = ACPT_S_GET_ACCEPT_SOCKET;
	  return 1;
      case ACPT_S_GET_ACCEPT_SOCKET:
	  if (b->next_bio != NULL) {
	      ctx->state = ACPT_S_OK;
	      goto again;
	  }
	  if ((i = do_accept(ctx->accept_sock, ctx->addr_family)) < 0) {
	      return i;
	  }
	  if ((new_b = BIO_new_socket(i, BIO_CLOSE)) == NULL) goto err;

	  BIO_set_callback(new_b, BIO_get_callback(b));
	  BIO_set_callback_arg(new_b, BIO_get_callback_arg(b));

	  if (ctx->nbio && !BIO_socket_nbio(i, 1)) {
	      pp_log("%s(): Couldn't set non-blocking mode on accepted socket.\n", fn);
	      goto err;
	  }

	  /* If the accept BIO has an bio_chain, we dup it and
	   * put the new socket at the end. */
	  if (ctx->bio_chain != NULL) {
	      BIO * d_new_b;
	      if ((d_new_b = BIO_dup_chain(ctx->bio_chain)) == NULL)
		  goto err;
	      if (!BIO_push(d_new_b, new_b)) goto err;
	      new_b = d_new_b;
	  }
	  if (BIO_push(b, new_b) == NULL) goto err;

	  ctx->state = ACPT_S_OK;
	  return 1;
    err:
	  if (new_b != NULL) {
	      BIO_free(new_b);
	  } else if (s != -1) {
	      close(s);
	  }
	  return 0;
      case ACPT_S_OK:
	  if (b->next_bio == NULL) {
	      ctx->state = ACPT_S_GET_ACCEPT_SOCKET;
	      goto again;
	  }
	  return 1;
      default:	
	  return 0;
    }
}

static int accept_read(BIO * b, char * out, int outl)
{
    pp_accept_ctx_t * ctx;
    int ret;

    if (b == NULL || !b->init) return -1;

    BIO_clear_retry_flags(b);
    ctx = (pp_accept_ctx_t *)b->ptr;

    while (b->next_bio == NULL) {
	ret = accept_state(b, ctx);
	if (ret <= 0) return ret;
    }

    ret = BIO_read(b->next_bio, out, outl);
    BIO_copy_next_retry(b);
    return ret;
}

static int accept_write(BIO * b, const char * in, int inl)
{
    pp_accept_ctx_t * ctx;
    int ret;

    if (b == NULL || !b->init) return -1;

    BIO_clear_retry_flags(b);
    ctx = (pp_accept_ctx_t *)b->ptr;

    while (b->next_bio == NULL) {
	ret = accept_state(b, ctx);
	if (ret <= 0) return ret;
    }

    ret = BIO_write(b->next_bio, in, inl);
    BIO_copy_next_retry(b);
    return ret;
}

static long accept_ctrl(BIO * b, int cmd, long num, void * ptr)
{
    pp_accept_ctx_t * ctx;
    long ret = 1;

    if (b == NULL || !b->init) return -1;

    ctx = (pp_accept_ctx_t *)b->ptr;

    switch (cmd) {
      case BIO_CTRL_RESET:
	  ret = 0;
	  ctx->state = ACPT_S_BEFORE;
	  accept_close_socket(b);
	  b->flags = 0;
	  break;
      case BIO_C_DO_STATE_MACHINE:
	  /* use this one to start the connection */
	  ret = (long)accept_state(b, ctx);
	  break;
      case BIO_C_SET_ACCEPT:
	  if (ptr != NULL) {
	      if (num == 0) {
		  if (ctx->param_addr != NULL)
		      OPENSSL_free(ctx->param_addr);
		  ctx->param_addr = BUF_strdup(ptr);
	      } else if (num == 1) {
		  ctx->accept_nbio = (ptr != NULL);
	      } else if (num == 2) {
		  if (ctx->bio_chain != NULL)
		      BIO_free(ctx->bio_chain);
		  ctx->bio_chain = (BIO *)ptr;
	      } else if (num == 3) {
		  ctx->addr_family = (int)ptr;
	      } else {
		  return -1;
	      }
	  } else {
	      return -1;
	  }
	  break;
      case BIO_C_SET_NBIO:
	  ctx->nbio = (int)num;
	  break;
      case BIO_C_SET_FD:
	  if (ptr != NULL) {
	      b->num = *((int *)ptr);
	      ctx->accept_sock = b->num;
	      ctx->state = ACPT_S_GET_ACCEPT_SOCKET;
	      b->shutdown = (int)num;
	  } else {
	      ret = -1;
	  }
	  break;
      case BIO_C_GET_FD:
	  if (ptr != NULL)
	      *((int *)ptr) = ctx->accept_sock;
	  ret = ctx->accept_sock;
	  break;
      case BIO_C_GET_ACCEPT:
	  if (ptr != NULL) {
	      *((char **)ptr) = ctx->param_addr;
	  } else {
	      ret = -1;
	  }
	  break;
      case BIO_CTRL_GET_CLOSE:
	  ret = b->shutdown;
	  break;
      case BIO_CTRL_SET_CLOSE:
	  b->shutdown = (int)num;
	  break;
      case BIO_CTRL_PENDING:
      case BIO_CTRL_WPENDING:
	  ret = 0;
	  break;
      case BIO_CTRL_FLUSH:
	  break;
      case BIO_C_SET_BIND_MODE:
	  ctx->bind_mode = (int)num;
	  break;
      case BIO_C_GET_BIND_MODE:
	  ret = (long)ctx->bind_mode;
	  break;
      case BIO_CTRL_DUP:
	  break;
      default:
	  ret = 0;
	  break;
    }
    return ret;
}

static int accept_puts(BIO * b, const char * str)
{
    return accept_write(b, str, strlen(str));
}

static void accept_close_socket(BIO * b)
{
    pp_accept_ctx_t * ctx;

    if (b == NULL || !b->init) return;

    ctx = (pp_accept_ctx_t *)b->ptr;

    if (ctx->accept_sock != -1) {
	close(ctx->accept_sock);
	ctx->accept_sock = -1;
	b->num = -1;
    }
}

static int get_accept_socket(char * host_port, int bind_mode, int addr_family)
{
    const char * fn = ___F;
    int ret = 0;
    struct sockaddr_in server_ipv4;
    struct sockaddr_in6 server_ipv6;
    struct sockaddr * server_p;
    socklen_t server_len;
    int s = -1;
    unsigned short port;
    char host[256];
    int bind_count = 0, n;

    if (addr_family != AF_INET && addr_family != AF_INET6) return 0;

    if ((n = sscanf(host_port, "[%255[^]]]:%hu", host, &port)) == 2) {
	if (addr_family != AF_INET6) return 0;
    } else if ((n = sscanf(host_port, "%255[^:]:%hu", host, &port)) != 2 &&
	       (n = sscanf(host_port, "%hu", &port) != 1)) {
	return 0;
    }

    if (n == 1) strcpy(host, "*");

    if (addr_family == AF_INET6) {
	server_p = (struct sockaddr *)&server_ipv6;
	server_len = sizeof(server_ipv6);
	memset(server_p, 0, server_len);
	server_ipv6.sin6_family = AF_INET6;
	server_ipv6.sin6_port = htons(port);
    } else {
	server_p = (struct sockaddr *)&server_ipv4;
	server_len = sizeof(server_ipv4);
	memset(server_p, 0, server_len);
	server_ipv4.sin_family = AF_INET;
	server_ipv4.sin_port = htons(port);
    }

    if (strcmp(host, "*")) {
	char * ip_addr_p;
	if (addr_family == AF_INET6)
	    ip_addr_p = (char *)&server_ipv6.sin6_addr;
	else
	    ip_addr_p = (char *)&server_ipv4.sin_addr;
        if (!_pp_bio_get_host_ip(host, ip_addr_p, addr_family)) goto err;
    }

    if ((s = socket(addr_family, SOCK_STREAM, IPPROTO_TCP)) == -1) {
	pp_log_err("%s(): Couldn't create accept socket", fn);
        goto err;
    }
    if (bind_mode == BIO_BIND_REUSEADDR) {
        int i = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&i, sizeof(i));
        bind_mode = BIO_BIND_NORMAL;
    }
bind_again:
    if (bind(s, server_p, server_len) == -1) {
	/*
	 * We cannot bind immediately after closing an accept socket - even if
	 * we use the SO_REUSEADDR socket option. So we wait <= 20s.
	 */
	if (errno == EADDRINUSE && bind_count++ < 20) {
	    usleep(1000000);
	    goto bind_again;
	}
	pp_log_err("%s(): Couldn't bind accept socket", fn);
        goto err;
    }
    if (listen(s, 10) == -1) {
	pp_log_err("%s(): Couldn't listen on accept socket", fn);
        goto err;
    }
    ret = 1;

 err:
    if (ret == 0 && s != -1) {
        close(s);
        s = -1;
    }
    return s;
}

static int do_accept(int sock, int addr_family)
{
    const char * fn = ___F;
    int ret = -1;
    static struct sockaddr_in from_ipv4;
    static struct sockaddr_in6 from_ipv6;
    struct sockaddr * from_p;
    socklen_t from_len;

    if (addr_family == AF_INET) {
	from_p = (struct sockaddr *)&from_ipv4;
	from_len = sizeof(from_ipv4);
    } else if (addr_family == AF_INET6) {
	from_p = (struct sockaddr *)&from_ipv6;
	from_len = sizeof(from_ipv6);
    } else {
	return 0;
    }

    memset(from_p, 0, from_len);
    if ((ret = accept(sock, from_p, &from_len)) == -1) {
	pp_log_err("%s(): Couldn't accept on accept socket", fn);
	goto end;
    }

end:
    return ret;
}

BIO * pp_bio_new_accept(char * host_port, int addr_family)
{
    BIO * b = NULL;

    if ((b = BIO_new(BIO_pp_s_accept())) != NULL &&
	BIO_set_accept_port(b, host_port) == 1 &&
	BIO_ctrl(b, BIO_C_SET_ACCEPT, 3, (void *)addr_family) == 1) {

	return b;
    }

    if (b != NULL) BIO_free(b);
    return NULL;
}
