#include <pp/cd.h>
#include <cd_intern.h>

typedef struct {
    pp_cd_as_cd_t       *cd;
    char                *scope;
    char                *prefix;
    pp_mallocator_t     *alloc;
    pp_hash_t           *qid_hash;
} get_qid_ctx_t;

static int add_qid(pp_cd_as_t* asn, char* name, void* ctx);

static int 
cd_is_enumerator(const char *enumerator, char *name, pp_cd_as_cd_t *cd);

static int
get_qid_for_cd (pp_cd_as_cd_t* cd,  pp_cd_as_op_t* op, void* ctx) {
    int ret;
    get_qid_ctx_t c;

    c.cd = cd;
    c.scope = strdup("");
    c.prefix = NULL;
    c.alloc = cd->alloc;
    c.qid_hash = (pp_hash_t*)ctx;

    ret = pp_cd_op_cd(cd, op, &c);
    
    free(c.scope);
    return ret;
}

static int
get_qid_for_section (pp_cd_as_section_t* s, pp_cd_as_op_t* op, void* ctx) {
    return pp_cd_op_list(s->decls, op, ctx);
}

static int
get_qid_for_type_fund (pp_cd_as_type_t* type, pp_cd_as_op_t* op UNUSED, 
			void* ctx) {
    return add_qid((pp_cd_as_t*)type, type->name, ctx);
}

static int
get_qid_for_type_alias(pp_cd_as_type_alias_t* a, pp_cd_as_op_t* op UNUSED,
			void* ctx) {
    return add_qid((pp_cd_as_t*)a, a->base.name, ctx);
}

static int
get_qid_for_type_struct(pp_cd_as_type_struct_t* s, pp_cd_as_op_t* op,
			 void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*)ctx;
    int ret;
    if (0 == (ret = add_qid((pp_cd_as_t*)s, s->base.name, ctx))) {
	pp_cd_qid_push_scope(s->base.name, &c->scope);
	ret = pp_cd_op_list(s->elements, op, ctx);
	pp_cd_qid_pop_scope(c->scope);
    }
    return ret;
}
    
static int
get_qid_for_type_enum(pp_cd_as_type_enum_t* e, pp_cd_as_op_t* op,
		       void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*)ctx;
    int ret;
    if (0 == (ret = add_qid((pp_cd_as_t*)e, e->base.name, ctx))) {
	pp_cd_qid_push_scope(e->base.name, &c->scope);
	ret = pp_cd_op_list(e->elements, op, ctx);
	pp_cd_qid_pop_scope(c->scope);
    }
    return ret;
}

static int
get_qid_for_type_string_tmpl(pp_cd_as_type_string_tmpl_t* st,
			      pp_cd_as_op_t* op UNUSED, void* ctx) {
    return add_qid((pp_cd_as_t*)st, st->base.name, ctx);
}

static int
get_qid_for_type_int_tmpl(pp_cd_as_type_int_tmpl_t* it, 
			   pp_cd_as_op_t* op UNUSED, void* ctx) {
    return add_qid((pp_cd_as_t*)it, it->base.name, ctx);
}

static int
get_qid_for_type_choice_tmpl(pp_cd_as_type_choice_tmpl_t* ct,
			      pp_cd_as_op_t* op, void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*)ctx;
    int ret;
    unsigned long i;
    pp_cd_as_type_t* ce;

    if (0 == (ret = add_qid((pp_cd_as_t*)ct, ct->base.name, ctx))) {
        pp_cd_qid_push_scope(ct->base.name, &c->scope);
        for (i = 0; i < vector_size(ct->elements); ++i) {
            ce = (pp_cd_as_type_t*)vector_get(ct->elements, i);
            ret += ce->base.execute((pp_cd_as_t*)ce, op, ctx);
        }
        ret += add_qid((pp_cd_as_t*)ct->tmpl_type, "_c_", ctx);
        pp_cd_qid_pop_scope(c->scope);
    }
    return ret;
}
    
static int
get_qid_for_type_vector_tmpl(pp_cd_as_type_vector_tmpl_t* v,
			      pp_cd_as_op_t* op UNUSED, void* ctx) {
    return add_qid((pp_cd_as_t*)v, v->base.name, ctx);
}
    
static int
get_qid_for_type_vector_tmpl_spec(pp_cd_as_type_vector_tmpl_spec_t* v UNUSED, 
				   pp_cd_as_op_t* op UNUSED, void* ctx UNUSED) {
    assert(0); //todo
    return 1;
}

static int
get_qid_for_type_vector_tmpl_inst(pp_cd_as_type_vector_tmpl_inst_t* v UNUSED,
				   pp_cd_as_op_t* op UNUSED, void* ctx UNUSED) {
    assert(0); //todo
    return 1;
}
    
static int
get_qid_for_enumerator(pp_cd_as_enumerator_t* er, pp_cd_as_op_t* op UNUSED,
			void* ctx) {
    return add_qid((pp_cd_as_t*)er, er->value, ctx);
}
 
static int
get_qid_for_property(pp_cd_as_property_t* p UNUSED, pp_cd_as_op_t* op UNUSED, 
		      void* ctx UNUSED) {
    assert(0); // shouldn't be called, property have no semantics.. ;-)
    return 0;
}

/**
 * Checks elements of a vector assignment.
 * @param ve      the value expression having a list as value
 * @param velist  the value expression list of ve
 * @param op      the operation, needed to process the elements
 * @param ctx     the op's context
 */
static int
get_qid_for_vector_value_expr_list(pp_cd_as_value_expr_t* ve,
				    pp_cd_value_expr_list_t* velist,
				    pp_cd_as_type_vector_tmpl_t* vesym,
				    pp_cd_as_op_t* op, void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*) ctx;
    unsigned long vel_size = vector_size(velist);
    unsigned int qkeycomps, i;
    int ret = 0;

    assert(PP_CD_IS_TYPE(&vesym->base.base, AS_TYPE_VECTOR_TMPL));

	
    // name elements with an int index
    for (i = 0; i < vel_size; ++i) {
	pp_cd_cfg_key_idx_t k;
	pp_cd_cfg_key_t* key;
	pp_cd_quali_cfg_key_t* qkey;
		
	k.type = AS_FUND_TYPE_INT;
	k.value.intidx = i;
	key = pp_cd_cfg_key_create(c->alloc, AS_CFG_KEY_VECTOR, NULL, k);
		
	qkey = vector_new_with_alloc(NULL, 1, 
			       (vector_elem_del_func)pp_cd_cfg_key_destroy, 
				     c->alloc);
	vector_add(qkey, key);
	((pp_cd_as_value_expr_t*)vector_get(velist, i))->qkey = qkey;
    }
	
    // and exec op for all elements, push namespace only, not the name itself
    qkeycomps = vector_size(ve->qkey);// - 1;
    for (i = 0; i < qkeycomps; ++i) {
	char* key = ((pp_cd_cfg_key_t*)vector_get(ve->qkey, i))->key;
	pp_cd_qid_push_scope(key, &c->scope);
    }
    ret += pp_cd_op_list(velist, op, c);
    pp_cd_qid_pop_scope_n(c->scope, qkeycomps);

    return ret;
}

/**
 * Checks elements of a struct assignment.
 * @param ve      the value expression having a list as value
 * @param velist  the value expression list of ve
 * @param tlist   the element type list of the type definition of ve
 * @param op      the operation, needed to process the elements
 * @param ctx     the op's context
 */
static int
get_qid_for_named_value_expr_list(pp_cd_as_value_expr_t* ve,
				   pp_cd_value_expr_list_t* velist,
				   pp_cd_type_list_t* tlist,
				   pp_cd_as_op_t* op, void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*) ctx;
    unsigned long vel_size = vector_size(velist);
    pp_cd_as_value_expr_t* velem;
    pp_cd_as_type_t* telem;
    unsigned int qkeycomps, i;
    int named, ret = 0;
    
    if (vel_size > 0) {
	velem = (pp_cd_as_value_expr_t*)vector_get(velist, 0);
	named = (velem->qkey !=NULL);
	

	// if all are unnamed, check count and name them
	if (!named) {
	    if (vector_size(tlist) != vel_size) {
		pp_cd_yyerrorf(ve->base.pos,
			       "value list doesn't match type's element count"
			       ": %d != %d", vector_size(tlist), vel_size);
		return 1;
	    }
	
	    for (i = 0; i < vel_size; ++i) {
	        pp_cd_cfg_key_idx_t k;
		pp_cd_quali_cfg_key_t* qkey;
		telem = (pp_cd_as_type_t*)vector_get(tlist, i);
		assert(telem->base.type & AS_TYPE);
		velem = (pp_cd_as_value_expr_t*)vector_get(velist, i);

		k.type = 0;
		qkey = vector_new_with_alloc(NULL, 1, 
					     (vector_elem_del_func)pp_cd_cfg_key_destroy, 
					     c->alloc);
		vector_add(qkey, pp_cd_cfg_key_create(c->alloc, 
						      AS_CFG_KEY_SCALAR, 
						      pp_strdup(c->alloc, telem->name), k));
		velem->qkey = qkey;
	    }
	}
	
	// and exec op for all elements
	for (qkeycomps = 0, i = 0; i < vector_size(ve->qkey); ++i) {
	    char* key = ((pp_cd_cfg_key_t*)vector_get(ve->qkey, i))->key;
	    if (key) { // there may be index only components
		pp_cd_qid_push_scope(key, &c->scope);
		++qkeycomps;
	    }
	}
	ret += pp_cd_op_list(velist, op, c);
	pp_cd_qid_pop_scope_n(c->scope, qkeycomps);
    }

    return ret;
}

/*
 * loop over cfg key components and build a qid
 * magic is as follows:
 *  - if key has an index, resolve aliases and
 *    we check wether symbol is a vector and index type is correct
 *  - if key has no index, resolve aliases and vectors
 *    but vectors only in case it is not the last element
 */
static pp_cd_as_type_t*
get_qid_for_cfg_key(pp_cd_quali_cfg_key_t* qkey, int pos, 
		     get_qid_ctx_t* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*)ctx;
    unsigned long i = 0, l = vector_size(qkey);
    char* qid;
    pp_cd_cfg_key_t* k;
    pp_cd_as_type_t* sym;
    pp_cd_as_type_vector_tmpl_t* vsym;
    const char* no_type_msg = "config type not declared: '%s'";

    assert(l > 0);
    k = vector_get(qkey, i);
    qid = pp_cd_qualified_id_create(pp_mallocator_heap(), ctx->scope, k->key);
    do {
	if (PP_CD_IS_CFG_KEY(k, AS_CFG_KEY_VECTOR)) {
	    if (NULL == (sym = pp_cd_resolve_type_symbol(c->cd, qid,
							 PP_CD_RESOLVE_ALIAS))) {
		pp_cd_yyerrorf(pos, no_type_msg, qid);
		goto bailout;
	    } 
	    if (!PP_CD_IS_TYPE(&sym->base, AS_TYPE_VECTOR_TMPL)) {
		pp_cd_yyerrorf(pos,
			       "indexed key for type that is not a vector: '%s'",
			       qid);
		sym = NULL;
		goto bailout;
	    }
	    vsym = (pp_cd_as_type_vector_tmpl_t*)sym;
	    if (vsym->idx_type != k->idx.type) {
		pp_cd_yyerrorf(pos,
			       "index type differs from declared one: %s != %s",
			       pp_cd_fund_type_name(k->idx.type), 
			       pp_cd_fund_type_name(vsym->idx_type));
		sym = NULL;
		goto bailout;
	    }
            /* set sym to vector element type */
            sym = (pp_cd_as_type_t*)(vsym->elem_type);
	} else {
	    if (NULL == (sym = pp_cd_resolve_type_symbol(c->cd, qid,
							 PP_CD_RESOLVE_ALIAS))) {
		pp_cd_yyerrorf(pos, no_type_msg, qid);
		goto bailout;
	    }
	}
	
	if (++i >= l) break;
	
	k = vector_get(qkey, i);
	qid = pp_cd_qualified_id_add(pp_mallocator_heap(), qid, k->key);
    } while (1);
    
 bailout:
    free(qid);
    return sym;
}

static int
get_qid_for_value_expr(pp_cd_as_value_expr_t* ve, pp_cd_as_op_t* op,
			void* ctx) {
    int ret;
    pp_cd_as_type_t* vesym;
    pp_cd_type_list_t* tlist;
    pp_cd_quali_cfg_key_t* qkey = ve->qkey;
    pp_cd_cfg_key_t* qkey_last_elem;
    get_qid_ctx_t* c = (get_qid_ctx_t*)ctx;

    // check whether expression is named, unnamed expr must have named before
    if (ve->qkey == NULL) {
	pp_cd_yyerrorf(ve->base.pos, "no identifier for value expression");
	return 1;
    }
    
    // check if cfg key is valid 
    if (NULL == (vesym = get_qid_for_cfg_key(qkey, ve->base.pos, ctx))) {
        return 1;
    }
    ve->type = vesym;
    qkey_last_elem = (pp_cd_cfg_key_t*)vector_get(qkey, vector_size(qkey) - 1);
    
    switch (ve->base.type & AS_EXPR) {
      case AS_EXPR_VALUE_LIST:
	  
	  /* is it a struct? */
	  if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_STRUCT)) {
	      tlist = ((pp_cd_as_type_struct_t*)vesym)->elements;
	      ret = get_qid_for_named_value_expr_list(ve, ve->value.listval,
						       tlist, op, ctx);
	      
	      /* or is it a vector, but no concrete element */
	  } else if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL) &&
		     PP_CD_IS_CFG_KEY(qkey_last_elem, AS_CFG_KEY_SCALAR)) {
	      ret = get_qid_for_vector_value_expr_list(ve, ve->value.listval,
					(pp_cd_as_type_vector_tmpl_t*)vesym,
					op, ctx);
	      
	      /* or is it a concrete vector element, that has a struct type */
	  } else if (PP_CD_IS_CFG_KEY(qkey_last_elem, AS_CFG_KEY_VECTOR) &&
		     PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL) &&
		     PP_CD_IS_TYPE(&((pp_cd_as_type_vector_tmpl_t*)vesym)
				   ->elem_type->base, AS_TYPE_STRUCT)) {
	      pp_cd_as_type_vector_tmpl_t* v = 
		  (pp_cd_as_type_vector_tmpl_t*)vesym;
	      tlist = ((pp_cd_as_type_struct_t*)v->elem_type)->elements;
	      ret = get_qid_for_named_value_expr_list(ve, ve->value.listval,
						       tlist, op, ctx);

	      /* or is it something else, that is not valid */
	  } else {
	      pp_cd_yyerrorf(ve->base.pos,
			     "value expression list given for non-struct "
			     "or non-vector type");
	      return 1;
	  }
	  break;
	  
      case AS_EXPR_CHOICE_LITERAL:
	  // really choice ?
	  if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_CHOICE_TMPL)) {
	      tlist = ((pp_cd_as_type_choice_tmpl_t*)vesym)->elements;
	  } else {
	      pp_cd_yyerrorf(ve->base.pos,
			     "value expression list given for no-choice type");
	      return 1;
	  }
	  // and call checker for value expressions lists
	  ret = get_qid_for_named_value_expr_list(ve,
					   ve->value.choiceval.choicelist,
					   tlist, op, ctx);
	  break;
	  
      case AS_EXPR_FKT_DECL:
		  {
			assert(0); // TODO
		  }
	  break;
	  
	  case AS_EXPR_ENUM_LITERAL:
		  {
			  char *quis = pp_cd_qualikey_to_string(c->scope, qkey);
			  ret = !cd_is_enumerator(ve->value.stringval, quis, c->cd);
			  if (ret) {
				  pp_cd_yyerrorf(ve->base.pos,
					  "undefined enumerator '%s' for enum '%s'",
					  ve->value.stringval, quis);
			  }
			  free(quis);
		  }
          break;

      default:
	  ret = cd_as_match_expr_type(ve->type) != (int)ve->base.type;
          break;
    }
    return ret;
}

pp_cd_as_op_t pp_cd_as_op_get_qid = {
    get_qid_for_cd,
    get_qid_for_section,
    get_qid_for_type_fund,
    get_qid_for_type_alias,
    get_qid_for_type_struct,
    get_qid_for_type_enum,
    get_qid_for_type_string_tmpl,
    get_qid_for_type_int_tmpl,
    get_qid_for_type_choice_tmpl,
    get_qid_for_type_vector_tmpl,
    get_qid_for_type_vector_tmpl_spec,
    get_qid_for_type_vector_tmpl_inst,
    get_qid_for_enumerator,
    get_qid_for_property,
    get_qid_for_value_expr,
    NULL
};

static int add_qid(pp_cd_as_t* asn, char* name, void* ctx) {
    get_qid_ctx_t* c = (get_qid_ctx_t*) ctx;
    char* sn, *tmp;

    if (c->prefix == NULL) {
	sn = pp_cd_qualified_id_create(pp_mallocator_heap(), c->scope, name);
    } else {
	tmp = pp_cd_qualified_id_create(pp_mallocator_heap(), c->scope, name);
	sn = pp_cd_qualified_id_create(pp_mallocator_heap(), c->prefix, tmp);
	free(tmp);
    }

    pp_hash_set_entry(c->qid_hash, sn, asn, NULL);

    free(sn);
    return 0;
}

static int 
cd_is_enumerator(const char *enumerator, char *name, pp_cd_as_cd_t *cd) {
    char *qid;
    pp_cd_as_t *ens;

    qid = pp_cd_qualified_id_create(pp_mallocator_heap(), name, enumerator);
    ens = pp_cd_resolve_symbol(cd, qid, PP_CD_RESOLVE_SIMPLE);
    free(qid);
    if (ens && PP_CD_IS_MISC(ens, AS_ENUMERATOR))
        return 1;
    return 0;
}


