/*
 * operation that
 * - sementically checks a configuration description,
 * - builds up the type and the config-var hashes
 * - and resolves references to other types as needed by aliases and others
 *
 * tbr@peppercon.de
 * (c) 2004 Peppercon AG
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pp/cd.h>
#include "cd_intern.h"

typedef struct {
    pp_cd_as_cd_t* cd;
    char* scope;
    pp_mallocator_t* alloc;
    const char *prefix;
} semcheck_ctx_t;

static int add_symbol(pp_cd_as_t* asn, const char* name, void* ctx);

static int 
cd_is_enumerator(const char *enumerator, char *name, pp_cd_as_cd_t *cd);

static int
semcheck_for_vector_value_expr_list_core(pp_cd_as_value_expr_t* ve,
				         pp_cd_value_expr_list_t* velist,
                                         pp_cd_as_type_vector_tmpl_t* vesym,
                                         pp_cd_as_op_t* op, void* ctx);

static int
semcheck_for_cd (pp_cd_as_cd_t* cd,  pp_cd_as_op_t* op, void* ctx UNUSED) {
    int ret;
    semcheck_ctx_t c;
    /* assert that op has not been run before */
    assert(cd->symbols == NULL);
    
    cd->symbols = pp_hash_create(200);
    c.cd = cd;
    c.scope = strdup("");
    c.alloc = cd->alloc;
    c.prefix = NULL;

    ret = pp_cd_op_cd(cd, op, &c);
    
    free(c.scope);
    return ret;;
}

static int
semcheck_for_section (pp_cd_as_section_t* s, pp_cd_as_op_t* op, void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    return pp_cd_op_list(s->decls, op, c);
}

static int
semcheck_for_type_fund (pp_cd_as_type_t* type, pp_cd_as_op_t* op UNUSED, 
			void* ctx) {
    add_symbol((pp_cd_as_t*)type, type->name, ctx);
    return 0;
}

static int
semcheck_for_type_alias(pp_cd_as_type_alias_t* a, pp_cd_as_op_t* op UNUSED,
			void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    int ret = add_symbol((pp_cd_as_t*)a, a->base.name, ctx);
    assert(a->ref_type == NULL);
    if (NULL == (a->ref_type = pp_cd_resolve_type_symbol(c->cd,
						 a->ref_type_name,
						 PP_CD_RESOLVE_SIMPLE))) {
	pp_cd_yyerrorf(a->base.base.pos, "undefined aliased type: '%s'",
		       a->ref_type_name);
	++ret;
    }
    return ret;
}

static int
semcheck_for_type_struct(pp_cd_as_type_struct_t* s, pp_cd_as_op_t* op,
			 void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    int ret;
    if (0 == (ret = add_symbol((pp_cd_as_t*)s, s->base.name, ctx))) {
	pp_cd_qid_push_scope(s->base.name, &c->scope);
	ret = pp_cd_op_list(s->elements, op, ctx);
	pp_cd_qid_pop_scope(c->scope);
    }
    return ret;
}
    
static int
semcheck_for_type_enum(pp_cd_as_type_enum_t* e, pp_cd_as_op_t* op,
		       void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    int ret;
    if (0 == (ret = add_symbol((pp_cd_as_t*)e, e->base.name, ctx))) {
	pp_cd_qid_push_scope(e->base.name, &c->scope);
	ret = pp_cd_op_list(e->elements, op, ctx);
	pp_cd_qid_pop_scope(c->scope);
    }
    return ret;
}

static int
semcheck_for_type_string_tmpl(pp_cd_as_type_string_tmpl_t* st,
			      pp_cd_as_op_t* op UNUSED, void* ctx) {
    int ret;
    const int max = 16 * 1024; // max 16 k long strings
    ret = add_symbol((pp_cd_as_t*)st, st->base.name, ctx);
    if (st->min > max || st->max > max) {
	pp_cd_yyerrorf(st->base.base.pos,
		       "string min/max length exceeds limit of %d bytes",
		       max);
	++ret;
    }
    if (st->min > st->max) {
	pp_cd_yyerrorf(st->base.base.pos,
		       "string min length greater than max (%d > %d) ",
		       st->min, st->max);
	++ret;
    }
    return ret;
}

static int
semcheck_for_type_int_tmpl(pp_cd_as_type_int_tmpl_t* it, 
			   pp_cd_as_op_t* op UNUSED, void* ctx) {
    int ret;
    ret = add_symbol((pp_cd_as_t*)it, it->base.name, ctx);
    if (it->min > it->max) {
	pp_cd_yyerrorf(it->base.base.pos,
		       "integer min value greater than max (%d > %d) ",
		       it->min, it->max);
	++ret;
    }
    return ret;
}

static int
semcheck_for_type_choice_tmpl(pp_cd_as_type_choice_tmpl_t* ct,
			      pp_cd_as_op_t* op, void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    int ret;
    unsigned long i;

    if (0 == (ret = add_symbol((pp_cd_as_t*)ct, ct->base.name, ctx))) {
	
	// check whether choice type exists
	assert(ct->tmpl_type == NULL);
	if (NULL == (ct->tmpl_type = pp_cd_resolve_type_symbol(c->cd,
						     ct->tmpl_type_name,
						     PP_CD_RESOLVE_ALIAS))) {
	    pp_cd_yyerrorf(ct->base.base.pos,
			   "undefined choice type: '%s'", ct->tmpl_type_name);
	    ++ret;
	    // and is an enum
	} else if (!PP_CD_IS_TYPE(&ct->tmpl_type->base, AS_TYPE_ENUM)) {
	    pp_cd_yyerrorf(ct->base.base.pos,
			   "unsupported choice type, must be enum");
	    ++ret;
	} else {
	    // check whether choices enumerators are defined and correct
	    pp_cd_qid_push_scope(ct->base.name, &c->scope);
	    for (i = 0; i < vector_size(ct->elements); ++i) {
		pp_cd_as_type_t* ce =
		    (pp_cd_as_type_t*)vector_get(ct->elements, i);
		if (!cd_is_enumerator(ce->name, ct->tmpl_type_name, c->cd)) {
		    pp_cd_yyerrorf(ce->base.pos,
				   "undefined choice enumerator: '%s'",
				   ce->name);
		    ++ret;
		} else {
		    ret += ce->base.execute((pp_cd_as_t*)ce, op, ctx);
		}
	    }
	    // add a pseudo symbol for the choice enum
	    ret += add_symbol((pp_cd_as_t*)ct->tmpl_type,
                              PP_CD_CHOICE_SELECT_COMP_STR, ctx);
	    pp_cd_qid_pop_scope(c->scope);
	    // check whether all enumerators have been declared
	    /*
	      if (vector_size(ct->elements) <
	      vector_size(((pp_cd_as_type_enum_t*)ct->tmpl_type)->elements))
	      {
	      pp_cd_yyerrorf(ct->base.base.pos,
	      "choice enumerators incomplete");
	      ++ret;
	      }
	    */
	}
    }
    return ret;
}
    
static int
semcheck_for_type_vector_tmpl(pp_cd_as_type_vector_tmpl_t* v,
			      pp_cd_as_op_t* op UNUSED, void* ctx) {
    int ret;
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    const unsigned int max_vec_elems = 1024;
    
    ret = add_symbol((pp_cd_as_t*)v, v->base.name, ctx);
	
    // check whether element type exists
    assert(v->elem_type == NULL);
    if (NULL == (v->elem_type = pp_cd_resolve_type_symbol(c->cd,
							  v->elem_type_name,
							  PP_CD_RESOLVE_ALIAS))) {
	pp_cd_yyerrorf(v->base.base.pos,
		       "undefined element type: '%s'", v->elem_type_name);
	++ret;
    }

    // and bound is ok
    if (v->bound == 0 || v->bound > max_vec_elems) {
	pp_cd_yyerrorf(v->base.base.pos,
		       "vector bound beyond limits, 0 < bound <= %d",
		       max_vec_elems);
	++ret;
    }
    
    pp_cd_qid_push_scope(v->base.name, &c->scope);

    // add element type pseudo symbol
    ret += add_symbol((pp_cd_as_t*)v, PP_CD_VECT_ELEMS_COMP_STR, ctx);

    if(v->idx_type == AS_FUND_TYPE_INT) {
        // create and add pseudo symbol for the vector size
        v->size_type = pp_cd_as_create_int_tmpl(pp_cd_malloc, -1, NULL, 
                                                0, (int)max_vec_elems);
        ret += add_symbol((pp_cd_as_t*)v->size_type, 
                          PP_CD_VECT_SIZE_COMP_STR, ctx);
    }
    
    // add another pseudo symbol for the vector default which is of elem_type
    ret += add_symbol((pp_cd_as_t*)v->elem_type, PP_CD_VECT_DEF_COMP_STR, ctx);

    pp_cd_qid_pop_scope(c->scope);
    
    return ret;
}
    
static int
semcheck_for_type_vector_tmpl_spec(pp_cd_as_type_vector_tmpl_spec_t* v, 
				   pp_cd_as_op_t* op UNUSED, void* ctx) {
    // TODO
    int ret = add_symbol((pp_cd_as_t*)v, v->base.name, ctx);
    return ret;
}

static int
semcheck_for_type_vector_tmpl_inst(pp_cd_as_type_vector_tmpl_inst_t* v,
				   pp_cd_as_op_t* op UNUSED, void* ctx) {
    // TODO
    int ret = add_symbol((pp_cd_as_t*)v, v->base.name, ctx);
    return ret;
}
    
static int
semcheck_for_enumerator(pp_cd_as_enumerator_t* er, pp_cd_as_op_t* op UNUSED,
			void* ctx) {
    int ret = add_symbol((pp_cd_as_t*)er, er->value, ctx);
    return ret;
}
 
static int
semcheck_for_property(pp_cd_as_property_t* p UNUSED, pp_cd_as_op_t* op UNUSED, 
		      void* ctx UNUSED) {
    assert(0); // shouldn't be called, property have no semantics.. ;-)
    return 0;
}

/**
 * Checks elements of a vector assignment.
 * @param ve      the value expression having a list as value
 * @param velist  the value expression list of ve
 * @param op      the operation, needed to process the elements
 * @param ctx     the op's context
 */
static int
semcheck_for_vector_value_expr_list(pp_cd_as_value_expr_t* ve,
				    pp_cd_value_expr_list_t* velist,
				    pp_cd_as_type_vector_tmpl_t* vesym,
				    pp_cd_as_op_t* op, void* ctx) {
    assert(PP_CD_IS_TYPE(&vesym->base.base, AS_TYPE_VECTOR_TMPL));

    // check that vector idx type is int, so we can auto-name the elements
    if (vesym->idx_type != AS_FUND_TYPE_INT) {
	pp_cd_yyerrorf(ve->base.pos, "value expression list for associative"
		       " (non-int-index) vector");
	return 1;
    }
    
    return semcheck_for_vector_value_expr_list_core(ve, velist, vesym, op, ctx);
}
    
static int
semcheck_for_vector_value_expr_list_core(pp_cd_as_value_expr_t* ve,
				         pp_cd_value_expr_list_t* velist,
                                         pp_cd_as_type_vector_tmpl_t* vesym,
                                         pp_cd_as_op_t* op, void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*) ctx;
    unsigned long vel_size = vector_size(velist);
    unsigned int qkeycomps, i;
    int ret = 0;

    // check that there are no named elements
    for (i = 0; i < vel_size; ++i) {
	if (((pp_cd_as_value_expr_t*)vector_get(velist, i))->qkey != NULL){
	    pp_cd_yyerrorf(ve->base.pos,
			   "named value expression in vector value list");
	    return 1;
	}
    }

    // check for vectors max size
    if (vel_size > vesym->bound) {
	pp_cd_yyerrorf(ve->base.pos, "too many elements in vector assignment");
	return 1;
    }
	
    // name elements with an int index
    for (i = 0; i < vel_size; ++i) {
	pp_cd_cfg_key_idx_t k;
	pp_cd_cfg_key_t* key;
	pp_cd_quali_cfg_key_t* qkey;
		
	k.type = AS_FUND_TYPE_INT;
	k.value.intidx = i;
	key = pp_cd_cfg_key_create(c->alloc, AS_CFG_KEY_VECTOR, NULL, k);

	/* FIXME: MEMORY LEAK! qkey has to be freed somewhere... */
	qkey = vector_new_with_alloc(NULL, 1, 
			       (vector_elem_del_func)pp_cd_cfg_key_destroy, 
				     c->alloc);
	vector_add(qkey, key);
	((pp_cd_as_value_expr_t*)vector_get(velist, i))->qkey = qkey;
    }
	
    // and exec op for all elements, push namespace only, not the name itself
    {
        int j = pp_cd_qualified_id_count_comps(c->scope);
        qkeycomps = vector_size(ve->qkey);
        for (i = 0; i < qkeycomps; ++i) {
            c->scope = pp_cd_qualified_id_add_from_cfg_key(c->alloc, c->scope,
                                                       vector_get(ve->qkey, i));
        }
        ret += pp_cd_op_list(velist, op, c);
        pp_cd_qid_pop_scope_n(c->scope, 
                              pp_cd_qualified_id_count_comps(c->scope) - j);
    }
    return ret;
}

/**
 * Checks elements of a struct assignment.
 * @param ve      the value expression having a list as value
 * @param velist  the value expression list of ve
 * @param tlist   the element type list of the type definition of ve
 * @param op      the operation, needed to process the elements
 * @param ctx     the op's context
 */
static int
semcheck_for_named_value_expr_list(pp_cd_as_value_expr_t* ve,
				   pp_cd_value_expr_list_t* velist,
				   pp_cd_type_list_t* tlist,
				   pp_cd_as_op_t* op, void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*) ctx;
    unsigned long vel_size = vector_size(velist);
    pp_cd_as_value_expr_t* velem;
    pp_cd_as_type_t* telem;
    unsigned int qkeycomps, i;
    int named, ret = 0;
    
    if (vel_size > 0) {
	velem = (pp_cd_as_value_expr_t*)vector_get(velist, 0);
	named = (velem->qkey !=NULL);
	
	// if some named and some unnamed, error
	for (i = 1; i < vel_size; ++i) {
	    velem = (pp_cd_as_value_expr_t*)vector_get(velist, i);
	    if (named != (velem->qkey != NULL)) {
		pp_cd_yyerrorf(ve->base.pos,
			       "mixed named and annonymous assignments");
		return 1;
	    }
	}

	// if all are unnamed, check count and name them
	if (!named) {
	    if (vector_size(tlist) != vel_size) {
		pp_cd_yyerrorf(ve->base.pos,
			       "value list doesn't match type's element count"
			       ": %d != %d", vector_size(tlist), vel_size);
		return 1;
	    }
	
	    for (i = 0; i < vel_size; ++i) {
	        pp_cd_cfg_key_idx_t k;
		pp_cd_quali_cfg_key_t* qkey;
		telem = (pp_cd_as_type_t*)vector_get(tlist, i);
		assert(telem->base.type & AS_TYPE);
		velem = (pp_cd_as_value_expr_t*)vector_get(velist, i);

		k.type = 0;
		qkey = vector_new_with_alloc(NULL, 1, 
					     (vector_elem_del_func)pp_cd_cfg_key_destroy, 
					     c->alloc);
		vector_add(qkey, pp_cd_cfg_key_create(c->alloc, 
						      AS_CFG_KEY_SCALAR, 
						      pp_strdup(c->alloc, telem->name), k));
		velem->qkey = qkey;
	    }
	}
	
	// and exec op for all elements
        {
            int j = pp_cd_qualified_id_count_comps(c->scope);
            qkeycomps = vector_size(ve->qkey);
            for (i = 0; i < qkeycomps; ++i) {
                c->scope = pp_cd_qualified_id_add_from_cfg_key(c->alloc,
                                             c->scope, vector_get(ve->qkey, i));
            }
            ret += pp_cd_op_list(velist, op, c);
            pp_cd_qid_pop_scope_n(c->scope, 
                                  pp_cd_qualified_id_count_comps(c->scope) - j);
        }
    }

    return ret;
}

/*
 * loop over cfg key components and build a qid
 * magic is as follows:
 *  - if key has an index, resolve aliases and
 *    we check wether symbol is a vector and index type is correct
 *  - if key has no index, resolve aliases and vectors
 *    but vectors only in case it is not the last element
 */
static pp_cd_as_type_t*
semcheck_for_cfg_key(pp_cd_quali_cfg_key_t* qkey, int pos, 
		     semcheck_ctx_t* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;
    unsigned long i = 0, l = vector_size(qkey);
    char* qid;
    pp_cd_cfg_key_t* k;
    pp_cd_as_type_t* sym;
    pp_cd_as_type_vector_tmpl_t* vsym;
    const char* no_type_msg = "config type not declared: '%s'";

    assert(l > 0);
    k = vector_get(qkey, i);
    qid = pp_cd_qualified_id_create_from_cfg_key(pp_mallocator_heap(),
                                                 ctx->scope, k);
    do {
	if (PP_CD_IS_CFG_KEY(k, AS_CFG_KEY_VECTOR)) {
            pp_cd_as_id_t idx_type;
            
	    if (NULL == (sym = pp_cd_resolve_type_symbol(c->cd, qid,
							 PP_CD_RESOLVE_ALIAS))) {
		pp_cd_yyerrorf(pos, no_type_msg, qid);
		goto bailout;
	    } 
	    if (!PP_CD_IS_TYPE(&sym->base, AS_TYPE_VECTOR_TMPL)) {
		pp_cd_yyerrorf(pos,
			       "indexed key for type that is not a vector: '%s'",
			       qid);
		sym = NULL;
		goto bailout;
	    }
	    vsym = (pp_cd_as_type_vector_tmpl_t*)sym;
            /* for nested vectors, we have to dive into the elem-type */
            if(PP_CD_IS_TYPE(&vsym->elem_type->base, AS_TYPE_VECTOR_TMPL)) {
                pp_cd_as_type_t* tsym = pp_cd_resolve_type_symbol(c->cd, qid,
                                                          PP_CD_RESOLVE_VECTOR);
                idx_type = ((pp_cd_as_type_vector_tmpl_t*)tsym)->idx_type;
            } else {
                idx_type = vsym->idx_type;
            }
	    if(idx_type != k->idx.type) {
		pp_cd_yyerrorf(pos,
			       "index type differs from declared one: %s != %s",
			       pp_cd_fund_type_name(k->idx.type), 
			       pp_cd_fund_type_name(idx_type));
		sym = NULL;
		goto bailout;
	    }
            /* set sym to vector element type */
            sym = (pp_cd_as_type_t*)(vsym->elem_type);
	} else {
	    if (NULL == (sym = pp_cd_resolve_type_symbol(c->cd, qid,
							 PP_CD_RESOLVE_ALIAS))) {
		pp_cd_yyerrorf(pos, no_type_msg, qid);
		goto bailout;
	    }
	}
	
	if (++i >= l) break;
	
	k = vector_get(qkey, i);
	qid = pp_cd_qualified_id_add_from_cfg_key(pp_mallocator_heap(), qid, k);
    } while (1);
    
 bailout:
    free(qid);
    return sym;
}

static int
semcheck_for_value_expr(pp_cd_as_value_expr_t* ve, pp_cd_as_op_t* op,
			void* ctx) {
    int ret;
    pp_cd_as_type_t* vesym;
    pp_cd_type_list_t* tlist;
    pp_cd_quali_cfg_key_t* qkey = ve->qkey;
    pp_cd_cfg_key_t* qkey_last_elem;
    semcheck_ctx_t* c = (semcheck_ctx_t*)ctx;

    // check whether expression is named, unnamed expr must have named before
    if (qkey == NULL) {
	pp_cd_yyerrorf(ve->base.pos, "no identifier for value expression");
	return 1;
    }
    
    // check if cfg key is valid 
    if (NULL == (vesym = semcheck_for_cfg_key(qkey, ve->base.pos, ctx))) {
        return 1;
    }
    ve->type = vesym;
    qkey_last_elem = (pp_cd_cfg_key_t*)vector_get(qkey, vector_size(qkey) - 1);
    
    switch (ve->base.type & AS_EXPR) {
      case AS_EXPR_VALUE_LIST:
	  /* is it a struct? */
	  if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_STRUCT)) {
	      tlist = ((pp_cd_as_type_struct_t*)vesym)->elements;
	      ret = semcheck_for_named_value_expr_list(ve, ve->value.listval,
						       tlist, op, ctx);
	      
	  /* or is it a vector, but no concrete element */
	  } else if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL) &&
		     PP_CD_IS_CFG_KEY(qkey_last_elem, AS_CFG_KEY_SCALAR)) {
	      ret = semcheck_for_vector_value_expr_list(ve, ve->value.listval,
					(pp_cd_as_type_vector_tmpl_t*)vesym,
					op, ctx);
	      
          /* or is it a nested vector */
	  } else if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL) &&
		     PP_CD_IS_CFG_KEY(qkey_last_elem, AS_CFG_KEY_VECTOR)) {
              ret = semcheck_for_vector_value_expr_list_core(ve,
                                            ve->value.listval,
                                            (pp_cd_as_type_vector_tmpl_t*)vesym,
                                            op, ctx);	      
	  
          /* or is it a concrete vector element, that has a struct type */
	  } else if (PP_CD_IS_CFG_KEY(qkey_last_elem, AS_CFG_KEY_VECTOR) &&
		     PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL) &&
		     PP_CD_IS_TYPE(&((pp_cd_as_type_vector_tmpl_t*)vesym)
				   ->elem_type->base, AS_TYPE_STRUCT)) {
	      pp_cd_as_type_vector_tmpl_t* v = 
		  (pp_cd_as_type_vector_tmpl_t*)vesym;
	      tlist = ((pp_cd_as_type_struct_t*)v->elem_type)->elements;
	      ret = semcheck_for_named_value_expr_list(ve, ve->value.listval,
						       tlist, op, ctx);

	  /* or is it something else, that is not valid */
	  } else {
	      pp_cd_yyerrorf(ve->base.pos,
			     "value expression list given for non-struct "
			     "or non-vector type");
	      return 1;
	  }
	  break;
	  
      case AS_EXPR_CHOICE_LITERAL:
	  // really choice ?
	  if (PP_CD_IS_TYPE(&vesym->base, AS_TYPE_CHOICE_TMPL)) {
	      tlist = ((pp_cd_as_type_choice_tmpl_t*)vesym)->elements;
	  } else {
	      pp_cd_yyerrorf(ve->base.pos,
			     "value expression list given for no-choice type");
	      return 1;
	  }
	  // and call checker for value expressions lists
	  ret = semcheck_for_named_value_expr_list(ve,
					   ve->value.choiceval.choicelist,
					   tlist, op, ctx);
	  break;
	  
      case AS_EXPR_FKT_DECL:
	  {
	    // checker for value function declarations
	    pp_cd_as_fkt_decl_t *fd = ve->value.fktval;
	    c->prefix = "_val_";
	    // why add value expression?	  if (0 == (ret = add_symbol((pp_cd_as_t*)ve, fd->base.name, ctx))) {
	    if (0 == (ret = add_symbol((pp_cd_as_t*)fd, fd->base.name, ctx))) {
		  pp_cd_type_list_t *params = fd->params;
		if( params ) {
		      pp_cd_qid_push_scope( fd->base.name, &c->scope );
		    ret += pp_cd_op_list( params, op, ctx );
		    pp_cd_qid_pop_scope( c->scope );
		}
	    }
	    c->prefix = NULL;
	  }
	  break;
	  
      case AS_EXPR_ENUM_LITERAL:
	  {
	    char *quis = pp_cd_qualikey_to_string(c->scope, qkey);
	    ret = !cd_is_enumerator(ve->value.stringval, quis, c->cd);
	    if (ret) {
		  pp_cd_yyerrorf(ve->base.pos,
		                 "undefined enumerator '%s' for enum '%s'",
			         ve->value.stringval, quis);
	    }
	    free(quis);
	  }
          break;

      default:
          /* care about nested vectors */
          if(PP_CD_IS_TYPE(&vesym->base, AS_TYPE_VECTOR_TMPL)) {
              ret = cd_as_match_expr_type((pp_cd_as_type_t*)
                                (&((pp_cd_as_type_vector_tmpl_t*)vesym)
                                ->elem_type->base.type)) != (int)ve->base.type;
          } else {
              ret = cd_as_match_expr_type(ve->type) != (int)ve->base.type;
          }
          if(ret) {
	      pp_cd_yyerrorf(ve->base.pos,
                             "type mismatch for value expression");
assert(0);
          }
          break;
    }
    return ret;
}

pp_cd_as_op_t pp_cd_as_op_semcheck = {
    semcheck_for_cd,
    semcheck_for_section,
    semcheck_for_type_fund,
    semcheck_for_type_alias,
    semcheck_for_type_struct,
    semcheck_for_type_enum,
    semcheck_for_type_string_tmpl,
    semcheck_for_type_int_tmpl,
    semcheck_for_type_choice_tmpl,
    semcheck_for_type_vector_tmpl,
    semcheck_for_type_vector_tmpl_spec,
    semcheck_for_type_vector_tmpl_inst,
    semcheck_for_enumerator,
    semcheck_for_property,
    semcheck_for_value_expr,
    NULL
};

static int add_symbol(pp_cd_as_t* asn, const char* name, void* ctx) {
    semcheck_ctx_t* c = (semcheck_ctx_t*) ctx;
    char* sn, *tmp;
    pp_cd_as_t* ds;
    int ret;

    if (c->prefix == NULL) {
	sn = pp_cd_qualified_id_create(pp_mallocator_heap(), c->scope, name);
    } else {
	tmp = pp_cd_qualified_id_create(pp_mallocator_heap(), c->scope, name);
	sn = pp_cd_qualified_id_create(pp_mallocator_heap(), c->prefix, tmp);
	free(tmp);
    }

    if (NULL != (ds = (pp_cd_as_t*)pp_hash_get_entry(c->cd->symbols, sn))) {
	pp_cd_yyerrorf(asn->pos, "symbol already declared: '%s'", sn);
	pp_cd_yyerrorf(ds->pos, "first declared here");
	ret = 1;
    } else {
	pp_hash_set_entry(c->cd->symbols, sn, asn, NULL);

#if __TWEB_DEBUG
//    printf("add symbol %s %p\n", sn, asn);
#endif

	ret = 0;
    }
    free(sn);
    return ret;
}

static int 
cd_is_enumerator(const char *enumerator, char *name, pp_cd_as_cd_t *cd) {
    char *qid;
    pp_cd_as_t *ens;

    qid = pp_cd_qualified_id_create(pp_mallocator_heap(), name, enumerator);
    ens = pp_cd_resolve_symbol(cd, qid, PP_CD_RESOLVE_SIMPLE);
    free(qid);
    if (ens && PP_CD_IS_MISC(ens, AS_ENUMERATOR))
        return 1;
    return 0;
}


