%{
#include <string.h>
#if !defined(_WIN32)
# include <stdarg.h>
#endif
#include <unistd.h>
#include <pp/cd.h>
#include "cdlparser.tab.h"

#define YY_NO_UNPUT
#define YY_STACK_USED 0
#define YY_ALWAYS_INTERACTIVE 0
#define YY_NEVER_INTERACTIVE 0
#define YY_MAIN 0
#define YYERROR_VERBOSE
#define YYDEBUG 1

extern int pp_cd_yylex   (void);

extern int pp_cd_yyparse(void);
extern int pp_cd_yydebug;

int pp_cd_yyerror(const char*);
int pp_cd_yyerror_base(const int pos, const char* text);

pp_cd_as_cd_t* pp_cd_as_yy_cd;
unsigned int pp_cd_yy_pos;
pp_mallocator_t* pp_cd_malloc;

typedef struct {
  YY_BUFFER_STATE buffer;
  unsigned int pos;
} incl_state_t;

static char* process_strlit(char* text);
static incl_state_t* create_incl_state(YY_BUFFER_STATE b, unsigned int pos);
static FILE* new_file(const char* fname, unsigned int* pos);
static void push_input(FILE* in, unsigned int inpos);
static int pop_input(void);
static const char* filename;
static int noerrors;
static int comnesting;
static vector_t incl_stack;
static vector_t filenames;
%}

%option noyywrap
%x COMMENT
%x INCLUDE

NUM      -?[0-9]+
HEXNUM   0[xX][0-9a-fA-F]+
ID       [_a-zA-Z][_a-zA-Z0-9]*

%%

"/*"                { BEGIN(COMMENT); 
                      comnesting = 1; }

<COMMENT>"/*"       { comnesting++; }

<COMMENT>"*/"       { if (--comnesting == 0) 
                         BEGIN(INITIAL); }

<COMMENT>.          /* eat the comment */

<COMMENT>\n         { ++pp_cd_yy_pos; }

<COMMENT><<EOF>>    { pp_cd_yyerror("unexpected EOF in comment");
                      yyterminate();
                    }
                    
\/\/.*\n            { /* eat the comment line */
                      ++pp_cd_yy_pos;
                    }

"include"           { BEGIN(INCLUDE); }

"boolean"           { return BOOLEAN_KW; }
"int"               { return INT_KW; }
"string"            { return STRING_KW; }
"enum"              { return ENUM_KW; }
"struct"            { return STRUCT_KW; }
"choice"            { return CHOICE_KW; }
"vector"            { return VECTOR_KW; }
"fktdecl"           { return FKTDECL_KW; }
"typedef"           { return TYPE_KW; }
"typedefs"          { return TYPES_KW; }
"configs"           { return CONFIGS_KW; }
"values"            { return VALUES_KW; }

"true"              { pp_cd_yylval.number = 1;
                      return BOOLEAN_LIT; }

"false"             { pp_cd_yylval.number = 0;
                      return BOOLEAN_LIT; }

{ID}	            { pp_cd_yylval.string = strdup(pp_cd_yytext);
	              return ID;
	            }

{NUM}               { pp_cd_yylval.number = atoi(pp_cd_yytext);
	              return INT_LIT;
	            }

{HEXNUM}            { pp_cd_yylval.number = strtol(pp_cd_yytext, NULL, 16);
                      return INT_LIT;
                    }

"<"|">"|";"|":"|","|"{"|"}"|"("|")"|"."|"="|"["|"]" { return pp_cd_yytext[0]; }

\"([^\"\n]|\\\"|\\\n)*\" { pp_cd_yylval.string = process_strlit(pp_cd_yytext);
	              return STRING_LIT;
	            }

\"[^\"\n]*\n        { pp_cd_yylval.string = pp_cd_yytext;
                      pp_cd_yyerror("unterminated string literal");
                      ++pp_cd_yy_pos;
	              return STRING_LIT;
	            }

<INCLUDE>[ \t]*     /* eat up whitespace */

<INCLUDE>[^ \t\n]+  { FILE* in; unsigned int inpos;
                      if (NULL == (in = new_file(pp_cd_yytext, &inpos))) {
                         pp_cd_yyerrorf(pp_cd_yy_pos, 
                                        "can't open include '%s': %s",
                                        pp_cd_yytext, strerror(errno));
                      } else {
                         push_input(in, inpos);
                      }
                      BEGIN(INITIAL);
                    }

\n                  { ++pp_cd_yy_pos; }

[ \t]+              /* eat up whitespace */

.                   { pp_cd_yyerrorf(++pp_cd_yy_pos, "invalid character: '%c'",
			             pp_cd_yytext[0]); }

<<EOF>>             { if (pop_input() == 0)
                         yyterminate();
                    }
%%

pp_cd_as_cd_t* pp_cd_parse_cdl(pp_mallocator_t* a, const char* fname) {
  pp_cd_as_cd_t* ret;
  pp_cd_yydebug = 0;
  filename = fname;
  pp_cd_malloc = a;
  yy_init = 1;

  vector_new(&filenames, 10, free);

  if (NULL != (pp_cd_yyin = new_file(filename, &pp_cd_yy_pos))) {
    noerrors = 0;
    vector_new(&incl_stack, 10, free);
    
    if (0 != pp_cd_yyparse() || noerrors > 0 ||
	((noerrors = pp_cd_execute_op((pp_cd_as_t*)pp_cd_as_yy_cd, 
                                      &pp_cd_as_op_semcheck, 0)) > 0)) {
       fprintf(stderr, "%s: encountered %d error%s\n", 
	               filename, noerrors, noerrors > 1 ? "s" : "");
       pp_cd_as_destroy(a, (pp_cd_as_t*) pp_cd_as_yy_cd);
       ret = pp_cd_as_yy_cd = NULL;
       errno = EINVAL;
    } else {
       ret = pp_cd_as_yy_cd;
       pp_cd_as_yy_cd = NULL;
    }
    vector_delete(&incl_stack);
  } else {
    perror (filename);
    ret = NULL;
  }
  vector_delete(&filenames);
  return ret;
}

int pp_cd_yyerror(const char* text) {
  pp_cd_yyerror_base(pp_cd_yy_pos, text);
  return 0;
}

int pp_cd_yyerror_base(const int pos, const char* text) {
  noerrors++;
  fprintf(stderr, "%s:%d: ERROR: %s.\n", 
         (char*)vector_get(&filenames, pos >> 24), pos & 0xffffff, text);
  return 0;
}

int pp_cd_yyerrorf(const int pos, const char *format, ...) {
  char* buf;
  va_list ap, aq;

  va_start(ap, format);
  va_copy(aq, ap);
  buf = alloca(vsnprintf(NULL, 0, format, ap) + 1);
  va_end(ap);
  vsprintf(buf, format, aq);
  va_end(aq);
  pp_cd_yyerror_base(pos, buf);
  return 0;
}

/* 
 * processes escapes and cuts off the leading and trailing quotes
 */
static char* process_strlit(char* text) {
  char *str, *s, c;
  int len = strlen(text) -2;
  ++text; text[len] = '\0';
  str = s = malloc(len + 1);
  while((c = *text++) != '\0') {
    if (c == '\\') {
      c = *text++;
      if (c == '\0') break;
      if (c == '\n') continue;
    }
    *s++ = c;
  }
  *s = '\0';
  return str;
}

#ifdef _WIN32
#define PATH_SEP '\\'
#else
#define PATH_SEP '/'
#endif

static FILE* new_file(const char* fname, unsigned int* pos) {
  FILE* in;
  const char *new_filename = fname;
  char new_fname[FILENAME_MAX];
  char *tmp;
  if(*fname != PATH_SEP && vector_size(&filenames) > 0) {
    /* no absolute path, get working dir from current_file */
    tmp = vector_get(&filenames, vector_size(&filenames) - 1);
    strcpy(new_fname, tmp);
    tmp = strrchr(new_fname, PATH_SEP);
    if(tmp) {
        if(tmp - new_fname + strlen(fname) + 2 > FILENAME_MAX)
            return NULL;
        strcpy(tmp + 1, fname);
        new_filename = new_fname;
    }
  }
  if (NULL != (in = fopen(new_filename, "r"))) {
    vector_add(&filenames, strdup(new_filename));
    *pos = ((vector_size(&filenames) - 1) << 24) + 1;
  }
  return in;
}

static void push_input(FILE* in, unsigned int inpos) {
  vector_add(&incl_stack, create_incl_state(YY_CURRENT_BUFFER, pp_cd_yy_pos));
  pp_cd_yy_switch_to_buffer(pp_cd_yy_create_buffer(in, YY_BUF_SIZE));
  pp_cd_yy_pos = inpos;
}

static int pop_input() {
  incl_state_t* is;
  int ss;

  fclose(pp_cd_yyin);
  pp_cd_yy_delete_buffer(YY_CURRENT_BUFFER);
  ss = vector_size(&incl_stack);
  if (ss == 0) return 0;

  is = (incl_state_t*)vector_get(&incl_stack, ss - 1);
  pp_cd_yy_switch_to_buffer(is->buffer);
  pp_cd_yy_pos = is->pos;
  vector_remove(&incl_stack, ss - 1);
  return 1;
}

static incl_state_t* create_incl_state(YY_BUFFER_STATE buffer, 
                                       unsigned int pos) {
  incl_state_t* s = malloc(sizeof(incl_state_t));
  s->buffer = buffer;
  s->pos = pos;
  return s;
}
