/**
 * A profile is a Configuration Description instance,
 * i.e. it contains values of configuration keys
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de>
 */

#include <stdlib.h>
#include <assert.h>
#include <pp/mallocator.h>
#include "acl_cache.h"
#include <pp/profile.h>

static int profile_uid_cnt = 0;
pthread_mutex_t profile_uid_cnt_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;

pp_profile_t* pp_profile_create(pp_mallocator_t* a, pp_profile_type_t type, 
                                const char* name) {
    pp_profile_t* p;

    assert(type);
    assert(name);
    
    p = (pp_profile_t *)pp_malloc(a, sizeof(pp_profile_t));

    MUTEX_LOCK(&profile_uid_cnt_mtx);
    p->id = ++profile_uid_cnt;
    MUTEX_UNLOCK(&profile_uid_cnt_mtx);

    p->type = type;
    p->name = pp_strdup(a, name);
    p->configs = pp_dict_str_new_with_alloc(a);
    p->ref_cnt = 1;
    p->allocator = a;
    p->acl = NULL;
    p->ldap_acl = NULL;

    pp_mallocator_mutex_init(a, &p->mtx, PP_MUTEX_RECURSIVE);
    pp_mallocator_mutex_init(a, &p->ref_cnt_mtx, PP_MUTEX_NORMAL);

    return p;
}

/*
 * profiles are reference counted,
 * so duplication means increasing the ref counter by one
 */
pp_profile_t* pp_profile_duplicate(pp_profile_t* prof) {
    pp_mutex_lock(&prof->ref_cnt_mtx);
    assert(prof->ref_cnt > 0);
    prof->ref_cnt++;
    pp_mutex_unlock(&prof->ref_cnt_mtx);
    return prof;
}

/*
 * profiles are reference counted,
 * so destrcution means decreasing the ref counter by one
 * and deleting them if it has reached zero
 */
void pp_profile_release(void * _p) {
    pp_profile_t * prof = (pp_profile_t *)_p;
    if (!prof) return;
    pp_mutex_lock(&prof->ref_cnt_mtx);
    assert(prof->ref_cnt > 0);
    if(--prof->ref_cnt == 0) {
        // we are the one and only, so unlock is safe
        pp_mutex_unlock(&prof->ref_cnt_mtx);
        pp_mutex_destroy(&prof->mtx);
        pp_mutex_destroy(&prof->ref_cnt_mtx);
        pp_free(prof->allocator, prof->name);
        pp_dict_destroy(prof->configs);
        pp_hash_delete(prof->acl);
	pp_acl_destroy(prof->ldap_acl);
        pp_free(prof->allocator, prof);
        return;
    }
    pp_mutex_unlock(&prof->ref_cnt_mtx);
}

/**
 * gets ACL for user uname from profile prof
 * perhaps there should be a default ACL later on, then we have to decide a
 * different hash-key than uname (e.g. qid?)
 * acl is locked until pp_profile_release_acl
 */
int pp_profile_get_acl(pp_acl_t **acl, pp_profile_t *prof, int gid) {
    pp_acl_t* nacl;
    if(prof->ldap_acl) {
	*acl=prof->ldap_acl;
	pp_mutex_lock(&prof->ldap_acl->mtx);
	return PP_SUC;
    }
    if(!(prof->acl))
        prof->acl = pp_hash_create_i(10);
    if(NULL == (nacl = acl_cache_get(prof->acl, gid)))
        return PP_ERR;
    pp_mutex_lock(&nacl->mtx);
    pp_profile_duplicate(prof);
    nacl->parent = (void*)prof;
    *acl = nacl;
    return PP_SUC;
}

/**
 * stores group-acl in profile
 * WARNING! This function is called by pp_ldap at profile creation time
 *          If using is in other context, care about thread safety!
 **/
void pp_profile_set_acl(pp_acl_t *acl, pp_profile_t *prof, int gid) {
    if(!(prof->acl))
        prof->acl = pp_hash_create_i(10);
    acl_cache_set(prof->acl, gid, acl);
}

void pp_profile_set_ldap_acl(pp_profile_t *prof,pp_acl_t *acl)
{
    prof->ldap_acl=acl;
}

/**
 * releases mutex lock from ACL
 * if there is a parent (usually should) which may only be a pp_profile at the
 * moment, it is released as well (pp_profile_get_acl duplicates prof)
 **/
void pp_profile_release_acl(void *v) {
    pp_acl_t *acl;

    if(!v)
        return;
    acl = (pp_acl_t*)v;
    pp_mutex_unlock(&acl->mtx);
    if(acl->parent)
        pp_profile_release(acl->parent);
}

#ifdef PP_CFG_DEBUG
#include <pp/cfg.h>
#include <pp/cfgx.h>

static void
profile_print_recursive(int recursion, vector_t* stack, const pp_dict_t* dict) {
    pp_ditr_t* ditr = pp_ditr_new(dict);
    const cfg_entry_t* ce;
    char *key;
    int i;
    
    /* we better assert max recursions, you never know */
    recursion++;
    assert(recursion < 10);
    
    for (pp_ditr_first(ditr); pp_ditr_valid(ditr); pp_ditr_next(ditr)) {
        key = (char*)pp_ditr_key(ditr);
        vector_add(stack, key);
	
	ce = pp_ditr_data(ditr);
	switch (ce->type) {
	  case CFG_DICT:
	      profile_print_recursive(recursion, stack, ce->value.dict);
	      break;
	  case CFG_STR:
              for(i = 0; i < recursion - 1; ++i)
                  printf("%s.", (char*)vector_get(stack, i));
              printf("%s = %s\n", key, ce->value.string);
	      break;
	  default:
	      assert(0);
	}
	
        vector_remove(stack, recursion - 1);
    }

    pp_ditr_destroy(ditr);
}

void
pp_profile_print(const pp_profile_t* prof) {
    vector_t stack;

    printf("PROFILE %s (id %d)\n", prof->name, prof->id);

    vector_new(&stack, 10, NULL);
    profile_print_recursive(0, &stack, prof->configs);

    vector_delete(&stack);
}

static int profile_validate_dict(const pp_dict_t* dict, int recursion) {
    pp_ditr_t* ditr = pp_ditr_new(dict);
    const cfg_entry_t* ce;
    int ret = PP_SUC;

    /* we better assert max recursions, you never know */
    ++recursion;
    assert(recursion < 25);
    
    if(!pp_ditr_first(ditr)) {
        ret = PP_ERR;
    } else for(; ret == PP_SUC && pp_ditr_valid(ditr); pp_ditr_next(ditr)) {
//        printf("processing key '%s' @ recursion level %d\n",
//               (char*)pp_ditr_key(ditr), recursion);
        ce = pp_ditr_data(ditr);
	switch (ce->type) {
            case CFG_DICT:
                ret = profile_validate_dict(ce->value.dict, recursion);
                break;
            case CFG_STR:
                ret = PP_SUC;
                break;
            default:
                ret = PP_ERR;
                break;
	}
    }
    
    pp_ditr_destroy(ditr);
    
    return ret;
}

int pp_profile_validate_type(pp_profile_type_t type) {
    pp_profile_t* p;
    int ret;

    if (PP_ERR == (ret = pp_cfg_get_profile(&p, type, NULL, NULL)))
	return ret;
    
    /* validate configs dict */
    ret = profile_validate_dict(p->configs, 0);
    
    return ret;
}
#endif /* PP_CFG_DEBUG */

int pp_profile_invalidate_acl_cache(pp_profile_t *prof) {
    pp_acl_t* acl;
    
    assert(prof);
    
    if(prof->ldap_acl) {
	/* not for LDAP!!! */
	return PP_ERR;
    }
    
    if(!prof->acl) {
        /* no profile cache yet, easy one */
        return PP_SUC;
    }
    
    for(acl = pp_hash_get_first_entry_i(prof->acl); acl;
        acl = pp_hash_get_next_entry_i(prof->acl)){
        /* be sure, noone else uses that ACL right now... */
        /* TODO race condition? */
        pp_mutex_lock(&acl->mtx);
    }
    
    /* ... and throw them away */
    pp_hash_clear_i(prof->acl);
    
    return PP_SUC;
}
