/**
 * Profile cache manages loading of profiles
 * from the local file system if necessary
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de>
 */

#include <stdlib.h>

#include <pp/base.h>
#include "profile_cache.h"
#ifndef PP_BOARD_PEMX
#include <liberic_config.h>
#endif


/*
 * the profile cache
 */
struct pp_profile_cache_s {
    pp_hash_t* profiles;
    pp_profile_cache_load_fkt load_profile;
    pp_profile_cache_diff_fkt diff_profile;
    pthread_mutex_t mtx;
};

/*
 * this structure is the hash entry of the local
 * profile cash. Together with the profile we need to keep
 * the file modification time to trigger reloads in case
 * somebody has changed the profile on storage
 */
typedef struct {
    time_t filemtime;
    pp_profile_t* profile;
} local_profile_entry_t;

static local_profile_entry_t* local_profile_entry_create(pp_profile_t* prof,
                                                         time_t filemtime);
static void local_profile_entry_destroy(void* p);

/*
 * Private methods
 * ----------------
 */

static local_profile_entry_t* local_profile_entry_create(pp_profile_t* prof,
                                                         time_t filemtime) {
    local_profile_entry_t* pe;

    pe = malloc(sizeof(local_profile_entry_t));
    pe->profile = prof;
    pe->filemtime = filemtime;

    return pe;
}

static void local_profile_entry_destroy(void* p) {
    local_profile_entry_t* pe = (local_profile_entry_t*)p;
    if (pe != NULL) {
        pp_profile_release(pe->profile);
        free(pe);
    }
    return;
}

/*
 * Public methods
 * ----------------
 */

pp_profile_cache_t*
pp_profile_cache_create(pp_profile_cache_load_fkt load,
                        pp_profile_cache_diff_fkt diff) {

    pp_profile_cache_t* pc = malloc(sizeof(pp_profile_cache_t));

    pc->load_profile = load;
    pc->diff_profile = diff;
    pc->profiles = pp_hash_create(10);

    /**
     * for updating profiles (external changes) we need recursive mutex
     */
//    pthread_mutex_init(&pc->mtx, PP_MUTEX_NORMAL);
    MUTEX_CREATE_RECURSIVE(&pc->mtx);

    return pc;
}

void
pp_profile_cache_destroy(pp_profile_cache_t* pc) {

    if (pc != NULL) {
        pthread_mutex_destroy(&pc->mtx);
	pp_hash_delete(pc->profiles);
	free(pc);
    }
}

pp_profile_t*
pp_profile_cache_load(pp_profile_cache_t* pc, pp_profile_type_t type, 
                      const char* fname) {
    local_profile_entry_t* pe;
    pp_profile_t* prof;
    time_t mtime = 0;

    MUTEX_LOCK(&pc->mtx);
    // try to find the profile in cache
    if (NULL == (pe = pp_hash_get_entry(pc->profiles, fname))
#ifndef PP_BOARD_PEMX
        || eric_config_file_changed_since(fname, &pe->filemtime)
#endif
        ) {

        if(pe && !eric_config_ext_change_ntfy && type == PP_PROFILE_LOCAL) {
            /* local profile exists and we are eric - update current profile */
            
            /* no file access while applying changes */
            eric_config_set_flush_lock();
            
            /* read new local profile from file and get current from cfg */
            if (NULL == (prof = pc->load_profile(type, fname))) {
                eric_config_release_flush_lock();
                return NULL;
            }

            /* unset external change semaphore - doesn't matter if set or not */
            pp_sem_unlock_nu(eric_config_ext_change_sem_id);
            
            /* generate diff of old config and current config */
            pc->diff_profile(pe->profile, prof);

            /* release currently read profile */
            pp_profile_release(prof);
            eric_config_release_flush_lock();

#ifndef PP_BOARD_PEMX
            eric_config_file_changed_since(fname, &pe->filemtime);
#endif
        } else {
            /* profile doesn't exist or we are external - load new profile */
            if (NULL == (prof = pc->load_profile(type, fname)))
                return NULL;

#ifndef PP_BOARD_PEMX
            eric_config_file_changed_since(fname, &mtime);
#endif

            pe = local_profile_entry_create(prof, mtime);
            
            pp_hash_set_entry(pc->profiles, fname, pe,
                              local_profile_entry_destroy);
        }
    }
    prof = pp_profile_duplicate(pe->profile);
    MUTEX_UNLOCK(&pc->mtx);

    return prof;
}

int profile_cache_update_timestamp(pp_profile_cache_t* pc, 
                                   pp_profile_type_t type,
                                   const char* fname) {
    int ret = PP_SUC;
    
    if(type == PP_PROFILE_LOCAL) {
        /* currently only makes sense for local profile */
        local_profile_entry_t* pe;

        assert(pc);
        MUTEX_LOCK(&pc->mtx);
        
        // try to find the profile in cache
        if (NULL == (pe = pp_hash_get_entry(pc->profiles, fname))) {
            ret = PP_ERR;
        } else {
#ifndef PP_BOARD_PEMX
            eric_config_file_changed_since(fname, &pe->filemtime);
#endif
        }
        
        MUTEX_UNLOCK(&pc->mtx);
    }
    
    return ret;
}


