#include "cim_common.h"
#include "namespace.h"
#include "class.h"
#include "provider_common.h"
/* alphabetically ordered */
#include "provider_account.h"
#include "provider_accountonsystem.h"
#include "provider_admin.h"
#include "provider_authorizedprivilege.h"
#include "provider_authorizedsubject.h"
#include "provider_authorizedtarget.h"
#include "provider_associatedsensor.h"
#include "provider_capabilities.h"
#include "provider_chassislocator.h"
#include "provider_collection.h"
#include "provider_component.h"
#include "provider_componentcs.h"
#include "provider_computersystem.h"
#include "provider_concretedependency.h"
#include "provider_coolingdevice.h"
#include "provider_dependency.h"
#include "provider_elementcapabilities.h"
#include "provider_elementconformstoprofile.h"
#include "provider_enabledlogicalelement.h"
#include "provider_fan.h"
#include "provider_group.h"
#include "provider_hostedcollection.h"
#include "provider_hosteddependency.h"
#include "provider_hostedservice.h"
#include "provider_log.h"
#include "provider_logicalelement.h"
#include "provider_logicaldevice.h"
#include "provider_logrecord.h"
#include "provider_managedelement.h"
#include "provider_managedsystemelement.h"
#include "provider_memberofcollection.h"
#include "provider_numericsensor.h"
#include "provider_powermanagementcapabilities.h"
#include "provider_powermanagementservice.h"
#include "provider_privilege.h"
#include "provider_recordforlog.h"
#include "provider_recordinlog.h"
#include "provider_recordlog.h"
#include "provider_registeredprofile.h"
#include "provider_sensor.h"
#include "provider_service.h"
#include "provider_system.h"
#include "provider_systemcomponent.h"
#include "provider_systemdevice.h"
#include "provider_useoflog.h"
#include "provider_useofmessagelog.h"

#ifdef PP_FEAT_CIM_PORT
#include "provider_port.h"
#endif
#ifdef PRODUCT_PDU
#include "provider_powerstrip.h"
#include "provider_outletgroup.h"
#include "provider_outlet.h"
#endif
#include "sensor_discovery.h"

/*
 * CIM instance repository
 *
 * Hash key:      CIM class name (e.g. CIM_ComputerSystem)
 * Hash element:  CIM class structure (pp_cim_class_t)
 */
pp_hash_t *pp_cim_repository = NULL;

/*
 * CIM instance repository, UFcT-indexed
 *
 * Hash key:      User Friendly Class Tag (e.g. system)
 * Hash element:  CIM class structure (pp_cim_class_t)
 */
pp_hash_t *pp_cim_repository_ufct = NULL;

/*
 * Repository lock (access control for both repository hashes)
 */
pthread_rwlock_t pp_cim_repository_lock;

/*
 * CLP namespace root
 */
pp_cim_instance_t *pp_cim_ns_root = NULL;

#ifdef PP_CIM_DEBUG
/*
 * Dump CIM instance tree to stdout.
 *
 * @param instance  List this instance and its children
 * @param indent    Initial indentation depth
 */
void pp_cim_list_recursive(pp_cim_instance_t *instance, int indent)
{
    int i;
    pp_cim_instance_t *c = instance->children;
    for (i = 0; i < indent; i++)
        printf("  ");
    if (instance->alias) {
        printf("%s %d (%s)\n", instance->alias,
            instance->instance_id, instance->cim_class->ufct);
    } else {
        printf("%s %d\n", instance->cim_class->ufct, instance->instance_id);
    }
    while (c) {
        pp_cim_list_recursive(c, indent + 1);
        c = c->next;
    }
}

void check_properties(int instnum, vector_t *propvals)
{
    unsigned int i;
    for (i = 0; i < vector_size(propvals); i++) {
        pp_cim_propval_t *pv = vector_get(propvals, i);
        if (pv->property->key && pv->data.null) {
            printf("  - Instance #%d: Key property %s undefined.\n", instnum,
                pv->property->name);
        }
        if (pv->property->array) {
            if (!pv->data.null && !pv->data.types.array) {
                printf("  - Instance #%d: Invalid NULL pointer in property %s.\n",
                    instnum, pv->property->name);
            } else if (!pv->data.null &&
		       (pv->property->type == PP_CIM_STRING || pv->property->type == PP_CIM_STRING_CONST)) {
                unsigned int j;
                for (j = 0; j < vector_size(pv->data.types.array); j++) {
                    pp_cim_data_t *d = vector_get(pv->data.types.array, j);
                    if (!d->null && !d->types.string_const) {
                        printf("  - Instance #%d: Invalid NULL pointer in %s array element.\n",
                            instnum, pv->property->name);
                    }
                }
            }
        } else if (pv->property->type == PP_CIM_STRING || pv->property->type == PP_CIM_STRING_CONST) {
            if (!pv->data.null && !pv->data.types.string_const) {
                printf("  - Instance #%d: Invalid NULL pointer in property %s.\n",
                    instnum, pv->property->name);
            }
        }
    }
}

void check_namespace()
{
    pp_cim_class_t *class = pp_hash_get_first_entry(pp_cim_repository);
    printf("CIM namespace check: %d classes\n",
        pp_hash_get_entry_count(pp_cim_repository));
    while (class) {
        unsigned int count = vector_size(class->instances);
        unsigned int i;
        printf("  %s: %d instances...\n", class->cim_name, count);
        for (i = 0; i < vector_size(class->instances); i++) {
            pp_cim_instance_t *inst = vector_get(class->instances, i);
            vector_t *propvals = pp_cim_get_properties(inst);
            check_properties(i, propvals);
            vector_delete(propvals);
        }
        class = pp_hash_get_next_entry(pp_cim_repository);
    }
}
#endif

/*
 * CIM/CLP namespace initialization.
 */
void pp_cim_ns_init()
{
    pp_cim_data_t cs_name, cs_dedicated, d;
    pp_cim_instance_t *inst, *system1, *system2;
#ifdef PP_FEAT_IPMI_SERVER
    pp_cim_instance_t *log1, *pwrmgtsvc1, *priv1;
    vector_t *sensors, *groups, *accounts;
#ifdef PRODUCT_PDU
    vector_t *outlets;
#endif
    unsigned int i;
#endif /* PP_FEAT_IPMI_SERVER */

    pp_cim_repository = pp_hash_create(5);
    pp_cim_repository_ufct = pp_hash_create(5);

    pthread_rwlock_init(&pp_cim_repository_lock, NULL);

    // Create all supported CIM classes
    pp_cim_class_new(&pp_cim_system_desc);
    pp_cim_class_new(&pp_cim_admindomain_desc);
    pp_cim_class_new(&pp_cim_managedelement_desc);
    pp_cim_class_new(&pp_cim_logicaldevice_desc);
    pp_cim_class_new(&pp_cim_coolingdevice_desc);
    pp_cim_class_new(&pp_cim_recordforlog_desc);
    pp_cim_class_new(&pp_cim_privilege_desc);
    pp_cim_class_new(&pp_cim_authorizedprivilege_desc);
    pp_cim_class_new(&pp_cim_managedsystemelement_desc);
    pp_cim_class_new(&pp_cim_logicalelement_desc);
    pp_cim_class_new(&pp_cim_enabledlogicalelement_desc);
    pp_cim_class_new(&pp_cim_log_desc);
    pp_cim_class_new(&pp_cim_service_desc);
    pp_cim_class_new(&pp_cim_powermanagementservice_desc);
    pp_cim_class_new(&pp_cim_capabilities_desc);
    pp_cim_class_new(&pp_cim_powermanagementcapabilities_desc);
    pp_cim_class_new(&pp_cim_registeredprofile_desc);
    pp_cim_class_new(&pp_cim_collection_desc);
    pp_cim_class_new(&pp_cim_group_desc);
    pp_cim_class_new(&pp_cim_computersystem_desc);
#ifdef PP_FEAT_IPMI_SERVER
    pp_cim_class_new(&pp_cim_fan_desc);
    pp_cim_class_new(&pp_cim_sensor_desc);
    pp_cim_class_new(&pp_cim_numericsensor_desc);
    pp_cim_class_new(&pp_cim_logrecord_desc);
    pp_cim_class_new(&pp_cim_recordlog_desc);
    pp_cim_class_new(&pp_cim_account_desc);
    pp_cim_class_new(&pp_cim_chassislocator_desc);
#endif
#ifdef PP_FEAT_CIM_PORT
    pp_cim_class_new(&pp_cim_port_desc);
#endif
#ifdef PRODUCT_PDU
    pp_cim_class_new(&pp_cim_powerstrip_desc);
    pp_cim_class_new(&pp_cim_outlet_desc);
    pp_cim_class_new(&pp_cim_outletgroup_desc);
#endif
    // Association classes
    pp_cim_class_new(&pp_cim_component_desc);
    pp_cim_class_new(&pp_cim_systemcomponent_desc);
    pp_cim_class_new(&pp_cim_componentcs_desc);
    pp_cim_class_new(&pp_cim_systemdevice_desc);
    pp_cim_class_new(&pp_cim_accountonsystem_desc);
    pp_cim_class_new(&pp_cim_dependency_desc);
    pp_cim_class_new(&pp_cim_concretedependency_desc);
    pp_cim_class_new(&pp_cim_hosteddependency_desc);
    pp_cim_class_new(&pp_cim_hostedservice_desc);
    pp_cim_class_new(&pp_cim_hostedcollection_desc);
#ifdef PP_FEAT_IPMI_SERVER
    pp_cim_class_new(&pp_cim_memberofcollection_desc);
#endif
    pp_cim_class_new(&pp_cim_useoflog_desc);
    pp_cim_class_new(&pp_cim_useofmessagelog_desc);
    pp_cim_class_new(&pp_cim_elementcapabilities_desc);
    pp_cim_class_new(&pp_cim_recordinlog_desc);
    pp_cim_class_new(&pp_cim_elementconformstoprofile_desc);
    pp_cim_class_new(&pp_cim_authorizedsubject_desc);
    pp_cim_class_new(&pp_cim_authorizedtarget_desc);
    pp_cim_class_new(&pp_cim_associatedsensor_desc);

    // Admin Domain/CLP namespace root
    pp_cim_ns_root = pp_cim_admindomain_new();

    cs_name.null = 0;
    cs_dedicated.null = 0;
    d.null = 0;

    // Host system
#ifdef PRODUCT_PDU
    system1 = inst = pp_cim_powerstrip_new();
    cs_name.types.string_const = "BaseSystem";
    provider_common_set_property(inst, "Name", cs_name, 1);
    cs_dedicated.types.array = vector_new(NULL, 1, NULL);
    d.types.unsigned_int = 1; // Dedicated == 1 -> "Unknown"
    vector_add(cs_dedicated.types.array, &d);
    provider_common_set_property(inst, "powerOnDealy", cs_dedicated, 1);
    vector_delete(cs_dedicated.types.array);
    d.types.boolean = 0;
    pp_cim_add_instance(inst, pp_cim_ns_root);
    pp_cim_component_new("CIM_SystemComponent", pp_cim_ns_root, inst);
#else /* !PRODUCT_PDU */
    system1 = inst = pp_cim_computersystem_new();
    cs_name.types.string_const = "BaseSystem";
//    provider_common_set_property(inst, "Name", cs_name, 1);
    cs_dedicated.types.array = vector_new(NULL, 1, NULL);
    d.types.unsigned_int = 0; // Dedicated == 0 -> "Not Dedicated"
    vector_add(cs_dedicated.types.array, &d);
    provider_common_set_property(inst, "Dedicated", cs_dedicated, 1);
    vector_delete(cs_dedicated.types.array);
    d.types.boolean = 0;
    provider_common_set_property(inst, "_BMC", d, 1);
    pp_cim_add_instance(inst, pp_cim_ns_root);
    pp_cim_component_new("CIM_SystemComponent", pp_cim_ns_root, inst);
#endif /* !PRODUCT_PDU */

    // Management controller
    system2 = inst = pp_cim_computersystem_new();
    cs_name.types.string_const = "Management";
//    provider_common_set_property(inst, "Name", cs_name, 1);
    cs_dedicated.types.array = vector_new(NULL, 1, NULL);
    d.types.unsigned_int = 14; // Dedicated == 14 -> "Management"
    vector_add(cs_dedicated.types.array, &d);
    provider_common_set_property(inst, "Dedicated", cs_dedicated, 1);
    vector_delete(cs_dedicated.types.array);
    d.types.boolean = 1;
    provider_common_set_property(inst, "_BMC", d, 1);
    pp_cim_add_instance(inst, pp_cim_ns_root);
    pp_cim_component_new("CIM_SystemComponent", pp_cim_ns_root, inst);
    pp_cim_component_new("CIM_ComponentCS", system1, inst);

#ifdef PP_FEAT_IPMI_SERVER
    // Power management service
    pwrmgtsvc1 = pp_cim_powermanagementservice_new();
    pp_cim_add_instance(pwrmgtsvc1, system2);
    pp_cim_dependency_new("CIM_HostedService", system2, pwrmgtsvc1);
    inst = pp_cim_powermanagementcapabilities_new();
    pp_cim_add_instance(inst, pwrmgtsvc1);
    pp_cim_elementcapabilities_new(pwrmgtsvc1, inst);

#ifndef PRODUCT_PDU
    // Fan
    inst = pp_cim_fan_new();
    d.types.string_const = "Fan1";
    provider_common_set_property(inst, "DeviceID", d, 1);
    pp_cim_add_instance(inst, system1);
    pp_cim_component_new("CIM_SystemDevice", system1, inst);

    // Chassis Locator
    inst = pp_cim_chassislocator_new();
    pp_cim_add_instance(inst, system1);
    pp_cim_component_new("CIM_SystemDevice", system1, inst);
#endif

    // Sensors
    sensors = pp_cim_sensor_discovery();
    if (sensors) {
        for (i = 0; i < vector_size(sensors); i++) {
            inst = vector_get(sensors, i);
            pp_cim_add_instance(inst, system1);
            pp_cim_component_new("CIM_SystemDevice", system1, inst);
            pp_cim_dependency_new("CIM_ConcreteDependency", system2, inst);
        }
        //vector_delete(sensors);
    }
#ifdef PRODUCT_PDU
    // Outlet
    outlets = pp_cim_outlet_discovery();

    if (outlets) {
        for (i = 0; i < vector_size(outlets); i++) {
            inst = vector_get(outlets, i);
            pp_cim_add_instance(inst, system1);
            pp_cim_component_new("CIM_SystemDevice", system1, inst);
            pp_cim_outlet_sensor_association(inst, sensors);
        }
        pp_cim_outletgroup_discovery(system1, outlets);

        // I need outlets to create association of outlets and accounts
        // so, delete it after outlet_privilege_discovery
        //vector_delete(outlets);
    }
#endif

    if (sensors)
        vector_delete(sensors);

    // IPMI Sensor Event Log
    log1 = pp_cim_recordlog_new();
    pp_cim_add_instance(log1, system2);
    pp_cim_dependency_new("CIM_UseOfMessageLog", system2, log1);

    // IPMI User ID Management Mapping
    priv1 = pp_cim_authorizedprivilege_new();
    pp_cim_add_instance(priv1, system2);
    pp_cim_authorizedtarget_new(priv1, pp_cim_ns_root);
    pp_cim_authorizedtarget_new(priv1, system2);
    groups = pp_cim_group_discovery();
    if (groups) {
        for (i = 0; i < vector_size(groups); i++) {
            inst = vector_get(groups, i);
            pp_cim_add_instance(inst, system2);
            // According to IPMI-to-CIM mapping CIM_SystemSpecificCollection
            // should be used for Group-System association. Unfortunately it
            // is not an association class...
            pp_cim_dependency_new("CIM_HostedCollection", system2, inst);
            pp_cim_authorizedsubject_new(priv1, inst);
        }
        vector_delete(groups);
    }
    accounts = pp_cim_account_discovery();
    if (accounts) {
        for (i = 0; i < vector_size(accounts); i++) {
            inst = vector_get(accounts, i);
            pp_cim_add_instance(inst, system2);
            pp_cim_component_new("CIM_AccountOnSystem", system2, inst);
        }
#ifdef PRODUCT_PDU
        if (outlets)
            pp_cim_outlet_privilege_discovery(system2, accounts, outlets);

        // register cfg change listener
        //cfg_chg_outlet_user(NULL);
        //pp_cfg_add_change_listener(cfg_chg_outlet_user, "user");

#endif
        vector_delete(accounts);
    }
#ifdef PRODUCT_PDU
    if (outlets)
        vector_delete(outlets);
#endif

    // IPMI Versioning Mapping
    inst = pp_cim_registeredprofile_new();
    pp_cim_elementconformstoprofile_new(inst, pp_cim_ns_root);
    pp_cim_elementconformstoprofile_new(inst, system1);
    pp_cim_elementconformstoprofile_new(inst, system2);

#endif /* PP_FEAT_IPMI_SERVER */

#ifdef PP_FEAT_CIM_PORT
    {
	vector_t *ports;
	u_int i;

	// Ports, create one sample port
	ports = pp_cim_ports();
	if (ports) {
	    for (i = 0; i < vector_size(ports); i++) {
		inst = vector_get(ports, i);
		pp_cim_add_instance(inst, system2); 
		pp_cim_component_new("CIM_SystemDevice", system2, inst);
	    }
	    vector_delete(ports);
	}
    }
#endif

#ifdef PP_CIM_DEBUG
    pp_cim_list_recursive(pp_cim_ns_root, 0);
    check_namespace();
#endif
}

/*
 * Delete a class, including all subclasses and instances.
 *
 * @param cim_class  CIM class structure
 */
static void pp_cim_class_delete_recursive(pp_cim_class_t *cim_class)
{
    unsigned int i;
    while (vector_size(cim_class->subclasses) > 0) {
        pp_cim_class_t *c = vector_get(cim_class->subclasses, 0);
        pp_cim_class_delete_recursive(c);
    }
    for (i = 0; i < vector_size(cim_class->instances); i++) {
        pp_cim_instance_t *inst = vector_get(cim_class->instances, i);
        inst->provider->deinit(inst);
    }
    pp_cim_class_delete(cim_class);
}

/*
 * Namespace deinitialization.
 */
void pp_cim_ns_deinit()
{
    vector_t *top = vector_new(NULL, 10, NULL);
    pp_cim_class_t *c;
    unsigned int i;

    pthread_rwlock_wrlock(&pp_cim_repository_lock);

    // find top-level classes
    c = pp_hash_get_first_entry(pp_cim_repository);
    while (c) {
        if (!c->superclass)
            vector_add(top, c);
        c = pp_hash_get_next_entry(pp_cim_repository);
    }
    // delete recursively
    for (i = 0; i < vector_size(top); i++) {
        c = vector_get(top, i);
        pp_cim_class_delete_recursive(c);
    }
    vector_delete(top);

    pp_hash_delete(pp_cim_repository);
    pp_hash_delete(pp_cim_repository_ufct);

    pp_cim_repository = NULL;
    pp_cim_repository_ufct = NULL;
    pp_cim_ns_root = NULL;

    pthread_rwlock_unlock(&pp_cim_repository_lock);
    pthread_rwlock_destroy(&pp_cim_repository_lock);
}

/*
 * Convert string to lower case
 *
 * @param string  Input string
 *
 * @return  Copy of input string, converted to lower case
 */
char *strdup_lc(const char *string)
{
    char *result;
    char *c;

    if (!string)
        return NULL;
    c = result = malloc(strlen(string) + 1);
    while (*string)
        *(c++) = tolower(*(string++));
    *c = '\0';

    return result;
}

/*
 * Get a list of all known CIM classes.
 *
 * @return  Class name vector; element type: (char *)
 */
vector_t *pp_cim_list_classes()
{
    const char *key;
    pp_hash_iterator_t it;

    pthread_rwlock_rdlock(&pp_cim_repository_lock);

    vector_t *result = vector_new(NULL,
        pp_hash_get_entry_count(pp_cim_repository),
        (vector_elem_del_func_simple)free);
    key = pp_hash_get_first_key_r(pp_cim_repository, &it);
    while (key) {
        vector_add(result, strdup(key));
        key = pp_hash_get_next_key_r(pp_cim_repository, &it);
    }

    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return result;
}

/*
 * CIM class lookup.
 *
 * @param cim_name  CIM class name (e.g. CIM_NumericSensor)
 *
 * @return  Pointer to class structure, NULL in case of error
 *
 * Note: The class and all parent classes will be locked. Call
 *       pp_cim_class_release() to release the lock.
 */
pp_cim_class_t *pp_cim_class_lookup(const char *cim_name)
{
    pp_cim_class_t *c;
    char *name = strdup_lc(cim_name);
    pthread_rwlock_rdlock(&pp_cim_repository_lock);
    c = pp_hash_get_entry(pp_cim_repository, name);
    free(name);
    if (c)
        pp_cim_class_lock(c);
    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return c;
}

/*
 * Lookup class by UFcT.
 *
 * @param ufct  User Friendly Class Tag (e.g. nsensor)
 *
 * @return  Pointer to class structure, NULL in case of error
 *
 * Note: The class and all parent classes will be locked. Call
 *       pp_cim_class_release() to release the lock.
 */
pp_cim_class_t *pp_cim_class_lookup_ufct(const char *ufct)
{
    pp_cim_class_t *c;
    char *name = strdup_lc(ufct);
    pthread_rwlock_rdlock(&pp_cim_repository_lock);
    c = pp_hash_get_entry(pp_cim_repository_ufct, name);
    free(name);
    if (c)
        pp_cim_class_lock(c);
    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return c;
}

/*
 * Add one CIM instance to another one's children.
 *
 * @param instance  Child instance
 * @param parent    Parent instance
 */
void pp_cim_add_instance(pp_cim_instance_t *instance, pp_cim_instance_t *parent)
{
    pp_cim_instance_t *n = parent->children;
    pp_cim_instance_t *last = NULL;
    int max_id = 0;

    assert(instance->cim_class->ufct);

    instance->parent = parent;
    instance->next = NULL;
    instance->children = NULL;

    while (n) {
        char *name1 = instance->alias ? instance->alias : instance->cim_class->ufct;
        char *name2 = n->alias ? n->alias : n->cim_class->ufct;
        if (strcasecmp(name1, name2) == 0) {
            if (max_id < n->instance_id)
                max_id = n->instance_id;
        }
        last = n;
        n = n->next;
    }
    instance->instance_id = max_id + 1;

    if (last) {
        last->next = instance;
    } else {
        parent->children = instance;
    }
}

/*
 * Relative UFiP lookup.
 *
 * @param start  Interpret UFiP relative to this CIM instance.
 * @param ufip   User Friendly Instance Path, element type: pp_clp_ufit_t
 *
 * @return  Target instance reference, NULL in case of error.
 */
pp_cim_instance_t *pp_cim_lookup_relative(pp_cim_instance_t *start, vector_t *ufip)
{
    pp_cim_instance_t *instance = start;
    unsigned int i;

    for (i = 0; i < vector_size(ufip); i++) {
        pp_clp_ufit_t *ufit = vector_get(ufip, i);
        if (strcmp(ufit->ufct, ".") == 0) {
            continue;
        } else if (strcmp(ufit->ufct, "..") == 0) {
            instance = instance->parent;
        } else {
            if (instance->cim_class->clp_update) {
                instance->cim_class->clp_update(instance->cim_class);
            }
            instance = instance->children;
            while (instance) {
                if (instance->alias) {
                    if (ufit->instance_id == instance->instance_id &&
                        strcasecmp(ufit->ufct, instance->alias) == 0) {
                        break;
                    }
                } else {
                    if (ufit->instance_id == instance->instance_id &&
                        strcasecmp(ufit->ufct, instance->cim_class->ufct) == 0) {
                        break;
                    }
                }
                instance = instance->next;
            }
        }
        if (instance == NULL)
            return NULL;
    }
    return instance;
}

/*
 * Absolute UFiP lookup.
 *
 * @param ufip   User Friendly Instance Path, element type: pp_clp_ufit_t
 *
 * @return  Target instance reference, NULL in case of error.
 */
pp_cim_instance_t *pp_cim_lookup_absolute(vector_t *ufip)
{
    return pp_cim_lookup_relative(pp_cim_ns_root, ufip);
}

/*
 * Get UFiT (User Friendly Instance Tag) for a given CIM instance.
 *
 * @param instance  CIM instance
 *
 * @return  UFiT; must be freed by caller.
 */
pp_clp_ufit_t *pp_cim_get_ufit(pp_cim_instance_t *instance)
{
    pp_clp_ufit_t *ufit = malloc(sizeof(pp_clp_ufit_t));
    ufit->ufct = instance->alias ?
        strdup(instance->alias) : strdup(instance->cim_class->ufct);
    ufit->instance_id = instance->instance_id;
    return ufit;
}

/*
 * Get UFiP for a given CIM instance.
 *
 * @param instance  CIM instance
 * @param ufip      User Friendly Instance Path, element type: pp_clp_ufit_t
 *
 * @return  PP_SUC or PP_ERR
 *
 * Note: The UFiP vector must be created and deleted by the caller.
 */
int pp_cim_get_ufip(pp_cim_instance_t *instance, vector_t *ufip)
{
    unsigned int i;
    vector_t *stack = vector_new(NULL, 10, NULL);

    if (!instance->parent && instance != pp_cim_ns_root) {
        // not in CLP namespace
        return PP_ERR;
    }

    while (instance->parent) {
        vector_add(stack, instance);
        instance = instance->parent;
    }

    for (i = vector_size(stack); i > 0; i--) {
        pp_cim_instance_t *inst = vector_get(stack, i - 1);
        pp_clp_ufit_t *ufit = pp_cim_get_ufit(inst);
        vector_add(ufip, ufit);
    }

    vector_delete(stack);

    return PP_SUC;
}

