#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_account.h"
#include "provider_group.h"
#include "provider_memberofcollection.h"

static void account_update(pp_cim_instance_t *instance);
static pp_cim_method_call_t *account_clp_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop);
static char *account_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *account_clp_help_class(pp_cim_class_t *cim_class, int verbose);

static int account_clp_setuserid(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static int account_clp_setpassword(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = account_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = account_clp_map_set,
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = account_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"SystemCreationClassName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {0, {.string_const = "CIM_ComputerSystem"}}},
    {"SystemName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {0, {.string_const = "Management"}}},
    {"CreationClassName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"Name", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"UserID", PP_CIM_STRING_CONST, 0, NULL, 0, 0, 1, 1, {1, {0}}},
    // UserPassword should be an OctetStream property, but neither CLP nor WS-Man could reasonably handle that
    {"UserPassword", PP_CIM_STRING_CONST, 0, NULL, 0, 0, 0, 1, {1, {0}}},
    {"_ID", PP_CIM_UNSIGNED, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {"_GroupAssociation", PP_CIM_REFERENCE, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {.name = NULL}
};

// CIM_Account::_CLP_SetUserID() and CIM_Account::_CLP_SetPassword arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t account_clp_set_args[] =
{
    {"Value", PP_CIM_STRING, 0, NULL, 1, 0}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"_CLP_SetUserID", PP_CIM_VOID, 1, account_clp_set_args, 1, account_clp_setuserid},
    {"_CLP_SetPassword", PP_CIM_VOID, 1, account_clp_set_args, 1, account_clp_setpassword},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_account_desc =
{
    .cim_name = "CIM_Account",
    .ufct = "account",
    .dispname = "CIM Account",
    .superclass = "CIM_LogicalElement",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = account_clp_help_class
};

// Create a new CIM_Account instance
pp_cim_instance_t *pp_cim_account_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_Account", &provider);
    return i;
}

int pp_cim_account_get_id(pp_cim_instance_t *instance)
{
    pp_cim_propval_t *propval =
        instance->provider->get_property(instance, "_ID");
    int id = propval->data.types.unsigned_int;
    pp_cim_propval_delete(propval);
    return id;
}

vector_t *pp_cim_account_discovery()
{
    vector_t *result = NULL;
    unsigned int num, i;
    pp_ipmi_return_t ipmi_ret;
    int ret, err;
    pp_cim_instance_t *group = pp_cim_group_getbyid(15); // No Access

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_SUMMARY,
        NULL, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return NULL;
    }

    num = ipmi_ret.data.usr_summary.max_users;
    result = vector_new(NULL, num, NULL);
    for (i = 1; i <= num; i++) {
        pp_cim_instance_t *inst = pp_cim_account_new();
        pp_cim_instance_t *assoc = pp_cim_memberofcollection_new(group, inst);
        pp_cim_data_t d = {0, {0}};
        pp_strstream_t *str;

        d.types.unsigned_int = i;
        provider_common_set_property(inst, "_ID", d, 1);

        d.types.reference = assoc;
        provider_common_set_property(inst, "_GroupAssociation", d, 1);

        str = pp_strstream_init(NULL);
        pp_strappendf(str, "IPMI User%d", i);
        d.types.string = pp_strstream_buf_and_free(str);
        provider_common_set_property(inst, "Name", d, 1);
        free(d.types.string);

        vector_add(result, inst);
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return result;
}

static void account_update(pp_cim_instance_t *instance)
{
    pp_cim_data_t d;
    pp_cim_propval_t *pv;
    pp_cim_instance_t *assoc;
    pp_ipmi_parameter_t ipmi_param;
    pp_ipmi_return_t ipmi_ret;
    int ret, err;

    provider_common_update(instance);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    ipmi_param.data.usr_info.chan = 0x0E; // current channel
    ipmi_param.data.usr_info.user_id = pp_cim_account_get_id(instance);
    ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_INFO,
        &ipmi_param, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return;
    }

    if (ipmi_ret.data.usr_info.name[0] != '\0') {
        d.null = 0;
        d.types.string_const = ipmi_ret.data.usr_info.name;
    } else {
        d.null = 1;
        d.types.string_const = NULL;
    }
    provider_common_set_property(instance, "UserID", d, 1);

    pv = provider_common_get_property(instance, "_GroupAssociation");
    assoc = pv->data.null ? NULL : pv->data.types.reference;
    pp_cim_propval_delete(pv);
    d.null = 0;
    d.types.reference = pp_cim_group_getbyid(ipmi_ret.data.usr_info.priv_limit);
    provider_common_set_property(assoc, "Collection", d, 1);

    pp_ipmi_cleanup_ret(&ipmi_ret);
}

static pp_cim_method_call_t *account_clp_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;

    if (strcasecmp(prop->property, "UserID") == 0) {
        methodptr = instance->provider->get_method(instance, "_CLP_SetUserID");
    } else if (strcasecmp(prop->property, "UserPassword") == 0) {
        methodptr = instance->provider->get_method(instance, "_CLP_SetPassword");
    } else {
        return NULL;
    }

    call = pp_cim_method_call_new(methodptr, instance);

    arg = malloc(sizeof(pp_cim_data_t));
    arg->null = 0;
    arg->types.string = strdup(prop->value);
    vector_add(call->args, arg);

    return call;
}

static const char clp_help_message_instance[] =
    "This instance stores information about a BMC user account. It supports\r\n"
    "the following CLP operations:\r\n"
    "- Changing the user name:   SET UserID=name\r\n"
    "- Disabling the account:    SET UserID=\"\"\r\n"
    "- Setting a new password:   SET UserPassword=secret\r\n"
    "- Changing the privilege:   SET .=>CIM_MemberOfCollection Collection=../group3\r\n";

static char *account_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message_instance);
}

static const char clp_help_message_class[] =
    "The CIM_Account class is used to store information about identities\r\n"
    "managed by a CIM_SecurityService.\r\n";

static char *account_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message_class);
}

static int account_clp_setuserid(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg = vector_get(args, 0);
    pp_cim_data_t data = {1, {0}};
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    memset(&ipmi_param, 0, sizeof(ipmi_param));

    if (!arg->null && arg->types.string[0] != '\0') {
        // rename user
        ipmi_param.data.usr_set_name.uid = pp_cim_account_get_id(instance);
        snprintf(ipmi_param.data.usr_set_name.name, sizeof(ipmi_param.data.usr_set_name.name), "%s", arg->types.string_const);
        ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_SET_NAME,
            &ipmi_param, NULL, &err, NULL);
        if (ret) {
            printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
            return PP_ERR;
        }
        pp_cim_data_copy(PP_CIM_STRING, 0, &data, arg);
    } else {
        // empty/null argument -> disable user
        ipmi_param.data.usr_disable.uid = pp_cim_account_get_id(instance);
        ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_DISABLE,
            &ipmi_param, NULL, &err, NULL);
        if (ret) {
            printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
            return PP_ERR;
        }
    }

    provider_common_set_property(instance, "UserID", data, 0);
    pp_cim_data_null(PP_CIM_STRING, 0, &data);

    return PP_SUC;
}

static int account_clp_setpassword(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    pp_cim_data_t *arg = vector_get(args, 0);
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    memset(&ipmi_param, 0, sizeof(ipmi_param));

    ipmi_param.data.usr_set_pwd.uid = pp_cim_account_get_id(instance);
    snprintf(ipmi_param.data.usr_set_pwd.pwd, sizeof(ipmi_param.data.usr_set_pwd.pwd), "%s", arg->types.string_const);
    ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_SET_PASSWORD,
        &ipmi_param, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return PP_ERR;
    }

    return PP_SUC;
}

