#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_computersystem.h"
#include "provider_system.h"
#include "provider_powerstrip.h"
#include <pp/ipmi.h>
#include <liberic_misc.h>

static pp_cim_method_call_t *powerstrip_map_reset(
    pp_cim_instance_t *instance);
static pp_cim_method_call_t *powerstrip_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop);
static char *powerstrip_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *powerstrip_clp_help_class(pp_cim_class_t *cim_class,
    int verbose);
static void powerstrip_update(pp_cim_instance_t *instance);
static int powerstrip_setpowerondelay(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static int powerstrip_reset(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);


// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = powerstrip_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = powerstrip_map_reset,
    .clp_map_set = powerstrip_map_set,
    .clp_map_start = NULL,
    .clp_map_stop = NULL,
    .clp_help = powerstrip_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"powerOnDelay", PP_CIM_UNSIGNED, 0, NULL, 0, 0, 1, 1, {1, {0}}},
    {.name = NULL}
};

// CIM_ComputerSystem::_ChassisControl() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t powerstrip_setpowerondelay_args[] =
{
    {"powerOnDelay", PP_CIM_UNSIGNED, 0, NULL, 1, 0}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"setPowerOnDelay", PP_CIM_VOID, 1, powerstrip_setpowerondelay_args, 1, powerstrip_setpowerondelay},
    {"Reset", PP_CIM_VOID, 0, NULL, 1, powerstrip_reset},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_powerstrip_desc =
{
    .cim_name = "CIM_PowerStrip",
    .ufct = "system",
    .dispname = "CIM PowerStrip",
    .superclass = "CIM_ComputerSystem",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = powerstrip_clp_help_class
};

// Create a new CIM_ComputerSystem instance
pp_cim_instance_t *pp_cim_powerstrip_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_PowerStrip", &provider);
    return i;
}

static pp_cim_method_call_t *powerstrip_map_reset(
    pp_cim_instance_t *instance)
{
/*
        pp_cim_propval_t *propval;
        pp_cim_methodptr_t *methodptr;
        pp_cim_method_call_t *call;
        pp_cim_data_t *arg;
        printf("%s\n", __FUNCTION__);

        propval = instance->provider->get_property(instance, "_BMC");
        if (propval->data.types.boolean) {
                // BMC reset
                methodptr = instance->provider->get_method(instance, "_BMC_Reset");
                call = pp_cim_method_call_new(methodptr, instance);
        } else {
                // Host system reset
                methodptr = instance->provider->get_method(instance, "_ChassisControl");
                call = pp_cim_method_call_new(methodptr, instance);

                arg = malloc(sizeof(pp_cim_data_t));
                arg->null = 0;
                arg->types.signed_int = PP_IPMI_CHASSIS_RESET;
                vector_add(call->args, arg);
        }
        pp_cim_propval_delete(propval);

*/
    printf("%s\n", __FUNCTION__);
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;

    methodptr = instance->provider->get_method(instance, "Reset");
    call = pp_cim_method_call_new(methodptr, instance);
    return call;
}

static pp_cim_method_call_t *powerstrip_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    pp_cim_method_call_t *call;
    pp_cim_data_t *arg;
    pp_cim_methodptr_t *methodptr;
    if (strcasecmp(prop->property, "powerOnDelay") == 0) {
        methodptr = provider_common_get_method(instance, "setPowerOnDelay");
        call = pp_cim_method_call_new(methodptr, instance);
        arg = pp_cim_data_parse(call->methodptr->method->args[0].type, prop->value);
        if (!arg)
            return NULL;
        vector_add(call->args, arg);
        return call;
    }
    return NULL;
}

static const char clp_help_message_instance_bmc[] =
    "This instance stores information about the IPMI Baseboard Management\r\n"
    "Controller (BMC). It supports the following CLP operations:\r\n"
    "  - Controller reset:  RESET\r\n";

static const char clp_help_message_instance_host[] =
    "This instance stores information about the host system. It supports\r\n"
    "the following CLP operations:\r\n"
    "  - Power on:      START\r\n"
    "  - Power off:     STOP\r\n"
    "  - System reset:  RESET\r\n";


static char *powerstrip_clp_help_instance(pp_cim_instance_t *instance,
    int verbose UNUSED)
{
    pp_cim_propval_t *propval;
    const char *message;

    propval = instance->provider->get_property(instance, "_BMC");
    if (propval->data.types.boolean) {
        message = clp_help_message_instance_bmc;
    } else {
        message = clp_help_message_instance_host;
    }
    pp_cim_propval_delete(propval);

    return strdup(message);
}

static const char clp_help_message_class[] =
    "The CIM_ComputerSystem class is used to store information about\r\n"
    "computer system, e.g. the management controller or the host system.\r\n";

static char *powerstrip_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message_class);
}
static void powerstrip_update(pp_cim_instance_t *instance)
{
    pp_cim_data_t data;
    printf("%s\n", __FUNCTION__);

    provider_common_update(instance);
    // get state from IPMI
    int err, ret;
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;

    memset(&params, 0, sizeof(params));
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_GET_POWER_ON_DELAY,
        &params, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return;
    }

    printf("%s: %d\n", __FUNCTION__, ipmi_ret.data.oem_pp_rpc_get_power_on_delay_rs.delay);

    // set to property
    data.null = 0;
    data.types.unsigned_int = ipmi_ret.data.oem_pp_rpc_get_power_on_delay_rs.delay;
    pp_cim_set_property(instance, "powerOnDelay", data);

    pp_ipmi_cleanup_ret(&ipmi_ret);
}
static int powerstrip_setpowerondelay(pp_cim_instance_t *instance UNUSED,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    //pp_cim_propval_t *pv;
    pp_cim_data_t *arg;

    if (vector_size(args) != 1)
        return PP_ERR;
    arg = vector_get(args, 0);

    // write to IPMI
    int err, ret;
    pp_ipmi_parameter_t params;

    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_set_power_on_delay.delay = arg->types.unsigned_int;
    //printf("%s :%d\n", __FUNCTION__, params.data.oem_pp_rpc_set_power_on_delay.delay);
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
        PP_IPMI_OEM_PP_RPC_SET_POWER_ON_DELAY,
        &params, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
    }
    /*
    pp_cim_set_property(instance, "powerOnDelay", *arg);

    // check again
    pv = pp_cim_get_property(instance, "powerOnDelay");
    if (!pv || pv->data.null) {
            pp_cim_propval_delete(pv);
            return PP_ERR;
    }
    */

    return PP_SUC;
}

static int powerstrip_reset(pp_cim_instance_t *instance UNUSED,
    vector_t *args UNUSED, pp_cim_data_t *result UNUSED)
{
    eric_misc_trigger_board_reset(0);

    return PP_SUC;
}

