#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_sensor.h"
#include "provider_logicaldevice.h"

#include "valmap_sensor.h"

// forward declarations
static void sensor_update(pp_cim_instance_t *instance);
static char *sensor_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *sensor_clp_help_class(pp_cim_class_t *cim_class, int verbose);


static int sensor_reset(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);
static int sensor_getmanufacturerid(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = sensor_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = pp_cim_logicaldevice_map_reset,
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = sensor_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"SensorType", PP_CIM_UNSIGNED, 0, sensortype_valmap, 0, 0, 1, 0, {1, {0}}},
    {"OtherSensorTypeDescription", PP_CIM_STRING_CONST, 0, NULL, 0, 0, 1, 0, {1, {0}}},
    {"CurrentState", PP_CIM_STRING_CONST, 0, NULL, 0, 0, 1, 0, {1, {0}}},
    {"PossibleStates", PP_CIM_STRING_CONST, 1, NULL, 0, 0, 1, 0, {1, {0}}},
    {"_IPMI_ID", PP_CIM_UNSIGNED, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {"_IPMI_Name", PP_CIM_STRING_CONST, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {"_EntityName", PP_CIM_STRING_CONST, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {"_EntityInst", PP_CIM_UNSIGNED, 0, NULL, 0, 1, 0, 0, {1, {0}}},
    {.name = NULL}
};

// CIM_Sensor::GetManufacturerID() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t sensor_getmanufacturerid_args[] =
{
    {"ManufacturerID", PP_CIM_UNSIGNED, 1, NULL, 0, 1},
    {"Format", PP_CIM_UNSIGNED, 0, NULL, 0, 1}
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"Reset", PP_CIM_UNSIGNED, 0, NULL, 0, sensor_reset}, // overwrites CIM_LogicalDevice::Reset()
    {"GetManufacturerID", PP_CIM_UNSIGNED, 2, sensor_getmanufacturerid_args, 0, sensor_getmanufacturerid}, // defined by IPMI-CIM mapping
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_sensor_desc =
{
    .cim_name = "CIM_Sensor",
    .ufct = "sensor",
    .dispname = "CIM Sensor",
    .superclass = "CIM_LogicalDevice",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = sensor_clp_help_class
};

// Create a new CIM_Sensor instance
pp_cim_instance_t *pp_cim_sensor_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_Sensor", &provider);
    return i;
}

pp_cim_instance_t *pp_cim_sensor_getbyid(unsigned int id)
{
    pp_cim_class_t *cim_class = pp_cim_class_lookup("CIM_Sensor");
    vector_t *instances = pp_cim_get_instances(cim_class);
    pp_cim_data_t d;
    pp_cim_instance_t *result = NULL;
    unsigned int i;

    d.null = 0;
    d.types.unsigned_int = id;

    for (i = 0; i < vector_size(instances); i++) {
        pp_cim_instance_t *inst = vector_get(instances, i);
        pp_cim_propval_t *pv = provider_common_get_property(inst, "_IPMI_ID");
        if (pv) {
            if (pp_cim_data_equal(PP_CIM_UNSIGNED, pv->data, d)) {
                pp_cim_instance_tag(inst);
                result = inst;
            }
            pp_cim_propval_delete(pv);
        }
        if (result)
            break;
    }

    vector_delete(instances);
    pp_cim_class_release(cim_class);

    return result;
}

void sensor_update_sdr(pp_cim_instance_t *instance,
    pp_ipmi_sdr_list_entry_t *sdr_entry)
{
    unsigned int i;
    char *current_state = NULL;
    vector_t *possible_states = NULL;

    switch (sdr_entry->type) {
        case PP_IPMI_SENSOR_TYPE_FULL:
            current_state = pp_strstream_buf(&sdr_entry->data.full.cim_current_state);
            possible_states = &sdr_entry->data.full.cim_possible_states;
            break;
        case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
            current_state = pp_strstream_buf(&sdr_entry->data.compact.cim_current_state);
            possible_states = &sdr_entry->data.compact.cim_possible_states;
            break;
        case PP_IPMI_SENSOR_TYPE_EVENTONLY:
            // nothing to do
            return;
        default:
            printf("Warning: Unhandled sensor type %d.\n", sdr_entry->type);
            return;
    }

    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *pv = vector_get(instance->prop_table, i);
        pp_cim_property_t *p = pv->property;
        if (current_state && strcasecmp(p->name, "CurrentState") == 0) {
            pp_cim_data_null(pv->property->type, 0, &pv->data);
            pv->data.null = 0;
            pv->data.types.string = strdup(current_state);
        } else if (possible_states && strcasecmp(p->name, "PossibleStates") == 0) {
            unsigned int j;
            unsigned int n = vector_size(possible_states);
            pp_cim_data_null(pv->property->type, 1, &pv->data);
            pv->data.null = 0;
            pv->data.types.array = vector_new(NULL, n, NULL);
            for (j = 0; j < n; j++) {
                pp_cim_data_t *el = malloc(sizeof(pp_cim_data_t));
                el->null = 0;
                el->types.string = strdup(vector_get(possible_states, j));
                vector_add(pv->data.types.array, el);
            }
        }
    }
}

static void sensor_update(pp_cim_instance_t *instance)
{
    pp_ipmi_sdr_list_entry_t *sdr_entry;
    pp_ipmi_return_t *ipmi_ret = malloc(sizeof(pp_ipmi_return_t));
    pp_ipmi_parameter_t ipmi_param;
    pp_cim_propval_t *propval;
    int ret, err;

    provider_common_update(instance);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.sensor_get_by_id_list = vector_new2(NULL, 1,
        sizeof(int), NULL);

    propval = instance->provider->get_property(instance, "_IPMI_ID");
    vector_add2(ipmi_param.data.sensor_get_by_name_list,
        &propval->data.types.unsigned_int);
    pp_cim_propval_delete(propval);

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
        PP_IPMI_SENSOR_SUBCMD_GET_BY_ID, &ipmi_param, ipmi_ret,
        &err, NULL);

    vector_delete(ipmi_param.data.sensor_get_by_id_list);

    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        free(ipmi_ret);
        return;
    }

    sdr_entry = vector_get(ipmi_ret->data.sensor_get_list, 0);
    sensor_update_sdr(instance, sdr_entry);

    pp_ipmi_cleanup_ret(ipmi_ret);
    free(ipmi_ret);
}

static const char clp_help_message[] =
    "The CIM_Sensor class is used to store information about IPMI sensors and to\r\n"
    "query the current sensor state. It supports the following CLP operations:\r\n"
    "  - Re-arm the sensor:   RESET\r\n";

static char *sensor_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static char *sensor_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

// Method: CIM_Sensor::Reset()
static int sensor_reset(pp_cim_instance_t *instance, vector_t *args UNUSED,
    pp_cim_data_t *result)
{
    pp_ipmi_parameter_t ipmi_param;
    pp_cim_propval_t *propval;
    int ret, err;

    memset(&ipmi_param, 0, sizeof(ipmi_param));

    pp_cim_debug("CIM_Sensor::Reset()\n");
    if (result) {
        result->null = 0; // "Completed with no error"
        result->types.unsigned_int = 0;
    }

    propval = instance->provider->get_property(instance, "_IPMI_ID");
    ipmi_param.data.rearm_sensor_events.sensor_id = propval->data.types.unsigned_int;
    pp_cim_propval_delete(propval);
    ipmi_param.data.rearm_sensor_events.all_events = 1;

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
        PP_IPMI_SENSOR_SUBCMD_REARM_EVENTS, &ipmi_param, NULL,
        &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        if (result)
            result->types.unsigned_int = 4; // "Failed"
    }

    return PP_SUC;
}

// Method: CIM_Sensor::GetManufacturerID()
static int sensor_getmanufacturerid(pp_cim_instance_t *instance UNUSED,
    vector_t *args, pp_cim_data_t *result)
{
    pp_ipmi_return_t *ipmi_ret = malloc(sizeof(pp_ipmi_return_t));
    int manuf_id;
    int ret, err;
    int i;
    pp_cim_data_t *d1, *d2;

    if (vector_size(args) != 2)
        return PP_ERR;

    ret = pp_ipmi_send_command(PP_IPMI_CMD_BMC,
        PP_IPMI_BMC_SUBCMD_INFO, NULL, ipmi_ret,
        &err, NULL);
    manuf_id = ipmi_ret->data.bmc_info.manufacturer_id;
    pp_ipmi_cleanup_ret(ipmi_ret);

    d1 = vector_get(args, 0);
    d1->null = 0;
    d1->types.array = vector_new(NULL, 3, NULL);
    for (i = 0; i <= 2; i++) {
        d2 = malloc(sizeof(pp_cim_data_t));
        d2->null = 0;
        d2->types.unsigned_int = manuf_id & 0xff;
        manuf_id = manuf_id >> 8;
        vector_add(d1->types.array, d2);
    }

    d1 = vector_get(args, 1);
    d1->null = 0;
    d1->types.unsigned_int = 1; // IANA

    if (result) {
        result->null = 0;
        result->types.unsigned_int = 0; // succeeded
    }
    return PP_SUC;
}

