#ifndef _PP_CLP_H
#define _PP_CLP_H

#define PP_CLP_QUIT -2
#define PP_CLP_SYN_ERR -1
#define PP_CLP_INV_OK 0

#include <pp/vector.h>
#include <liberic_term.h>

/* tokenizer state */
typedef enum {
    PP_CLP_TOK_NORMAL,
    PP_CLP_TOK_TOKEN,
    PP_CLP_TOK_QUOTED
} pp_clp_tok_state_t;

#define MAX_TOKENSIZE 128

/* structure containing the CLP parser's session variables */
typedef struct {
    /* terminal to use for output */
    term_cl_t *term;
    /* current default target */
    vector_t default_target; // element type: pp_clp_ufit_t
    /* CLP prompt */
    char *prompt;
    /* vector of CLP command line tokens */
    vector_t tokens; // element type: char *
    /* continuation of multi-line command expected */
    int cont;
    /* current token (in case of escaped line break within token) */
    char token[MAX_TOKENSIZE];
    int tpos;
    /* current state the tokenizer is in */
    pp_clp_tok_state_t tok_state;
} pp_clp_session_t;

pp_clp_session_t *pp_clp_open(term_cl_t *term);
void pp_clp_close(pp_clp_session_t *session);
int pp_clp_invoke(pp_clp_session_t *session, char *cmd_buf);
int parse_ufip(const char *path, vector_t *ufip, int *relative, int *instance, int *isclass);

/* the CLP verbs */
typedef enum pp_clp_verb_s {
    CLP_VERB_CD,
    CLP_VERB_CREATE,
    CLP_VERB_DELETE,
    CLP_VERB_DUMP,
    CLP_VERB_EXIT,
    CLP_VERB_HELP,
    CLP_VERB_LOAD,
    CLP_VERB_RESET,
    CLP_VERB_SET,
    CLP_VERB_SHOW,
    CLP_VERB_START,
    CLP_VERB_STOP,
    CLP_VERB_VERSION,
    CLP_NUM_VERBS
} pp_clp_verb_t;

typedef enum pp_clp_option_s {
     CLP_OPTION_ALL         = 0x0001,
     CLP_OPTION_DEFAULT     = 0x0002,
     CLP_OPTION_DESTINATION = 0x0004,
     CLP_OPTION_DISPLAY     = 0x0008,
     CLP_OPTION_EXAMINE     = 0x0010,
     CLP_OPTION_FORCE       = 0x0020,
     CLP_OPTION_HELP        = 0x0040,
     CLP_OPTION_KEEP        = 0x0080,
     CLP_OPTION_LEVEL       = 0x0100,
     CLP_OPTION_OUTPUT      = 0x0200,
     CLP_OPTION_RESETSTATE  = 0x0400,
     CLP_OPTION_SOURCE      = 0x0800,
     CLP_OPTION_STATE       = 0x1000,
     CLP_OPTION_VERSION     = 0x2000,
     CLP_OPTION_WAIT        = 0x4000
} pp_clp_option_t;

/* the option and its value list */
typedef struct {
    /* the option */
    pp_clp_option_t option;
    /* vector of option arguments (char *) */
    vector_t values;
} pp_clp_option_value_t;

typedef enum {
    CLP_TARGET_DEFAULT     = 0x00,
    CLP_TARGET_INSTANCE    = 0x01,
    CLP_TARGET_SET         = 0x02,
    CLP_TARGET_ASSOCIATION = 0x04,
    CLP_TARGET_CLASS       = 0x08,
    CLP_TARGET_VERB        = 0x10
} pp_clp_target_type_t;

typedef struct {
    char *ufct;
    int instance_id;
} pp_clp_ufit_t;
void pp_clp_ufit_delete(pp_clp_ufit_t *ufit);

typedef struct {
    /* target type */
    pp_clp_target_type_t type;
    /* first UFiP (prior to first '=>' separator) */
    vector_t *ufip1;
    /* first UFiP relative or absolute */
    int ufip1_relative;
    /* Association class name (TARGET_ASSOCIATION) or UFcT (TARGET_CLASS) */
    char *ufct;
    /* second UFiP (after second '=>' separator) */
    vector_t *ufip2;
    /* second UFiP relative or absolute */
    int ufip2_relative;
    /* CLP verb id (if type == CLP_TARGET_VERB) */
    pp_clp_verb_t verb;
    /* -level option argument, -1 == all */
    int level;
} pp_clp_target_t;

/* the property and its value */
typedef struct {
    /* property name */
    char *property;
    /* property value */
    char *value;
} pp_clp_property_value_t;

typedef enum {
    CLP_OUTPUT_TEXT,
    CLP_OUTPUT_CLPCSV,
    CLP_OUTPUT_KEYWORD,
    CLP_OUTPUT_CLPXML
} pp_clp_output_format_t;

typedef enum {
    CLP_OUTPUT_ERROR,
    CLP_OUTPUT_TERSE,
    CLP_OUTPUT_VERBOSE
} pp_clp_output_level_t;

/* CLP display options */
typedef struct {
    int associations;
    vector_t association_names; // element type: char *
    int properties;
    vector_t property_names; // element type: char *
    char *filter_property;
    char *filter_value;
    int targets;
    vector_t target_names; // element type: char *
    int verbs;
} pp_clp_display_options_t;

/* CLP output format */
typedef struct {
    pp_clp_output_format_t format;
    pp_clp_output_level_t level;
    char language[4];
    int end;       // 0=begin, 1=end
    int reverse;   // 0=default, 1=reverse order
    int count;     // -1=all
    int range_min; // -1=unspecified
    int range_max; // -1=unspecified
} pp_clp_output_options_t;

/* the CLP command structure */
typedef struct {
    /* the verb type */
    pp_clp_verb_t verb;
    /* option vector, items are of type pp_clp_option_value_t */
    vector_t options;
    /* the target */
    pp_clp_target_t *target;
    /* property vector, items are of type pp_clp_property_value_t */
    vector_t properties;
    /* global option flags */
    pp_clp_display_options_t display;
    pp_clp_output_options_t output;
    int opt_all;
    int opt_default;
    int opt_examine;
    int opt_help;
    int opt_version;
} pp_clp_cmd_t;

#endif /* _PP_CLP_H */

