#ifndef _DC_CRYPTO_H
#define _DC_CRYPTO_H

#include "dc_common.h"

#include <openssl/bio.h>
#include <openssl/rsa.h>

#define PP_DC_SIGNED 1
#define PP_DC_ENCRYPTED 2

typedef struct { u_int8_t ui[8]; } pp_dc_keyid_t;

typedef enum {
    PP_DC_HASH_MD5=1,
    PP_DC_HASH_SHA1=2
} pp_dc_hash_algo_t;

typedef enum {
    PP_DC_PUB_RSA=1  //,
    //    PP_DC_PUB_ELGAMAL=2
} pp_dc_pub_algo_t;

typedef enum {
    PP_DC_SYM_AES_ECB=1,
    PP_DC_SYM_AES_CBC=2

} pp_dc_sym_algo_t;

typedef enum {
    PP_DC_TAG_NULL=0,
    PP_DC_TAG_PLAIN=1,       // simple text
    PP_DC_TAG_SIGNATURE=2,   // signature
    PP_DC_TAG_ESK=3,         // encrypted secret key
    PP_DC_TAG_ENCRYPTED=4,   // encrypted data
    PP_DC_TAG_COMPRESSED=5,  // compressed data
    PP_DC_TAG_HASH=6,        // hash value
    PP_DC_TAG_SKEY=7,        // secret key
    PP_DC_TAG_PUBKEY=8,      // public key
    PP_DC_TAG_PRIVKEY=9      // private key
} pp_dc_tag_t;

typedef struct pp_dc_signature_s {
    u_int8_t hash_algo;
    u_int8_t pub_algo;
    pp_dc_keyid_t keyid;
    size_t size;
    u_int8_t *data;
} pp_dc_signature_t;

typedef struct pp_dc_esk_s {
    u_int8_t pub_algo;
    pp_dc_keyid_t keyid;
    size_t size;
    u_int8_t *data;
} pp_dc_esk_t;

typedef struct pp_dc_skey_s {
    u_int8_t algo;
    size_t size;
    u_int8_t *data;
} pp_dc_skey_t;

typedef struct pp_dc_hash_s {
    u_int8_t algo;
    size_t size;
    u_int8_t *data;
} pp_dc_hash_t;

typedef union {
    void *gen;
    RSA *rsa;
} pp_dc_pubssl_t;

typedef struct pp_dc_pubkey_s {
    u_int8_t algo;
    pp_dc_keyid_t keyid;
    pp_dc_pubssl_t key;
} pp_dc_pubkey_t;

typedef struct pp_dc_privkey_s {
    u_int8_t algo;
    pp_dc_keyid_t keyid;
    pp_dc_pubssl_t key;
} pp_dc_privkey_t;

typedef struct pp_dc_plain_s {
    size_t size;
    u_int8_t *data;
} pp_dc_plain_t;

typedef struct pp_dc_cnode_s pp_dc_cnode_t;

typedef struct pp_dc_encrypted_s {
    size_t size;
    u_int8_t *data;
    pp_dc_cnode_t *sub;
    pp_dc_skey_t *skey;
} pp_dc_encrypted_t;

typedef struct pp_dc_compressed_s {
    pp_dc_cnode_t *sub;
} pp_dc_compressed_t;

typedef union {
    void *gen;
    pp_dc_signature_t  *signature;
    pp_dc_esk_t        *esk;
    pp_dc_skey_t       *skey;
    pp_dc_pubkey_t     *pubkey;
    pp_dc_privkey_t    *privkey;
    pp_dc_plain_t      *plain;
    pp_dc_compressed_t *compressed;
    pp_dc_encrypted_t  *encrypted;
} pp_dc_union_t;

//typedef 
struct pp_dc_cnode_s {
    pp_dc_tag_t tag;
    pp_dc_union_t un;
    struct pp_dc_cnode_s *next;
};
//pp_dc_cnode_t;



#define DECLARE_CRYPTO_TYPE(X) \
EXTERN_C pp_dc_##X##_t* pp_dc_##X##_new(void); \
EXTERN_C void pp_dc_##X##_free (pp_dc_##X##_t *p); \
EXTERN_C int pp_dc_##X##_parse (BIO *bio,pp_dc_##X##_t **p,size_t size); \
EXTERN_C int pp_dc_##X##_build (BIO *bio,pp_dc_##X##_t *p); \
EXTERN_C pp_dc_cnode_t* pp_dc_##X##2cnode (pp_dc_##X##_t *p); \
EXTERN_C pp_dc_##X##_t* pp_dc_cnode2##X (pp_dc_cnode_t* cnode)



/**************************************************
 * CNode
 **************************************************/ 

EXTERN_C pp_dc_cnode_t* pp_dc_cnode_new(void);
EXTERN_C pp_dc_cnode_t* pp_dc_cnode_create(pp_dc_tag_t tag,void* data);

EXTERN_C void pp_dc_cnode_free(pp_dc_cnode_t *node);

EXTERN_C void pp_dc_cnode_print(pp_dc_cnode_t *node);

EXTERN_C int pp_dc_cnode_parse(BIO *bio,pp_dc_cnode_t **node);
EXTERN_C int pp_dc_cnode_parse_mem(char* data,size_t size,pp_dc_cnode_t **node);

EXTERN_C int pp_dc_cnode_build(BIO *bio,pp_dc_cnode_t *node);
EXTERN_C int pp_dc_cnode_build_mem(char* data,size_t *size,pp_dc_cnode_t *node);

EXTERN_C size_t pp_dc_cnode_getdata(pp_dc_cnode_t *node,char **data);


/**************************************************
 * Plain
 **************************************************/ 

DECLARE_CRYPTO_TYPE(plain);

EXTERN_C pp_dc_plain_t* pp_dc_plain_gen(char* data,size_t size);


/**************************************************
 * Compressed
 **************************************************/ 

DECLARE_CRYPTO_TYPE(compressed);

EXTERN_C pp_dc_compressed_t* pp_dc_compressed_gen(pp_dc_cnode_t *node);


/**************************************************
 * Signature
 **************************************************/ 

DECLARE_CRYPTO_TYPE(signature);

EXTERN_C pp_dc_signature_t* pp_dc_signature_sign(pp_dc_cnode_t *node,pp_dc_hash_algo_t algo,pp_dc_privkey_t *privkey);
EXTERN_C int pp_dc_signature_verify(pp_dc_cnode_t *node,pp_dc_signature_t *signature,pp_dc_pubkey_t *pubkey);


/**************************************************
 * Secret Key
 **************************************************/ 

DECLARE_CRYPTO_TYPE(skey);

EXTERN_C BIO* pp_dc_skey2bio(pp_dc_skey_t *skey,int enc);
EXTERN_C pp_dc_skey_t* pp_dc_skey_gen(void);


/**************************************************
 * ESK
 **************************************************/ 

DECLARE_CRYPTO_TYPE(esk);

EXTERN_C pp_dc_skey_t* pp_dc_esk2skey(pp_dc_esk_t *esk,pp_dc_privkey_t *privkey);
EXTERN_C pp_dc_esk_t* pp_dc_skey2esk(pp_dc_skey_t *skey,pp_dc_pubkey_t *pubkey);


/**************************************************
 * Encrypted
 **************************************************/ 

DECLARE_CRYPTO_TYPE(encrypted);

EXTERN_C pp_dc_encrypted_t* pp_dc_encrypted_gen(pp_dc_cnode_t *node,pp_dc_skey_t *skey);
EXTERN_C pp_dc_cnode_t* pp_dc_encrypted_decrypt(pp_dc_encrypted_t *encrypted,pp_dc_skey_t *skey);


/**************************************************
 * Public Key
 **************************************************/ 

DECLARE_CRYPTO_TYPE(pubkey);

EXTERN_C pp_dc_keyid_t pp_dc_pubkey_keyid(pp_dc_pubkey_t *pubkey);
EXTERN_C int pp_dc_pubkey_encrypt(char *indata,size_t insize,pp_dc_pubkey_t *pubkey,char **outdata,size_t *outsize);
EXTERN_C int pp_dc_pubkey_decrypt(char *indata,size_t insize,pp_dc_pubkey_t *pubkey,char **outdata,size_t *outsize);


/**************************************************
 * Private Key
 **************************************************/ 

DECLARE_CRYPTO_TYPE(privkey);

EXTERN_C pp_dc_pubkey_t* pp_dc_privkey2pubkey(pp_dc_privkey_t *privkey);
EXTERN_C pp_dc_privkey_t* pp_dc_privkey_gen(void);
pp_dc_keyid_t pp_dc_privkey_keyid(pp_dc_privkey_t *privkey);
EXTERN_C int pp_dc_privkey_encrypt(char *indata,size_t insize,pp_dc_privkey_t *privkey,char **outdata,size_t *outsize);
EXTERN_C int pp_dc_privkey_decrypt(char *indata,size_t insize,pp_dc_privkey_t *privkey,char **outdata,size_t *outsize);


/**************************************************
 * Utilities
 **************************************************/ 

EXTERN_C int pp_dc_sign_and_encrypt(char *data,size_t size,int state,pp_dc_privkey_t *privkey,pp_dc_pubkey_t *pubkey,pp_dc_cnode_t **node);
EXTERN_C size_t pp_dc_decrypt_and_verify(pp_dc_cnode_t *node,int *state,pp_dc_privkey_t *privkey,pp_dc_pubkey_t *pubkey,char **data);
EXTERN_C int pp_dc_decrypt(pp_dc_cnode_t *node,pp_dc_privkey_t *privkey);
EXTERN_C int pp_dc_verify(pp_dc_cnode_t *node,pp_dc_pubkey_t *pubkey);

#endif /* _DC_CRYPTO_H */
