#include <sys/mman.h>
#include <stdlib.h>
#include <pp/hal_common.h>

#include "debug.h"
#include "mempool.h"


u_char *mem_pool;
static u_char init_done = 0;

static u_long var_start_ofs = 0;

static int max_fix_descriptors = 0;
static mem_desc_t *fix_descriptors = NULL;
static u_char     fix_desc_count = 0;

static struct list_head desc_list;
static unsigned char nr_desc = 0;

static u_int32_t redistribute_memory(u_int32_t size);

int
mempool_init(void)
{    
    /* 
     * mempool is shared between vsc's, so we can use fd of the first one
     * when a new client connects, the mempool is devided among present clients
     */
    int grab_vsc_fd = pp_grab_fd( 0 );
    mem_pool = mmap(0, VSC_MEM_POOL_SIZE, PROT_READ|PROT_WRITE, MAP_SHARED, grab_vsc_fd, 0);
    if (mem_pool == MAP_FAILED) {
	return -1;
    }

    max_fix_descriptors = pp_hal_common_get_vsc_cnt() * 5;
    assert( max_fix_descriptors > 0 );

    fix_descriptors = (mem_desc_t *)malloc( sizeof(mem_desc_t) * max_fix_descriptors );

    INIT_LIST_HEAD(&desc_list);
    
    return 0;
}

void
mempool_cleanup(void)
{
    if (mem_pool != MAP_FAILED) {
	munmap(mem_pool, VSC_MEM_POOL_SIZE);
    }
    free(fix_descriptors);
}

int
mempool_reserve_fix_desc(pp_grab_mem_desc_id_t id, u_int32_t size, u_char video_link)
{
    static const char* fn = ___F;
    mem_desc_t *desc;
    int ret = -1;

    /* fixed descriptors are quite static, check this */
    if (init_done) {
	DCH(D_ERROR, "%s: not allowed after grabber started, only during init (id=%d)\n", fn, id);
	goto bail;
    }
    if (fix_desc_count >= max_fix_descriptors) {
	DCH(D_ERROR, "%s: maximum of %d fix mem descriptors reached (id=%d)\n", fn, max_fix_descriptors, id);
	
	goto bail;
    }
    if (var_start_ofs + size >= VSC_MEM_POOL_SIZE) {
	DCH(D_ERROR, "%s: out of physically continuous pool memory (id=%d)\n", fn, id);
	goto bail;
    }
    
    desc = &fix_descriptors[fix_desc_count];

    desc->video_link = video_link;
    desc->id	 = id;
    desc->offset = var_start_ofs;
    desc->ptr    = mem_pool + desc->offset;
    desc->size   = size;
 
    var_start_ofs += size;
    fix_desc_count++;

    DCH(D_VERBOSE, "%s: reserved fix id %d @ ofs 0x%08x, size %d\n", fn, id, desc->offset, size);
    ret = 0;
    
 bail:
    return ret;
}

void
mempool_start_var_alloc(void)
{
    init_done = 1;
}

mem_desc_t*
mempool_get_desc(u_char id, u_char video_link)
{
    static const char* fn = ___F;
    mem_desc_t *new_desc = NULL;
    u_int32_t block_size, ofs;
   
    if (id != GRAB_VARIABLE_MEM_DESC) {	
	/* return one of the fixed descriptors */
	int i;
	for (i = 0; i < fix_desc_count; i++) {
	    if ((fix_descriptors[i].id == id) && (fix_descriptors[i].video_link== video_link)){
		new_desc = &fix_descriptors[i];
		break;
	    }
	}	    
	
	if (new_desc == NULL) {
	    D(D_ERROR, "%s: unknown fix descriptor id\n", fn);
	    goto bail;
	}

	new_desc->id	 = id;
	new_desc->in_use = 1;
	goto bail;
    }

    /* create a variable descriptor */
    block_size = (VSC_MEM_POOL_SIZE - var_start_ofs) / (nr_desc + 1);    

    /* align blocksize to 4 bytes */
    if (block_size % 4) block_size = (block_size / 4 + 1) * 4;
    
    /* we need at least one tile line of maximum resolution to fit
       in the request buffer, although thats already quite inefficient */
    if (block_size < VSC_MAX_RES_X / PP_FB_TILE_WIDTH * PP_FB_TILE_SIZE) {
	D(D_ERROR, "%s: out of physically continuous pool memory\n", fn);
	goto bail;
    }

    ofs = redistribute_memory(block_size);

    new_desc = malloc(sizeof(mem_desc_t));
    new_desc->id     = id;
    new_desc->size   = block_size;
    new_desc->offset = ofs;
    new_desc->ptr    = mem_pool + ofs;
    new_desc->video_link= video_link;
    
    list_add(&new_desc->listnode, &desc_list);
    nr_desc++;

    D(D_VERBOSE, "%s: new @ %p, size %d, offset 0x%08x, video_link=%d\n",
      fn, new_desc->ptr, new_desc->size, new_desc->offset, new_desc->video_link);
    
 bail:
    return new_desc;
}

void
mempool_release_desc(mem_desc_t *desc)
{
    u_int32_t block_size;
    
    assert(desc);

    /* nothing to do for fixed blocks */
    if (desc->id != GRAB_VARIABLE_MEM_DESC) {
	desc->in_use = 0;
	return;
    }
    
    list_del(&desc->listnode);
    nr_desc--;

    if (nr_desc != 0) {
	block_size = (VSC_MEM_POOL_SIZE - var_start_ofs) / nr_desc;

	/* align blocksize to 4 bytes */
	if (block_size % 4) block_size = (block_size / 4 + 1) * 4;
	
	redistribute_memory(block_size);
    }
    
    free(desc);
}

static u_int32_t
redistribute_memory(u_int32_t size)
{
    struct list_head *node, *tmp;
    u_int32_t ofs;
	
    /* redistribute available pool memory */
    ofs = var_start_ofs;
   
    // for all clients of all video_links 
    list_for_each_safe(node, tmp, &desc_list) {
	mem_desc_t *desc = list_entry(node, mem_desc_t, listnode);

	desc->size	= size;
	desc->offset    = ofs;
	desc->ptr	= mem_pool + ofs;
	
	ofs += size;
    }

    return ofs;
}
