#include <string.h>
#include <openssl/hmac.h>
#include <pp/hal_common.h>
#include <pp/features.h>
#include "flashid.h"

u_char flashid_sig_key[] = PP_HAL_COMMON_FLASHID_SIG_KEY_INITIALIZER;

char *
pp_hal_common_get_flash_factory_id(void)
{
    return get_flash_factory_id();
}
    
int
pp_hal_common_is_flash_id_matching(void)
{
    char *fact = get_flash_factory_id();
    char *user = get_flash_user_id();
    int ret = 0, i, fact_len, user_len, outlen;
    unsigned char user_array[PP_HAL_COMMON_MAX_FLASHID_LEN / 2];
    unsigned char fact_array[PP_HAL_COMMON_MAX_FLASHID_LEN / 2];
    unsigned char hmac_out[HMAC_MAX_MD_CBLOCK];

#ifndef PP_FEAT_CHECK_FLASHID
/* FIXME: FlashID check disabled! */
    ret = 1;
    goto bail;
#endif

#ifdef PP_FEAT_CREATE_FLASHID
    goto bail;
#endif
   
#ifdef PP_FEAT_CHECK_WLAN_INSTEAD_OF_FLASH_ID
    {
        int check;
        check = WEXITSTATUS(pp_system("cat /proc/pci | grep -i 168c:0013"));
        if (!check) {
	    ret = 1;
	    goto bail;
	}
    }
#endif /* PP_FEAT_CHECK_WLAN_INSTEAD_OF_FLASH_ID */
   
    if (fact == NULL || user == NULL) goto bail;

    fact_len = strlen(fact);
    user_len = strlen(user);
    
    // first check the size
    if (fact_len != 16 && user_len != 16) goto bail;

    for (i = 0; i < fact_len; i += 2) {
	u_char fact_tmp[3];
	u_char user_tmp[3];

	memcpy(fact_tmp, &fact[i], 2);
	memcpy(user_tmp, &user[i], 2);

        fact_tmp[2] = '\0';
        user_tmp[2] = '\0';
			 
	fact_array[i/2] = strtol(fact_tmp, NULL, 16);
	user_array[i/2] = strtol(user_tmp, NULL, 16);
    }

    HMAC(EVP_md5(), flashid_sig_key, sizeof(flashid_sig_key),
	 fact_array, sizeof(fact_array), hmac_out, &outlen);

    // compare the first 8 bytes
    if (memcmp(user_array, hmac_out, 8) == 0) ret = 1;
    
 bail:
    free(user);
    free(fact);
    return ret;    
}

#ifdef PP_FEAT_CREATE_FLASHID
/**
 * only temporary for Firmware-Version 040000, disable afterwards.
 * used to update old devices with a firmware that writes the flashID
 * before continuing to update to one (040001) which checks it
 */
int
pp_hal_common_create_flash_id(void)
{
    char *fact = get_flash_factory_id();
    char *lock = get_flash_lock();
    char  user[PP_HAL_COMMON_MAX_FLASHID_LEN + 1];
    unsigned char hmac_out[HMAC_MAX_MD_CBLOCK];    
    unsigned char fact_array[PP_HAL_COMMON_MAX_FLASHID_LEN / 2];
    int ret = PP_ERR, fact_len, i, outlen;
    
    if (fact == NULL || lock == NULL) goto bail;

    if (strstr(lock, "1")) {
	pp_log("Flash already locked, nothing to do\n");
	goto bail;
    }

    fact_len = strlen(fact);
    if (fact_len != 16) goto bail;

    for (i = 0; i < fact_len; i += 2) {
	unsigned char fact_tmp[3];
	memcpy(fact_tmp, &fact[i], 2);
	
	fact_tmp[2]= '\0';
	fact_array[i/2] = strtol(fact_tmp, NULL, 16);
    }

    HMAC(EVP_md5(), flashid_sig_key, sizeof(flashid_sig_key),
	 fact_array, sizeof(fact_array), hmac_out, &outlen);

    printf("Calculated FlashID ");
    for (i = 0; i < outlen && i < 16; i++) {
	sprintf(&user[2*i], "%02X", hmac_out[i]);
	printf("%02X", hmac_out[i]);
    }
    printf("\n");

    if (PP_FAILED(set_flash_user_id(user))) {
	pp_log_err("Could not write flash user id\n");
	goto bail;
    }

    if (PP_FAILED(set_flash_user_id_lock())) {
	pp_log_err("Could not lock flash user id\n");
	goto bail;
    }
    
    ret = PP_SUC;
    
 bail:
    free(fact);
    free(lock);
    return ret;
}
#endif /* PP_FEAT_CREATE_FLASHID */
