#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/poll.h>
#include <liberic_pthread.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/hal_icpmmd.h>
#include <pp_gpio.h>

/******************************************************************************
 * internal declarations
 */

#define ICPMMD_DEFAULT_DELAY_MS 2

typedef struct {
    int gpio_fd0;
    int gpio_fd1;
} icpmmd_ctx_t;

typedef struct {
    icpmmd_switch_type_t switch_type;
    u_int node;
    u_int delay_ms;
} icpmmd_switch_arg_t;

static pthread_mutex_t switch_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static pthread_t icpmmd_kvm_switch_button_handler_thread;
static pthread_t icpmmd_switch_non_block_thread;
static volatile int icpmmd_switch_non_block_thread_running = 0;
static u_int sel_node = 0;
static u_int node_cnt = 12;

static void * icpmmd_kvm_switch_button_handler(void * arg);
static icpmmd_ctx_t * icpmmd_create_context(void);
static void icpmmd_free_context(icpmmd_ctx_t * ctx);
static void icpmmd_select_func(icpmmd_ctx_t * ctx, icpmmd_switch_type_t switch_type);
static void icpmmd_select_node(icpmmd_ctx_t * ctx, u_int node);
static void icpmmd_select_all_nodes(icpmmd_ctx_t * ctx);
static void icpmmd_control_activate(icpmmd_ctx_t * ctx, u_int delay_ms);
static void icpmmd_control_deactivate(icpmmd_ctx_t * ctx);
static void icpmmd_switch(icpmmd_switch_type_t switch_type, u_int node,
			  u_int delay_ms);
static void* icpmmd_switch_non_block_thread_func(void* icpmmd_switch_arg);

/******************************************************************************
 * external API
 */

int
pp_hal_icpmmd_switch_kvm(u_int node)
{
    if (PP_FAILED(pp_hal_icpmmd_switch(ICPMMD_SWITCH_TYPE_KVM, node, 0, 0))) {
	return PP_ERR;
    }
    sel_node = node;
    return PP_SUC;
}

int 
pp_hal_icpmmd_reset_bmc(u_int node)
{
    return pp_hal_icpmmd_switch(ICPMMD_SWITCH_TYPE_BMC_RESET, node, 0, 0);
}

int 
pp_hal_icpmmd_power_blade(u_int node, u_int delay_ms, int non_block)
{
    return pp_hal_icpmmd_switch(ICPMMD_SWITCH_TYPE_POWER, node, delay_ms,
				non_block);
}

int 
pp_hal_icpmmd_reset_blade(u_int node, u_int delay_ms, int non_block)
{
    return pp_hal_icpmmd_switch(ICPMMD_SWITCH_TYPE_RESET, node, delay_ms,
				non_block);
}

int 
pp_hal_icpmmd_switch(icpmmd_switch_type_t switch_type, u_int node,
		     u_int delay_ms, int non_block)
{
    if (icpmmd_switch_non_block_thread_running) {
	errno = EBUSY;
	return PP_ERR;
    }
    
    if (non_block) {
	icpmmd_switch_arg_t* arg = malloc(sizeof(icpmmd_switch_arg_t));
	arg->switch_type = switch_type;
	arg->node = node;
	arg->delay_ms = delay_ms;
	if (0 != eric_pthread_create(&icpmmd_switch_non_block_thread, 0, 65536,
				     icpmmd_switch_non_block_thread_func,
				     arg)) {
	    pp_log("%s(): Cannot create non blocking switch thread.\n", ___F);
	    free(arg);
	    return PP_ERR;
	}
    } else {
	icpmmd_switch(switch_type, node, delay_ms);
    }
    return PP_SUC;
}

int 
pp_hal_icpmmd_get_selected_blade(unsigned char* sel_node_copy)
{
    if (icpmmd_switch_non_block_thread_running) {
	errno = EBUSY;
	return PP_ERR;
    }
    MUTEX_LOCK(&switch_mtx);
    *sel_node_copy = sel_node;
    MUTEX_UNLOCK(&switch_mtx);
    return PP_SUC;
}


/******************************************************************************
 * library internal API
 */

void
icpmmd_init(void)
{
    /* only if we are in eric process ... */
    if (pp_i_am_eric) {
	MUTEX_LOCK(&switch_mtx);
	pp_hal_icpmmd_switch_kvm(sel_node);
	MUTEX_UNLOCK(&switch_mtx);
	/* start the kvm switch button handler thread */
	if (eric_pthread_create(&icpmmd_kvm_switch_button_handler_thread, 0, 65536,
				icpmmd_kvm_switch_button_handler, NULL) != 0) {
	    pp_log("%s(): Cannot create kvm switch button handler thread.\n", ___F);
	    //return -1;
	    return;
	}
    }
}

void
icpmmd_cleanup(void)
{
    /* only if we are in eric process ... */
    if (pp_i_am_eric) {
    }
}

/******************************************************************************
 * private routines
 */
static void* icpmmd_switch_non_block_thread_func(void* a) {
    icpmmd_switch_arg_t* arg = a;
    icpmmd_switch_non_block_thread_running = 1;
    icpmmd_switch(arg->switch_type, arg->node, arg->delay_ms);
    icpmmd_switch_non_block_thread_running = 0;
    free(arg);
    return 0;
}

static void
icpmmd_switch(icpmmd_switch_type_t switch_type, u_int node,
		     u_int delay_ms)
{
    icpmmd_ctx_t * ctx;

    MUTEX_LOCK(&switch_mtx);

    ctx = icpmmd_create_context();
    if (ctx) {
	icpmmd_select_func(ctx, switch_type);
	if (switch_type == ICPMMD_SWITCH_TYPE_KVM) {
	    icpmmd_select_all_nodes(ctx);
	    icpmmd_control_deactivate(ctx);
	    sel_node = node;
	}
	icpmmd_select_node(ctx, node);
	if (switch_type == ICPMMD_SWITCH_TYPE_POWER
	    || switch_type == ICPMMD_SWITCH_TYPE_RESET) {
	    icpmmd_control_activate(ctx, delay_ms);
	} else {
	    icpmmd_control_activate(ctx, ICPMMD_DEFAULT_DELAY_MS);
	}
	icpmmd_free_context(ctx);
    }

    MUTEX_UNLOCK(&switch_mtx);
}

static void *
icpmmd_kvm_switch_button_handler(void * arg UNUSED)
{
    icpmmd_ctx_t * ctx = icpmmd_create_context();

    if (gpio_ioctl_irqconf(ctx->gpio_fd1, 0, 0, ~0x30, 0x30) != 0) {
	pp_log_err("%s(): gpio_ioctl_irqconf() failed", ___F);
	return NULL;
    }

    if (gpio_ioctl_watch(ctx->gpio_fd1, 0x30) != 0) {
	pp_log_err("%s(): gpio_ioctl_watch() failed", ___F);
	return NULL;
    }

    if (ctx) {
	struct pollfd fds[1];

	fds[0].fd = ctx->gpio_fd1;
	fds[0].events = POLLIN;

	while (1) {
	    int r;
	    u_long val;
	    if ((r = poll(fds, 1, 1000)) < 0) {
		pp_log_err("%s(): poll() failed", ___F);
	    } else if (r > 0 && fds[0].revents & POLLIN) {
		/* determine if up or down is pressed */
		usleep(10000); /* wait for a stable signal */
		gpio_ioctl_in(fds[0].fd, &val);
		/* up/down exchanged on icp request, board labels wrong now */
		int up = (val & 0x30) == 0x10 ? 1 : 0;
		int down = (val & 0x30) == 0x20 ? 1 : 0;
		/* switch kvm channel according to pressed button */
		MUTEX_LOCK(&switch_mtx);
		if (up) {
		    pp_hal_icpmmd_switch_kvm((sel_node + 1) % node_cnt);
		} else if (down) {
		    /* we add node_cnt to prevent negative numbers */
		    pp_hal_icpmmd_switch_kvm((sel_node + node_cnt - 1) % node_cnt);
		}
		MUTEX_UNLOCK(&switch_mtx);
	    }
	}
	icpmmd_free_context(ctx);
    }
    return NULL;
}

static icpmmd_ctx_t *
icpmmd_create_context(void)
{
    icpmmd_ctx_t * ctx = malloc(sizeof(icpmmd_ctx_t));

    if (ctx) {
	ctx->gpio_fd0 = ctx->gpio_fd1 = -1;

	if ((ctx->gpio_fd0 = open("/dev/gpio-xr17-0", O_RDWR)) < 0) {
	    pp_log_err("%s(): open(/dev/gpio-xr17-0) failed", ___F);
	    goto bail;
	}
	
	if ((ctx->gpio_fd1 = open("/dev/gpio-xr17-1", O_RDWR)) < 0) {
	    pp_log_err("%s(): open(/dev/gpio-xr17-1) failed", ___F);
	    goto bail;
	}
    }

    return ctx;

 bail:
    icpmmd_free_context(ctx);
    return NULL;
}

static void
icpmmd_free_context(icpmmd_ctx_t * ctx)
{
    if (ctx) {
	if (ctx->gpio_fd0 >= 0) close(ctx->gpio_fd0);
	if (ctx->gpio_fd1 >= 0) close(ctx->gpio_fd1);
	free(ctx);
    }
}

static void
icpmmd_select_func(icpmmd_ctx_t * ctx, icpmmd_switch_type_t switch_type)
{
    assert(ctx);

    /* Configure the SELx lines */
    gpio_ioctl_out(ctx->gpio_fd1, switch_type & 0x7, ~switch_type & 0x7, 0);
}

static void
icpmmd_select_node(icpmmd_ctx_t * ctx, u_int node)
{
    assert(ctx);

    /* CLEAR_7474_N -> 1 */
    gpio_ioctl_set(ctx->gpio_fd0, 0x80);

    /* Configure BLADE_ADDRx lines (BLADE_ADDR4 always zero for now) */
    gpio_ioctl_out(ctx->gpio_fd0, node & 0xf, 0x10 | (~node & 0xf), 0);
}

static void
icpmmd_select_all_nodes(icpmmd_ctx_t * ctx)
{
    assert(ctx);

    /* CLEAR_7474_N -> 0 */
    gpio_ioctl_clr(ctx->gpio_fd0, 0x80);
}

static void
icpmmd_control_activate(icpmmd_ctx_t * ctx, u_int delay_ms)
{
    assert(ctx);

    /* FF_D_IN_N -> 0; STROBE_N -> 0 */
    gpio_ioctl_clr(ctx->gpio_fd0, 0x60);

    /* wait <delay_ms> ms */
    usleep(delay_ms * 1000);

    /* STROBE_N -> 1 */
    gpio_ioctl_set(ctx->gpio_fd0, 0x40);

    /* wait 2 ms */
    usleep(ICPMMD_DEFAULT_DELAY_MS * 1000);
}

static void
icpmmd_control_deactivate(icpmmd_ctx_t * ctx)
{
    /* FF_D_IN_N -> 1; STROBE_N -> 0 */
    gpio_ioctl_set(ctx->gpio_fd0, 0x20);
    gpio_ioctl_clr(ctx->gpio_fd0, 0x40);

    /* wait 2 ms */
    usleep(ICPMMD_DEFAULT_DELAY_MS * 1000);
    
    /* STROBE_N -> 1 */
    gpio_ioctl_set(ctx->gpio_fd0, 0x40);

    /* wait 2 ms */
    usleep(ICPMMD_DEFAULT_DELAY_MS * 1000);
}
