/**
 * Wrapper for libintl use in erla firmware
 *
 * Author: Thomas Rendelmann <thre@peppercon.de>
 * Copyright 2003 Peppercon AG
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libintl.h>
#include <locale.h>
#include <sys/stat.h>

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/intl.h>

#include "language.h"

#define LOCALEDIR	"/usr/po"
#define POFILE_DIR	"LC_MESSAGES"
#define PACKAGE		"erla"
#define MOFILE_EXT	"mo"

#define noDEBUG
#ifdef DEBUG
#define DBG(fmt, x...)  pp_log("libpp_intl: " fmt, ##x)
#else
#define DBG(fmt, x...)  ((void)0)
#endif

static int language_ch(pp_cfg_chg_ctx_t *ctx);

/* all available translations */
static vector_t *translations;

/* initialize and cleanup the library */
int pp_intl_init() {
    const char * lang;
    int i;

    // initialize the libintl library
    setenv("LANG", PP_INTL_DEFAULT_LANGUAGE, 1);
    setlocale (LC_ALL, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
    
    // find all available translations
    translations = vector_new(NULL, 1, NULL);
    vector_add_const(translations, PP_INTL_DEFAULT_LANGUAGE);
    for (i = 0; languages[i].shortname; ++i) {
        char path[PATH_MAX + 1];
        struct stat statbuf;
        
        snprintf(path, sizeof(path), "%s/%s/%s/%s.%s",
            LOCALEDIR, languages[i].shortname, POFILE_DIR, PACKAGE, MOFILE_EXT);
        
        if (stat(path, &statbuf) == 0) {
            DBG("Found language %s\n", languages[i].shortname);
            vector_add_const(translations, languages[i].shortname);
        }
    }
    
    // determine the current language setting...
    lang = pp_intl_get_language_shortname();
    // ...and apply it
    pp_intl_set_language(lang);

    pp_cfg_add_change_listener(language_ch, "language");

    return 0;
}

void pp_intl_cleanup(void) {
    vector_delete(translations);
}

/* get the current language setting */
const char * pp_intl_get_language_shortname() {
    unsigned int i;
    char *lang;
    
    pp_cfg_get(&lang, "language");
    for (i = 0; i < vector_size(translations); ++i) {
        const char *l = vector_get_const(translations, i);
	if (!strcmp(lang, l)) {
	    free(lang);
	    return l;
	}
    }
    free(lang);
    return PP_INTL_DEFAULT_LANGUAGE;
}

const char * pp_intl_get_language_fullname() {
    return pp_intl_get_fullname_from_shortname(pp_intl_get_language_shortname());
}

/* get all available translations */
int pp_intl_get_no_translations() {
    return vector_size(translations);
}

const char * pp_intl_get_translation_shortname(int no) {
    if(no >= pp_intl_get_no_translations()) {
    	no = 0;
    }
    return vector_get_const(translations, no);
}

const char * pp_intl_get_translation_fullname(int no) {
    return pp_intl_get_fullname_from_shortname(pp_intl_get_translation_shortname(no));
}

const char * pp_intl_get_translation_utf8name(int no) {
    return pp_intl_get_utf8name_from_shortname(pp_intl_get_translation_shortname(no));
}

/* get all possible languages */
int pp_intl_get_no_possible_languages(void) {
    int i;
    for (i = 0; languages[i].shortname; ++i) ;
    return i;
}

const char * pp_intl_get_possible_language_shortname(int no) {
    if(no >= pp_intl_get_no_possible_languages()) {
    	no = 0;
    }
    return languages[no].shortname;
}

const char * pp_intl_get_possible_language_fullname(int no) {
    return pp_intl_get_fullname_from_shortname(pp_intl_get_possible_language_shortname(no));
}

const char * pp_intl_get_possible_language_utf8name(int no) {
    return pp_intl_get_utf8name_from_shortname(pp_intl_get_possible_language_shortname(no));
}

/* set a new language for the interface */
extern int  _nl_msg_cat_cntr;

void pp_intl_set_language(const char * language) {
    setenv ("LANG", language, 1);
    ++_nl_msg_cat_cntr;
    
    DBG("language: %s\n", language);
}

/* get the character set of the currently set language */
const char * pp_intl_get_charset() {
    const char *cs = getcharset();	// from libintl
    const char *l;
    
    // if no character set is set and we use english (means, we have no
    // translation active), default to UTF-8
    if (cs == NULL &&
        (l = pp_intl_get_language_shortname()) &&
        !strcmp(l, PP_INTL_DEFAULT_LANGUAGE)) {
        cs = "UTF-8";
    }
    
    return cs;
}

/* do the actual translation */
const char * pp_intl_translate(const char * s) {
    return (s ? ((s[0] != '\0') ? gettext(s) : s) : NULL);
}

/* other functions */
const char *pp_intl_get_fullname_from_shortname(const char *shortname) {
    int i;
    
    for (i = 0; languages[i].shortname; ++i) {
	if (!strcmp(shortname, languages[i].shortname)) {
	    return languages[i].fullname;
	}
    }
    
    return NULL;
}

const char *pp_intl_get_utf8name_from_shortname(const char *shortname) {
    int i;
    
    for (i = 0; languages[i].shortname; ++i) {
	if (!strcmp(shortname, languages[i].shortname)) {
	    return languages[i].utf8name;
	}
    }
    
    return NULL;
}

/* language change listener */
static int language_ch(pp_cfg_chg_ctx_t *ctx UNUSED) {
    pp_intl_set_language(pp_intl_get_language_shortname());
    return 0;
}
