#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <pp/base.h>
#include <pp/hash.h>
#include <pp/ipc.h>
#include "ipc_socket.h"

#define MAX_REQ_PDU (PP_IPC_SOCKET_MTU + sizeof(pp_ipc_req_head_t))

struct pp_ipc_req_ctx_s {
    int socket;
    struct sockaddr_un sa;
    socklen_t msg_namelen;
    int msg_flags;
};

static int initialized = 0;
static pthread_t listener_thread;
static volatile int terminate;

pthread_mutex_t handlers_mtx = PTHREAD_MUTEX_INITIALIZER;
pp_hash_i_t *request_handlers;

static void *listener_thread_func(void *data UNUSED)
{
    int sock;
    struct sockaddr_un sa;
    struct iovec iov;
    struct msghdr msg;

    pp_ipc_req_head_t *request;

    sa.sun_family = AF_UNIX;
    strcpy(sa.sun_path, PP_IPC_SOCKET_PATH);
    unlink(PP_IPC_SOCKET_PATH);
    if ( (sock = socket(AF_UNIX, SOCK_DGRAM, 0)) < 0
	|| (bind(sock, (struct sockaddr *)&sa, sizeof(sa))) < 0) {
	pp_log_err("Failed to open control socket");
	return NULL;
    }

    request = malloc(MAX_REQ_PDU);
    iov.iov_base = request;
    iov.iov_len = MAX_REQ_PDU;
    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &sa;
    msg.msg_namelen = sizeof(sa);
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;

    for (;;) {
	pp_ipc_req_handler_t handler;
	int length = recvmsg(sock, &msg, 0);
	if (terminate) break;
	if (length < 0) {
	    pp_log_err("recvmsg() failed");
	    break;
	} else if (length < (int)sizeof(pp_ipc_req_head_t)) {
	    pp_log("Short control request received.\n");
	    continue;
	}

	pthread_mutex_lock(&handlers_mtx);
	handler = pp_hash_get_entry_i(request_handlers, request->request_type);
	if (!handler) {
	    pp_log("Unhandled control request type %d.\n", request->request_type);
	} else {
	    pp_ipc_req_ctx_t ctx;
	    ctx.socket = sock;
	    ctx.sa = sa;
	    ctx.msg_namelen = msg.msg_namelen;
	    ctx.msg_flags = msg.msg_flags;
	    handler(&ctx, request->request_type,
		    length - sizeof(pp_ipc_req_head_t), request->data);
	}
	pthread_mutex_unlock(&handlers_mtx);
    }

    free(request);
    close(sock);
    return NULL;
}

int ipc_socket_init(void)
{
    if (initialized) return PP_ERR;
    terminate = 0;
    if (pthread_create(&listener_thread, NULL, listener_thread_func, NULL) != 0) {
	pp_log_err("Failed to create control socket listener thread");
	return PP_ERR;
    };
    request_handlers = pp_hash_create_i(7);
    initialized = 1;
    return PP_SUC;
}

int ipc_socket_cleanup(void)
{
    int sock;
    struct sockaddr_un sa;
    struct msghdr msg;

    if (!initialized) return PP_ERR;

    terminate = 1;

    // send an empty message to interrupt the recvmsg() call
    sock = socket(AF_UNIX, SOCK_DGRAM, 0);
    sa.sun_family = AF_UNIX;
    strcpy(sa.sun_path, PP_IPC_SOCKET_PATH);
    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &sa;
    msg.msg_namelen = sizeof(sa);
    sendmsg(sock, &msg, 0);
    close(sock);

    pthread_join(listener_thread, NULL);
    pp_hash_delete_i(request_handlers);
    initialized = 0;
    return PP_SUC;
}

/**
 * Register a handler function for a particular request type
 *
 * @param reqtype  Request type code
 * @param handler  Handler function
 *
 * @return  PP_SUC or PP_ERR
 *
 * Note: The message context that will be passed to the handler is
 *       invalidated after the handler returns. It must be duplicated
 *       with pp_ipc_ctx_dup() for response messages to be generated
 *       outside of the handler.
 */
int pp_ipc_register_handler(pp_ipc_req_type_t reqtype,
	pp_ipc_req_handler_t handler)
{
    int ret = PP_ERR;
    pthread_mutex_lock(&handlers_mtx);
    if (pp_hash_get_entry_i(request_handlers, reqtype)) {
	pp_log("Handler for control request type %d already registered.\n", reqtype);
    } else {
	pp_hash_set_entry_i(request_handlers, reqtype, handler, NULL);
	ret = PP_SUC;
    }
    pthread_mutex_unlock(&handlers_mtx);
    return ret;
}

/**
 * Unregister a handler function
 *
 * @param reqtype  Request type code
 */
void pp_ipc_unregister_handler(pp_ipc_req_type_t reqtype)
{
    pthread_mutex_lock(&handlers_mtx);
    pp_hash_delete_entry_i(request_handlers, reqtype);
    pthread_mutex_unlock(&handlers_mtx);
}

/**
 * Send a response to the orignator of the request
 *
 * @param ctx     Message context; see note above
 * @param status  Response status code
 * @param length  Length of payload data in bytes
 * @param data    Response payload buffer
 *
 * @return  PP_SUC or PP_ERR
 */
int pp_ipc_send_response(pp_ipc_req_ctx_t *ctx,
	pp_ipc_rsp_status_t status, int length, unsigned char *data)
{
    struct iovec iov[2];
    struct msghdr msg;

    iov[0].iov_base = &status;
    iov[0].iov_len = sizeof(status);
    iov[1].iov_base = data;
    iov[1].iov_len = length;

    memset(&msg, 0, sizeof(msg));
    msg.msg_name = &ctx->sa;
    msg.msg_namelen = ctx->msg_namelen;
    msg.msg_iov = iov;
    msg.msg_iovlen = 2;
    msg.msg_flags = ctx->msg_flags;

    if (sendmsg(ctx->socket, &msg, 0) < 0) {
	pp_log_err("sendmsg() failed");
	return PP_ERR;
    }
    return PP_SUC;
}

/**
 * Duplicate a request context for later responses
 *
 * @param ctx  Context to duplicate
 *
 * @return  New context, NULL in case of error; must be free'd by caller.
 */
pp_ipc_req_ctx_t *pp_ipc_ctx_dup(pp_ipc_req_ctx_t *ctx)
{
    pp_ipc_req_ctx_t *ret = malloc(sizeof(pp_ipc_req_ctx_t));
    memcpy(ret, ctx, sizeof(pp_ipc_req_ctx_t));
    return ret;
}

