#include <stdio.h>

#ifdef _WIN32
#include <pp/strstream.h>
#include <sleep.h>
#else
#include <pp/base.h>
#endif
#include <pp/ipmi.h>

#include <config.h>

#include "ipmi_test.h"

#ifdef PP_IPMI_DEBUG
/*
#include "ipmi_params.h"
static void cmdline_test(void) {
    t_ipmitool_commandline *cmdline = init_cmdline();
    char *cmd = "bla blubb doof=dumm \"hello world\" bla2 \"hello again\"";
    
    if (!cmdline_add_command(cmdline, cmd)) {
    	int i;
    	printf("%d commands:\n", cmdline->argc);
    	for (i = 0; i < cmdline->argc; i++) {
    	    printf("%s\n", cmdline->argv[i]);
    	}
    } else {
    	printf("Error parsing commandline.\n");
    }
}

static int pp_ipmi_test_bmc_info(void) {
    pp_ipmi_return_t ipmi_ret;
    int size;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    if (pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_INFO, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Error in sending BMC INFO command.\n");
    	goto bail;
    }
    
    printf("Device ID                 : %i\n", ipmi_ret.data.bmc_info.device_id);
    printf("Device Revision           : %i\n", ipmi_ret.data.bmc_info.device_revision);
    printf("Firmware Revision         : %s\n", ipmi_ret.data.bmc_info.firmware_revision.buf);
    printf("IPMI Version              : %s\n", ipmi_ret.data.bmc_info.ipmi_version.buf);
    printf("Manufacturer ID           : %lu\n", ipmi_ret.data.bmc_info.manufacturer_id);
    printf("Product ID                : %ld (0x%04lx)\n", ipmi_ret.data.bmc_info.product_id, ipmi_ret.data.bmc_info.product_id);
    printf("Device Available          : %s\n", ipmi_ret.data.bmc_info.device_available ? "yes" : "no");
    printf("Provides Device SDRs      : %s\n", ipmi_ret.data.bmc_info.provides_sdrs ? "yes" : "no");
    printf("Additional Device Support :\n");
    size = vector_size(ipmi_ret.data.bmc_info.additional_device_support);
    if (size == 0) {
    	printf("    none.\n");
    } else {
    	int i;
    	for (i = 0; i < size; i++) {
    	    printf("    %s\n", (char *)vector_get(ipmi_ret.data.bmc_info.additional_device_support, i));
    	}
    }
    printf("Aux Firmware Rev Info     : \n");
    printf("    0x%02x\n    0x%02x\n    0x%02x\n    0x%02x\n",
		ipmi_ret.data.bmc_info.aux_fw_info[0], ipmi_ret.data.bmc_info.aux_fw_info[1],
		ipmi_ret.data.bmc_info.aux_fw_info[2], ipmi_ret.data.bmc_info.aux_fw_info[3]);
    
 bail:
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_chassis_power_status(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POWER_STATUS, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("Power Status              : %s\n", ipmi_ret.data.chassis_power_status.status_string.buf);
    } else {
    	printf("IPMI: Error in sending CHASSIS POWER STATUS command.\n");
    }
    
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_chassis_power_down(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    
    ipmi_param.data.chassis_power = PP_IPMI_CHASSIS_POWER_DOWN;
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL, &ipmi_param, &ipmi_ret, NULL, NULL)) {
    	printf("Successfully powered off.");
    } else {
    	printf("IPMI: Error in sending CHASSIS POWER CONTROL command.\n");
    }
    
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_chassis_power_up(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    
    ipmi_param.data.chassis_power = PP_IPMI_CHASSIS_POWER_UP;
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL, &ipmi_param, &ipmi_ret, NULL, NULL)) {
    	printf("Successfully powered off.");
    } else {
    	printf("IPMI: Error in sending CHASSIS POWER CONTROL command.\n");
    }
    
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_chassis_status(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_STATUS, NULL, &ipmi_ret, NULL, NULL)) {
    	pp_ipmi_chassis_status_t *status = &ipmi_ret.data.chassis_status;
    	printf("Power Status         : %s\n", status->power_status_string.buf);
    	printf("Power Overload       : %s\n", status->power_overload ? "true" : "false");
    	printf("Power Interlock      : %s\n", status->power_interlock_active ? "active" : "inactive");
    	printf("Main Power Fault     : %s\n", status->main_power_fault ? "true" : "false");
    	printf("Power Control Fault  : %s\n", status->power_control_fault ? "true" : "false");
    	printf("Power Restore Policy : %s\n", status->power_restore_policy.string.buf);
    	printf("Last Power Event     : %s\n", status->last_power_event.buf);
    	printf("Chassis Intrusion    : %s\n", status->chassis_intrusion_active ? "active" : "inactive");
    	printf("Front-Panel Lockout  : %s\n", status->front_panel_lockout_active ? "active" : "inactive");
    	printf("Drive Fault          : %s\n", status->drive_fault ? "true" : "false");
    	printf("Cooling/Fan Fault    : %s\n", status->cooling_fan_fault ? "true" : "false");
    	if (status->front_panel_control_info_valid) {
    	    if (!status->front_panel_control) {
    	    	printf("Front Panel Control  : none\n");
    	    } else {
    	    	printf("Sleep Button Disable : %s\n", status->sleep_allowed ? "allowed" : "not allowed");
    	    	printf("Diag Button Disable  : %s\n", status->diag_allowed ? "allowed" : "not allowed");
    	    	printf("Reset Button Disable : %s\n", status->reset_allowed ? "allowed" : "not allowed");
    	    	printf("Power Button Disable : %s\n", status->power_allowed ? "allowed" : "not allowed");
    	    	printf("Sleep Button Disabled: %s\n", status->sleep_disabled ? "true" : "false");
    	    	printf("Diag Button Disabled : %s\n", status->diag_disabled ? "true" : "false");
    	    	printf("Reset Button Disabled: %s\n", status->reset_disabled ? "true" : "false");
    	    	printf("Power Button Disabled: %s\n", status->power_disabled ? "true" : "false");
    	    }
    	}
    } else {
    	printf("IPMI: Error in sending CHASSIS STATUS command.\n");
    }
    
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_chassis_poh(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POH, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("POH Counter               : %s\n", ipmi_ret.data.chassis_poh.buf);
    } else {
    	printf("IPMI: Error in sending CHASSIS POH command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sel_info(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_INFO, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("SEL Information\n");
    	printf("Version          : %x\n", ipmi_ret.data.sel_info.version);
    	printf("Entries          : %d\n", ipmi_ret.data.sel_info.entries);
    	printf("Free Space       : %d\n", ipmi_ret.data.sel_info.free_space);
    	printf("Last Add Time    : %s %s\n", 
    		ipmi_ret.data.sel_info.last_add_time.is_unspecified ? "Unspecified" : ipmi_ret.data.sel_info.last_add_time.string_date.buf,
    		ipmi_ret.data.sel_info.last_add_time.is_unspecified ? "" : ipmi_ret.data.sel_info.last_add_time.string_time.buf);
    	printf("Last Del Time    : %s %s\n", 
    		ipmi_ret.data.sel_info.last_del_time.is_unspecified ? "Unspecified" : ipmi_ret.data.sel_info.last_del_time.string_date.buf,
    		ipmi_ret.data.sel_info.last_del_time.is_unspecified ? "" : ipmi_ret.data.sel_info.last_del_time.string_time.buf);
    	printf("Overflow         : %s\n", ipmi_ret.data.sel_info.overflow ? "true" : "false");
    	printf("Delete cmd       : %ssupported\n", ipmi_ret.data.sel_info.delete_supported ? "" : "un");
    	printf("Parial add cmd   : %ssupported\n", ipmi_ret.data.sel_info.partial_add_supported ? "" : "un");
    	printf("Reserve cmd      : %ssupported\n", ipmi_ret.data.sel_info.reserve_supported ? "" : "un");
    	printf("Get Alloc Info   : %ssupported\n", ipmi_ret.data.sel_info.get_alloc_info_supported ? "" : "un");
    	if (ipmi_ret.data.sel_info.get_alloc_info_supported) {
    	    if (ipmi_ret.data.sel_info.allocation_info.alloc_units.specified) {
    	    	printf("# of Alloc Units : %d\n", ipmi_ret.data.sel_info.allocation_info.alloc_units.units);
    	    } else {
    	    	printf("# of Alloc Units : Unspecified\n");
    	    }
    	    if (ipmi_ret.data.sel_info.allocation_info.alloc_unit_size.specified) {
    	    	printf("Alloc Unit Size  : %d\n", ipmi_ret.data.sel_info.allocation_info.alloc_unit_size.unit_size);
    	    } else {
    	    	printf("Alloc Unit Size  : Unspecified\n");
    	    }
    	    printf("# Free Units     : %d\n", ipmi_ret.data.sel_info.allocation_info.free_units);
    	    printf("Largest Free Blk : %d\n", ipmi_ret.data.sel_info.allocation_info.largest_free_block);
    	    printf("Max Record Size  : %d\n", ipmi_ret.data.sel_info.allocation_info.max_record_size);
    	}
    } else {
    	printf("IPMI: Error in sending SEL INFO command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sel_clear(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_CLEAR, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("Cleared System Event Log.\n");
    } else {
    	printf("IPMI: Error in sending SEL CLEAR command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sel_get_time(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_GET_TIME, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("SEL time: %s %s (0x%x).\n", ipmi_ret.data.sel_get_time.string_date.buf,
    	       ipmi_ret.data.sel_get_time.string_time.buf, (unsigned int)ipmi_ret.data.sel_get_time.timestamp);
    } else {
    	printf("IPMI: Error in sending SEL GET_TIME command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sel_list(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    
    ipmi_param.data.sel_list.range = PP_IPMI_SEL_LIST_RANGE_FROMTO;
    ipmi_param.data.sel_list.from = 0;
    ipmi_param.data.sel_list.to = 100000;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_LIST, &ipmi_param, &ipmi_ret, NULL, NULL)) {
    	int size, i;
    	
    	size = vector_size(ipmi_ret.data.sel_list);
    	
    	printf("Fetched System Event Log.\n");
    	printf("SEL has %u entries.\n", size);
    	
    	for (i = 0; i < size; i++) {
    	    pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);
    	    
    	    if (!entry) {
    	    	printf("Error: could not read entry %d\n", i);
    	    	continue;
    	    }
    	    
    	    printf("Entry %d\n", i);
    	    printf("  Record ID           : %04x\n", entry->record_id);
    	    printf("  Record type         : %s\n", entry->record_type_string.buf);
    	    
    	    if (entry->timestamp.is_valid) {
    	    	printf("  Timestamp           : %s %s\n",
    	    	       entry->timestamp.is_pre_init ? "Pre-Init" : entry->timestamp.string_date.buf,
    	    	       entry->timestamp.is_pre_init ? "" : entry->timestamp.string_time.buf);
    	    }
    	    
    	    if (!entry->further_data_is_valid) {
    	    	continue;
    	    }
    	    
    	    printf("  Generator ID        : %04x\n", entry->generator_id);
    	    printf("  EvM Revision        : %02x\n", entry->evm_revision);
    	    printf("  Sensor Type         : %s\n", entry->sensor_type.buf);
    	    printf("  Sensor Number       : %02x\n", entry->sensor_num);
    	    printf("  Sensor Name         : %s\n", entry->sensor_name.buf);
    	    printf("  Event Type          : %s\n", entry->event_type.buf);
    	    printf("  Event Direction     : %s\n", entry->event_direction.buf);
    	    printf("  Event Data          : %02x%02x%02x\n", entry->event_data[0], entry->event_data[1], entry->event_data[2]);
    	    printf("  Description         : %s\n", entry->description.buf);
    	}
    } else {
    	printf("IPMI: Error in sending SEL LIST command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sdr_info(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_INFO, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("SEL Information\n");
    	printf("Version          : %x\n", ipmi_ret.data.sdr_info.version);
    	printf("Record Count     : %d\n", ipmi_ret.data.sdr_info.record_count);
    	printf("Free Space       : %s\n", ipmi_ret.data.sdr_info.free_space.string.buf);
    	printf("Last Add Time    : %s %s\n", 
    		ipmi_ret.data.sdr_info.most_resent_addition.is_unspecified ? "Unspecified" : ipmi_ret.data.sdr_info.most_resent_addition.string_date.buf,
    		ipmi_ret.data.sdr_info.most_resent_addition.is_unspecified ? "" : ipmi_ret.data.sdr_info.most_resent_addition.string_time.buf);
    	printf("Last Del Time    : %s %s\n", 
    		ipmi_ret.data.sdr_info.most_resent_erase.is_unspecified ? "Unspecified" : ipmi_ret.data.sdr_info.most_resent_erase.string_date.buf,
    		ipmi_ret.data.sdr_info.most_resent_erase.is_unspecified ? "" : ipmi_ret.data.sdr_info.most_resent_erase.string_time.buf);
    	printf("Overflow         : %s\n", ipmi_ret.data.sdr_info.overflow ? "true" : "false");
    	printf("Update Support   : %s\n", ipmi_ret.data.sdr_info.modal_update_support.buf);
    	printf("Delete cmd       : %ssupported\n", ipmi_ret.data.sdr_info.delete_supported ? "" : "un");
    	printf("Parial add cmd   : %ssupported\n", ipmi_ret.data.sdr_info.partial_add_supported ? "" : "un");
    	printf("Reserve cmd      : %ssupported\n", ipmi_ret.data.sdr_info.reserve_supported ? "" : "un");
    	printf("Get Alloc Info   : %ssupported\n", ipmi_ret.data.sdr_info.get_alloc_info_supported ? "" : "un");
    	if (ipmi_ret.data.sdr_info.get_alloc_info_supported) {
    	    if (ipmi_ret.data.sdr_info.allocation_info.alloc_units.specified) {
    	    	printf("# of Alloc Units : %d\n", ipmi_ret.data.sdr_info.allocation_info.alloc_units.units);
    	    } else {
    	    	printf("# of Alloc Units : Unspecified\n");
    	    }
    	    if (ipmi_ret.data.sdr_info.allocation_info.alloc_unit_size.specified) {
    	    	printf("Alloc Unit Size  : %d\n", ipmi_ret.data.sdr_info.allocation_info.alloc_unit_size.unit_size);
    	    } else {
    	    	printf("Alloc Unit Size  : Unspecified\n");
    	    }
    	    printf("# Free Units     : %d\n", ipmi_ret.data.sdr_info.allocation_info.free_units);
    	    printf("Largest Free Blk : %d\n", ipmi_ret.data.sdr_info.allocation_info.largest_free_block);
    	    printf("Max Record Size  : %d\n", ipmi_ret.data.sdr_info.allocation_info.max_record_size);
    	}
    } else {
    	printf("IPMI: Error in sending SEL INFO command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sdr_list(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t params;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&params, 0, sizeof(params));
    params.data.sdr_list = PP_IPMI_SDR_LIST_FULL;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_LIST, &params, &ipmi_ret, NULL, NULL)) {
    	int size, i;
    	
    	size = vector_size(ipmi_ret.data.sdr_list);

    	printf("Successfully queried SDR list.\n");
    	printf("SDR has %u entries.\n", size);
    	
    	for (i = 0; i < size; i++) {
    	    pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sdr_list, i);

    	    if (!entry) {
    	    	printf("Error: could not read entry %d\n", i);
    	    	continue;
    	    }
    	    
    	    printf("Entry %d\n", i);
    	    printf("  ID                  : %02X\n", entry->sdr_id);
    	    printf("  Name                : %s\n", entry->sdr_name.buf ? entry->sdr_name.buf : "Unknown");
    	    if (entry->entity.is_valid) {
    	    	printf("  Entity ID           : %d.%d (%s)\n", entry->entity.id, entry->entity.instance, entry->entity.string.buf ? entry->entity.string.buf : "");
    	    }
    	    printf("  Sensor (%s) Type  : %s\n", entry->sdr_type_main.buf ? entry->sdr_type_main.buf : "", entry->sdr_type_sub.buf ? entry->sdr_type_sub.buf : "");
    	    
    	    if (entry->type != PP_IPMI_SENSOR_TYPE_FULL) {
    	    	printf("  non-full sensor, discarding.\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.reading_present != PP_IPMI_SENSOR_READING_OK) {
    	    	printf("  Sensor Reading      : Not present\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_DISCRETE) {
    	    	printf("  Sensor Reading      : 0x%x\n", (unsigned int) entry->data.full.reading.value);
    	    } else {
    	    	printf("  Sensor Reading      : %.*f (+/- %.*f) %s\n",
                       (entry->data.full.reading.value==(int)entry->data.full.reading.value) ? 0 : 3, 
                       entry->data.full.reading.value, 
                       (entry->data.full.tolerance.value==(int)entry->data.full.tolerance.value) ? 0 : 3, 
                       entry->data.full.tolerance.value, 
                       entry->data.full.unit.buf ? entry->data.full.unit.buf : "");
    	    }
    	    
    	    if (entry->data.full.status != PP_IPMI_SENSOR_STATUS_NO_READING) {
    	    	printf("  Sensor Status       : %s\n", entry->data.full.status_string.buf ? entry->data.full.status_string.buf : "Unknown");
    	    }
    	}
    } else {
    	printf("IPMI: Error in sending SDR LIST command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sensor_by_name(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t param;
    vector_t *sensor_vec = vector_new(NULL, 0, NULL);
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    vector_add(sensor_vec, "Planar Temp 1");
    vector_add(sensor_vec, "Planar 5V");
    vector_add(sensor_vec, "blabla");
    vector_add(sensor_vec, "3.3V");
    
    memset(&param, 0, sizeof(param));
    param.data.sensor_get_by_name_list = sensor_vec;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SENSOR, PP_IPMI_SENSOR_SUBCMD_GET_BY_NAME, &param, &ipmi_ret, NULL, NULL)) {
    	int size, i;
    	
    	size = vector_size(ipmi_ret.data.sdr_list);

    	printf("Successfully queried sensor list.\n");
    	printf("sensor list has %u entries.\n", size);
    	
    	for (i = 0; i < size; i++) {
    	    pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sdr_list, i);

    	    if (!entry) {
    	    	printf("Error: could not read entry %d\n", i);
    	    	continue;
    	    }
    	    
    	    printf("Entry %d\n", i);
    	    printf("  ID                  : %02X\n", entry->sdr_id);
    	    printf("  Name                : %s\n", entry->sdr_name.buf ? entry->sdr_name.buf : "Unknown");
    	    if (entry->entity.is_valid) {
    	    	printf("  Entity ID           : %d.%d (%s)\n", entry->entity.id, entry->entity.instance, entry->entity.string.buf ? entry->entity.string.buf : "");
    	    }
    	    printf("  Sensor (%s) Type  : %s\n", entry->sdr_type_main.buf ? entry->sdr_type_main.buf : "", entry->sdr_type_sub.buf ? entry->sdr_type_sub.buf : "");
    	    
    	    if (entry->type != PP_IPMI_SENSOR_TYPE_FULL) {
    	    	printf("  non-full sensor, discarding.\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.reading_present != PP_IPMI_SENSOR_READING_OK) {
    	    	printf("  Sensor Reading      : Not present\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_DISCRETE) {
    	    	printf("  Sensor Reading      : 0x%x\n", (unsigned int) entry->data.full.reading.value);
    	    } else {
    	    	printf("  Sensor Reading      : %.*f (+/- %.*f) %s\n",
                       (entry->data.full.reading.value==(int)entry->data.full.reading.value) ? 0 : 3, 
                       entry->data.full.reading.value, 
                       (entry->data.full.tolerance.value==(int)entry->data.full.tolerance.value) ? 0 : 3, 
                       entry->data.full.tolerance.value, 
                       entry->data.full.unit.buf ? entry->data.full.unit.buf : "");
    	    }
    	    
    	    if (entry->data.full.status != PP_IPMI_SENSOR_STATUS_NO_READING) {
    	    	printf("  Sensor Status       : %s\n", entry->data.full.status_string.buf ? entry->data.full.status_string.buf : "Unknown");
    	    }
    	}
    } else {
    	printf("IPMI: Error in sending sensor LIST command.\n");
    }

    vector_delete(sensor_vec);
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_test_sensor_by_id(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t param;
    vector_t *sensor_vec = vector_new2(NULL, 0, sizeof(unsigned int), NULL);
	unsigned int no;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
	no = 1;
    vector_add2(sensor_vec, &no);
	no = 5;
    vector_add2(sensor_vec, &no);
	no = 27;
    vector_add2(sensor_vec, &no);
    
    memset(&param, 0, sizeof(param));
    param.data.sensor_get_by_name_list = sensor_vec;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SENSOR, PP_IPMI_SENSOR_SUBCMD_GET_BY_ID, &param, &ipmi_ret, NULL, NULL)) {
    	int size, i;
    	
    	size = vector_size(ipmi_ret.data.sdr_list);

    	printf("Successfully queried sensor list.\n");
    	printf("sensor list has %u entries.\n", size);
    	
    	for (i = 0; i < size; i++) {
    	    pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sdr_list, i);

    	    if (!entry) {
    	    	printf("Error: could not read entry %d\n", i);
    	    	continue;
    	    }
    	    
    	    printf("Entry %d\n", i);
    	    printf("  ID                  : %02X\n", entry->sdr_id);
    	    printf("  Name                : %s\n", entry->sdr_name.buf ? entry->sdr_name.buf : "Unknown");
    	    if (entry->entity.is_valid) {
    	    	printf("  Entity ID           : %d.%d (%s)\n", entry->entity.id, entry->entity.instance, entry->entity.string.buf ? entry->entity.string.buf : "");
    	    }
    	    printf("  Sensor (%s) Type  : %s\n", entry->sdr_type_main.buf ? entry->sdr_type_main.buf : "", entry->sdr_type_sub.buf ? entry->sdr_type_sub.buf : "");
    	    
    	    if (entry->type != PP_IPMI_SENSOR_TYPE_FULL) {
    	    	printf("  non-full sensor, discarding.\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.reading_present != PP_IPMI_SENSOR_READING_OK) {
    	    	printf("  Sensor Reading      : Not present\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_DISCRETE) {
    	    	printf("  Sensor Reading      : 0x%x\n", (unsigned int) entry->data.full.reading.value);
    	    } else {
    	    	printf("  Sensor Reading      : %.*f (+/- %.*f) %s\n",
                       (entry->data.full.reading.value==(int)entry->data.full.reading.value) ? 0 : 3, 
                       entry->data.full.reading.value, 
                       (entry->data.full.tolerance.value==(int)entry->data.full.tolerance.value) ? 0 : 3, 
                       entry->data.full.tolerance.value, 
                       entry->data.full.unit.buf ? entry->data.full.unit.buf : "");
    	    }
    	    
    	    if (entry->data.full.status != PP_IPMI_SENSOR_STATUS_NO_READING) {
    	    	printf("  Sensor Status       : %s\n", entry->data.full.status_string.buf ? entry->data.full.status_string.buf : "Unknown");
    	    }
    	}
    } else {
    	printf("IPMI: Error in sending sensor LIST command.\n");
    }

    vector_delete(sensor_vec);
    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_kvm_test(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_KVM, PP_IPMI_KVM_SUBCMD_RESET, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent KVM RESET command.\n");
    } else {
    	printf("IPMI: Error in sending KVM RESET command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_bmc_reset_test(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t params;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&params, 0, sizeof(params));
    params.data.bmc_reset = PP_IPMI_BMC_RESET_COLD;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_RESET, &params, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent BMC RESET command.\n");
    } else {
    	printf("IPMI: Error in sending BMC RESET command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_user_test(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t params;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&params, 0, sizeof(params));
    params.is_cmdline = 1;
    params.data.cmdline = "23";

    if (!pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_SET_NAME, &params, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent USER SET_NAME command.\n");
    } else {
    	printf("IPMI: Error in sending USER SET_NAME command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_sel_delete_test(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t params;
    int i;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&params, 0, sizeof(params));
    params.data.sel_delete_list = vector_new2(NULL, 0, sizeof(unsigned int), NULL);
    
    i = 1;
    vector_add2(params.data.sel_delete_list, &i);
    i = 3;
    vector_add2(params.data.sel_delete_list, &i);

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_DELETE, &params, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent SEL DELETE command.\n");
    } else {
    	printf("IPMI: Error in sending SEL DELETE command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    vector_delete(params.data.sel_delete_list);
    
    return 0;
}

static int pp_ipmi_bmc_get_enables_test(void) {
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_GET_ENABLES, NULL, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent BMC GET_ENABLES command.\n");
    	printf("Receive Message Queue Interrupt          : %sabled\n", ipmi_ret.data.bmc_enables.receive_msg_int_enabled ? "en" : "dis");
    	printf("Event Message Buffer Full Interrupt      : %sabled\n", ipmi_ret.data.bmc_enables.event_msg_buf_int_enabled ? "en" : "dis");
    	printf("Event Message Buffer                     : %sabled\n", ipmi_ret.data.bmc_enables.event_msg_buf_enabled ? "en" : "dis");
    	printf("System Event Logging                     : %sabled\n", ipmi_ret.data.bmc_enables.sel_enabled ? "en" : "dis");
    	printf("OEM 0                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem0_enabled ? "en" : "dis");
    	printf("OEM 1                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem1_enabled ? "en" : "dis");
    	printf("OEM 2                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem2_enabled ? "en" : "dis");
    } else {
    	printf("IPMI: Error in sending BMC GET_ENABLES command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_bmc_set_enables_test(void) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    
    ipmi_param.data.bmc_enables.receive_msg_int_enabled = 0;
    ipmi_param.data.bmc_enables.event_msg_buf_int_enabled = 0;
    ipmi_param.data.bmc_enables.event_msg_buf_enabled = 0;
    ipmi_param.data.bmc_enables.sel_enabled = 0;
    ipmi_param.data.bmc_enables.oem0_enabled = 0;
    ipmi_param.data.bmc_enables.oem1_enabled = 0;
    ipmi_param.data.bmc_enables.oem2_enabled = 0;

    if (!pp_ipmi_send_command(PP_IPMI_CMD_BMC, PP_IPMI_BMC_SUBCMD_SET_ENABLES, &ipmi_param, &ipmi_ret, NULL, NULL)) {
    	printf("IPMI: Successfully sent BMC SET_ENABLES command.\n");
    	printf("Receive Message Queue Interrupt          : %sabled\n", ipmi_ret.data.bmc_enables.receive_msg_int_enabled ? "en" : "dis");
    	printf("Event Message Buffer Full Interrupt      : %sabled\n", ipmi_ret.data.bmc_enables.event_msg_buf_int_enabled ? "en" : "dis");
    	printf("Event Message Buffer                     : %sabled\n", ipmi_ret.data.bmc_enables.event_msg_buf_enabled ? "en" : "dis");
    	printf("System Event Logging                     : %sabled\n", ipmi_ret.data.bmc_enables.sel_enabled ? "en" : "dis");
    	printf("OEM 0                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem0_enabled ? "en" : "dis");
    	printf("OEM 1                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem1_enabled ? "en" : "dis");
    	printf("OEM 2                                    : %sabled\n", ipmi_ret.data.bmc_enables.oem2_enabled ? "en" : "dis");
    } else {
    	printf("IPMI: Error in sending BMC SET_ENABLES command.\n");
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}

static int pp_ipmi_sensor_list_test() {
    int error;
    pp_ipmi_return_t ipmi_ret;
    
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
    if (!pp_ipmi_send_command(PP_IPMI_CMD_SENSOR, PP_IPMI_SENSOR_SUBCMD_LIST, NULL, &ipmi_ret, &error, NULL)) {
    	int size, i;
    	
    	size = vector_size(ipmi_ret.data.sdr_list);

    	printf("Successfully queried sensor list.\n");
    	printf("sensor list has %u entries.\n", size);
    	
    	for (i = 0; i < size; i++) {
    	    pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sensor_list, i);

    	    if (!entry) {
    	    	printf("Error: could not read entry %d\n", i);
    	    	continue;
    	    }
    	    
    	    printf("Entry %d\n", i);
    	    printf("  ID                  : %02X\n", entry->sdr_id);
    	    printf("  Name                : %s\n", entry->sdr_name.buf ? entry->sdr_name.buf : "Unknown");
    	    if (entry->entity.is_valid) {
    	    	printf("  Entity ID           : %d.%d (%s)\n", entry->entity.id, entry->entity.instance, entry->entity.string.buf ? entry->entity.string.buf : "");
    	    }
    	    printf("  Sensor (%s) Type  : %s\n", entry->sdr_type_main.buf ? entry->sdr_type_main.buf : "", entry->sdr_type_sub.buf ? entry->sdr_type_sub.buf : "");
    	    
    	    if (entry->type != PP_IPMI_SENSOR_TYPE_FULL) {
    	    	printf("  non-full sensor, discarding.\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.reading_present != PP_IPMI_SENSOR_READING_OK) {
    	    	printf("  Sensor Reading      : Not present\n");
    	    	continue;
    	    }
    	    
    	    if (entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_DISCRETE) {
    	    	printf("  Sensor Reading      : 0x%x\n", (unsigned int) entry->data.full.reading.value);
    	    } else {
    	    	printf("  Sensor Reading      : %.*f (+/- %.*f) %s\n",
                       (entry->data.full.reading.value==(int)entry->data.full.reading.value) ? 0 : 3, 
                       entry->data.full.reading.value, 
                       (entry->data.full.tolerance.value==(int)entry->data.full.tolerance.value) ? 0 : 3, 
                       entry->data.full.tolerance.value, 
                       entry->data.full.unit.buf ? entry->data.full.unit.buf : "");
    	    }
    	    
    	    if (entry->data.full.status != PP_IPMI_SENSOR_STATUS_NO_READING) {
    	    	printf("  Sensor Status       : %s\n", entry->data.full.status_string.buf ? entry->data.full.status_string.buf : "Unknown");
    	    }
    	}
    } else {
    	printf("IPMI: Error in sending sensor LIST command :%s.\n",
    	    pp_ipmi_get_error_string(error));
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return 0;
}
*/

#endif // PP_IPMI_DEBUG

int ipmi_test(void) {
#ifdef PP_IPMI_DEBUG
    //sleep(2);
    //cmdline_test();
    //pp_ipmi_test_bmc_info();
    //pp_ipmi_test_chassis_power_status();
    //pp_ipmi_test_chassis_power_up();
    //pp_ipmi_test_chassis_power_down();
    //pp_ipmi_test_chassis_poh();
    //pp_ipmi_test_sel_get_time();
    //pp_ipmi_test_sel_list();
    //pp_ipmi_test_sel_list();
    //pp_ipmi_test_sel_list();
    //pp_ipmi_test_sel_list();
    //pp_ipmi_test_sel_clear();
    //pp_ipmi_test_sdr_list();
    //pp_ipmi_test_sensor_by_name();
    //pp_ipmi_test_sensor_by_id();
    //pp_ipmi_test_chassis_status();
    //pp_ipmi_test_sel_info();
    //pp_ipmi_test_sdr_info();
    //pp_ipmi_kvm_test();
    //pp_ipmi_bmc_reset_test();
    //pp_ipmi_user_test();
    //pp_ipmi_sel_delete_test();
    //pp_ipmi_bmc_get_enables_test();
    //pp_ipmi_bmc_set_enables_test();
    //pp_ipmi_sensor_list_test();
    
    //int i;
    //for (i = 0; i < 100; i++) pp_ipmi_test_sensor_by_name();
#endif

    return 0;
}

