/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#ifdef WIN32
#include <winsock2.h>
#include <windows.h>
#define bzero(addr, size)	memset(addr, 0, size)
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#include <errno.h>
#include <sys/types.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#include <signal.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_channel.h>
#include <ipmitool/ipmi_constants.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#include <ipmi_params.h>

void printf_channel_usage (void);

/**
 * ipmi_1_5_authtypes
 *
 * Create a string describing the supported authentication types as 
 * specificed by the parameter n
 */
static const char *
ipmi_1_5_authtypes(uint8_t n)
{
	uint32_t i;
	static char supportedTypes[128];

	bzero(supportedTypes, 128);

	for (i = 0; ipmi_authtype_vals[i].val != 0; i++) {
		if (n & ipmi_authtype_vals[i].val) {
			strcat(supportedTypes, ipmi_authtype_vals[i].str);
			strcat(supportedTypes, " ");
		}
	}

	return supportedTypes;
}



/**
 * ipmi_get_channel_auth_cap
 *
 * return 0 on success
 *        -1 on failure
 */
int
ipmi_get_channel_auth_cap(struct ipmi_intf * intf,
			  uint8_t channel,
			  uint8_t priv,
                          int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct get_channel_auth_cap_rsp auth_cap;
	uint8_t msg_data[2];

	msg_data[0] = channel | 0x80; // Ask for IPMI v2 data as well
	msg_data[1] = priv;

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;            // 0x06
	req.msg.cmd      = IPMI_GET_CHANNEL_AUTH_CAP; // 0x38
	req.msg.data     = msg_data;
	req.msg.data_len = 2;

	rsp = intf->sendrecv(intf, &req, error);

	if ((rsp == NULL) || (rsp->ccode > 0))	{
		/*
		 * It's very possible that this failed because we asked for IPMI v2 data
		 * Ask again, without requesting IPMI v2 data
		 */
		msg_data[0] &= 0x7F;
		
		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Unable to Get Channel Authentication Capabilities\n");
			return -1;
		}
		if (rsp->ccode > 0) {
			ipmi_printf("Get Channel Authentication Capabilities failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
  			return -1;
  		}
	}

	memcpy(&auth_cap, rsp->data, sizeof(struct get_channel_auth_cap_rsp));


	ipmi_printf("Channel number             : %d\n",
		   auth_cap.channel_number);
	ipmi_printf("IPMI v1.5  auth types      : %s\n",
		   ipmi_1_5_authtypes(auth_cap.enabled_auth_types));

	if (auth_cap.v20_data_available)
		ipmi_printf("KG status                  : %s\n",
			   (auth_cap.kg_status) ? "non-zero" : "default (all zeroes)");

	ipmi_printf("Per message authentication : %sabled\n",
		   (auth_cap.per_message_auth) ? "en" : "dis");
	ipmi_printf("User level authentication  : %sabled\n",
		   (auth_cap.user_level_auth) ? "en" : "dis");

	ipmi_printf("Non-null user names exist  : %s\n",
		   (auth_cap.non_null_usernames) ? "yes" : "no");
	ipmi_printf("Null user names exist      : %s\n",
		   (auth_cap.null_usernames) ? "yes" : "no");
	ipmi_printf("Anonymous login enabled    : %s\n",
		   (auth_cap.anon_login_enabled) ? "yes" : "no");

	if (auth_cap.v20_data_available) {
		ipmi_printf("Channel supports IPMI v1.5 : %s\n",
			   (auth_cap.ipmiv15_support) ? "yes" : "no");
		ipmi_printf("Channel supports IPMI v2.0 : %s\n",
			   (auth_cap.ipmiv20_support) ? "yes" : "no");
	}

	/*
	 * If there is support for an OEM authentication type, there is some
	 * information.
	 */
	if (auth_cap.enabled_auth_types & IPMI_1_5_AUTH_TYPE_BIT_OEM) {
		ipmi_printf("IANA Number for OEM        : %d\n",
			   auth_cap.oem_id[0]      | 
			   auth_cap.oem_id[1] << 8 | 
			   auth_cap.oem_id[2] << 16);
		ipmi_printf("OEM Auxiliary Data         : 0x%x\n",
			   auth_cap.oem_aux_data);
	}

    return 0;
}



/**
 * ipmi_get_channel_info
 *
 * returns 0 on success
 *         -1 on failure
 *
 */
int
ipmi_get_channel_info(struct ipmi_intf * intf, uint8_t channel, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[2];
	struct get_channel_info_rsp   channel_info;
	struct get_channel_access_rsp channel_access;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;        // 0x06
	req.msg.cmd   = IPMI_GET_CHANNEL_INFO; // 0x42
	req.msg.data = &channel;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Unable to Get Channel Info\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Channel Info failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}


	memcpy(&channel_info, rsp->data, sizeof(struct get_channel_info_rsp));


	ipmi_printf("Channel 0x%x info:\n", channel_info.channel_number);

	ipmi_printf("  Channel Medium Type   : %s\n",
		   val2str(channel_info.channel_medium, ipmi_channel_medium_vals));

	ipmi_printf("  Channel Protocol Type : %s\n",
		   val2str(channel_info.channel_protocol, ipmi_channel_protocol_vals));

	ipmi_printf("  Session Support       : ");
	switch (channel_info.session_support) {
		case 0x00:
			ipmi_printf("session-less\n");
			break;
		case 0x01:
			ipmi_printf("single-session\n");
			break;
		case 0x02:
			ipmi_printf("multi-session\n");
			break;
		case 0x03:
		default:
			ipmi_printf("session-based\n");
			break;
	}

	ipmi_printf("  Active Session Count  : %d\n",
		   channel_info.active_sessions);

	ipmi_printf("  Protocol Vendor ID    : %d\n",
		   channel_info.vendor_id[0]      |
		   channel_info.vendor_id[1] << 8 |
		   channel_info.vendor_id[2] << 16);



	memset(&req, 0, sizeof(req));
	rqdata[0] = channel & 0xf;

	/* get volatile settings */

	rqdata[1] = 0x80; /* 0x80=active */
	req.msg.netfn = IPMI_NETFN_APP;          // 0x06
	req.msg.cmd   = IPMI_GET_CHANNEL_ACCESS; // 0x41
	req.msg.data = rqdata;
	req.msg.data_len = 2;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Get Channel Access (volatile)\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Channel Access (volatile) failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(&channel_access, rsp->data, sizeof(struct get_channel_access_rsp));


	ipmi_printf("  Volatile(active) Settings\n");
	ipmi_printf("    Alerting            : %sabled\n",
		   (channel_access.alerting) ? "dis" : "en");
	ipmi_printf("    Per-message Auth    : %sabled\n",
		   (channel_access.per_message_auth) ? "dis" : "en");
	ipmi_printf("    User Level Auth     : %sabled\n",
		   (channel_access.user_level_auth) ? "dis" : "en");

	ipmi_printf("    Access Mode         : ");
	switch (channel_access.access_mode) {
		case 0:
			ipmi_printf("disabled\n");
			break;
		case 1:
			ipmi_printf("pre-boot only\n");
			break;
		case 2:
			ipmi_printf("always available\n");
			break;
		case 3:
			ipmi_printf("shared\n");
			break;
		default:
			ipmi_printf("unknown\n");
			break;
	}

	/* get non-volatile settings */

	rqdata[1] = 0x40; /* 0x40=non-volatile */
	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Get Channel Access (non-volatile)\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Channel Access (non-volatile) failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(&channel_access, rsp->data, sizeof(struct get_channel_access_rsp));

	ipmi_printf("  Non-Volatile Settings\n");
	ipmi_printf("    Alerting            : %sabled\n",
		   (channel_access.alerting) ? "dis" : "en");
	ipmi_printf("    Per-message Auth    : %sabled\n",
		   (channel_access.per_message_auth) ? "dis" : "en");
	ipmi_printf("    User Level Auth     : %sabled\n",
		   (channel_access.user_level_auth) ? "dis" : "en");

	ipmi_printf("    Access Mode         : ");
	switch (channel_access.access_mode) {
		case 0:
			ipmi_printf("disabled\n");
			break;
		case 1:
			ipmi_printf("pre-boot only\n");
			break;
		case 2:
			ipmi_printf("always available\n");
			break;
		case 3:
			ipmi_printf("shared\n");
			break;
		default:
			ipmi_printf("unknown\n");
			break;
	}

	return 0;
}

static int
ipmi_get_user_access(struct ipmi_intf * intf, pp_ipmi_chan_getaccess_param_t* param, pp_ipmi_channel_access_t *chan_access, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[2];
	struct get_user_access_rsp user_access;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = IPMI_GET_USER_ACCESS;
	req.msg.data = rqdata;
	req.msg.data_len = 2;

	rqdata[0] = param->channel & 0xf;
	rqdata[1] = param->user_id & 0x3f;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Get User Access (channel %d id %d)\n",
			rqdata[0], rqdata[1]);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get User Access (channel %d id %d) failed: %s\n",
			rqdata[0], rqdata[1],
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	
	memcpy(&user_access, rsp->data, sizeof(struct get_user_access_rsp));
	
	chan_access->callin = user_access.callin_callback ? 1 : -1;
	chan_access->link = user_access.link_auth ? 1 : -1;
	chan_access->ipmi = user_access.ipmi_messaging ? 1 : -1;
	chan_access->privilege = priv_level_to_enum(user_access.privilege_limit);
	chan_access->privilege_raw = user_access.privilege_limit;

	return 0;
}

static int
ipmi_set_user_access(struct ipmi_intf * intf, pp_ipmi_chan_setaccess_param_t* param, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[2];
	struct get_user_access_rsp user_access;
	struct set_user_access_data set_access;

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = IPMI_GET_USER_ACCESS;
	req.msg.data = rqdata;
	req.msg.data_len = 2;

	rqdata[0] = param->channel & 0xf;
	rqdata[1] = param->user_id & 0x3f;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Get User Access (channel %d id %d)\n",
			rqdata[0], rqdata[1]);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get User Access (channel %d id %d) failed: %s\n",
			rqdata[0], rqdata[1],
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(&user_access, rsp->data, sizeof(struct get_user_access_rsp));

	bzero(&set_access,sizeof(set_access));
	set_access.change_bits = 1;
	set_access.callin_callback = user_access.callin_callback;
	set_access.link_auth = user_access.link_auth;
	set_access.ipmi_messaging = user_access.ipmi_messaging;
	set_access.channel = param->channel;
	set_access.user_id = param->user_id;
	set_access.privilege_limit = user_access.privilege_limit;
	set_access.session_limit = 0;

	if (param->access.callin != 0) {
		set_access.callin_callback = (param->access.callin < 0)? 0 : 1;
	}
	if (param->access.link != 0) {
		set_access.link_auth = (param->access.link < 0)? 0 : 1;
	}
	if (param->access.ipmi != 0) {
		set_access.ipmi_messaging = (param->access.ipmi < 0)? 0 : 1;
	}
	set_access.privilege_limit = (uint8_t) enum_to_priv_level(param->access.privilege);

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = IPMI_SET_USER_ACCESS;
	req.msg.data = (uint8_t *) &set_access;
	req.msg.data_len = 4;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Set User Access (channel %d id %d)\n",
			set_access.channel, set_access.user_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set User Access (channel %d id %d) failed: %s\n",
			set_access.channel, set_access.user_id,
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}



static const char *
iana_string(uint32_t iana)
{
	static char s[10];

	if (iana)
	{
		sprintf(s, "%06x", iana);
		return s;
	}
	else
		return "N/A";
}


static int
ipmi_get_channel_cipher_suites(struct ipmi_intf * intf,
			       const char * which,
			       const char * payload_type,
			       uint8_t channel,
			       int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	uint8_t  oem_record;
	uint8_t  rqdata[3];
	uint32_t iana;
	uint8_t  auth_alg, integrity_alg, crypt_alg;
	uint8_t  cipher_suite_id;
	uint8_t  list_index = 0;
	uint8_t  cipher_suite_data[1024]; // 0x40 sets * 16 bytes per set
	uint16_t offset = 0;
	uint16_t cipher_suite_data_length = 0; // how much was returned, total

	memset(cipher_suite_data, 0, sizeof(cipher_suite_data));
	
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;                 // 0x06
	req.msg.cmd   = IPMI_GET_CHANNEL_CIPHER_SUITES; // 0x54
	req.msg.data = rqdata;
	req.msg.data_len = 3;

	rqdata[0] = channel;
	rqdata[1] = ((strncmp(payload_type, "ipmi", 4) == 0)? 0: 1);
	rqdata[2] = ((strncmp(which, "all", 3) == 0)? 0x80: 0);

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Get Channel Cipher Suites\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Channel Cipher Suites failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}


	// Grab the returned channel number once.  We assume it's the same
	// in future calls.
	if (rsp->data_len >= 1)
		channel = rsp->data[0];
		

	while ((rsp->data_len > 1) && (list_index < 0x3F))
	{
		//
		// We got back cipher suite data -- store it.
		//
		//printf("copying data to offset %d\n", offset);
		//printbuf(rsp->data + 1, rsp->data_len - 1, "this is the data");
		memcpy(cipher_suite_data + offset, rsp->data + 1, rsp->data_len - 1);
		offset += rsp->data_len - 1;
		
		//
		// Increment our list for the next call
		//
		++list_index;
		rqdata[2] =  (rqdata[2] & 0x80) + list_index; 

		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Unable to Get Channel Cipher Suites\n");
			return -1;
		}
		if (rsp->ccode > 0) {
			ipmi_printf("Get Channel Cipher Suites failed: %s\n",
					val2str(rsp->ccode, completion_code_vals));
			return -1;
		}
	}


	//
	// We can chomp on all our data now.
	//
	cipher_suite_data_length = offset;
	offset = 0;

	printf("ID   IANA    Auth Alg        Integrity Alg   Confidentiality Alg\n");
	
	while (offset < cipher_suite_data_length)
	{
		if (cipher_suite_data[offset++] == 0xC0)
		{
			oem_record = 0; // standard type
			iana       = 0;

			// Verify that we have at least a full record left
			if ((cipher_suite_data_length - offset) < 4) // id + 3 algs
			{
				ipmi_printf("Incomplete data record in cipher suite data\n");
				return -1;
			}
			
			cipher_suite_id = cipher_suite_data[offset++];
				
		}
		else if (cipher_suite_data[offset++] == 0xC1)
		{
			oem_record = 1; // OEM record type    

			// Verify that we have at least a full record left
			if ((cipher_suite_data_length - offset) < 4) // id + iana + 3 algs
			{
				ipmi_printf("Incomplete data record in cipher suite data\n");
				return -1;
			}

			cipher_suite_id = cipher_suite_data[offset++];

			//
			// Grab the IANA
			//
			iana =
				cipher_suite_data[offset]            | 
				(cipher_suite_data[offset + 1] << 8) | 
				(cipher_suite_data[offset + 2] << 16);
			offset += 3;
		}
		else
		{
			ipmi_printf("Bad start of record byte in cipher suite data\n");
			return -1;
		}

		//
		// Grab the algorithms for this cipher suite.  I guess we can't be
		// sure of what order they'll come in.  Also, I suppose we default
		// to the NONE algorithm if one were absent.  This part of the spec is
		// poorly written -- I have read the errata document.  For now, I'm only
		// allowing one algorithm per type (auth, integrity, crypt) because I
		// don't I understand how it could be otherwise.
		//
		auth_alg      = IPMI_AUTH_RAKP_NONE;
		integrity_alg = IPMI_INTEGRITY_NONE;
		crypt_alg     = IPMI_CRYPT_NONE;
		
		while (((cipher_suite_data[offset] & 0xC0) != 0xC0) &&
			   ((cipher_suite_data_length - offset) > 0))
		{
			switch (cipher_suite_data[offset] & 0xC0)
			{
			case 0x00:
				// Authentication algorithm specifier
				auth_alg = cipher_suite_data[offset++] & 0x3F;
				break;
			case 0x40:
				// Interity algorithm specifier
				integrity_alg = cipher_suite_data[offset++] & 0x3F;
				break;
			case 0x80:
				// Confidentiality algorithm specifier
				crypt_alg = cipher_suite_data[offset++] & 0x3F;
				break;
			}
		}


		//
		// We have everything we need to spit out a cipher suite record
		//
		printf("%-4d %-7s %-15s %-15s %-15s\n",
		       cipher_suite_id,
		       iana_string(iana),
		       val2str(auth_alg, ipmi_auth_algorithms),
		       val2str(integrity_alg, ipmi_integrity_algorithms),
		       val2str(crypt_alg, ipmi_encryption_algorithms));
	}
	
	
	return 0;
}



uint8_t
ipmi_get_channel_medium(struct ipmi_intf * intf, uint8_t channel, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct get_channel_info_rsp info;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = IPMI_GET_CHANNEL_INFO;
	req.msg.data = &channel;
	req.msg.data_len = 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Channel Info command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get Channel Info command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(&info, rsp->data, sizeof(struct get_channel_info_rsp));

	if (verbose) {
		ipmi_printf("Channel type: %s\n",
			val2str(info.channel_medium, ipmi_channel_medium_vals));
	}

	return info.channel_medium;
}

uint8_t
ipmi_current_channel_medium(struct ipmi_intf * intf, int * error)
{
	return ipmi_get_channel_medium(intf, 0xE, error);
}

void
printf_channel_usage()
{
	ipmi_printf("Channel Commands: authcap   <channel number> <max privilege>\n");
	ipmi_printf("                  getaccess <channel number> [user id]\n");
	ipmi_printf("                  setaccess <channel number> "
		"<user id> [callin=on|off] [ipmi=on|off] [link=on|off] [privilege=level]\n");
	ipmi_printf("                  info      [channel number]\n");
	ipmi_printf("                  getciphers <all | supported> <ipmi | sol> [channel]\n\n");
	ipmi_printf("Possible privilege levels are:\n");
	ipmi_printf("   1   Callback level\n");
	ipmi_printf("   2   User level\n");
	ipmi_printf("   3   Operator level\n");
	ipmi_printf("   4   Administrator level\n");
	ipmi_printf("   5   OEM Proprietary level\n");
	ipmi_printf("  15   No access\n");
}


int
ipmi_channel_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int retval = 0;

	t_ipmitool_commandline *cmdline = get_cmdline(params);

	if (!cmdline) {
		ipmi_printf("Error: could not allocate commandline.\n");
		return -1;
	}

	switch ((pp_ipmi_channel_subcommand_t) subcmd) {
		case PP_IPMI_CHANNEL_SUBCMD_AUTHCAP:
			if (cmdline->argc != 2) {
				printf_channel_usage();
				retval = -1;
			} else {
				retval = ipmi_get_channel_auth_cap(intf,
	                                               (uint8_t)strtol(cmdline->argv[0], NULL, 0),
	                                               (uint8_t)strtol(cmdline->argv[1], NULL, 0), error);
	                }
	                break;
	        case PP_IPMI_CHANNEL_SUBCMD_GET_ACCESS:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'Channel Get User Access' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}
			retval = ipmi_get_user_access(intf, &params->data.channel_getaccess,
			        &ipmi_ret->data.channel_getaccess, error);
			break;
		case PP_IPMI_CHANNEL_SUBCMD_SET_ACCESS:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'Channel Set User Access' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}
			retval = ipmi_set_user_access(intf, &params->data.channel_setaccess, error);
			break;
		case PP_IPMI_CHANNEL_SUBCMD_INFO:
			if (cmdline->argc > 1) {
				printf_channel_usage();
				retval = -1;
			} else {
				uint8_t ch = 0xe;
				if (cmdline->argc == 1)
					ch = (uint8_t)strtol(cmdline->argv[0], NULL, 0);
				retval = ipmi_get_channel_info(intf, ch, error);
			}
			break;
		case PP_IPMI_CHANNEL_SUBCMD_GET_CIPHERS:
			if ((cmdline->argc < 2) || (cmdline->argc > 3)                                         ||
			    (strncmp(cmdline->argv[0], "all", 3) && strncmp(cmdline->argv[0], "supported", 9)) ||
			    (strncmp(cmdline->argv[1], "ipmi", 4) && strncmp(cmdline->argv[1], "sol",  3)))
				printf_channel_usage();
			else {
				uint8_t ch = 0xe;
				if (cmdline->argc == 3)
					ch = (uint8_t)strtol(cmdline->argv[2], NULL, 0);
				retval = ipmi_get_channel_cipher_suites(intf,
									cmdline->argv[0], // all | supported
									cmdline->argv[1], // ipmi | sol
									ch,
									error);
			}
			break;
		default:
			ipmi_printf("Invalid CHANNEL command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			retval = -1;
	}

	cleanup_cmdline(cmdline);
	return retval;
}


