/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <errno.h>
#include <sys/types.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#include <signal.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_sel.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_channel.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_event.h>
#include <ipmi_params.h>

#ifdef WIN32
#define strncasecmp _strnicmp
#endif

static int
ipmi_send_platform_event(struct ipmi_intf * intf, struct platform_event_msg * emsg, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[8];
	uint8_t chmed;

	memset(&req, 0, sizeof(req));
	memset(rqdata, 0, 8);

	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = 0x02;
	req.msg.data = rqdata;

	chmed = ipmi_current_channel_medium(intf, error);
	if (chmed == IPMI_CHANNEL_MEDIUM_SYSTEM) {
		/* system interface, need extra generator ID */
		req.msg.data_len = 8;
		rqdata[0] = 0x20;
		memcpy(rqdata+1, emsg, sizeof(struct platform_event_msg));
	}
	else {
		req.msg.data_len = 7;
		memcpy(rqdata, emsg, sizeof(struct platform_event_msg));
	}

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Platform Event Message command failed\n");
		return -1;
	}
	else if (rsp->ccode > 0) {
		ipmi_printf("Platform Event Message command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}

#define EVENT_THRESH_STATE_LNC_LO	0
#define EVENT_THRESH_STATE_LNC_HI	1
#define EVENT_THRESH_STATE_LCR_LO	2
#define EVENT_THRESH_STATE_LCR_HI	3
#define EVENT_THRESH_STATE_LNR_LO	4
#define EVENT_THRESH_STATE_LNR_HI	5
#define EVENT_THRESH_STATE_UNC_LO	6
#define EVENT_THRESH_STATE_UNC_HI	7
#define EVENT_THRESH_STATE_UCR_LO	8
#define EVENT_THRESH_STATE_UCR_HI	9
#define EVENT_THRESH_STATE_UNR_LO	10
#define EVENT_THRESH_STATE_UNR_HI	11

static const struct valstr ipmi_event_thresh_lo[] = {
	{ EVENT_THRESH_STATE_LNC_LO, "lnc" },
	{ EVENT_THRESH_STATE_LCR_LO, "lcr" },
	{ EVENT_THRESH_STATE_LNR_LO, "lnr" },
	{ EVENT_THRESH_STATE_UNC_LO, "unc" },
	{ EVENT_THRESH_STATE_UCR_LO, "ucr" },
	{ EVENT_THRESH_STATE_UNR_LO, "unr" },
	{ 0, NULL  },
};
static const struct valstr ipmi_event_thresh_hi[] = {
	{ EVENT_THRESH_STATE_LNC_HI, "lnc" },
	{ EVENT_THRESH_STATE_LCR_HI, "lcr" },
	{ EVENT_THRESH_STATE_LNR_HI, "lnr" },
	{ EVENT_THRESH_STATE_UNC_HI, "unc" },
	{ EVENT_THRESH_STATE_UCR_HI, "ucr" },
	{ EVENT_THRESH_STATE_UNR_HI, "unr" },
	{ 0, NULL  },
};

static int
ipmi_send_platform_event_num(struct ipmi_intf * intf, int num, int *error)
{
	struct platform_event_msg emsg;

	memset(&emsg, 0, sizeof(struct platform_event_msg));

	/* IPMB/LAN/etc */
	switch (num) {
	case 1:			/* temperature */
		ipmi_printf("Sending SAMPLE event: Temperature - "
		       "Upper Critical - Going High\n");
		emsg.evm_rev       = 0x04;
		emsg.sensor_type   = 0x01;
		emsg.sensor_num    = 0x30;
		emsg.event_dir     = EVENT_DIR_ASSERT;
		emsg.event_type    = 0x01;
		emsg.event_data[0] = EVENT_THRESH_STATE_UCR_HI;
		emsg.event_data[1] = 0xff;
		emsg.event_data[2] = 0xff;
		break;
	case 2:			/* voltage error */
		ipmi_printf("Sending SAMPLE event: Voltage Threshold - "
		       "Lower Critical - Going Low\n");
		emsg.evm_rev       = 0x04;
		emsg.sensor_type   = 0x02;
		emsg.sensor_num    = 0x60;
		emsg.event_dir     = EVENT_DIR_ASSERT;
		emsg.event_type    = 0x01;
		emsg.event_data[0] = EVENT_THRESH_STATE_LCR_LO;
		emsg.event_data[1] = 0xff;
		emsg.event_data[2] = 0xff;
		break;
	case 3:			/* correctable ECC */
		ipmi_printf("Sending SAMPLE event: Memory - Correctable ECC\n");
		emsg.evm_rev       = 0x04;
		emsg.sensor_type   = 0x0c;
		emsg.sensor_num    = 0x53;
		emsg.event_dir     = EVENT_DIR_ASSERT;
		emsg.event_type    = 0x6f;
		emsg.event_data[0] = 0x00;
		emsg.event_data[1] = 0xff;
		emsg.event_data[2] = 0xff;
		break;
	default:
		ipmi_printf("Invalid event number: %d\n", num);
		return -1;
	}

	return ipmi_send_platform_event(intf, &emsg, error);
}

static int
ipmi_event_find_offset(uint8_t code,
		       struct ipmi_event_sensor_types * evt,
		       char * desc)
{
	if (desc == NULL || code == 0)
		return 0x00;

	while (evt->type) {
		if (evt->code == code && evt->desc != NULL &&
		    strncasecmp(desc, evt->desc, __maxlen(desc, (char *)evt->desc)) == 0)
			return evt->offset;
		evt++;
	}

	ipmi_printf("Unable to find matching event offset for '%s'\n", desc);
	return -1;
}


static int
ipmi_event_fromsensor(struct ipmi_intf * intf, char * id, char * state, char * evdir, int * error)
{
	struct ipmi_rs * rsp;
	struct sdr_record_list * sdr;
	struct platform_event_msg emsg;
	int off;

	if (id == NULL) {
		ipmi_printf("No sensor ID supplied\n");
		return -1;
	}

	memset(&emsg, 0, sizeof(struct platform_event_msg));
	emsg.evm_rev = 0x04;

	if (evdir == NULL)
		emsg.event_dir = EVENT_DIR_ASSERT;
	else if (strncasecmp(evdir, "assert", 6) == 0)
		emsg.event_dir = EVENT_DIR_ASSERT;
	else if (strncasecmp(evdir, "deassert", 8) == 0)
		emsg.event_dir = EVENT_DIR_DEASSERT;
	else {
		ipmi_printf("Invalid event direction %s.  Must be 'assert' or 'deassert'\n", evdir);
		return -1;
	}

	ipmi_printf("Finding sensor %s... ", id);
	sdr = ipmi_sdr_find_sdr_byid(intf, id, error);
	if (sdr == NULL) {
		ipmi_printf("not found!\n");
		return -1;
	}
	ipmi_printf("ok\n");

	switch (sdr->type)
	{
	case SDR_RECORD_TYPE_FULL_SENSOR:

		emsg.sensor_type   = sdr->record.full->sensor.type;
		emsg.sensor_num    = sdr->record.full->keys.sensor_num;
		emsg.event_type    = sdr->record.full->event_type;
		break;

	case SDR_RECORD_TYPE_COMPACT_SENSOR:

		emsg.sensor_type = sdr->record.compact->sensor.type;
		emsg.sensor_num  = sdr->record.compact->keys.sensor_num;
		emsg.event_type  = sdr->record.compact->event_type;
		break;

	default:
		ipmi_printf("Unknown sensor type for id '%s'\n", id);
		return -1;
	}

	emsg.event_data[1] = 0xff;
	emsg.event_data[2] = 0xff;

	switch (emsg.event_type)
	{
	/*
	 * Threshold Class
	 */
	case 1:
	{
		int dir = 0;
		int hilo = 0;
		off = 1;

		if (0 != strncasecmp(state, "lnr", 3) &&
		    0 != strncasecmp(state, "lcr", 3) &&
		    0 != strncasecmp(state, "lnc", 3) &&
		    0 != strncasecmp(state, "unc", 3) &&
		    0 != strncasecmp(state, "ucr", 3) &&
		    0 != strncasecmp(state, "unr", 3))
		{
			ipmi_printf("Invalid threshold identifier %s\n", state);
			return -1;
		}

		if (state[0] == 'u')
			hilo = 1;
		else
			hilo = 0;

		if (emsg.event_dir == EVENT_DIR_ASSERT)
			dir = hilo;
		else
			dir = !hilo;

		if ((emsg.event_dir == EVENT_DIR_ASSERT   && hilo == 1) ||
		    (emsg.event_dir == EVENT_DIR_DEASSERT && hilo == 0))
			emsg.event_data[0] = (uint8_t)(str2val(state, ipmi_event_thresh_hi) & 0xf);
		else if ((emsg.event_dir == EVENT_DIR_ASSERT   && hilo == 0) ||
			 (emsg.event_dir == EVENT_DIR_DEASSERT && hilo == 1))
			emsg.event_data[0] = (uint8_t)(str2val(state, ipmi_event_thresh_lo) & 0xf);
		else {
			ipmi_printf("Invalid Event\n");
			return -1;
		}

		rsp = ipmi_sdr_get_sensor_thresholds(intf, emsg.sensor_num, error);

		if (rsp != NULL && rsp->ccode == 0) {

			/* threshold reading */
			emsg.event_data[2] = rsp->data[(emsg.event_data[0] / 2) + 1];

			rsp = ipmi_sdr_get_sensor_hysteresis(intf, emsg.sensor_num, error);
			if (rsp != NULL && rsp->ccode == 0)
				off = dir ? rsp->data[0] : rsp->data[1];
			if (off <= 0)
				off = 1;

			/* trigger reading */
			if (dir) {
				if ((emsg.event_data[2] + off) > 0xff)
					emsg.event_data[1] = 0xff;
				else
					emsg.event_data[1] = emsg.event_data[2] + off;
			}
			else {
				if ((emsg.event_data[2] - off) < 0)
					emsg.event_data[1] = 0;
				else
					emsg.event_data[1] = emsg.event_data[2] - off;
			}

			/* trigger in byte 2, threshold in byte 3 */
			emsg.event_data[0] |= 0x50;
		}
	}
	break;

	/*
	 * Digital Discrete
	 */
	case 3: case 4: case 5: case 6: case 8: case 9:
	{
		size_t x;
		const char * digi_on[] = { "present", "assert", "limit",
					   "fail", "yes", "on", "up" };
		const char * digi_off[] = { "absent", "deassert", "nolimit",
					    "nofail", "no", "off", "down" };
		off = 0;
		for (x = 0; x < sizeof(digi_on)/sizeof(*digi_on); x++) {
			if (strncasecmp(state, digi_on[x], strlen(digi_on[x])) == 0) {
				emsg.event_data[0] = 1;
				off = 1;
				break;
			}
			else if (strncasecmp(state, digi_off[x], strlen(digi_off[x])) == 0) {
				emsg.event_data[0] = 0;
				off = 1;
				break;
			}
		}
		if (off == 0) {
			off = ipmi_event_find_offset(
				emsg.event_type, generic_event_types, state);
			if (off < 0)
				return -1;
			emsg.event_data[0] = off;
		}
	}
	break;

	/*
	 * Generic Discrete
	 */
	case 2: case 7: case 10: case 11: case 12:
	{
		off = ipmi_event_find_offset(
			emsg.event_type, generic_event_types, state);
		if (off < 0)
			return -1;
		emsg.event_data[0] = off;
	}
	break;
		
	/*
	 * Sensor-Specific Discrete
	 */
	case 0x6f:
	{
		off = ipmi_event_find_offset(
			emsg.sensor_type, sensor_specific_types, state);
		if (off < 0)
			return -1;
		emsg.event_data[0] = off;
	}
	break;

	default:
		return -1;

	}

	return ipmi_send_platform_event(intf, &emsg, error);
}

static void
ipmi_event_usage(void)
{
	ipmi_printf("\n");
	ipmi_printf("usage: event <num>\n");
	ipmi_printf("   Send generic test events\n");
	ipmi_printf("   1 : Temperature - Upper Critical - Going High\n");
	ipmi_printf("   2 : Voltage Threshold - Lower Critical - Going Low\n");
	ipmi_printf("   3 : Memory - Correctable ECC\n");
	ipmi_printf("\n");
	ipmi_printf("usage: event <sensorid> <state> [event_dir]\n");
	ipmi_printf("   sensorid  : Sensor ID to use for event data\n");
	ipmi_printf("   state     : Sensor state, use 'list' to see possible states for sensor\n");
	ipmi_printf("   event_dir : assert, deassert [default=assert]\n");
	ipmi_printf("\n");
}

int ipmi_event_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret UNUSED, int * error)
{
	int ret = -1;

	t_ipmitool_commandline *cmdline = get_cmdline(params);

	if (!cmdline) {
		ipmi_printf("Error: could not allocate commandline.\n");
		return -1;
	}

	switch ((pp_ipmi_event_subcommand_t) subcmd) {
		case PP_IPMI_EVENT_SUBCMD_GENERATE:
			if (!cmdline->argc) {
				ipmi_event_usage();
			} else if (strlen(cmdline->argv[0]) == 1) {
				switch (cmdline->argv[0][0]) {
					case '1':	ret = ipmi_send_platform_event_num(intf, 1, error);	break;
					case '2':	ret = ipmi_send_platform_event_num(intf, 2, error);	break;
					case '3':	ret = ipmi_send_platform_event_num(intf, 3, error);	break;
					default:	ipmi_event_usage();					break;
				}
			} else if (cmdline->argc < 2) {
				ret = ipmi_event_fromsensor(intf, cmdline->argv[0], NULL, NULL, error);
			} else if (cmdline->argc < 3) {
				ret = ipmi_event_fromsensor(intf, cmdline->argv[0], cmdline->argv[1], NULL, error);
			} else {
				ret = ipmi_event_fromsensor(intf, cmdline->argv[0], cmdline->argv[1], cmdline->argv[2], error);
			}
			break;
		default:
			ipmi_printf("Invalid event command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	}

	cleanup_cmdline(cmdline);
	return ret;
}

