/**
 * \file ipmi_oem_pp_rpc.c
 *
 * Description: Peppercon IPMI OEM commands
 *
 * (c) 2006 Peppercon AG
 *                        
 */

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_oem_pp_rpc.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>

static int ipmi_oem_pp_rpc_set_power_on_delay(struct ipmi_intf * intf,
                                       pp_ipmi_parameter_t *params,
                                       pp_ipmi_return_t *ret UNUSED,
                                       int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC Power On Delay Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[1];
    b[0] = (unsigned char)params->data.oem_pp_rpc_set_power_on_delay.delay;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_POWER_ON_DELAY;
    req.msg.data_len = 1;
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
} 

static int ipmi_oem_pp_rpc_get_power_on_delay(struct ipmi_intf * intf,
                                              pp_ipmi_parameter_t *params UNUSED,
                                              pp_ipmi_return_t *ret,
                                              int * error)
{
    const char *cmd_name = "[OEM Peppercon] Get RPC Power On Delay Status";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;


    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_GET_POWER_ON_DELAY;
    req.msg.data_len = 0;
    req.msg.data = NULL;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    ret->data.oem_pp_rpc_get_power_on_delay_rs.delay = rsp->data[0];

    return 0;
}


static int ipmi_oem_pp_rpc_set_receptacle_state(struct ipmi_intf * intf,
     			                        pp_ipmi_parameter_t *params,
                     	   		        pp_ipmi_return_t *ret UNUSED,
                  			        int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC Receptacle State Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[2];
    b[0] = (unsigned char)params->data.oem_pp_rpc_set_recep_state.receptacle;
    b[1] = (unsigned char)params->data.oem_pp_rpc_set_recep_state.status;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_RECEPTACLE_STATE;
    req.msg.data_len = 2;
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}

static int ipmi_oem_pp_rpc_get_receptacle_state(struct ipmi_intf * intf,
                                                pp_ipmi_parameter_t *params,
                                                pp_ipmi_return_t *ret,
                                                int * error)
{
    const char *cmd_name = "[OEM Peppercon] Get RPC Receptacle State";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;
    pp_ipmi_oem_pp_rpc_get_receptacle_state_result_t *result;

    unsigned char b[1];
    b[0] = params->data.oem_pp_rpc_get_recep_state.receptacle;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_GET_RECEPTACLE_STATE;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    result = &ret->data.oem_pp_rpc_get_recep_state_rs;
    result->status =        rsp->data[0]       & 0x01;
    result->soft_breaker = (rsp->data[0] >> 1) & 0x01;
    result->red =          (rsp->data[0] >> 4) & 0x01;
    result->green =        (rsp->data[0] >> 5) & 0x01;
    result->blinking =     (rsp->data[0] >> 6) & 0x01;

    return 0;
}

static int ipmi_oem_pp_rpc_set_group_state(struct ipmi_intf * intf,
                                           pp_ipmi_parameter_t *params,
                                           pp_ipmi_return_t *ret UNUSED,
                                           int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC Group State Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[2];
    b[0] = (unsigned char)params->data.oem_pp_rpc_set_group_state.group;
    b[1] = (unsigned char)params->data.oem_pp_rpc_set_group_state.status;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_GROUP_STATE;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}


static int ipmi_oem_pp_rpc_set_group_membership(struct ipmi_intf * intf,
                                        pp_ipmi_parameter_t *params,
                                        pp_ipmi_return_t *ret UNUSED,
                                        int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC Group Membership Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[5];
    b[0] = (unsigned char)params->data.oem_pp_rpc_set_group_membership.group;
    b[1] = (unsigned char)params->data.oem_pp_rpc_set_group_membership.enable;
    b[2] = (unsigned char)params->data.oem_pp_rpc_set_group_membership.mem[0];
    b[3] = (unsigned char)params->data.oem_pp_rpc_set_group_membership.mem[1];
    b[4] = (unsigned char)params->data.oem_pp_rpc_set_group_membership.mem[2];

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_GROUP_MEMBERSHIP;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}


static int ipmi_oem_pp_rpc_get_group_membership(struct ipmi_intf * intf,
                                                pp_ipmi_parameter_t *params,
                                                pp_ipmi_return_t *ret,
                                                int * error)
{
    const char *cmd_name = "[OEM Peppercon] Get RPC Group Membership Value";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[1];
    b[0] = (unsigned char)params->data.oem_pp_rpc_get_group_membership.group;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_GET_GROUP_MEMBERSHIP;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    ret->data.oem_pp_rpc_get_group_membership_rs.enable = rsp->data[0];
    ret->data.oem_pp_rpc_get_group_membership_rs.mem[0] = rsp->data[1];
    ret->data.oem_pp_rpc_get_group_membership_rs.mem[1] = rsp->data[2];
    ret->data.oem_pp_rpc_get_group_membership_rs.mem[2] = rsp->data[3];

    return 0;
}

static int ipmi_oem_pp_rpc_set_group_power_on_delay(struct ipmi_intf * intf,
                                                    pp_ipmi_parameter_t *params,
                                                    pp_ipmi_return_t *ret UNUSED,
                                                    int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC GROUP Power On Delay Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[2];
    b[0] = (unsigned char)params->data.oem_pp_rpc_set_group_power_on_delay.group;
    b[1] = (unsigned char)params->data.oem_pp_rpc_set_group_power_on_delay.delay;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_GROUP_POWER_ON_DELAY;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}

static int ipmi_oem_pp_rpc_get_group_power_on_delay(struct ipmi_intf * intf,
                                                    pp_ipmi_parameter_t *params,
                                                    pp_ipmi_return_t *ret,
                                                    int * error)
{
    const char *cmd_name = "[OEM Peppercon] Get RPC GROUP Power On Delay Value";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[1];
    b[0] = params->data.oem_pp_rpc_get_group_power_on_delay.group;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_GET_GROUP_POWER_ON_DELAY;
    req.msg.data_len = 1;
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    ret->data.oem_pp_rpc_get_group_power_on_delay_rs.delay = rsp->data[0];

    return 0;
}

static int ipmi_oem_pp_rpc_set_receptacle_acl(struct ipmi_intf * intf,
                                        pp_ipmi_parameter_t *params,
                                        pp_ipmi_return_t *ret UNUSED,
                                        int * error)
{
    const char *cmd_name = "[OEM Peppercon] Set RPC Receptacle ACL Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;
    int i;	
    unsigned char b[258]; /*b[2] ~ b[257] are for ACL entries*/

    b[0] = (unsigned char)params->data.oem_pp_rpc_set_recep_acl.receptacle;
    b[1] = (unsigned char)params->data.oem_pp_rpc_set_recep_acl.num_acl;
    for(i = 0; i < (int)b[1]; i++)
        b[i+2] = (unsigned char)params->data.oem_pp_rpc_set_recep_acl.acl[i];

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_SET_RECEPTACLE_ACL;
    req.msg.data_len = b[1]+2;
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}

static int ipmi_oem_pp_rpc_get_receptacle_acl(struct ipmi_intf * intf,
                                                pp_ipmi_parameter_t *params,
                                                pp_ipmi_return_t *ret,
                                                int * error)
{
    const char *cmd_name = "[OEM Peppercon] Get RPC Receptacle ACL Value";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;
    int i;
    unsigned char b[1];
    b[0] = params->data.oem_pp_rpc_get_recep_acl.receptacle;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_PP;
    req.msg.cmd = IPMI_CMD_PP_RPC_GET_RECEPTACLE_ACL;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }
    ret->data.oem_pp_rpc_get_recep_acl_rs.num_acl = rsp->data[0];
    for(i = 0; i < (int)rsp->data[0]; i++)
        ret->data.oem_pp_rpc_get_recep_acl_rs.acl[i] = rsp->data[i+1];

    return 0;
}


int ipmi_oem_pp_rpc_main(struct ipmi_intf * intf,
                         int subcmd,
                         pp_ipmi_parameter_t *params,
                         pp_ipmi_return_t *ret,
                         int * error)
{
    switch ((pp_ipmi_oem_pp_rpc_subcommand_t) subcmd) {

	case PP_IPMI_OEM_PP_RPC_SET_POWER_ON_DELAY:
	    return ipmi_oem_pp_rpc_set_power_on_delay(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_GET_POWER_ON_DELAY:
	    return ipmi_oem_pp_rpc_get_power_on_delay(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_STATE:
	    return ipmi_oem_pp_rpc_set_receptacle_state(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE:
	    return ipmi_oem_pp_rpc_get_receptacle_state(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_SET_GROUP_STATE:
	    return ipmi_oem_pp_rpc_set_group_state(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_SET_GROUP_MEMBERSHIP:
	    return ipmi_oem_pp_rpc_set_group_membership(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_GET_GROUP_MEMBERSHIP:
	    return ipmi_oem_pp_rpc_get_group_membership(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_SET_GROUP_POWER_ON_DELAY:
	    return ipmi_oem_pp_rpc_set_group_power_on_delay(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_GET_GROUP_POWER_ON_DELAY:
	    return ipmi_oem_pp_rpc_get_group_power_on_delay(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_ACL:
	    return ipmi_oem_pp_rpc_set_receptacle_acl(intf, params, ret, error);
	    break;
	case PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_ACL:
	    return ipmi_oem_pp_rpc_get_receptacle_acl(intf, params, ret, error);
	    break;
	default:
	    ipmi_printf("Invalid RPC OEM Peppercon command: %d\n", subcmd);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	    return -1;
    }
}
