/**
 * \file ipmi_oem_rackable.c
 *
 * Description: OEM commands for Rackable Roamer
 *
 * (c) 2006 Raritan Computer Inc., Ingo van Lil <ingo.lil@raritan.com>
 */

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_oem_rackable.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmi_return.h>

static const struct valstr oem_rackable_cmd_strs[] = {
    { BMC_OEM_RACKABLE_COLOR_LED_IDENTIFY, "Color LED Identify" },
    { BMC_OEM_RACKABLE_LCD_SET_SLOT,       "LCD Set Slot" },
    { BMC_OEM_RACKABLE_LCD_GET_SLOT,       "LCD Get Slot" },
    { BMC_OEM_RACKABLE_LCD_SET_STATE,      "LCD Set State" },
    { BMC_OEM_RACKABLE_LCD_GET_STATE,      "LCD Get State" },
    { 0xffff, NULL }
};

static const struct errstr {
    uint8_t cmd;
    uint8_t err;
    const char *str;
    int error_code;
} oem_rackable_err_strs[] = {
    { 0xff, 0xff, NULL, 0 }
};

static const char *cmd_str(unsigned char netfn, unsigned char cmd)
{
    return netfn == IPMI_NETFN_OEM_RACKABLE ?
	val2str(cmd, oem_rackable_cmd_strs) : "";
}

static const char *err_str(unsigned char err, unsigned char netfn, unsigned char cmd)
{
    const char *str = NULL;

    if (netfn == IPMI_NETFN_OEM_RACKABLE) {
	const struct errstr *errs = oem_rackable_err_strs;
	for (errs = oem_rackable_err_strs; errs->str != NULL; errs++) {
	    if (errs->cmd == cmd && errs->err == err) {
		str = errs->str;
		break;
	    }
	}
    }

    return str ? str : val2str(err, completion_code_vals);
}

static int err_code(unsigned char err, unsigned char netfn, unsigned char cmd)
{
    if (netfn == IPMI_NETFN_OEM_RACKABLE) {
	const struct errstr *errs = oem_rackable_err_strs;
	for (errs = oem_rackable_err_strs; errs->str != NULL; errs++) {
	    if (errs->cmd == cmd && errs->err == err) {
		return errs->error_code;
	    }
	}
    }

    return err;
}

#define SENDRECV(intf, req, rsp, err) \
    rsp = intf->sendrecv(intf, &req, err); \
    if (rsp == NULL) { \
	ipmi_printf("%s command failed\n", \
		cmd_str(req.msg.netfn, req.msg.cmd)); \
	goto exit; \
    } \
    if (rsp->ccode > 0) { \
	ipmi_printf("%s command failed: %s\n", \
		cmd_str(req.msg.netfn, req.msg.cmd), \
		err_str(rsp->ccode, req.msg.netfn, req.msg.cmd)); \
	ipmi_reset_error(error); \
	ipmi_set_error(err, err_code(rsp->ccode, req.msg.netfn, req.msg.cmd)); \
	goto exit; \
    }

static int ipmi_oem_rackable_lcd_set_slot(struct ipmi_intf *intf,
	int slot, int active, const char *msg, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;
    char buffer[50];

    memset(buffer, 0, sizeof(buffer));
    buffer[0] = (active? 0x80 : 0) | (slot & 0x7f);
    if (active) {
	strncpy(buffer + 1, msg, sizeof(buffer) - 2);
    }

    req.msg.netfn = IPMI_NETFN_OEM_RACKABLE;
    req.msg.cmd = BMC_OEM_RACKABLE_LCD_SET_SLOT;
    req.msg.data = (unsigned char *)buffer;
    req.msg.data_len = 1 + strlen(buffer + 1);
    SENDRECV(intf, req, rsp, error);
    ret = 0;

exit:
    return ret;
}

static int ipmi_oem_rackable_lcd_get_slot(struct ipmi_intf *intf,
	pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;
    unsigned char buffer;
    char *msg;

    buffer = params->data.oem_rackable_lcd_get_slot.slot;
    req.msg.netfn = IPMI_NETFN_OEM_RACKABLE;
    req.msg.cmd = BMC_OEM_RACKABLE_LCD_GET_SLOT;
    req.msg.data = &buffer;
    req.msg.data_len = 1;
    SENDRECV(intf, req, rsp, error);
    ret = 0;

    msg = malloc(rsp->data_len);
    memset(msg, 0, rsp->data_len);
    memcpy(msg, rsp->data + 1, rsp->data_len - 1);
    ipmi_init_oem_rackable_get_slot(ipmi_ret);
    ipmi_ret->data.oem_rackable_lcd_get_slot.active = rsp->data[0];
    ipmi_ret->data.oem_rackable_lcd_get_slot.message = msg;

exit:
    return ret;
}

static int ipmi_oem_rackable_lcd_set_state(struct ipmi_intf *intf,
	int set_slot, int slot,
	int set_mode, pp_ipmi_oem_rackable_lcd_mode_t mode,
	int set_interval, int interval,
	int set_immediate, const char *immediate, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;

    char *buffer;
    int len = 5;
    if (immediate) len += strlen(immediate);
    buffer = malloc(len);
    memset(buffer, 0, len);

    buffer[0] = (set_mode? 0x01 : 0)
	      | (set_slot? 0x02 : 0)
	      | (set_interval? 0x04 : 0)
	      | (set_immediate? 0x08 : 0);
    buffer[1] = mode;
    buffer[2] = slot;
    *(unsigned short *)&buffer[3] = cpu_to_le16(interval);
    if (immediate) strncpy(buffer + 5, immediate, len - 5);

    req.msg.netfn = IPMI_NETFN_OEM_RACKABLE;
    req.msg.cmd = BMC_OEM_RACKABLE_LCD_SET_STATE;
    req.msg.data = (unsigned char *)buffer;
    req.msg.data_len = len;
    SENDRECV(intf, req, rsp, error);
    ret = 0;

exit:
    return ret;
}

static int ipmi_oem_rackable_lcd_get_state(struct ipmi_intf *intf,
	pp_ipmi_return_t *ipmi_ret, int *error)
{
    struct ipmi_rq req;
    struct ipmi_rs *rsp;
    int ret = -1;
    char *msg;
    unsigned short interval;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_RACKABLE;
    req.msg.cmd = BMC_OEM_RACKABLE_LCD_GET_STATE;
    SENDRECV(intf, req, rsp, error);
    ret = 0;

    msg = malloc(rsp->data_len - 4);
    memset(msg, 0, rsp->data_len - 4);
    memcpy(msg, rsp->data + 5, rsp->data_len - 5);
    ipmi_init_oem_rackable_get_state(ipmi_ret);
    ipmi_ret->data.oem_rackable_lcd_get_state.mode = rsp->data[0];
    ipmi_ret->data.oem_rackable_lcd_get_state.current_slot = rsp->data[1];
    ipmi_ret->data.oem_rackable_lcd_get_state.active_slots = rsp->data[2];
    interval = *(unsigned short *)&rsp->data[3];
    ipmi_ret->data.oem_rackable_lcd_get_state.interval = le16_to_cpu(interval);
    ipmi_ret->data.oem_rackable_lcd_get_state.immediate = msg;

exit:
    return ret;
}

int ipmi_oem_rackable_main(struct ipmi_intf *intf, int subcmd,
	pp_ipmi_parameter_t *params, pp_ipmi_return_t *ret, int *error)
{
    pp_ipmi_cleanup_ret(ret);
    switch ((pp_ipmi_kvm_subcommand_t) subcmd) {
	case PP_IPMI_OEM_RACKABLE_LCD_SET_SLOT:
	    return ipmi_oem_rackable_lcd_set_slot(intf,
		    params->data.oem_rackable_lcd_set_slot.slot,
		    params->data.oem_rackable_lcd_set_slot.active,
		    params->data.oem_rackable_lcd_set_slot.message,
		    error);
	case PP_IPMI_OEM_RACKABLE_LCD_GET_SLOT:
	    return ipmi_oem_rackable_lcd_get_slot(intf, params, ret, error);
	case PP_IPMI_OEM_RACKABLE_LCD_SET_STATE:
	    return ipmi_oem_rackable_lcd_set_state(intf,
		    params->data.oem_rackable_lcd_set_state.set_slot,
		    params->data.oem_rackable_lcd_set_state.slot,
		    params->data.oem_rackable_lcd_set_state.set_mode,
		    params->data.oem_rackable_lcd_set_state.mode,
		    params->data.oem_rackable_lcd_set_state.set_interval,
		    params->data.oem_rackable_lcd_set_state.interval,
		    params->data.oem_rackable_lcd_set_state.set_immediate,
		    params->data.oem_rackable_lcd_set_state.immediate,
		    error);
	case PP_IPMI_OEM_RACKABLE_LCD_GET_STATE:
	    return ipmi_oem_rackable_lcd_get_state(intf, ret, error);
	default:
	    ipmi_printf("Invalid OEM Rackable command: %d\n", subcmd);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	    return -1;
    }
}

