/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <config.h>

#include <string.h>
#include <math.h>
#include <time.h>

#include <ipmitool/helper.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_sel.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_fru.h>
#include <ipmitool/ipmi_sensor.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/bswap.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmi_return.h>

static const struct valstr event_dir_vals[] = {
	{ 0, N_("Assertion Event") },
	{ 1, N_("Deassertion Event") },
	{ 0, NULL },
};

struct ipmi_event_sensor_types generic_event_types[] = {
	/* Threshold Based States */
	{ 0x01, 0x00, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Non-critical going low") },
	{ 0x01, 0x01, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Non-critical going high") },
	{ 0x01, 0x02, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Critical going low") },
	{ 0x01, 0x03, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Critical going high") },
	{ 0x01, 0x04, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Non-recoverable going low") },
	{ 0x01, 0x05, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Lower Non-recoverable going high") },
	{ 0x01, 0x06, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Non-critical going low") },
	{ 0x01, 0x07, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Non-critical going high") },
	{ 0x01, 0x08, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Critical going low") },
	{ 0x01, 0x09, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Critical going high") },
	{ 0x01, 0x0a, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Non-recoverable going low") },
	{ 0x01, 0x0b, 0xff, IPMI_EVENT_CLASS_THRESHOLD, N_("Threshold"), N_("Upper Non-recoverable going high") },
	/* DMI-based "usage state" States */
	{ 0x02, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Usage State"), N_("Transition to Idle") },
	{ 0x02, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Usage State"), N_("Transition to Active") },
	{ 0x02, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Usage State"), N_("Transition to Busy") },
	/* Digital-Discrete Event States */
	{ 0x03, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("State Deasserted") },
	{ 0x03, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("State Asserted") },
	{ 0x04, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Predictive Failure Deasserted") },
	{ 0x04, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Predictive Failure Asserted") },
	{ 0x05, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Limit Not Exceeded") },
	{ 0x05, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Limit Exceeded") },
	{ 0x06, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Performance Met") },
	{ 0x06, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Digital State"),  N_("Performance Lags") },
	/* Severity Event States */
	{ 0x07, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to OK") },
	{ 0x07, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Non-critical from OK") },
	{ 0x07, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Critical from less severe") },
	{ 0x07, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Non-recoverable from less severe") },
	{ 0x07, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Non-critical from more severe") },
	{ 0x07, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Critical from Non-recoverable") },
	{ 0x07, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Transition to Non-recoverable") },
	{ 0x07, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Monitor") },
	{ 0x07, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Severity State"), N_("Informational") },
	/* Availability Status States */
	{ 0x08, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Availability State"),  N_("Device Absent") },
	{ 0x08, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Availability State"),  N_("Device Present") },
	{ 0x09, 0x00, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Availability State"),  N_("Device Disabled") },
	{ 0x09, 0x01, 0xff, IPMI_EVENT_CLASS_DIGITAL, N_("Availability State"),  N_("Device Enabled") },
	{ 0x0a, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Running") },
	{ 0x0a, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to In Test") },
	{ 0x0a, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Power Off") },
	{ 0x0a, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to On Line") },
	{ 0x0a, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Off Line") },
	{ 0x0a, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Off Duty") },
	{ 0x0a, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Degraded") },
	{ 0x0a, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Transition to Power Save") },
	{ 0x0a, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Availability State"), N_("Install Error") },
	/* Redundancy States */
	{ 0x0b, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Fully Redundant") },
	{ 0x0b, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Redundancy Lost") },
	{ 0x0b, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Redundancy Degraded") },
	{ 0x0b, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Non-Redundant: Sufficient from Redundant") },
	{ 0x0b, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Non-Redundant: Sufficient from Insufficient") },
	{ 0x0b, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Non-Redundant: Insufficient Resources") },
	{ 0x0b, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Redundancy Degraded from Fully Redundant") },
	{ 0x0b, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Redundancy State"), N_("Redundancy Degraded from Non-Redundant") },
	/* ACPI Device Power States */
	{ 0x0c, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("ACPI Device Power State"), N_("D0 Power State") },
	{ 0x0c, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("ACPI Device Power State"), N_("D1 Power State") },
	{ 0x0c, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("ACPI Device Power State"), N_("D2 Power State") },
	{ 0x0c, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("ACPI Device Power State"), N_("D3 Power State") },
	/* END */
	{ 0x00, 0x00, 0xff, 0x00, NULL, NULL },
};

struct ipmi_event_sensor_types sensor_specific_types[] = {
	{ 0x00, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Reserved"),		NULL },
	{ 0x01, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Temperature"),	NULL },
	{ 0x02, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Voltage"),		NULL },
	{ 0x03, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Current"),		NULL },
	{ 0x04, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Fan"),		NULL },

	{ 0x05, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("General Chassis intrusion") },
	{ 0x05, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("Drive Bay intrusion") },
	{ 0x05, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("I/O Card area intrusion") },
	{ 0x05, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("Processor area intrusion") },
	{ 0x05, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("System unplugged from LAN") },
	{ 0x05, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("Unauthorized dock/undock") },
	{ 0x05, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Physical Security"), N_("FAN area intrusion") },
                                                        
	{ 0x06, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Front Panel Lockout violation attempted") },
	{ 0x06, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Pre-boot password violation - user password") },
	{ 0x06, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Pre-boot password violation - setup password") },
	{ 0x06, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Pre-boot password violation - network boot password") },
	{ 0x06, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Other pre-boot password violation") },
	{ 0x06, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Security"), N_("Out-of-band access password violation") },

	{ 0x07, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("IERR") },
	{ 0x07, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("Thermal Trip") },
	{ 0x07, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("FRB1/BIST failure") },
	{ 0x07, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("FRB2/Hang in POST failure") },
	{ 0x07, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("FRB3/Processor startup/init failure") },
	{ 0x07, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("Configuration Error") },
	{ 0x07, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("SM BIOS Uncorrectable CPU-complex Error") },
	{ 0x07, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("Presence detected") },
	{ 0x07, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("Disabled") },
	{ 0x07, 0x09, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Processor"), N_("Terminator presence detected") },
                                                                     
	{ 0x08, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("Presence detected") },
	{ 0x08, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("Failure detected") },
	{ 0x08, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("Predictive failure") },
	{ 0x08, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("Power Supply AC lost") },
	{ 0x08, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("AC lost or out-of-range") },
	{ 0x08, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Supply"), N_("AC out-of-range, but present") },
                                                                     
	{ 0x09, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Power off/down") },
	{ 0x09, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Power cycle") },
	{ 0x09, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("240VA power down") },
	{ 0x09, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Interlock power down") },
	{ 0x09, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("AC lost") },
	{ 0x09, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Soft-power control failure") },
	{ 0x09, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Failure detected") },
	{ 0x09, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Power Unit"), N_("Predictive failure") },

	{ 0x0a, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Cooling Device"), NULL },
	{ 0x0b, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Other Units-based Sensor"), NULL },

	{ 0x0c, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Correctable ECC") },
	{ 0x0c, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Uncorrectable ECC") },
	{ 0x0c, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Parity") },
	{ 0x0c, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Memory Scrub Failed") },
	{ 0x0c, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Memory Device Disabled") },
	{ 0x0c, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Memory"), N_("Correctable ECC logging limit reached") },

	{ 0x0d, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Drive Slot"), NULL },
	{ 0x0e, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("POST Memory Resize"), NULL },

	{ 0x0f, 0x00, 0x00, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unspecified") },
	{ 0x0f, 0x00, 0x01, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("No system memory installed") },
	{ 0x0f, 0x00, 0x02, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("No usable system memory") },
	{ 0x0f, 0x00, 0x03, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable IDE device failure") },
	{ 0x0f, 0x00, 0x04, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable system-board failure") },
	{ 0x0f, 0x00, 0x05, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable diskette failure") },
	{ 0x0f, 0x00, 0x06, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable hard-disk controller failure") },
	{ 0x0f, 0x00, 0x07, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable PS/2 or USB keyboard failure") },
	{ 0x0f, 0x00, 0x08, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Removable boot media not found") },
	{ 0x0f, 0x00, 0x09, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unrecoverable video controller failure") },
	{ 0x0f, 0x00, 0x0a, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("No video device selected") },
	{ 0x0f, 0x00, 0x0b, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("BIOS corruption detected") },
	{ 0x0f, 0x00, 0x0c, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("CPU voltage mismatch") },
	{ 0x0f, 0x00, 0x0d, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("CPU speed mismatch failure") },
	{ 0x0f, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Error"), N_("Unknown Error") },

	{ 0x0f, 0x01, 0x00, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Unspecified") },
	{ 0x0f, 0x01, 0x01, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Memory initialization") },
	{ 0x0f, 0x01, 0x02, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Hard-disk initialization") },
	{ 0x0f, 0x01, 0x03, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Secondary CPU Initialization") },
	{ 0x0f, 0x01, 0x04, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("User authentication") },
	{ 0x0f, 0x01, 0x05, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("User-initiated system setup") },
	{ 0x0f, 0x01, 0x06, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("USB resource configuration") },
	{ 0x0f, 0x01, 0x07, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("PCI resource configuration") },
	{ 0x0f, 0x01, 0x08, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Option ROM initialization") },
	{ 0x0f, 0x01, 0x09, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Video initialization") },
	{ 0x0f, 0x01, 0x0a, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Cache initialization") },
	{ 0x0f, 0x01, 0x0b, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("SMBus initialization") },
	{ 0x0f, 0x01, 0x0c, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Keyboard controller initialization") },
	{ 0x0f, 0x01, 0x0d, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Management controller initialization") },
	{ 0x0f, 0x01, 0x0e, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Docking station attachment") },
	{ 0x0f, 0x01, 0x0f, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Enabling docking station") },
	{ 0x0f, 0x01, 0x10, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Docking station ejection") },
	{ 0x0f, 0x01, 0x11, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Disabling docking station") },
	{ 0x0f, 0x01, 0x12, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Calling operating system wake-up vector") },
	{ 0x0f, 0x01, 0x13, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("System boot initiated") },
	{ 0x0f, 0x01, 0x14, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Motherboard initialization") },
	{ 0x0f, 0x01, 0x15, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("reserved") },
	{ 0x0f, 0x01, 0x16, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Floppy initialization") },
	{ 0x0f, 0x01, 0x17, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Keyboard test") },
	{ 0x0f, 0x01, 0x18, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Pointing device test") },
	{ 0x0f, 0x01, 0x19, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Primary CPU initialization") },
	{ 0x0f, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Hang"), N_("Unknown Hang") },

	{ 0x0f, 0x02, 0x00, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Unspecified") },
	{ 0x0f, 0x02, 0x01, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Memory initialization") },
	{ 0x0f, 0x02, 0x02, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Hard-disk initialization") },
	{ 0x0f, 0x02, 0x03, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Secondary CPU Initialization") },
	{ 0x0f, 0x02, 0x04, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("User authentication") },
	{ 0x0f, 0x02, 0x05, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("User-initiated system setup") },
	{ 0x0f, 0x02, 0x06, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("USB resource configuration") },
	{ 0x0f, 0x02, 0x07, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("PCI resource configuration") },
	{ 0x0f, 0x02, 0x08, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Option ROM initialization") },
	{ 0x0f, 0x02, 0x09, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Video initialization") },
	{ 0x0f, 0x02, 0x0a, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Cache initialization") },
	{ 0x0f, 0x02, 0x0b, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("SMBus initialization") },
	{ 0x0f, 0x02, 0x0c, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Keyboard controller initialization") },
	{ 0x0f, 0x02, 0x0d, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Management controller initialization") },
	{ 0x0f, 0x02, 0x0e, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Docking station attachment") },
	{ 0x0f, 0x02, 0x0f, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Enabling docking station") },
	{ 0x0f, 0x02, 0x10, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Docking station ejection") },
	{ 0x0f, 0x02, 0x11, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Disabling docking station") },
	{ 0x0f, 0x02, 0x12, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Calling operating system wake-up vector") },
	{ 0x0f, 0x02, 0x13, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("System boot initiated") },
	{ 0x0f, 0x02, 0x14, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Motherboard initialization") },
	{ 0x0f, 0x02, 0x15, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("reserved") },
	{ 0x0f, 0x02, 0x16, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Floppy initialization") },
	{ 0x0f, 0x02, 0x17, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Keyboard test") },
	{ 0x0f, 0x02, 0x18, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Pointing device test") },
	{ 0x0f, 0x02, 0x19, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Primary CPU initialization") },
	{ 0x0f, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Firmware Progress"), N_("Unknown Progress") },

	{ 0x10, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Event Logging Disabled"), N_("Correctable memory error logging disabled") },
	{ 0x10, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Event Logging Disabled"), N_("Event logging disabled") },
	{ 0x10, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Event Logging Disabled"), N_("Log area reset/cleared") },
	{ 0x10, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Event Logging Disabled"), N_("All event logging disabled") },
	{ 0x10, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Event Logging Disabled"), N_("Log full") },

	{ 0x11, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("BIOS Reset") },
	{ 0x11, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS Reset") },
	{ 0x11, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS Shut Down") },
	{ 0x11, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS Power Down") },
	{ 0x11, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS Power Cycle") },
	{ 0x11, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS NMI/Diag Interrupt") },
	{ 0x11, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS Expired") },
	{ 0x11, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 1"), N_("OS pre-timeout Interrupt") },

	{ 0x12, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("System Reconfigured") },
	{ 0x12, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("OEM System boot event") },
	{ 0x12, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("Undetermined system hardware failure") },
	{ 0x12, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("Entry added to auxiliary log") },
	{ 0x12, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("PEF Action") },
	{ 0x12, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Event"), N_("Timestamp Clock Sync.") },

	{ 0x13, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("NMI/Diag Interrupt") },
	{ 0x13, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("Bus Timeout") },
	{ 0x13, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("I/O Channel check NMI") },
	{ 0x13, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("Software NMI") },
	{ 0x13, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("PCI PERR") },
	{ 0x13, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("PCI SERR") },
	{ 0x13, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("EISA failsafe timeout") },
	{ 0x13, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("Bus Correctable error") },
	{ 0x13, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("Bus Uncorrectable error") },
	{ 0x13, 0x09, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Critical Interrupt"), N_("Fatal NMI") },

	{ 0x14, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Button"), N_("Power Button pressed") },
	{ 0x14, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Button"), N_("Sleep Button pressed") },
	{ 0x14, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Button"), N_("Reset Button pressed") },

	{ 0x15, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Module/Board"), NULL },
	{ 0x16, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Microcontroller/Coprocessor"), NULL },
	{ 0x17, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Add-in Card"), NULL },
	{ 0x18, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Chassis"), NULL },
	{ 0x19, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Chip Set"), NULL },
	{ 0x1a, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Other FRU"), NULL },
	{ 0x1b, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Cable/Interconnect"), NULL },
	{ 0x1c, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Terminator"), NULL },

	{ 0x1d, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Boot Initiated"), N_("Initiated by power up") },
	{ 0x1d, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Boot Initiated"), N_("Initiated by hard reset") },
	{ 0x1d, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Boot Initiated"), N_("Initiated by warm reset") },
	{ 0x1d, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Boot Initiated"), N_("User requested PXE boot") },
	{ 0x1d, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System Boot Initiated"), N_("Automatic boot to diagnostic") },

	{ 0x1e, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Boot Error"), N_("No bootable media") },
	{ 0x1e, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Boot Error"), N_("Non-bootable disk in drive") },
	{ 0x1e, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Boot Error"), N_("PXE server not found") },
	{ 0x1e, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Boot Error"), N_("Invalid boot sector") },
	{ 0x1e, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Boot Error"), N_("Timeout waiting for selection") },

	{ 0x1f, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("A: boot completed") },
	{ 0x1f, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("C: boot completed") },
	{ 0x1f, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("PXE boot completed") },
	{ 0x1f, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("Diagnostic boot completed") },
	{ 0x1f, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("CD-ROM boot completed") },
	{ 0x1f, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("ROM boot completed") },
	{ 0x1f, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Boot"), N_("boot completed - device not specified") },

	{ 0x20, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Critical Stop"), N_("Stop during OS load/init") },
	{ 0x20, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("OS Critical Stop"), N_("Run-time stop") },

	{ 0x21, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Fault Status asserted") },
	{ 0x21, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Identify Status asserted") },
	{ 0x21, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Device installed/attached") },
	{ 0x21, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Ready for device installation") },
	{ 0x21, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Ready for device removal") },
	{ 0x21, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Slot Power is off") },
	{ 0x21, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Device removal request") },
	{ 0x21, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Interlock asserted") },
	{ 0x21, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Slot/Connector"), N_("Slot is disabled") },

	{ 0x22, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S0/G0: working") },
	{ 0x22, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S1: sleeping with system hw & processor context maintained") },
	{ 0x22, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S2: sleeping, processor context lost") },
	{ 0x22, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S3: sleeping, processor & hw context lost, memory retained") },
	{ 0x22, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S4: non-volatile sleep/suspend-to-disk") },
	{ 0x22, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S5/G2: soft-off") },
	{ 0x22, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S4/S5: soft-off") },
	{ 0x22, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("G3: mechanical off") },
	{ 0x22, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("Sleeping in S1/S2/S3 state") },
	{ 0x22, 0x09, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("G1: sleeping") },
	{ 0x22, 0x0a, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("S5: entered by override") },
	{ 0x22, 0x0b, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("Legacy ON state") },
	{ 0x22, 0x0c, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("Legacy OFF state") },
	{ 0x22, 0x0e, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("System ACPI Power State"), N_("Unknown") },

	{ 0x23, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("Timer expired") },
	{ 0x23, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("Hard reset") },
	{ 0x23, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("Power down") },
	{ 0x23, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("Power cycle") },
	{ 0x23, 0x04, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("reserved") },
	{ 0x23, 0x05, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("reserved") },
	{ 0x23, 0x06, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("reserved") },
	{ 0x23, 0x07, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("reserved") },
	{ 0x23, 0x08, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Watchdog 2"), N_("Timer interrupt") },

	{ 0x24, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Alert"), N_("Platform generated page") },
	{ 0x24, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Alert"), N_("Platform generated LAN alert") },
	{ 0x24, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Alert"), N_("Platform Event Trap generated") },
	{ 0x24, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Platform Alert"), N_("Platform generated SNMP trap, OEM format") },

	{ 0x25, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Entity Presence"), N_("Present") },
	{ 0x25, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Entity Presence"), N_("Absent") },
	{ 0x25, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Entity Presence"), N_("Disabled") },

	{ 0x26, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Monitor ASIC/IC"), NULL },

	{ 0x27, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("LAN"), N_("Heartbeat Lost") },
	{ 0x27, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("LAN"), N_("Heartbeat") },

	{ 0x28, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Management Subsystem Health"), N_("Sensor access degraded or unavailable") },
	{ 0x28, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Management Subsystem Health"), N_("Controller access degraded or unavailable") },
	{ 0x28, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Management Subsystem Health"), N_("Management controller off-line") },
	{ 0x28, 0x03, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Management Subsystem Health"), N_("Management controller unavailable") },

	{ 0x29, 0x00, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Battery"), N_("Low") },
	{ 0x29, 0x01, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Battery"), N_("Failed") },
	{ 0x29, 0x02, 0xff, IPMI_EVENT_CLASS_DISCRETE, N_("Battery"), N_("Presence Detected") },

	{ 0xC0, 0x00, 0xff, 0x00, N_("OEM"), NULL },
	{ 0x00, 0x00, 0x00, 0x00, NULL, NULL },
};

static int
ipmi_sel_entry_used(pp_ipmi_return_t* ret, unsigned int rec_id)
{
    int i, s;
    if (!ret->data.sel_list) return 0;

    for (i = 0, s = vector_size(ret->data.sel_list); i < s; i++) {
        pp_ipmi_sel_list_entry_t *e = vector_get(ret->data.sel_list, i);
        if (e->record_id == rec_id) return 1;
    }
    return 0;
}

static int
ipmi_sel_entry_list_id_used(pp_ipmi_return_t* ret, unsigned int rec_id)
{
    int i, s;
    if (!ret->data.sel_list) return 0;

    for (i = 0, s = vector_size(ret->data.sel_list_ids); i < s; i++) {
        unsigned int record_id= vector_get_u_int(ret->data.sel_list_ids, i);
        if (record_id == rec_id) return 1;
    }
    return 0;
}

static const char *
ipmi_get_event_type(uint8_t code)
{
        if (code == 0)
                return _("Unspecified");
        if (code == 1)
                return _("Threshold");
        if (code >= 0x02 && code <= 0x0b)
                return _("Generic Discrete");
        if (code == 0x6f)
                return _("Sensor-specific Discrete");
        if (code >= 0x70 && code <= 0x7f)
                return _("OEM");
        return _("Reserved");
}

static int
ipmi_get_event_type_code(uint8_t code)
{
        if (code == 0)
                return PP_IPMI_EVENT_TYPE_UNSPECIFIED;
        if (code == 1)
                return PP_IPMI_EVENT_TYPE_THRESHOLD;
        if (code >= 0x02 && code <= 0x0b)
                return PP_IPMI_EVENT_TYPE_GENERIC_DISCRETE;
        if (code == 0x6f)
                return PP_IPMI_EVENT_TYPE_SENSOR_SPECIFIC_DISCRETE;
        if (code >= 0x70 && code <= 0x7f)
                return PP_IPMI_EVENT_TYPE_OEM;
        return PP_IPMI_EVENT_TYPE_RESERVED;
}

static void
ipmi_get_event_desc(struct sel_event_record * rec, char ** desc, int * error)
{
	uint8_t code, offset;
	struct ipmi_event_sensor_types *evt;

        if (desc == NULL)
	        return;
	*desc = NULL;

	if (rec->event_type == 0x6f) {
		evt = sensor_specific_types;
		code = rec->sensor_type;
	} else {
		evt = generic_event_types;
		code = rec->event_type;
	}

	offset = rec->event_data[0] & 0xf;

	while (evt->type) {
		if ((evt->code == code && evt->offset == offset)    &&
                    ((evt->data == ALL_OFFSETS_SPECIFIED) ||
                     ((rec->event_data[0] & DATA_BYTE2_SPECIFIED_MASK) &&
                      (evt->data == rec->event_data[1]))))
		{
                    if (evt->desc) {
			const char *__d = _(evt->desc);
			*desc = (char *)malloc(strlen(__d) + 48);
			if (*desc == NULL) {
				ipmi_printf("%s: malloc failure\n", __FUNCTION__);
				ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
				return;
			}
			memset(*desc, 0, strlen(__d)+48);
			sprintf(*desc, "%s", __d);
			return;
                    } else {
                        return;
                    }
		}
		evt++;
	}
}

static const char *
ipmi_sel_get_sensor_type(uint8_t code)
{
	struct ipmi_event_sensor_types *st;
	for (st = sensor_specific_types; st->type != NULL; st++)
		if (st->code == code)
			return _(st->type);
	return _("Unknown");
}

static const char *
ipmi_sel_get_sensor_type_offset(uint8_t code, uint8_t offset)
{
	struct ipmi_event_sensor_types *st;
	for (st = sensor_specific_types; st->type != NULL; st++)
		if (st->code == code && st->offset == (offset&0xf))
			return _(st->type);
	return ipmi_sel_get_sensor_type(code);
}

static int
ipmi_sel_get_info(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint16_t e, f;
	int pctfull = 0;
	pp_ipmi_sel_info_t *info;

	if (ipmi_init_sel_info(ipmi_ret)) {
		ipmi_printf("Error: Could not allocate return structure.\n");
		return -1;
	}

	info = &ipmi_ret->data.sel_info;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_GET_SEL_INFO;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get SEL Info command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SEL Info command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "sel_info");

	info->version = ((rsp->data[0] & 0xf0) >> 4) | (rsp->data[0] & 0xf);
	/* save the entry count and free space to determine percent full */
	e = buf2short(rsp->data + 1);
	f = buf2short(rsp->data + 3);
	info->entries = e;
	info->free_space = f;
	if (e) {
		e *= 16;
		f += e;
		pctfull = (int)(100 * ( (double)e / (double)f ));
	}
	info->percent_used = pctfull;
	ipmi_set_timestamp(buf2long(rsp->data + 5), &info->last_add_time);
	ipmi_set_timestamp(buf2long(rsp->data + 9), &info->last_del_time);
	info->overflow = rsp->data[13] & 0x80 ? 1 : 0;
	info->delete_supported		= rsp->data[13] & 0x8 ? 1 : 0;
	info->partial_add_supported	= rsp->data[13] & 0x4 ? 1 : 0;
	info->reserve_supported		= rsp->data[13] & 0x2 ? 1 : 0;
	info->get_alloc_info_supported	= rsp->data[13] & 0x1 ? 1 : 0;

	/* get sel allocation info if supported */
	if (rsp->data[13] & 1) {
		uint16_t unit_size, alloc_units;
		
		memset(&req, 0, sizeof(req));
		req.msg.netfn = IPMI_NETFN_STORAGE;
		req.msg.cmd = IPMI_CMD_GET_SEL_ALLOC_INFO;

		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Get SEL Allocation Info command failed\n");
			return -1;
		}
		if (rsp->ccode > 0) {
			ipmi_printf("Get SEL Allocation Info command failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
			return -1;
		}

		alloc_units = buf2short(rsp->data);
		if (alloc_units) {
			info->allocation_info.alloc_units.specified = 1;
			info->allocation_info.alloc_units.units = alloc_units;
		} else {
			info->allocation_info.alloc_units.specified = 0;
		}
		unit_size = buf2short(rsp->data + 2);
		if (unit_size) {
			info->allocation_info.alloc_unit_size.specified = 1;
			info->allocation_info.alloc_unit_size.unit_size = unit_size;
		} else {
			info->allocation_info.alloc_unit_size.specified = 0;
		}
		info->allocation_info.free_units = buf2short(rsp->data + 4);
		info->allocation_info.largest_free_block = buf2short(rsp->data + 6);
		info->allocation_info.max_record_size =rsp->data[8];
	}

	return 0;
}

static uint16_t
ipmi_sel_get_std_entry(struct ipmi_intf * intf, uint16_t id,
		       struct sel_event_record * evt, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;
	uint8_t msg_data[6];
	uint16_t next;

	memset(msg_data, 0, 6);
	msg_data[0] = 0x00;	/* no reserve id, not partial get */
	msg_data[1] = 0x00;
	msg_data[2] = id & 0xff;
	msg_data[3] = (id >> 8) & 0xff;
	msg_data[4] = 0x00;	/* offset */
	msg_data[5] = 0xff;	/* length */

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_GET_SEL_ENTRY;
	req.msg.data = msg_data;
	req.msg.data_len = 6;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get SEL Entry %x command failed\n", id);
		return 0;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SEL Entry %x command failed: %s\n",
			id, val2str(rsp->ccode, completion_code_vals));
		return 0;
	}

	/* save next entry id */
	next = (rsp->data[1] << 8) | rsp->data[0];

	if (verbose > 2)
		ipmi_printf("SEL Entry: %s\n", buf2str(rsp->data+2, rsp->data_len-2));


	evt->record_id = (rsp->data[3] << 8) | rsp->data[2];

	if (rsp->data[4] >= 0xc0) {
		ipmi_printf("Entry %x not a standard SEL entry", id);
		return next;
	}

	/* save response into SEL event structure */
	memset(evt, 0, sizeof(*evt));
	evt->record_type = rsp->data[4];
	evt->timestamp = (rsp->data[8] << 24) |	(rsp->data[7] << 16) |
		(rsp->data[6] << 8) | rsp->data[5];
	evt->gen_id = (rsp->data[10] << 8) | rsp->data[9];
	evt->evm_rev = rsp->data[11];
	evt->sensor_type = rsp->data[12];
	evt->sensor_num = rsp->data[13];
	evt->event_type = rsp->data[14] & 0x7f;
	evt->event_dir = (rsp->data[14] & 0x80) >> 7;
	evt->event_data[0] = rsp->data[15];
	evt->event_data[1] = rsp->data[16];
	evt->event_data[2] = rsp->data[17];

	return next;
}

static uint16_t
ipmi_sel_get_ids(struct ipmi_intf * intf, uint16_t id,
		 uint16_t *record_id, int * error)
{
	struct ipmi_rq req;
	struct ipmi_rs * rsp;
	uint8_t msg_data[6];
	uint16_t next;

	memset(msg_data, 0, 6);
	msg_data[0] = 0x00;	/* no reserve id, not partial get */
	msg_data[1] = 0x00;
	msg_data[2] = id & 0xff;
	msg_data[3] = (id >> 8) & 0xff;
	msg_data[4] = 0x00;	/* offset */
	msg_data[5] = 0x02;	/* length; we don't need the whole data */

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_GET_SEL_ENTRY;
	req.msg.data = msg_data;
	req.msg.data_len = 6;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get SEL Entry %x command failed\n", id);
		return 0;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SEL Entry %x command failed: %s\n",
			id, val2str(rsp->ccode, completion_code_vals));
		return 0;
	}

	/* save next entry id */
	next = (rsp->data[1] << 8) | rsp->data[0];

	if (verbose > 2)
		ipmi_printf("SEL Entry: %s\n", buf2str(rsp->data+2, rsp->data_len-2));


	/* save response into SEL event structure */
	*record_id = (rsp->data[3] << 8) | rsp->data[2];

	return next;
}

static void
ipmi_sel_print_std_entry(struct ipmi_intf * intf, struct sel_event_record * evt, pp_ipmi_return_t *ipmi_ret, vector_t *raw_sdr, int * error)
{
        char * description;
        pp_ipmi_sel_list_entry_t * entry = ipmi_init_sel_list_entry(ipmi_ret);
        struct sdr_record_list * sensor;
	
	if (!evt)
		return;

	if (!entry) {
		ipmi_printf("Error: could not allocate entry return structure.\n");
		return;
	}

	entry->record_id = evt->record_id;
	entry->record_type = evt->record_type;

	if (evt->record_type == 0xf0) {
		pp_strappendf(&entry->record_type_string,
			      _("Linux kernel panic (OEM record %02x) (%s)"),
			      evt->record_type, (char *) evt + 5);
	}
	else if (evt->record_type >= 0xc0) {
		pp_strappendf(&entry->record_type_string, _("OEM record %02x"), evt->record_type);
	}
	else {
		pp_strappendf(&entry->record_type_string, _("SEL record %02x"), evt->record_type);
	}

	if (evt->record_type < 0xe0)
	{
		ipmi_set_timestamp(evt->timestamp, &entry->timestamp);
	}

	if (evt->record_type >= 0xc0)
	{
		entry->further_data_is_valid = 0;
		return;
	}

	entry->further_data_is_valid = 1;
	int owner_id = evt->gen_id & 0xff;
	int owner_lun = (evt->gen_id >> 8) & 0xff;
	sensor = ipmi_sdr_find_sdr_bysensorno(intf, evt->sensor_num, owner_id, owner_lun, raw_sdr, error);

	entry->generator_id = evt->gen_id;
	entry->evm_revision = evt->evm_rev;
	pp_strappend(&entry->sensor_type, ipmi_sel_get_sensor_type_offset(evt->sensor_type, evt->event_data[0]));
	entry->sensor_type_code = evt->sensor_type;
	entry->sensor_num = evt->sensor_num;
	if (sensor != NULL) {
		pp_strappend(&entry->sensor_name, sdr_get_sensor_name(sensor));
	} else {
                pp_strappend(&entry->sensor_name, _("Unknown"));
        }
	pp_strappend(&entry->event_type, ipmi_get_event_type(evt->event_type));
	entry->event_type_code = ipmi_get_event_type_code(evt->event_type);
	pp_strappend(&entry->event_direction, val2str(evt->event_dir, event_dir_vals));
	entry->event_direction_code = evt->event_dir? PP_IPMI_EVENT_DIRECTION_DEASSERTION : PP_IPMI_EVENT_DIRECTION_ASSERTION;
	memcpy(entry->event_data, evt->event_data, 3);
	ipmi_get_event_desc(evt, &description, error);
	pp_strappend(&entry->description, description ? description : ""); /* no descr avail */
	free(description);

	if (sensor != NULL && evt->event_type == 1 && sensor->type == SDR_RECORD_TYPE_FULL_SENSOR) {
		if (((evt->event_data[0] >> 6) & 3) == 1) {
			entry->trigger_reading.valid = 1;
			entry->trigger_reading.value = (float)sdr_convert_sensor_reading(
				sensor->record.full, evt->event_data[1]);
			entry->trigger_reading.cim_value = sdr_convert_sensor_reading_cim(
				sensor->record.full, evt->event_data[1]);
		}

		/* trigger threshold in event data byte 3 */
		if (((evt->event_data[0] >> 4) & 3) == 1) {
			entry->threshold_reading.valid = 1;
			entry->threshold_reading.value = (float)sdr_convert_sensor_reading(
				sensor->record.full, evt->event_data[2]);
			entry->threshold_reading.cim_value = sdr_convert_sensor_reading_cim(
				sensor->record.full, evt->event_data[2]);
		}
		
		entry->above_threshold = ((evt->event_data[0] & 0xf) % 2) ? 1 : 0;
		pp_strappend(&entry->unit,
			ipmi_sdr_get_unit_string(sensor->record.full->unit.modifier,
						sensor->record.full->unit.type.base,
						sensor->record.full->unit.type.modifier));
	}
}

static int
ipmi_sel_list_entries(struct ipmi_intf * intf,
		      pp_ipmi_sel_range_t range,
		      time_t from, time_t to,
		      unsigned int count_total,
		      vector_t *raw_sdr,
		      pp_ipmi_return_t *ipmi_ret,
		      int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	int ret = 0;
	uint16_t next_id = 0, curr_id = 0;
	struct sel_event_record evt;
	time_t sel_time = 0;
	int count = 0;
	int n=0;

	if (range == PP_IPMI_SEL_LIST_RANGE_COUNT_LAST) {
		count = -(int)count_total;
	} else if (range == PP_IPMI_SEL_LIST_RANGE_COUNT_FIRST) {
		count = count_total;
	} else if (range != PP_IPMI_SEL_LIST_RANGE_ALL) {
		pp_ipmi_return_t ipmi_ret_loc;

		memset(&ipmi_ret_loc, 0, sizeof(ipmi_ret_loc));
		ret = pp_ipmi_cmd_run(intf, PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_GET_TIME, NULL, &ipmi_ret_loc, error);

		if (ret) {
			ipmi_printf("Error: could not read SEL time.\n");
			return ret;
		}

		sel_time = ipmi_ret_loc.data.sel_get_time.timestamp;
	}

	if (ipmi_init_sel_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_GET_SEL_INFO;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get SEL Info command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SEL Info command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "sel_info");

	if (rsp->data[1] == 0 && rsp->data[2] == 0) {
		ipmi_printf("SEL has no entries\n");
		return 0;
	}

	// do we really need the SEL reservation here???
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_RESERVE_SEL;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Reserve SEL command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Reserve SEL command failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (count < 0) {
		/** Show only the most recent 'count' records. */
		int delta;
		uint16_t entries;

		req.msg.cmd = IPMI_CMD_GET_SEL_INFO;
		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Get SEL Info command failed\n");
			return -1;
		}
		if (rsp->ccode > 0) {
			ipmi_printf("Get SEL Info command failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
			return -1;
		}
		entries = buf2short(rsp->data + 1);
		if (-count > entries)
			count = -entries;

		/* Get first record. */
		next_id = ipmi_sel_get_std_entry(intf, 0, &evt, error);

		delta = next_id - evt.record_id;

		/* Get last record. */
		next_id = ipmi_sel_get_std_entry(intf, 0xffff, &evt, error);

		next_id = evt.record_id + count * delta + delta;
	}

	while (next_id != 0xffff) {
		curr_id = next_id;
		if (verbose > 1)
			ipmi_printf("SEL Next ID: %04x\n", curr_id);

		next_id = ipmi_sel_get_std_entry(intf, curr_id, &evt, error);
		if (next_id == 0) {
			/*
			 * usually next_id of zero means end but
			 * retry because some hardware has quirks
			 * and will return 0 randomly.
			 */
			next_id = ipmi_sel_get_std_entry(intf, curr_id, &evt, error);
			if (next_id == 0)
				break;
		}

		// skip not-requested entries
		if (evt.record_type < 0xe0) {
			if (
			    (range == PP_IPMI_SEL_LIST_RANGE_FROMTO &&
			     ((time_t)evt.timestamp < from || (time_t)evt.timestamp > to)) ||
			    (range == PP_IPMI_SEL_LIST_RANGE_LASTDAY &&
			     (time_t)evt.timestamp < (sel_time - 24 * 60 * 60)) ||
			    (range == PP_IPMI_SEL_LIST_RANGE_LAST3DAYS &&
			     (time_t)evt.timestamp < (sel_time - 3 * 24 * 60 * 60)) ||
			    (range == PP_IPMI_SEL_LIST_RANGE_LASTWEEK &&
			     (time_t)evt.timestamp < (sel_time - 7 * 24 * 60 * 60)) ||
			    (range == PP_IPMI_SEL_LIST_RANGE_LASTMONTH &&
			     (time_t)evt.timestamp < (sel_time - 30 * 24 * 60 * 60))
			   ) {
				continue;
			}
			     
		}

		ipmi_sel_print_std_entry(intf, &evt, ipmi_ret, raw_sdr, error);

		/* is next id valid? prevent double reading */
		if (ipmi_sel_entry_used(ipmi_ret, next_id)) {
			ipmi_printf("Error: SEL corrupted (cyclically linked)\n");
			ret = -1;
			break;
		}

		if (++n == count) {
			break;
		}
	}
	
	ipmi_sdr_list_empty(intf);
	return ret;
}

static int
ipmi_sel_show_entry(struct ipmi_intf * intf,
		    vector_t * get_list,
		    vector_t * sdr_raw_list,
		    pp_ipmi_return_t *ipmi_ret,
		    int * error)
{
	size_t no_entries = vector_size(get_list);
	size_t i;
	struct sel_event_record evt;
	int rc = 0;
	
	if (ipmi_init_sel_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	for (i = 0; i < no_entries; i++) {
		unsigned int id = vector_get_u_int(get_list, i);

		/* lookup SEL entry based on ID */
		ipmi_sel_get_std_entry(intf, (uint16_t)id, &evt, error);
		if (evt.sensor_num == 0 && evt.sensor_type == 0) {
			ipmi_printf("SEL Entry 0x%x not found\n", id);
			rc = -1;
			continue;
		}

		/* lookup SEL entry based on sensor number and type */
		ipmi_sel_print_std_entry(intf, &evt, ipmi_ret, sdr_raw_list, error);
	}
	
	if (rc) {
	    return rc;
	}
        if (ipmi_ret->data.sel_list == NULL || vector_size(ipmi_ret->data.sel_list) == 0) {
            return -1;
        }
	
	return 0;
}

static uint16_t
ipmi_sel_reserve(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_RESERVE_SEL;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to reserve SEL\n");
		return 0;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Unable to reserve SEL: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		return 0;
	}

	return (rsp->data[0] | (rsp->data[1] << 8));
}



/*
 * ipmi_sel_get_time
 *
 * return 0 on success,
 *        -1 on error
 */
static int
ipmi_sel_get_time(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	uint32_t timei;
	time_t mytime;

	if (ipmi_init_sel_get_time(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure for SEL GET_TIME.\n");
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd   = IPMI_GET_SEL_TIME;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Get SEL Time command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get SEL Time command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	if (rsp->data_len != 4) {
		ipmi_printf("Get SEL Time command failed: "
			"Invalid data length %d\n", rsp->data_len);
		return -1;
	}
	
	memcpy(&timei, rsp->data, 4);
#ifdef WORDS_BIGENDIAN
	timei = BSWAP_32(timei);
#endif
	mytime = (time_t)timei;

	ipmi_set_timestamp(mytime, &ipmi_ret->data.sel_get_time);

	return 0;
}



/*
 * ipmi_sel_set_time
 *
 * return 0 on success,
 *        -1 on error
 */
static int
ipmi_sel_set_time(struct ipmi_intf * intf, time_t mytime, int * error)
{
	struct ipmi_rs     * rsp;
	struct ipmi_rq       req;
	uint32_t	     timei;

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_STORAGE;
	req.msg.cmd      = IPMI_SET_SEL_TIME;
	timei = (uint32_t)mytime;
	req.msg.data = (uint8_t *)&timei;	
	req.msg.data_len = 4;

#ifdef WORDS_BIGENDIAN
	timei = BSWAP_32(timei);
#endif

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Set SEL Time command failed\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set SEL Time command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}



static int
ipmi_sel_clear(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint16_t reserve_id;
	uint8_t msg_data[6];

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	reserve_id = ipmi_sel_reserve(intf, error);
	if (reserve_id == 0) {
		ipmi_printf("Error: could not reserve sel.\n");
		return -1;
	}

	memset(msg_data, 0, 6);
	msg_data[0] = reserve_id & 0xff;
	msg_data[1] = reserve_id >> 8;
	msg_data[2] = 'C';
	msg_data[3] = 'L';
	msg_data[4] = 'R';
	msg_data[5] = 0xaa;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_STORAGE;
	req.msg.cmd = IPMI_CMD_CLEAR_SEL;
	req.msg.data = msg_data;
	req.msg.data_len = 6;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to clear SEL\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Unable to clear SEL: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (verbose) {
		ipmi_printf("Clearing SEL.  Please allow a few seconds to erase.\n");
	}
	return 0;
}

static int
ipmi_sel_delete(struct ipmi_intf * intf, vector_t *entries, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint16_t id;
	uint8_t msg_data[4];
	unsigned int i;
	int rc = 0;
	size_t no_entries = vector_size(entries);

	id = ipmi_sel_reserve(intf, error);
	if (id == 0) {
		return -1;
	}

	memset(msg_data, 0, 4);
	msg_data[0] = id & 0xff;
	msg_data[1] = id >> 8;

	for (i = 0; i < no_entries; i++) {
		id = vector_get_u_int(entries, i);
		msg_data[2] = id & 0xff;
		msg_data[3] = id >> 8;

		memset(&req, 0, sizeof(req));
		req.msg.netfn = IPMI_NETFN_STORAGE;
		req.msg.cmd = IPMI_CMD_DELETE_SEL_ENTRY;
		req.msg.data = msg_data;
		req.msg.data_len = 4;

		rsp = intf->sendrecv(intf, &req, error);
		if (rsp == NULL) {
			ipmi_printf("Unable to delete entry %d\n", id);
			rc = -1;
		}
		else if (rsp->ccode > 0) {
			ipmi_printf("Unable to delete entry %d: %s\n", id,
				val2str(rsp->ccode, completion_code_vals));
			rc = -1;
		}
		else {
			if (verbose) {
				ipmi_printf("Deleted entry %d\n", id);
			}
		}
	}
	
	return rc;
}

static int ipmi_sel_list_ids(struct ipmi_intf * intf, unsigned int start_from, int count,
                             pp_ipmi_return_t *ipmi_ret, int * error) {
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint16_t next_id = 0, curr_id = 0;
    uint16_t record_id;
    int n = 0;
    int ret = 0;

    if (ipmi_init_sel_list_ids(ipmi_ret)) {
	ipmi_printf("Error: could not allocate return structure.\n");
        return -1;
    }

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_STORAGE;
    req.msg.cmd = IPMI_CMD_GET_SEL_INFO;
    
    rsp = intf->sendrecv(intf, &req, error);
    if (rsp == NULL) {
    	ipmi_printf("Get SEL Info command failed\n");
    	return -1;
    }
    if (rsp->ccode > 0) {
    	ipmi_printf("Get SEL Info command failed: %s\n",
    	       val2str(rsp->ccode, completion_code_vals));
    	return -1;
    }
    if (verbose > 2)
    	printbuf(rsp->data, rsp->data_len, "sel_info");
    
    if (rsp->data[1] == 0 && rsp->data[2] == 0) {
    	ipmi_printf("SEL has no entries\n");
    	return 0;
    }
    
    next_id = start_from;

    while (next_id != 0xffff) {
	curr_id = next_id;
	if (verbose > 1)
	    ipmi_printf("SEL Next ID: %04x\n", curr_id);

	next_id = ipmi_sel_get_ids(intf, curr_id, &record_id, error);

	if (next_id == 0) {
	    /*
	     * usually next_id of zero means end but
	     * retry because some hardware has quirks
	     * and will return 0 randomly.
	     */
	    next_id = ipmi_sel_get_ids(intf, curr_id, &record_id, error);
	    if (next_id == 0)
		break;
	}

	/* is next id valid? prevent double reading */
	if (ipmi_sel_entry_list_id_used(ipmi_ret, next_id)) {
	    ipmi_printf("Error: SEL corrupted (cyclically linked)\n");
	    ret = -1;
	    break;
	}

	vector_add_u_int(ipmi_ret->data.sel_list_ids, record_id);

	if (++n == count) {
	    break;
	}
    }

    return ret;
}

int ipmi_sel_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	switch ((pp_ipmi_sel_subcommand_t) subcmd) {
		case PP_IPMI_SEL_SUBCMD_INFO:
			return ipmi_sel_get_info(intf, ipmi_ret, error);
		case PP_IPMI_SEL_SUBCMD_CLEAR:
			return ipmi_sel_clear(intf, error);
		case PP_IPMI_SEL_SUBCMD_LIST:
		{
			pp_ipmi_sel_range_t range = PP_IPMI_SEL_LIST_RANGE_ALL;
			time_t from = 0, to = 0;
			unsigned int count = 0;
			vector_t *raw_sdr = NULL;
			if (params && !params->is_cmdline) {
				range = params->data.sel_list.range;
				from = params->data.sel_list.from;
				to = params->data.sel_list.to;
				count = params->data.sel_list.count;
				raw_sdr = params->data.sel_list.sdr_raw_list;
			}
			return ipmi_sel_list_entries(intf, range, from, to, count, raw_sdr, ipmi_ret, error);
		}
		case PP_IPMI_SEL_SUBCMD_DELETE:
			if (!params || params->is_cmdline || !params->data.sel_delete_list) {
				ipmi_printf("No SEL entries to delete!\n");
				return -1;
			}
			return ipmi_sel_delete(intf, params->data.sel_delete_list, error);
		case PP_IPMI_SEL_SUBCMD_GET:
			if (!params || params->is_cmdline || !params->data.sel_get_list.id_list) {
				ipmi_printf("No SEL entries to get!\n");
				return -1;
			}
			return ipmi_sel_show_entry(intf, params->data.sel_get_list.id_list,
			    params->data.sel_get_list.sdr_raw_list, ipmi_ret, error);
		case PP_IPMI_SEL_SUBCMD_GET_TIME:
			return ipmi_sel_get_time(intf, ipmi_ret, error);
		case PP_IPMI_SEL_SUBCMD_SET_TIME:
			if (!params || params->is_cmdline) {
				ipmi_printf("No parameter for SEL SET TIME!\n");
				return -1;
			}
			return ipmi_sel_set_time(intf, params->data.sel_set_time, error);
		case PP_IPMI_SEL_SUBCMD_LIST_IDS:
			if (!params || params->is_cmdline) {
				ipmi_printf("No parameter for SEL LIST IDS!\n");
				return -1;
			}
			return ipmi_sel_list_ids(intf, params->data.sel_list_ids.start_from,
			    params->data.sel_list_ids.count, ipmi_ret, error);
		default:
			ipmi_printf("Invalid SEL command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			return -1;
	}
}


