/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <string.h>
#include <math.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_sel.h>
#include <ipmitool/ipmi_sensor.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_error.h>
#include <ipmi_return.h>
#include <ipmi_params.h>

#define READING_UNAVAILABLE	0x20

static
struct ipmi_rs *
ipmi_sensor_set_sensor_thresholds(struct ipmi_intf * intf, 
                                  uint8_t sensor,
                                  uint8_t threshold,
                                  uint8_t setting,
                                  int * error)
{
	struct ipmi_rq req;
	static struct sensor_set_thresh_rq set_thresh_rq;
	
	memset(&set_thresh_rq, 0, sizeof(set_thresh_rq));
	set_thresh_rq.sensor_num = sensor;
	set_thresh_rq.set_mask   = threshold;
	if (threshold == UPPER_NON_RECOV_SPECIFIED)
		set_thresh_rq.upper_non_recov = setting;
	else if (threshold == UPPER_CRIT_SPECIFIED)
		set_thresh_rq.upper_crit = setting;
	else if (threshold == UPPER_NON_CRIT_SPECIFIED)
		set_thresh_rq.upper_non_crit = setting;
	else if (threshold == LOWER_NON_CRIT_SPECIFIED)
		set_thresh_rq.lower_non_crit = setting;
	else if (threshold == LOWER_CRIT_SPECIFIED)
		set_thresh_rq.lower_crit = setting;
	else if (threshold == LOWER_NON_RECOV_SPECIFIED)
		set_thresh_rq.lower_non_recov = setting;
	else
		return NULL;
	
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = SET_SENSOR_THRESHOLDS;
	req.msg.data = (uint8_t *)&set_thresh_rq;
	req.msg.data_len = sizeof(set_thresh_rq);
	
	return intf->sendrecv(intf, &req, error);
}

const struct valstr threshold_vals[] = {
	{ UPPER_NON_RECOV_SPECIFIED,	"Upper Non-Recoverable" },
	{ UPPER_CRIT_SPECIFIED,		"Upper Critical" },
	{ UPPER_NON_CRIT_SPECIFIED,	"Upper Non-Critical" },
	{ LOWER_NON_RECOV_SPECIFIED,	"Lower Non-Recoverable" },
	{ LOWER_CRIT_SPECIFIED,		"Lower Critical" },
	{ LOWER_NON_CRIT_SPECIFIED,	"Lower Non-Critical" },
	{ 0x00,				NULL },
};


static int
__ipmi_sensor_set_threshold(struct ipmi_intf * intf,
			    uint8_t num, uint8_t mask, uint8_t setting, int * error)
{
	struct ipmi_rs * rsp;

	rsp = ipmi_sensor_set_sensor_thresholds(intf, num, mask, setting, error);

	if (rsp == NULL) {
		ipmi_printf("Error setting threshold\n");
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Error setting threshold: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}

static int
ipmi_sensor_set_threshold(struct ipmi_intf * intf, vector_t *thresholds, 
                          int is_raw, int * error)
{
    char * id = NULL;
    uint8_t settingMask;
    struct sdr_record_list * sdr;
    size_t no_thresholds = vector_size(thresholds);
    unsigned int i;
    unsigned int sent = 0;

    ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

    for (i = 0; i < no_thresholds; i++) {
    	pp_ipmi_sensor_set_th_param_t *th = NULL;
        pp_ipmi_sensor_set_raw_th_param_t *raw_th = NULL;
        pp_ipmi_threshold_type_t threshold_type;
        
        if(!is_raw) {
            th = vector_get2(thresholds, i);

            if (!th) {
                continue;
            }

            threshold_type = th->threshold_type;
     	    id = th->sensor_name;
        } else {
            raw_th = vector_get2(thresholds, i);

            if (!raw_th) {
                continue;
            }

            threshold_type = raw_th->threshold_type;
        }

    	if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV) {
            settingMask = UPPER_NON_RECOV_SPECIFIED;
    	} else if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT) {
            settingMask = UPPER_CRIT_SPECIFIED;
    	} else if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT) {
            settingMask = UPPER_NON_CRIT_SPECIFIED;
    	} else if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV) {
            settingMask = LOWER_NON_RECOV_SPECIFIED;
    	} else if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT) {
            settingMask = LOWER_CRIT_SPECIFIED;
    	} else if (threshold_type == PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT) {
            settingMask = LOWER_NON_CRIT_SPECIFIED;
    	} else {
            ipmi_printf("Valid threshold not specified!\n");
            continue;
    	}

        if(!is_raw) {
            if (verbose) {
                ipmi_printf("Locating sensor record...\n");
            }
    
            /* lookup by sensor name */
            sdr = ipmi_sdr_find_sdr_byid(intf, id, NULL, error);
            if (sdr) {
                if (sdr->type != SDR_RECORD_TYPE_FULL_SENSOR) {
                    ipmi_printf("Invalid sensor type %02x\n", sdr->type);
                } else {
                    uint8_t raw;
                    int ret = 0;
                    
                    if (th->use_cim) {
                        if (verbose) {
                            ipmi_printf("Setting sensor \"%s\" %s threshold to %d (CIM)\n",
                                    sdr->record.full->id_string, val2str(settingMask, threshold_vals), th->cim_value);
                        }
                        
                        raw = sdr_convert_sensor_value_to_raw_cim(sdr->record.full, th->cim_value);
                    } else {
                        if (verbose) {
                            ipmi_printf("Setting sensor \"%s\" %s threshold to %.3f\n",
                                    sdr->record.full->id_string, val2str(settingMask, threshold_vals), th->value);
                        }
                        
                        raw = sdr_convert_sensor_value_to_raw(sdr->record.full, th->value);
                    }
                    
                    ret = __ipmi_sensor_set_threshold(intf, 
                                sdr->record.full->keys.sensor_num, settingMask,
                                raw, error);
                    if (ret == 0) {
                        sent++;
                    }
                }
            } else {
                ipmi_printf("Sensor data record not found!\n");
            }
        } else {
            /* raw */
            int ret = 0;
            
            ret = __ipmi_sensor_set_threshold(intf, raw_th->sensor_id,
                                              settingMask, raw_th->raw_value,
                                              error);
            if (ret == 0) {
                sent++;
            }
        }
    }

    ipmi_sdr_list_empty(intf);
    
    return sent == no_thresholds ? 0 : -1;
}

static int
sensor_print_sensor_raw(struct ipmi_intf * intf, int sensor_num,
                        pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_raw_reading_list_entry_t *entry = ipmi_init_sensor_get_raw_list_entry(sensor_num, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_sensor_raw_reading(intf, entry, error);
	} else {
		return -1;
	}
}

static int
sensor_print_sensor_thresholds(struct ipmi_intf * intf, int sensor_num,
                               pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_raw_thresholds_list_entry_t *entry = ipmi_init_sensor_get_thresholds_list_entry(sensor_num, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_sensor_raw_thresholds(intf, entry, error);
	} else {
		return -1;
	}
}

static int
sensor_print_sensor_full(struct ipmi_intf * intf, struct sdr_record_full_sensor * rec, pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_FULL, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_sensor_full(intf, rec, 0, entry, error);
	} else {
		return -1;
	}
}

static int
sensor_print_sensor_compact(struct ipmi_intf * intf, struct sdr_record_compact_sensor * rec, pp_ipmi_return_t *ipmi_ret, int * error)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_sensor_compact(intf, rec, entry, error);
	} else {
		return -1;
	}
}

static int
sensor_print_sensor_eventonly(struct ipmi_intf * intf, struct sdr_record_eventonly_sensor * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_EVENTONLY, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_sensor_eventonly(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sensor_print_fru_locator(struct ipmi_intf * intf, struct sdr_record_fru_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_fru_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sensor_print_mc_locator(struct ipmi_intf * intf, struct sdr_record_mc_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_mc_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int
sensor_print_generic_locator(struct ipmi_intf * intf, struct sdr_record_generic_locator * rec, pp_ipmi_return_t *ipmi_ret)
{
	pp_ipmi_sdr_list_entry_t *entry = ipmi_init_sensor_get_list_entry(PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR, ipmi_ret);
	if (entry) {
		return ipmi_sdr_print_generic_locator(intf, rec, entry);
	} else {
		return -1;
	}
}

static int ipmi_sensor_get(struct ipmi_intf * intf, int by_name, vector_t *sensors, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct sdr_record_list * sdr;
	unsigned int i, found;
	size_t no_sensors = vector_size(sensors);

	if (no_sensors < 1) {
		if (verbose) {
			ipmi_printf("No sensors to query.\n");
		}
		return 0;
	}

	if (ipmi_init_sensor_get_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	if (verbose) {
		ipmi_printf("Locating sensor record...\n");
	}

	/* lookup by sensor name */
	for (i = 0, found = 0; i < no_sensors; i++) {
		if (by_name) {
			char *sensor_name = vector_get(sensors, i);
			if (!sensor_name) {
				continue;
			}

			sdr = ipmi_sdr_find_sdr_byid(intf, sensor_name, NULL, error);

			if (!sdr) {
				ipmi_printf("Sensor data record \"%s\" not found!\n", sensor_name);
				continue;
			}
		} else {
			unsigned int sensor_id;
			unsigned int *sensor_id_addr = vector_get2(sensors, i);
			if (!sensor_id_addr) {
				continue;
			}

			sensor_id = *sensor_id_addr;
			sdr = ipmi_sdr_find_sdr_bysensorno(intf, sensor_id, -1, -1, NULL, error);

			if (!sdr) {
				ipmi_printf("Sensor data record 0x%x not found!\n", sensor_id);
				continue;
			}
		}
		switch (sdr->type) {
			case SDR_RECORD_TYPE_FULL_SENSOR:
				sensor_print_sensor_full(intf, sdr->record.full, ipmi_ret, error);
				break;
			case SDR_RECORD_TYPE_COMPACT_SENSOR:
				sensor_print_sensor_compact(intf, sdr->record.compact, ipmi_ret, error);
				break;
			case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
				sensor_print_sensor_eventonly(intf, sdr->record.eventonly, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
				sensor_print_fru_locator(intf, sdr->record.fruloc, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
				sensor_print_generic_locator(intf, sdr->record.genloc, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
				sensor_print_mc_locator(intf, sdr->record.mcloc, ipmi_ret);
				break;
		}
		found++;
	}

	ipmi_sdr_list_empty(intf);

	return found ? 0 : -1;
}

static int ipmi_sensor_get_raw(struct ipmi_intf * intf, vector_t *sensors,
                               pp_ipmi_return_t *ipmi_ret, int * error)
{
	unsigned int i, found;
	size_t no_sensors = vector_size(sensors);

	if (no_sensors < 1) {
		if (verbose) {
			ipmi_printf("No sensors to query.\n");
		}
		return 0;
	}

	if (ipmi_init_sensor_get_raw_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	if (verbose) {
		ipmi_printf("Locating sensor record...\n");
	}

	/* lookup by sensor name */
	for (i = 0, found = 0; i < no_sensors; i++) {
            unsigned int sensor_id;
            unsigned int *sensor_id_addr = vector_get2(sensors, i);
            
            if (!sensor_id_addr) {
                    continue;
            }

            sensor_id = *sensor_id_addr;

            sensor_print_sensor_raw(intf, sensor_id, ipmi_ret, error);
            found++;
	}

	ipmi_sdr_list_empty(intf);

	return found ? 0 : -1;
}

static int ipmi_sensor_get_thresholds(struct ipmi_intf * intf,
                                      vector_t *sensors,
                                      pp_ipmi_return_t *ipmi_ret, int * error)
{
	unsigned int i, found;
	size_t no_sensors = vector_size(sensors);

	if (no_sensors < 1) {
		if (verbose) {
			ipmi_printf("No sensors to query.\n");
		}
		return 0;
	}

	if (ipmi_init_sensor_get_thresh_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	if (verbose) {
		ipmi_printf("Locating sensor record...\n");
	}

	/* lookup by sensor name */
	for (i = 0, found = 0; i < no_sensors; i++) {
            unsigned int sensor_id;
            unsigned int *sensor_id_addr = vector_get2(sensors, i);
            
            if (!sensor_id_addr) {
                    continue;
            }

            sensor_id = *sensor_id_addr;

            sensor_print_sensor_thresholds(intf, sensor_id, ipmi_ret, error);
            found++;
	}

	ipmi_sdr_list_empty(intf);

	return found ? 0 : -1;
}

static int
ipmi_sensor_list(struct ipmi_intf * intf, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct sdr_get_rs * header;
	struct ipmi_sdr_iterator * itr;
	int rc = 0;

	if (ipmi_init_sensor_list(ipmi_ret)) {
		ipmi_printf("Error: could not allocate return structure.\n");
		return -1;
	}

	if (verbose) {
		ipmi_printf("Querying SDR for sensor list\n");
	}

	itr = ipmi_sdr_start(intf, error);
	if (itr == NULL) {
		ipmi_printf("Unable to open SDR for reading\n");
		return -1;
	}

	while ((header = ipmi_sdr_get_next_header(intf, itr, error)) != NULL)
	{
		int r = 0;
		uint8_t * rec;

		rec = ipmi_sdr_get_record(intf, header, itr, error);
		if (rec == NULL)
			continue;

		switch(header->type)
		{
			case SDR_RECORD_TYPE_FULL_SENSOR:
				r = sensor_print_sensor_full(intf, (struct sdr_record_full_sensor *)rec, ipmi_ret, error);
				break;
			case SDR_RECORD_TYPE_COMPACT_SENSOR:
				r = sensor_print_sensor_compact(intf, (struct sdr_record_compact_sensor *)rec, ipmi_ret, error);
				break;
			case SDR_RECORD_TYPE_EVENTONLY_SENSOR:
				r = sensor_print_sensor_eventonly(intf, (struct sdr_record_eventonly_sensor *)rec, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_FRU_DEVICE_LOCATOR:
				r = sensor_print_fru_locator(intf, (struct sdr_record_fru_locator *)rec, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_GENERIC_DEVICE_LOCATOR:
				r = sensor_print_generic_locator(intf, (struct sdr_record_generic_locator *)rec, ipmi_ret);
				break;
			case SDR_RECORD_TYPE_MC_DEVICE_LOCATOR:
				r = sensor_print_mc_locator(intf, (struct sdr_record_mc_locator *)rec, ipmi_ret);
				break;
		}
		free(rec);

		/* save any errors */
		rc = (r == 0) ? rc : r; 
	}

	ipmi_sdr_end(intf, itr);

	return rc;
}

static int ipmi_sensor_rearm(struct ipmi_intf *intf, pp_ipmi_sensor_rearm_param_t rearm_param, int *error)
{
	const char *cmd_name = "[SENSOR] Re-arm sensor events";
	struct ipmi_rs *rsp;
	struct ipmi_rq req;

	unsigned char b[6];
	int count = 6;
	b[0] = rearm_param.sensor_id;
	if (rearm_param.all_events) {
	    b[1] = 0;
	    count = 2;
	} else {
	    b[1] = 1;
	    if (rearm_param.discrete) {
		b[2] = rearm_param.assertions.discrete & 0xff;
		b[3] = (rearm_param.assertions.discrete >> 8) & 0x7f;
		b[4] = rearm_param.deassertions.discrete & 0xff;
		b[5] = (rearm_param.deassertions.discrete >> 8) & 0x7f;
	    } else {
		b[2] = (rearm_param.assertions.threshold.unc_high? 1<<7 : 0)
		     | (rearm_param.assertions.threshold.unc_low?  1<<6 : 0)
		     | (rearm_param.assertions.threshold.lnr_high? 1<<5 : 0)
		     | (rearm_param.assertions.threshold.lnr_low?  1<<4 : 0)
		     | (rearm_param.assertions.threshold.lcr_high? 1<<3 : 0)
		     | (rearm_param.assertions.threshold.lcr_low?  1<<2 : 0)
		     | (rearm_param.assertions.threshold.lnc_high? 1<<1 : 0)
		     | (rearm_param.assertions.threshold.lnc_low?  1<<0 : 0);
		b[3] = (rearm_param.assertions.threshold.unr_high? 1<<3 : 0)
		     | (rearm_param.assertions.threshold.unr_low?  1<<2 : 0)
		     | (rearm_param.assertions.threshold.ucr_high? 1<<1 : 0)
		     | (rearm_param.assertions.threshold.ucr_low?  1<<0 : 0);
		b[4] = (rearm_param.deassertions.threshold.unc_high? 1<<7 : 0)
		     | (rearm_param.deassertions.threshold.unc_low?  1<<6 : 0)
		     | (rearm_param.deassertions.threshold.lnr_high? 1<<5 : 0)
		     | (rearm_param.deassertions.threshold.lnr_low?  1<<4 : 0)
		     | (rearm_param.deassertions.threshold.lcr_high? 1<<3 : 0)
		     | (rearm_param.deassertions.threshold.lcr_low?  1<<2 : 0)
		     | (rearm_param.deassertions.threshold.lnc_high? 1<<1 : 0)
		     | (rearm_param.deassertions.threshold.lnc_low?  1<<0 : 0);
		b[5] = (rearm_param.deassertions.threshold.unr_high? 1<<3 : 0)
		     | (rearm_param.deassertions.threshold.unr_low?  1<<2 : 0)
		     | (rearm_param.deassertions.threshold.ucr_high? 1<<1 : 0)
		     | (rearm_param.deassertions.threshold.ucr_low?  1<<0 : 0);
	    }
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_SE;
	req.msg.cmd = REARM_SENSOR_EVENTS;
	req.msg.data_len = count;
	req.msg.data = b;

	rsp = intf->sendrecv(intf, &req, error);
	if (!rsp) {
		ipmi_printf("'%s' failed\n", cmd_name);
		return -1;
	}
	if (rsp->ccode) {
		ipmi_printf("'%s' failed: %s\n", cmd_name,
		       val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}

int
ipmi_sensor_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int ret = -1;
	switch ((pp_ipmi_sensor_subcommand_t) subcmd) {
		case PP_IPMI_SENSOR_SUBCMD_LIST:
			ret = ipmi_sensor_list(intf, ipmi_ret, error);
			break;
		case PP_IPMI_SENSOR_SUBCMD_THRESHOLD:
			if (!params || params->is_cmdline || !params->data.sensor_threshold_list) {
				ipmi_printf("Error: no parameters for SENSOR THRESHOLD command.\n");
			} else {
				ret = ipmi_sensor_set_threshold(intf, params->data.sensor_threshold_list, 0, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_SET_RAW_TH:
			if (!params || params->is_cmdline || !params->data.sensor_raw_threshold_list) {
				ipmi_printf("Error: no parameters for SENSOR SET RAW THRESHOLD command.\n");
			} else {
				ret = ipmi_sensor_set_threshold(intf, params->data.sensor_raw_threshold_list, 1, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_GET_BY_NAME:
			if (!params || params->is_cmdline || !params->data.sensor_get_by_name_list) {
				ipmi_printf("Error: no parameters for SENSOR GET command.\n");
			} else {
				ret = ipmi_sensor_get(intf, 1, params->data.sensor_get_by_name_list, ipmi_ret, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_GET_BY_ID:
			if (!params || params->is_cmdline || !params->data.sensor_get_by_id_list) {
				ipmi_printf("Error: no parameters for SENSOR GET command.\n");
			} else {
				ret = ipmi_sensor_get(intf, 0, params->data.sensor_get_by_id_list, ipmi_ret, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_REARM_EVENTS:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameters for SENSOR REARM command.\n");
			} else {
				ret = ipmi_sensor_rearm(intf, params->data.rearm_sensor_events, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID:
			if (!params || params->is_cmdline ||
                            !params->data.sensor_get_by_id_list) {
				ipmi_printf("Error: no parameters for SENSOR GET RAW command.\n");
			} else {
				ret = ipmi_sensor_get_raw(intf,
                                             params->data.sensor_get_by_id_list,
                                                         ipmi_ret, error);
			}
			break;
		case PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID:
			if (!params || params->is_cmdline ||
                            !params->data.sensor_get_by_id_list) {
				ipmi_printf("Error: no parameters for SENSOR GET RAW command.\n");
			} else {
				ret = ipmi_sensor_get_thresholds(intf,
                                             params->data.sensor_get_by_id_list,
                                             ipmi_ret, error);
			}
			break;
		default:
			ipmi_printf("Invalid sensor command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
	}

	return ret;
}

