/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <sys/types.h>
#include <signal.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#include <ipmitool/helper.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_user.h>
#include <ipmitool/ipmi_constants.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/bswap.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#include <ipmi_params.h>

#define IPMI_PASSWORD_DISABLE_USER  0x00
#define IPMI_PASSWORD_ENABLE_USER   0x01
#define IPMI_PASSWORD_SET_PASSWORD  0x02
#define IPMI_PASSWORD_TEST_PASSWORD 0x03


/*
 * ipmi_get_user_access
 *
 * param intf           [in]
 * param channel_number [in]
 * param user_id        [in]
 * param user_access    [out]
 *
 * return 0 on succes
 *        1 on failure
 */
static int
ipmi_get_user_access(struct ipmi_intf * intf,
		     uint8_t            channel_number,
		     uint8_t            user_id,
		     struct             user_access_rsp * user_access,
			 int * error)
					 
{
	struct ipmi_rs           * rsp;
	struct ipmi_rq             req;
	uint8_t                    msg_data[2];

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;       /* 0x06 */
	req.msg.cmd      = IPMI_GET_USER_ACCESS; /* 0x44 */
	req.msg.data     = msg_data;
	req.msg.data_len = 2;


	/* The channel number will remain constant throughout this function */
	msg_data[0] = channel_number;
	msg_data[1] = user_id;
	
	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Get User Access command failed "
			"(channel %d, user %d)\n", channel_number, user_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get User Access command failed "
			"(channel %d, user %d): %s\n", channel_number, user_id,
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(user_access, rsp->data, sizeof(struct user_access_rsp));

	return 0;
}



/*
 * ipmi_get_user_name
 *
 * param intf           [in]
 * param channel_number [in]
 * param user_id        [in]
 * param user_name      [out]
 *
 * return 0 on succes
 *        1 on failure
 */
static int
ipmi_get_user_name(struct ipmi_intf * intf,
		   uint8_t            user_id,
		   char             * user_name,
		   int              * error)
					 
{
	struct ipmi_rs           * rsp;
	struct ipmi_rq             req;
	uint8_t                    msg_data[1];

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;     /* 0x06 */
	req.msg.cmd      = IPMI_GET_USER_NAME; /* 0x45 */
	req.msg.data     = msg_data;
	req.msg.data_len = 1;

	msg_data[0] = user_id;
	
	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Get User Name command failed (user %d)\n",
			user_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get User Name command failed (user %d): %s\n",
			user_id, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	memcpy(user_name, rsp->data, 16);

	return 0;
}

#if 0
static void
dump_user_access(uint8_t                  user_id,
		 const             char * user_name,
		 struct user_access_rsp * user_access)
{
	static int printed_header = 0;

	if (! printed_header)
	{
		ipmi_printf("ID  Name             Callin  Link Auth  IPMI Msg   "
			   "Channel Priv Limit\n");
		printed_header = 1;
	}

	ipmi_printf("%-4d%-17s%-8s%-11s%-11s%-s\n",
		   user_id,
		   user_name,
		   user_access->no_callin_access? "false": "true ",
		   user_access->link_auth_access? "true ": "false",
		   user_access->ipmi_messaging_access? "true ": "false",
		   val2str(user_access->channel_privilege_limit,
				                      ipmi_privlvl_vals));
}
#endif /* 0 */

static int
ipmi_user_list(struct ipmi_intf   * intf,
               uint8_t            channel_number,
               pp_ipmi_usr_list_t * usr_list,
               int                * error)
{
        int i;
        unsigned char max = 64; /* Absolute maximum allowed by spec */
        memset(usr_list, 0, sizeof(*usr_list));

        for (i = 1; i < max; i++) {
            struct user_access_rsp user_access;

            if (ipmi_get_user_access(intf, channel_number, i, &user_access, error))
                return -1;

            usr_list->priv_limit[i] = priv_level_to_enum(user_access.channel_privilege_limit);
            usr_list->access[i] =
                (user_access.no_callin_access ? 0 : PP_IPMI_USR_ACCESS_CALL_IN)
                | (user_access.link_auth_access ? PP_IPMI_USR_ACCESS_LINK_AUTH : 0)
                | (user_access.ipmi_messaging_access ? PP_IPMI_USR_ACCESS_IPMI_MESSAGING : 0);

            if (ipmi_get_user_name(intf, i, usr_list->name[i], error))
                return -1;

            if (max < user_access.maximum_ids) max = user_access.maximum_ids;
        }
        return 0;
}

static int
ipmi_user_info(struct ipmi_intf   * intf,
	       uint8_t            channel_number,
	       uint8_t            user_id,
	       pp_ipmi_usr_info_t * usr_info,
	       int                * error)
{
        struct user_access_rsp user_access;
	if (ipmi_get_user_access(intf, channel_number, user_id, &user_access, error))
		return -1;

	usr_info->priv_limit = priv_level_to_enum(user_access.channel_privilege_limit);
	usr_info->access =
	    (user_access.no_callin_access ? 0 : PP_IPMI_USR_ACCESS_CALL_IN)
	    | (user_access.link_auth_access ? PP_IPMI_USR_ACCESS_LINK_AUTH : 0)
	    | (user_access.ipmi_messaging_access ? PP_IPMI_USR_ACCESS_IPMI_MESSAGING : 0);

	if (ipmi_get_user_name(intf, user_id, usr_info->name, error))
	    return -1;

	return 0;
}

static int
ipmi_user_summary(struct ipmi_intf      * intf,
		  uint8_t               channel_number,
		  pp_ipmi_usr_summary_t * usr_summary,
		  int                   * error)
{
	struct user_access_rsp     user_access;

	if (ipmi_get_user_access(intf,
	                         channel_number,
	                         1, &user_access, error)) {
		return -1;
	}

	usr_summary->max_users = user_access.maximum_ids;
	usr_summary->enabled_users = user_access.enabled_user_count;
	usr_summary->fixed_names = user_access.fixed_name_count;

	return 0;
}

/*
 * ipmi_user_set_username
 */
static int
ipmi_user_set_username(struct ipmi_intf * intf,
		       uint8_t            user_id,
		       const char       * name,
                       int              * error)
{
	struct ipmi_rs           * rsp;
	struct ipmi_rq             req;
	uint8_t                    msg_data[17];

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;       /* 0x06 */
	req.msg.cmd      = IPMI_SET_USER_NAME;   /* 0x45 */
	req.msg.data     = msg_data;
	req.msg.data_len = 17;

	/* The channel number will remain constant throughout this function */
	msg_data[0] = user_id;
	memset(msg_data + 1, 0, 16);
	strcpy((char *)msg_data + 1, name);

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Set User Name command failed (user %d, name %s)\n",
			user_id, name);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set User Name command failed (user %d, name %s): %s\n",
			user_id, name, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}

/*
 * ipmi_user_set_password
 *
 * This function is responsible for 4 things
 * Enabling/Disabling users
 * Setting/Testing passwords
 */
static int
ipmi_user_set_password(struct ipmi_intf * intf,
		       uint8_t            user_id,
		       uint8_t            operation,
		       const char       * password,
		       int                is_twenty_byte_password,
                       int              * error)
{
	struct ipmi_rs       * rsp;
	struct ipmi_rq         req;
	char	             * msg_data;

	int password_length = (is_twenty_byte_password? 20 : 16);

	msg_data = (char*)malloc(password_length + 2);
	if (msg_data == NULL) {
		ipmi_printf("%s: malloc error.\n", __FUNCTION__);
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;         /* 0x06 */
	req.msg.cmd      = IPMI_SET_USER_PASSWORD;  /* 0x47 */
	req.msg.data     = (uint8_t *)msg_data;
	req.msg.data_len = password_length + 2;


	/* The channel number will remain constant throughout this function */
	msg_data[0] = user_id;
	
	if (is_twenty_byte_password)
		msg_data[0] |= 0x80;
	
	msg_data[1] = operation;

	memset(msg_data + 2, 0, password_length);

	if (password != NULL)
		strncpy(msg_data + 2, password, password_length);

	rsp = intf->sendrecv(intf, &req, error);

	free(msg_data);

	if (rsp == NULL) {
		ipmi_printf("Set User Password command failed (user %d)\n",
			user_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set User Password command failed (user %d): %s\n",
			user_id, val2str(rsp->ccode, completion_code_vals));
		return rsp->ccode;
	}


	return 0;
}

/*
 * ipmi_user_test_password
 *
 * Call ipmi_user_set_password, and interpret the result
 */
static int
ipmi_user_test_password(struct ipmi_intf * intf,
			uint8_t            user_id,
			const char       * password,
			int                is_twenty_byte_password,
			int              * error)
{
	int ret;

	ret = ipmi_user_set_password(intf,
				     user_id,
				     IPMI_PASSWORD_TEST_PASSWORD,
				     password,
				     is_twenty_byte_password,
				     error);

	switch (ret) {
	case 0:
		if (verbose) {
			ipmi_printf("Test Password: Success\n");
		}
		break;
	case 0x80:
		if (verbose) {
			ipmi_printf("Test Password: password incorrect\n");
		}
		break;
	case 0x81:
		ipmi_printf("Failure: wrong password size\n");
		break;
	default:
		ipmi_printf("Unknown error\n");
	}

	return ((ret == 0) ? 0 : -1);
}

/*
 * ipmi_user_main
 *
 * Upon entry to this function argv should contain our arguments
 * specific to this subcommand
 */
int
ipmi_user_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	int retval = 0;

	t_ipmitool_commandline *cmdline = get_cmdline(params);

	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

	switch ((pp_ipmi_user_subcommand_t) subcmd) {
		case PP_IPMI_USER_SUBCMD_SUMMARY:
		{
			uint8_t channel;

			if (!params) {
				channel = 0x0E; /* Ask about the current channel */
			} else if (!params->is_cmdline) {
				channel = params->data.usr_summary.chan;
			} else {
				ipmi_printf("Error: no parameter for 'User Summary' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}

			retval = ipmi_user_summary(intf, channel, &ipmi_ret->data.usr_summary, error);
			break;
		}
		case PP_IPMI_USER_SUBCMD_INFO:
		{
                        if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'User Info' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}

			retval = ipmi_user_info(intf,
						params->data.usr_info.chan,
						params->data.usr_info.user_id,
						&ipmi_ret->data.usr_info,
						error);
			break;
		}
		case PP_IPMI_USER_SUBCMD_LIST:
		{
                        if (!params || params->is_cmdline) {
                                ipmi_printf("Error: no parameter for 'User List' command.\n");
                                ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
                                break;
                        }
                        retval = ipmi_user_list(intf,
                                                params->data.usr_list.chan,
                                                &ipmi_ret->data.usr_list,
                                                error);
                        break;
		}
		case PP_IPMI_USER_SUBCMD_SET_PASSWORD:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'User Set Password' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}
			retval = ipmi_user_set_password(intf,
							params->data.usr_set_pwd.uid,
							IPMI_PASSWORD_SET_PASSWORD,
							params->data.usr_set_pwd.pwd,
							params->data.usr_set_pwd.size20,
							error);
			if (retval > 0) {
				retval = -1;
			}
			break;
		case PP_IPMI_USER_SUBCMD_TEST_PASSWORD:
                        if (!params || params->is_cmdline) {
                                ipmi_printf("Error: no parameter for 'User Test Password' command.\n");
                                ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
                                break;
                        }
                        if (params->data.usr_test_pwd.uid == 0) {
                                ipmi_printf("Error. Invalid user ID: %d\n", params->data.usr_test_pwd.uid);
                                ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
                                break;
                        }
                        if (strlen(params->data.usr_test_pwd.pwd)
                            > (size_t)(params->data.usr_test_pwd.size20 ? 20 : 16)) {
                                ipmi_printf("Error. Password too long: %d\n", params->data.usr_test_pwd.uid);
                                ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
                                break;
                        }
                        retval = ipmi_user_test_password(intf,
                                                         params->data.usr_test_pwd.uid,
                                                         params->data.usr_test_pwd.pwd,
                                                         params->data.usr_test_pwd.size20,
                                                         error);
			break;
		case PP_IPMI_USER_SUBCMD_SET_NAME:
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'User Set Name' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}
			retval = ipmi_user_set_username(intf,
							params->data.usr_set_name.uid,
							params->data.usr_set_name.name,
							error);
			if (retval > 0) {
				retval = -1;
			}
			break;
		case PP_IPMI_USER_SUBCMD_ENABLE:
		case PP_IPMI_USER_SUBCMD_DISABLE:
		{
			uint8_t user_id;
			uint8_t operation;
			char null_password[16]; /* Not used, but required */
	
			memset(null_password, 0, sizeof(null_password));
	
			if (!params || params->is_cmdline) {
				ipmi_printf("Error: no parameter for 'User Enable/Disable' command.\n");
				ipmi_set_error(error, IPMI_ERROR_NO_PARAMETER);
				break;
			}

			if (subcmd == PP_IPMI_USER_SUBCMD_DISABLE) {
			    operation = IPMI_PASSWORD_DISABLE_USER;
			    user_id = params->data.usr_disable.uid;
			} else {
			    operation = IPMI_PASSWORD_ENABLE_USER;
			    user_id = params->data.usr_enable.uid;
			}
	
			retval = ipmi_user_set_password(intf,
							user_id,
							operation,
							null_password,
							0, /* This field is ignored */
							error);
			if (retval > 0) {
				retval = -1;
			}
			break;
		}
		default:
			ipmi_printf("Invalid USER command: %d\n", subcmd);
			ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
			retval = -1;
	}

	cleanup_cmdline(cmdline);

	return retval;
}

