/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <stdio.h>

#ifndef _WIN32
#include <inttypes.h>
#include <unistd.h>
#endif

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <signal.h>
#include <fcntl.h>
#include <time.h>

#ifdef WIN32

#include <winsock2.h>
#include <windows.h>

#else // WIN32

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#define closesocket(sock)	close(sock)

#endif // WIN32

#include <ipmitool/helper.h>
#include <ipmitool/bswap.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_oem.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_constants.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>

#include <config.h>

#include "lan.h"
#include "rmcp.h"
#include "asf.h"
#include "auth.h"

struct ipmi_rq_entry * ipmi_req_entries;
static struct ipmi_rq_entry * ipmi_req_entries_tail;

static int curr_seq = 0;

static int ipmi_lan_send_packet(struct ipmi_intf * intf, uint8_t * data, int data_len, int * error);
static struct ipmi_rs * ipmi_lan_recv_packet(struct ipmi_intf * intf, int * error);
static struct ipmi_rs * ipmi_lan_poll_recv(struct ipmi_intf * intf, int * error);
static int ipmi_lan_setup(struct ipmi_intf * intf, int * error);
static int ipmi_lan_keepalive(struct ipmi_intf * intf, int * error);
void ipmi_lan_abort(struct ipmi_intf * intf, int * error);

struct ipmi_intf ipmi_lan_intf = {
#ifndef WIN32
	.name        = "lan",
	.desc        = "IPMI v1.5 LAN Interface",

	.fd          = 0,
	.opened      = 0,
	.abort       = 0,
	.noanswer    = 0,
	.cancel      = 0,

	.session     = NULL,
	.oem         = NULL,
	.my_addr     = 0,
	.bmc_addr    = IPMI_BMC_SLAVE_ADDR,
	.target_addr = IPMI_BMC_SLAVE_ADDR,

	.setup       = ipmi_lan_setup,
	.open        = ipmi_lan_open,
	.close       = ipmi_lan_close,
	.do_abort    = ipmi_lan_abort,
	.sendrecv    = ipmi_lan_send_cmd,
	.sendrsp     = ipmi_lan_send_rsp,
	.recv_sol    = NULL,
	.send_sol    = NULL,
	.keepalive   = ipmi_lan_keepalive,
#else
	/* name:              */  "lan",
	/* desc:              */  "IPMI v1.5 LAN Interface",

	/* fd:                */  0,
	/* opened:            */  0,
	/* abort:             */  0,
	/* noanswer:          */  0,
	/* cancel:            */  0,
	/* intf_ctx:          */  NULL,

	/* cmd_wfd:           */  0,
	/* rsp_rfd:           */  0,
	/* loopi_pipe_prefix: */  0,

	/* session:           */  NULL,
	/* oem:               */  NULL,
	/* my_addr:           */  0,
	/* bmc_addr:          */  IPMI_BMC_SLAVE_ADDR,
	/* target_addr:       */  IPMI_BMC_SLAVE_ADDR,

	/* setup:             */  ipmi_lan_setup,
	/* open:              */  ipmi_lan_open,
	/* close:             */  ipmi_lan_close,
	/* do_abort:          */  ipmi_lan_abort,
	/* sendrecv:          */  ipmi_lan_send_cmd,
	/* sendrsp:           */  ipmi_lan_send_rsp,
	/* recv_sol:          */  NULL,
	/* send_sol:          */  NULL,
	/* keepalive:         */  ipmi_lan_keepalive,
#endif
};

static struct ipmi_rq_entry *
ipmi_req_add_entry(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	struct ipmi_rq_entry * e;

	e = malloc(sizeof(struct ipmi_rq_entry));
	if (e == NULL) {
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		return NULL;
	}

	memset(e, 0, sizeof(struct ipmi_rq_entry));
	memcpy(&e->req, req, sizeof(struct ipmi_rq));

	e->intf = intf;

	if (ipmi_req_entries == NULL)
		ipmi_req_entries = e;
	else
		ipmi_req_entries_tail->next = e;

	ipmi_req_entries_tail = e;
	if (verbose > 3)
		ipmi_printf("added list entry seq=0x%02x cmd=0x%02x",
			e->rq_seq, e->req.msg.cmd);
  	return e;
}

static struct ipmi_rq_entry *
ipmi_req_lookup_entry(uint8_t seq, uint8_t cmd)
{
	struct ipmi_rq_entry * e = ipmi_req_entries;
	while (e && (e->rq_seq != seq || e->req.msg.cmd != cmd)) {
		if (e == e->next)
			return NULL;
		e = e->next;
	}
	return e;
}

static void
ipmi_req_remove_entry(uint8_t seq, uint8_t cmd)
{
	struct ipmi_rq_entry * p, * e;

	e = p = ipmi_req_entries;

	while (e && (e->rq_seq != seq || e->req.msg.cmd != cmd)) {
		p = e;
		e = e->next;
	}
	if (e) {
		if (verbose > 3)
			ipmi_printf("removed list entry seq=0x%02x cmd=0x%02x\n", seq, cmd);
		p->next = (p->next == e->next) ? NULL : e->next;
		if (ipmi_req_entries == e) {
			if (ipmi_req_entries != p)
				ipmi_req_entries = p;
			else
				ipmi_req_entries = NULL;
		}
		if (ipmi_req_entries_tail == e) {
			if (ipmi_req_entries_tail != p)
				ipmi_req_entries_tail = p;
			else
				ipmi_req_entries_tail = NULL;
		}
		if (e->msg_data)
			free(e->msg_data);
		free(e);
	}
}

static void
ipmi_req_clear_entries(void)
{
	struct ipmi_rq_entry * p, * e;

	e = ipmi_req_entries;
	while (e) {
		if (verbose > 3)
			ipmi_printf("cleared list entry seq=0x%02x cmd=0x%02x\n",
			       e->rq_seq, e->req.msg.cmd);
		p = e->next;
		free(e);
		e = p;
	}

	ipmi_req_entries = NULL;
}

static int
get_random(void *data, int len)
{
#ifdef WIN32
	int i;
	for (i = 0; i < len; i++) {
		((char *)data)[i] = rand() & 0xff;
	}
	return len;

#else // WIN32

	int fd = open("/dev/urandom", O_RDONLY);
	int rv;

	if (fd < 0 || len < 0)
		return errno;

	rv = read(fd, data, len);

	close(fd);
	return rv;
#endif // WIN32
}

int ipmi_lan_send_packet(struct ipmi_intf * intf, uint8_t * data, int data_len, int * error UNUSED)
{
	if (verbose > 2)
		printbuf(data, data_len, "send_packet");

	return send(intf->fd, data, data_len, 0);
}

static int ipmi_lan_data_available(struct ipmi_intf * intf, int * error UNUSED)
{
	fd_set myFdSet;
	struct timeval myTime;

	myTime.tv_sec = intf->session->timeout;
	myTime.tv_usec = 0;

	FD_ZERO(&myFdSet);
#ifdef WIN32
	FD_SET((unsigned int)intf->fd, &myFdSet);
#else
	FD_SET(intf->fd, &myFdSet);
#endif

	if(select(intf->fd + 1, &myFdSet, NULL, NULL, &myTime) != 1) {
		return 0;
	}

	if(FD_ISSET(intf->fd, &myFdSet)) {
		return 1;
	}
	return 0;
}

struct ipmi_rs * ipmi_lan_recv_packet(struct ipmi_intf * intf, int * error)
{
	static struct ipmi_rs rsp;
	int rc;

	if (!ipmi_lan_data_available(intf, error)) {
		return NULL;
	}
	rc = recv(intf->fd, &rsp.data[0], IPMI_BUF_SIZE, 0);

	/* the first read may return ECONNREFUSED because the rmcp ping
	 * packet--sent to UDP port 623--will be processed by both the
	 * BMC and the OS.
	 *
	 * The problem with this is that the ECONNREFUSED takes
	 * priority over any other received datagram; that means that
	 * the Connection Refused shows up _before_ the response packet,
	 * regardless of the order they were sent out.  (unless the
	 * response is read before the connection refused is returned)
	 */
	if (rc < 0) {
		if (!ipmi_lan_data_available(intf, error)) {
			return NULL;
		}
		rc = recv(intf->fd, &rsp.data[0], IPMI_BUF_SIZE, 0);
		if (rc < 0) {
			ipmi_set_error(error, IPMI_ERROR_NETWORK);
			ipmi_printf("recv failed");
			return NULL;
		}
	}
	rsp.data[rc] = '\0';
	rsp.data_len = rc;

	return &rsp;
}

/*
 * parse response RMCP "pong" packet
 *
 * return -1 if ping response not received
 * returns 0 if IPMI is NOT supported
 * returns 1 if IPMI is supported
 *
 * udp.source	= 0x026f	// RMCP_UDP_PORT
 * udp.dest	= ?		// udp.source from rmcp-ping
 * udp.len	= ?
 * udp.check	= ?
 * rmcp.ver	= 0x06		// RMCP Version 1.0
 * rmcp.__res	= 0x00		// RESERVED
 * rmcp.seq	= 0xff		// no RMCP ACK
 * rmcp.class	= 0x06		// RMCP_CLASS_ASF
 * asf.iana	= 0x000011be	// ASF_RMCP_IANA
 * asf.type	= 0x40		// ASF_TYPE_PONG
 * asf.tag	= ?		// asf.tag from rmcp-ping
 * asf.__res	= 0x00		// RESERVED
 * asf.len	= 0x10		// 16 bytes
 * asf.data[3:0]= 0x000011be	// IANA# = RMCP_ASF_IANA if no OEM
 * asf.data[7:4]= 0x00000000	// OEM-defined (not for IPMI)
 * asf.data[8]	= 0x81		// supported entities
 * 				// [7]=IPMI [6:4]=RES [3:0]=ASF_1.0
 * asf.data[9]	= 0x00		// supported interactions (reserved)
 * asf.data[f:a]= 0x000000000000
 */
static int
ipmi_handle_pong(struct ipmi_intf * intf UNUSED, struct ipmi_rs * rsp)
{
	struct rmcp_pong * pong;

	if (rsp == NULL)
		return -1;

	pong = (struct rmcp_pong *)rsp->data;

	if (verbose > 1)
		ipmi_printf("Received IPMI/RMCP response packet: \n"
		       "  IPMI%s Supported\n"
		       "  ASF Version %s\n"
		       "  RMCP Version %s\n"
		       "  RMCP Sequence %d\n"
		       "  IANA Enterprise %d\n\n",
		       (pong->sup_entities & 0x80) ? "" : " NOT",
		       (pong->sup_entities & 0x01) ? "1.0" : "unknown",
		       (pong->rmcp.ver == 6) ? "1.0" : "unknown",
		       pong->rmcp.seq,
		       ntohl(pong->iana));

	return (pong->sup_entities & 0x80) ? 1 : 0;
}

/* build and send RMCP presence ping packet
 *
 * RMCP ping
 *
 * udp.source	= ?
 * udp.dest	= 0x026f	// RMCP_UDP_PORT
 * udp.len	= ?
 * udp.check	= ?
 * rmcp.ver	= 0x06		// RMCP Version 1.0
 * rmcp.__res	= 0x00		// RESERVED
 * rmcp.seq	= 0xff		// no RMCP ACK
 * rmcp.class	= 0x06		// RMCP_CLASS_ASF
 * asf.iana	= 0x000011be	// ASF_RMCP_IANA
 * asf.type	= 0x80		// ASF_TYPE_PING
 * asf.tag	= ?		// ASF sequence number
 * asf.__res	= 0x00		// RESERVED
 * asf.len	= 0x00
 *
 */
int
ipmi_lan_ping(struct ipmi_intf * intf, int * error)
{
	struct asf_hdr asf_ping;
	struct rmcp_hdr rmcp_ping;
	uint8_t * data;
	int len = sizeof(rmcp_ping) + sizeof(asf_ping);
	int rv;

	memset(&asf_ping, 0, sizeof(asf_ping));
	asf_ping.iana = htonl(ASF_RMCP_IANA);
	asf_ping.type = ASF_TYPE_PING;

	memset(&rmcp_ping, 0, sizeof(rmcp_ping));
	rmcp_ping.ver = RMCP_VERSION_1;
	rmcp_ping.class = RMCP_CLASS_ASF;
	rmcp_ping.seq = 0xff;

	data = malloc(len);
	if (data == NULL) {
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		return -1;
	}
	memset(data, 0, len);
	memcpy(data, &rmcp_ping, sizeof(rmcp_ping));
	memcpy(data+sizeof(rmcp_ping), &asf_ping, sizeof(asf_ping));

	if (verbose > 1)
		ipmi_printf("Sending IPMI/RMCP presence ping packet\n");

	rv = ipmi_lan_send_packet(intf, data, len, error);

	free(data);

	if (rv < 0) {
		ipmi_set_error(error, IPMI_ERROR_NETWORK);
		ipmi_printf("Unable to send IPMI presence ping packet\n");
		return -1;
	}

	if (ipmi_lan_poll_recv(intf, error) == 0)
		return 0;

	return 1;
}

/*
 * The "thump" functions are used to send an extra packet following each
 * request message.  This may kick-start some BMCs that get confused with
 * bad passwords or operate poorly under heavy network load.
 */
static void ipmi_lan_thump_first(struct ipmi_intf * intf, int *error)
{
	/* is this random data? */
	uint8_t data[16] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			     0x07, 0x20, 0x18, 0xc8, 0xc2, 0x01, 0x01, 0x3c };
	ipmi_lan_send_packet(intf, data, 16, error);
}

static void ipmi_lan_thump(struct ipmi_intf * intf, int *error)
{
	uint8_t data[10] = "thump";
	ipmi_lan_send_packet(intf, data, 10, error);
}

static struct ipmi_rs *
ipmi_lan_poll_recv(struct ipmi_intf * intf, int * error)
{
	struct rmcp_hdr rmcp_rsp;
	struct ipmi_rs * rsp;
	struct ipmi_rq_entry * entry;
	int x=0, rv;

	if (!intf->session) {
		return NULL;
	}

	rsp = ipmi_lan_recv_packet(intf, error);

	while (rsp != NULL) {

		/* parse response headers */
		memcpy(&rmcp_rsp, rsp->data, 4);

		switch (rmcp_rsp.class) {
		case RMCP_CLASS_ASF:
			/* ping response packet */
			rv = ipmi_handle_pong(intf, rsp);
			return (rv <= 0) ? NULL : rsp;
		case RMCP_CLASS_IPMI:
			/* handled by rest of function */
			break;
		default:
			if (verbose > 1)
				ipmi_printf("Invalid RMCP class: %x\n", rmcp_rsp.class);
			rsp = ipmi_lan_recv_packet(intf, error);
			continue;
		}

		x = 4;
		rsp->session.authtype = rsp->data[x++];
		memcpy(&rsp->session.seq, rsp->data+x, 4);
		x += 4;
		memcpy(&rsp->session.id, rsp->data+x, 4);
		x += 4;

		if (intf->session->active && (rsp->session.authtype || intf->session->authtype))
			x += 16;

		rsp->session.msglen = rsp->data[x++];
		rsp->payload.ipmi_response.rq_addr = rsp->data[x++];
		rsp->payload.ipmi_response.netfn   = rsp->data[x] >> 2;
		rsp->payload.ipmi_response.rq_lun  = rsp->data[x++] & 0x3;
		x++;		/* checksum */
		rsp->payload.ipmi_response.rs_addr = rsp->data[x++];
		rsp->payload.ipmi_response.rq_seq  = rsp->data[x] >> 2;
		rsp->payload.ipmi_response.rs_lun  = rsp->data[x++] & 0x3;
		rsp->payload.ipmi_response.cmd     = rsp->data[x++]; 
		rsp->ccode          = rsp->data[x++];

		if (verbose > 2) {
			printbuf(rsp->data, rsp->data_len, "ipmi message header");
			ipmi_printf("<< IPMI Response Session Header\n");
			ipmi_printf("<<   Authtype   : %s\n",
			       val2str(rsp->session.authtype, ipmi_authtype_session_vals));
			ipmi_printf("<<   Sequence   : 0x%08lx\n", (long)rsp->session.seq);
			ipmi_printf("<<   Session ID : 0x%08lx\n", (long)rsp->session.id);
			
			ipmi_printf("<< IPMI Response Message Header\n");
			ipmi_printf("<<   Rq Addr    : %02x\n", rsp->payload.ipmi_response.rq_addr);
			ipmi_printf("<<   NetFn      : %02x\n", rsp->payload.ipmi_response.netfn);
			ipmi_printf("<<   Rq LUN     : %01x\n", rsp->payload.ipmi_response.rq_lun);
			ipmi_printf("<<   Rs Addr    : %02x\n", rsp->payload.ipmi_response.rs_addr);
			ipmi_printf("<<   Rq Seq     : %02x\n", rsp->payload.ipmi_response.rq_seq);
			ipmi_printf("<<   Rs Lun     : %01x\n", rsp->payload.ipmi_response.rs_lun);
			ipmi_printf("<<   Command    : %02x\n", rsp->payload.ipmi_response.cmd);
			ipmi_printf("<<   Compl Code : 0x%02x\n", rsp->ccode);
		}

		/* now see if we have outstanding entry in request list */
		entry = ipmi_req_lookup_entry(rsp->payload.ipmi_response.rq_seq,
					      rsp->payload.ipmi_response.cmd);
		if (entry) {
			if (verbose > 2)
				ipmi_printf("IPMI Request Match found\n");
			if (intf->target_addr != intf->bmc_addr) {
				if (verbose && rsp->data_len)
					ipmi_printf("Bridged cmd %02x resp: %s",
						rsp->payload.ipmi_response.cmd,
						buf2str(&rsp->data[x],rsp->data_len));
				/* bridged command: lose extra header */
				if (rsp->payload.ipmi_response.cmd == 0x34) {
					entry->req.msg.cmd = entry->req.msg.target_cmd;
					rsp = ipmi_lan_recv_packet(intf, error);
					continue;
				} else {
					//x += sizeof(rsp->payload.ipmi_response);
					if (verbose && rsp->data[x-1] != 0)
						ipmi_printf("WARNING: Bridged "
							"cmd ccode = 0x%02x",
						       rsp->data[x-1]);
				}
			}
			ipmi_req_remove_entry(rsp->payload.ipmi_response.rq_seq,
					      rsp->payload.ipmi_response.cmd);
		} else {
			if (verbose)
				ipmi_printf("WARNING: IPMI Request Match NOT FOUND!\n");
			rsp = ipmi_lan_recv_packet(intf, error);
			continue;
		}			

		break;
	}

	/* shift response data to start of array */
	if (rsp && rsp->data_len > x) {
		rsp->data_len -= x + 1;
		memmove(rsp->data, rsp->data + x, rsp->data_len);
		memset(rsp->data + rsp->data_len, 0, IPMI_BUF_SIZE - rsp->data_len);
	}

	return rsp;
}

/*
 * IPMI LAN Request Message Format
 * +--------------------+
 * |  rmcp.ver          | 4 bytes
 * |  rmcp.__reserved   |
 * |  rmcp.seq          |
 * |  rmcp.class        |
 * +--------------------+
 * |  session.authtype | 9 bytes
 * |  session.seq   |
 * |  session.id    |
 * +--------------------+
 * | [session.authcode] | 16 bytes (AUTHTYPE != none)
 * +--------------------+
 * |  message length    | 1 byte
 * +--------------------+
 * |  message.rs_addr   | 6 bytes
 * |  message.netfn_lun |
 * |  message.checksum  |
 * |  message.rq_addr   |
 * |  message.rq_seq    |
 * |  message.cmd       |
 * +--------------------+
 * | [request data]     | data_len bytes
 * +--------------------+
 * |  checksum          | 1 byte
 * +--------------------+
 */
static struct ipmi_rq_entry *
ipmi_lan_build_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	struct rmcp_hdr rmcp;
	uint8_t * msg, * temp;
	int cs, mp, tmp;
	int ap = 0;
	int len = 0;
	int cs2 = 0;
	struct ipmi_rq_entry * entry;
	struct ipmi_session * s = intf->session;
	int bridge_request = 0;
	int my_addr = intf->my_addr ? intf->my_addr : IPMI_REMOTE_SWID;

	if (!intf->session) {
		return NULL;
	}

	memset(&rmcp, 0, sizeof(rmcp));
	rmcp.ver = RMCP_VERSION_1;
	rmcp.class = RMCP_CLASS_IPMI;
	rmcp.seq = 0xff;

	if (curr_seq >= 64)
		curr_seq = 0;

	entry = ipmi_req_add_entry(intf, req, error);
	if (entry == NULL)
		return NULL;
	
	len = req->msg.data_len + 29;
	if (s->active && s->authtype)
		len += 16;
	msg = malloc(len);
	if (msg == NULL) {
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		return NULL;
	}
	memset(msg, 0, len);

	/* rmcp header */
	memcpy(msg, &rmcp, sizeof(rmcp));
	len = sizeof(rmcp);

	/* ipmi session header */
	msg[len++] = s->active ? s->authtype : 0;

	msg[len++] = (uint8_t)s->in_seq & 0xff;
	msg[len++] = (uint8_t)(s->in_seq >> 8) & 0xff;
	msg[len++] = (uint8_t)(s->in_seq >> 16) & 0xff;
	msg[len++] = (uint8_t)(s->in_seq >> 24) & 0xff;
	memcpy(msg+len, &s->session_id, 4);
	len += 4;

	/* ipmi session authcode */
	if (s->active && s->authtype) {
		ap = len;
		memcpy(msg+len, s->authcode, 16);
		len += 16;
	}

	/* message length */
	if (intf->target_addr == intf->bmc_addr) {
		msg[len++] = req->msg.data_len + 7;
		cs = mp = len;
	}
	else {
		/* bridged request: encapsulate w/in Send Message */
		bridge_request = 1;
		msg[len++] = req->msg.data_len + 15;
		cs = mp = len;
		msg[len++] = intf->bmc_addr;
		msg[len++] = IPMI_NETFN_APP << 2;
		tmp = len - cs;
		msg[len++] = ipmi_csum(msg+cs, tmp);
		cs2 = len;
		msg[len++] = my_addr;
		msg[len++] = curr_seq << 2;
		msg[len++] = 0x34;			/* Send Message rqst */
		entry->req.msg.target_cmd = entry->req.msg.cmd;	/* Save target command */
		entry->req.msg.cmd = 0x34;		/* (fixup request entry) */
		msg[len++] = 0x40;			/* Track request, Channel=IPMB */
		cs = len;
	}

	/* ipmi message header */
	msg[len++] = intf->target_addr;
	msg[len++] = req->msg.netfn << 2;
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);
	cs = len;

	if (!bridge_request)
		msg[len++] = IPMI_REMOTE_SWID;
	else  /* Bridged message */
		msg[len++] = my_addr;

	entry->rq_seq = curr_seq++;
	msg[len++] = entry->rq_seq << 2;
	msg[len++] = req->msg.cmd;

	if (verbose > 2) {
		ipmi_printf(">> IPMI Request Session Header\n");
		ipmi_printf(">>   Authtype   : %s\n",
		       val2str(s->authtype, ipmi_authtype_session_vals));
		ipmi_printf(">>   Sequence   : 0x%08lx\n", (long)s->in_seq);
		ipmi_printf(">>   Session ID : 0x%08lx\n", (long)s->session_id);
		
		ipmi_printf(">> IPMI Request Message Header\n");
		ipmi_printf(">>   Rs Addr    : %02x\n", intf->target_addr);
		ipmi_printf(">>   NetFn      : %02x\n", req->msg.netfn);
		ipmi_printf(">>   Rs LUN     : %01x\n", 0);
		ipmi_printf(">>   Rq Addr    : %02x\n", my_addr);
		ipmi_printf(">>   Rq Seq     : %02x\n", entry->rq_seq);
		ipmi_printf(">>   Rq Lun     : %01x\n", 0);
		ipmi_printf(">>   Command    : %02x\n", req->msg.cmd);
	}

	/* message data */
	if (req->msg.data_len) {
 		memcpy(msg+len, req->msg.data, req->msg.data_len);
		len += req->msg.data_len;
	}

	/* second checksum */
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);

	/* bridged request: 2nd checksum */
	if (bridge_request) {
		tmp = len - cs2;
		msg[len++] = ipmi_csum(msg+cs2, tmp);
	}

	if (s->active) {
		/*
		 * s->authcode is already copied to msg+ap but some
		 * authtypes require portions of the ipmi message to
		 * create the authcode so they must be done last.
		 */
		switch (s->authtype) {
		case IPMI_SESSION_AUTHTYPE_MD5:
			temp = ipmi_auth_md5(s, msg+mp, msg[mp-1]);
			memcpy(msg+ap, temp, 16);
			break;
		case IPMI_SESSION_AUTHTYPE_MD2:
			temp = ipmi_auth_md2(s, msg+mp, msg[mp-1]);
			memcpy(msg+ap, temp, 16);
			break;
		}
	}

	if (s->in_seq) {
		s->in_seq++;
		if (s->in_seq == 0)
			s->in_seq++;
	}

	entry->msg_len = len;
	entry->msg_data = msg;

	return entry;
}

struct ipmi_rs *
ipmi_lan_send_cmd(struct ipmi_intf * intf, struct ipmi_rq * req, int * error)
{
	struct ipmi_rq_entry * entry;
	struct ipmi_rs * rsp = NULL;
	int try = 0;

	if (intf->opened == 0 && intf->open != NULL) {
		if (intf->open(intf, error) < 0)
			return NULL;
	}

	entry = ipmi_lan_build_cmd(intf, req, error);
	if (entry == NULL) {
		ipmi_set_error(error, IPMI_ERROR_INTERNAL);
		ipmi_printf("Aborting send command, unable to build.\n");
		return NULL;
	}

	for (;;) {
		if (ipmi_lan_send_packet(intf, entry->msg_data, entry->msg_len, error) < 0) {
			if (++try >= IPMI_LAN_RETRY) {
				ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
				ipmi_printf("No response (send packet).\n");
				break;
			}
			usleep(5000);
			continue;
		}

		/* if we are set to noanswer we do not expect response */
		if (intf->noanswer)
			break;

		if (ipmi_oem_active(intf, "intelwv2"))
			ipmi_lan_thump(intf, error);

		usleep(100);

		rsp = ipmi_lan_poll_recv(intf, error);
		if (rsp)
			break;

		if (intf->cancel) {
			break;
		}

		usleep(5000);
		if (++try >= intf->session->retry) {
			ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
			ipmi_printf("No response (recv packet).\n");
			break;
		}
	}

	if (rsp && rsp->ccode) {
		ipmi_set_error(error, rsp->ccode);
	}
	return rsp;
}

static uint8_t * ipmi_lan_build_rsp(struct ipmi_intf * intf, struct ipmi_rs * rsp, int * llen, int * error)
{
	struct rmcp_hdr rmcp;
	struct ipmi_session * s = intf->session;
	int cs, mp, ap = 0, tmp;
	int len;
	uint8_t * msg;

	if (!intf->session) {
		return NULL;
	}

	memset(&rmcp, 0, sizeof(rmcp));
	rmcp.ver = RMCP_VERSION_1;
	rmcp.class = RMCP_CLASS_IPMI;
	rmcp.seq = 0xff;

	len = rsp->data_len + 22;
	if (s->active)
		len += 16;

	msg = malloc(len);
	if (msg == NULL) {
		ipmi_printf("%s: malloc failure\n", __FUNCTION__);
		ipmi_set_error(error, IPMI_ERROR_NOT_ENOUGH_MEMORY);
		return NULL;
	}
	memset(msg, 0, len);

	/* rmcp header */
	memcpy(msg, &rmcp, 4);
	len = sizeof(rmcp);

	/* ipmi session header */
	msg[len++] = s->active ? s->authtype : 0;

	if (s->in_seq) {
		s->in_seq++;
		if (s->in_seq == 0)
			s->in_seq++;
	}
	memcpy(msg+len, &s->in_seq, 4);
	len += 4;
	memcpy(msg+len, &s->session_id, 4);
	len += 4;

	/* session authcode, if session active and authtype is not none */
	if (s->active && s->authtype) {
		ap = len;
		memcpy(msg+len, s->authcode, 16);
		len += 16;
	}

	/* message length */
	msg[len++] = rsp->data_len + 8;

	/* message header */
	cs = mp = len;
	msg[len++] = intf->my_addr ? intf->my_addr : IPMI_REMOTE_SWID;
	msg[len++] = rsp->msg.netfn << 2;
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);
	cs = len;
	msg[len++] = intf->bmc_addr;
	msg[len++] = (rsp->msg.seq << 2) | (rsp->msg.lun & 3);
	msg[len++] = rsp->msg.cmd;

	/* completion code */
	msg[len++] = rsp->ccode;

	/* message data */
	if (rsp->data_len) {
		memcpy(msg+len, rsp->data, rsp->data_len);
		len += rsp->data_len;
	}

	/* second checksum */
	tmp = len - cs;
	msg[len++] = ipmi_csum(msg+cs, tmp);

	if (s->active) {
		uint8_t * d;
		switch (s->authtype) {
		case IPMI_SESSION_AUTHTYPE_MD5:
			d = ipmi_auth_md5(s, msg+mp, msg[mp-1]);
			memcpy(msg+ap, d, 16);
			break;
		case IPMI_SESSION_AUTHTYPE_MD2:
			d = ipmi_auth_md2(s, msg+mp, msg[mp-1]);
			memcpy(msg+ap, d, 16);
			break;
		}			
	}

	*llen = len;
	return msg;
}

int ipmi_lan_send_rsp(struct ipmi_intf * intf, struct ipmi_rs * rsp, int * error)
{
	uint8_t * msg;
	int len, rv;

	if (!intf->session) {
		return -1;
	}

	msg = ipmi_lan_build_rsp(intf, rsp, &len, error);
	if (len <= 0 || msg == NULL) {
		ipmi_set_error(error, IPMI_ERROR_INTERNAL);
		ipmi_printf("ipmi_lan_send_rsp: invalid response packet\n");
		if (msg != NULL)
			free(msg);
		return -1;
	}

	rv = sendto(intf->fd, msg, len, 0,
		    (struct sockaddr *)&intf->session->addr,
		    intf->session->addrlen);
	if (rv < 0) {
		ipmi_set_error(error, IPMI_ERROR_NETWORK);
		ipmi_printf("ipmi_lan_send_rsp: packet send failed\n");
		if (msg != NULL)
			free(msg);
		return -1;
	}

	if (msg != NULL)
		free(msg);
	return 0;
}

/* send a get device id command to keep session active */
static int
ipmi_lan_keepalive(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 1;

	if (!intf->opened)
		return 0;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL)
		return -1;
	if (rsp->ccode > 0)
		return -1;

	return 0;
}

/*
 * IPMI Get Channel Authentication Capabilities Command
 */
static int
ipmi_get_auth_capabilities_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipmi_session * s = intf->session;
	uint8_t msg_data[2];

	if (!intf->session) {
		return -1;
	}

	msg_data[0] = IPMI_LAN_CHANNEL_E;
	msg_data[1] = s->privlvl;

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_APP;
	req.msg.cmd      = 0x38;
	req.msg.data     = msg_data;
	req.msg.data_len = 2;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_set_error(error, IPMI_ERROR_NO_CONNECTION);
		if (verbose)
			ipmi_printf("error in Get Auth Capabilities Command\n");
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "get_auth_capabilities");

	if (rsp->ccode > 0) {
		ipmi_set_error(error, IPMI_ERROR_NO_CONNECTION);
		ipmi_printf("Get Auth Capabilities error: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (verbose > 1) {
		ipmi_printf("Channel %02x Authentication Capabilities:\n",
		       rsp->data[0]);
		ipmi_printf("  Privilege Level : %s\n",
		       val2str(req.msg.data[1], ipmi_privlvl_vals));
		ipmi_printf("  Auth Types      : ");
		if (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_NONE)
			ipmi_printf("NONE ");
		if (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_MD2)
			ipmi_printf("MD2 ");
		if (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_MD5)
			ipmi_printf("MD5 ");
		if (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_PASSWORD)
			ipmi_printf("PASSWORD ");
		if (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_OEM)
			ipmi_printf("OEM ");
		ipmi_printf("\n");
		ipmi_printf("  Per-msg auth    : %sabled\n",
		       (rsp->data[2] & 1<<4) ? "dis" : "en");
		ipmi_printf("  User level auth : %sabled\n",
		       (rsp->data[2] & 1<<3) ? "dis" : "en");
		ipmi_printf("  Non-null users  : %sabled\n",
		       (rsp->data[2] & 1<<2) ? "en" : "dis");
		ipmi_printf("  Null users      : %sabled\n",
		       (rsp->data[2] & 1<<1) ? "en" : "dis");
		ipmi_printf("  Anonymous login : %sabled\n",
		       (rsp->data[2] & 1<<0) ? "en" : "dis");
		ipmi_printf("\n");
	}

	s->authstatus = rsp->data[2];

	if (s->password &&
	    (s->authtype_set == 0 ||
	     s->authtype_set == IPMI_SESSION_AUTHTYPE_MD5) &&
	    (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_MD5))
	{
		s->authtype = IPMI_SESSION_AUTHTYPE_MD5;
	}
	else if (s->password &&
		 (s->authtype_set == 0 ||
		  s->authtype_set == IPMI_SESSION_AUTHTYPE_MD2) &&
		 (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_MD2))
	{
		s->authtype = IPMI_SESSION_AUTHTYPE_MD2;
	}
	else if (s->password &&
		 (s->authtype_set == 0 ||
		  s->authtype_set == IPMI_SESSION_AUTHTYPE_PASSWORD) &&
		 (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_PASSWORD))
	{
		s->authtype = IPMI_SESSION_AUTHTYPE_PASSWORD;
	}
	else if (s->password &&
		 (s->authtype_set == 0 ||
		  s->authtype_set == IPMI_SESSION_AUTHTYPE_OEM) &&
		 (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_OEM))
	{
		s->authtype = IPMI_SESSION_AUTHTYPE_OEM;
	}
	else if ((s->authtype_set == 0 ||
		  s->authtype_set == IPMI_SESSION_AUTHTYPE_NONE) &&
		 (rsp->data[1] & 1<<IPMI_SESSION_AUTHTYPE_NONE))
	{
		s->authtype = IPMI_SESSION_AUTHTYPE_NONE;
	}
	else {
		if (!(rsp->data[1] & 1<<s->authtype_set))
			ipmi_printf("Authentication type %s not supported!\n",
			       val2str(s->authtype_set, ipmi_authtype_session_vals));
		else
			ipmi_printf("No supported authtypes found!\n");
		ipmi_set_error(error, IPMI_ERROR_INTERNAL);
		return -1;
	}

	if (verbose > 1)
		ipmi_printf("Proceeding with AuthType %s\n",
		       val2str(s->authtype, ipmi_authtype_session_vals));

	return 0;
}

/*
 * IPMI Get Session Challenge Command
 * returns a temporary session ID and 16 byte challenge string
 */
static int
ipmi_get_session_challenge_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipmi_session * s = intf->session;
	uint8_t msg_data[17];

	if (!intf->session) {
		return -1;
	}

	memset(msg_data, 0, 17);
	msg_data[0] = s->authtype;
	memcpy(msg_data+1, s->username, 16);

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x39;
	req.msg.data		= msg_data;
	req.msg.data_len	= 17; /* 1 byte for authtype, 16 for user */

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Get Session Challenge command failed\n");
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "get_session_challenge");

	if (rsp->ccode > 0) {
		ipmi_printf("Get Session Challenge error: ");
		switch (rsp->ccode) {
		case 0x81:
			ipmi_printf("Invalid user name\n");
			ipmi_set_error(error, IPMI_ERROR_AUTH_FAILED);
			break;
		case 0x82:
			ipmi_printf("NULL user name not enabled\n");
			ipmi_set_error(error, IPMI_ERROR_AUTH_FAILED);
			break;
		default:
			ipmi_reset_error(error);
			ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
			ipmi_printf("Get Session Challenge command failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
		}
		return -1;
	}

	memcpy(&s->session_id, rsp->data, 4);
	memcpy(s->challenge, rsp->data + 4, 16);

	if (verbose > 1) {
		ipmi_printf("Opening Session\n");
		ipmi_printf("  Session ID      : %08lx\n", (long)s->session_id);
	}


	return 0;
}

/*
 * IPMI Activate Session Command
 */
static int
ipmi_activate_session_cmd(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipmi_session * s = intf->session;
	uint8_t msg_data[22];

	if (!intf->session) {
		return -1;
	}

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 0x3a;

	msg_data[0] = s->authtype;
	msg_data[1] = s->privlvl;

	/* supermicro oem authentication hack */
	if (ipmi_oem_active(intf, "supermicro")) {
		uint8_t * special = ipmi_auth_special(s);
		memcpy(s->authcode, special, 16);
		memset(msg_data + 2, 0, 16);
		ipmi_printf("  OEM Auth        : %s",
			buf2str(special, 16));
	} else {
		memcpy(msg_data + 2, s->challenge, 16);
	}

	/* setup initial outbound sequence number */
	get_random(msg_data+18, 4);

	req.msg.data = msg_data;
	req.msg.data_len = 22;

	s->active = 1;

	if (verbose > 1) {
		ipmi_printf("  Privilege Level : %s\n",
		       val2str(msg_data[1], ipmi_privlvl_vals));
		ipmi_printf("  Auth Type       : %s\n",
		       val2str(s->authtype, ipmi_authtype_session_vals));
	}

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		ipmi_printf("Activate Session command failed\n");
		s->active = 0;
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "activate_session");

	if (rsp->ccode) {
		ipmi_printf("Activate Session error: ");
		switch (rsp->ccode) {
		case 0x81:
			ipmi_printf("No session slot available\n");
			break;
		case 0x82:
			ipmi_printf("No slot available for given user - "
			       "limit reached\n");
			break;
		case 0x83:
			ipmi_printf("No slot available to support user "
			       "due to maximum privilege capacity\n");
			break;
		case 0x84:
			ipmi_printf("Session sequence number out of range\n");
			break;
		case 0x85:
			ipmi_printf("Invalid session ID in request\n");
			break;
		case 0x86:
			ipmi_set_error(error, IPMI_ERROR_NO_PRIVILEGE);
			ipmi_printf("Requested privilege level exceeds limit\n");
			break;
		case 0xd4:
			ipmi_set_error(error, IPMI_ERROR_NO_PRIVILEGE);
			ipmi_printf("Insufficient privilege level\n");
			break;
		default:
			ipmi_printf("%s\n", val2str(rsp->ccode, completion_code_vals));
		}
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		return -1;
	}

	memcpy(&s->session_id, rsp->data + 1, 4);
	s->in_seq = rsp->data[8] << 24 | rsp->data[7] << 16 | rsp->data[6] << 8 | rsp->data[5];
	if (s->in_seq == 0)
		++s->in_seq;

	if (s->authstatus & IPMI_AUTHSTATUS_PER_MSG_DISABLED)
		s->authtype = IPMI_SESSION_AUTHTYPE_NONE;
	else if (s->authtype != (rsp->data[0] & 0xf)) {
		ipmi_printf("Invalid Session AuthType %s in response\n",
			val2str(s->authtype, ipmi_authtype_session_vals));
		return -1;
  	}

	if (verbose > 1) {
		ipmi_printf("\nSession Activated\n");
		ipmi_printf("  Auth Type       : %s\n",
		       val2str(rsp->data[0], ipmi_authtype_session_vals));
		ipmi_printf("  Max Priv Level  : %s\n",
		       val2str(rsp->data[9], ipmi_privlvl_vals));
		ipmi_printf("  Session ID      : %08lx\n", (long)s->session_id);
		ipmi_printf("  Inbound Seq     : %08lx\n", (long)s->in_seq);
		ipmi_printf("\n");
	}

	return 0;
}


/*
 * IPMI Set Session Privilege Level Command
 */
static int
ipmi_set_session_privlvl_cmd(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t privlvl;

	if (!intf->session) {
		return -1;
	}

	privlvl = intf->session->privlvl;

	if (privlvl <= IPMI_SESSION_PRIV_USER)
		return 0;	/* no need to set higher */

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x3b;
	req.msg.data		= &privlvl;
	req.msg.data_len	= 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		ipmi_printf("Set Session Privilege Level to %s failed\n",
			val2str(privlvl, ipmi_privlvl_vals));
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "set_session_privlvl");

	if (rsp->ccode > 0) {
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		ipmi_printf("Set Session Privilege Level to %s failed: %s\n",
			val2str(privlvl, ipmi_privlvl_vals),
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}
	if (verbose > 1)
		ipmi_printf("Set Session Privilege Level to %s\n\n",
		       val2str(rsp->data[0], ipmi_privlvl_vals));
	return 0;
}

static int
ipmi_close_session_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[4];
	uint32_t session_id;

	if (!intf->session) {
		return -1;
	}

	session_id = intf->session->session_id;

	if (intf->session->active == 0)
		return -1;

	intf->target_addr = intf->bmc_addr;

	memcpy(&msg_data, &session_id, 4);

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x3c;
	req.msg.data		= msg_data;
	req.msg.data_len	= 4;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Close Session command failed\n");
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "close_session");

	if (rsp->ccode == 0x87) {
		ipmi_printf("Failed to Close Session: invalid session ID %08lx\n",
		       (long)session_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Close Session command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (verbose > 1)
		ipmi_printf("\nClosed Session %08lx\n\n", (long)session_id);

	return 0;
}

/*
 * IPMI LAN Session Activation (IPMI spec v1.5 section 12.9)
 *
 * 1. send "RMCP Presence Ping" message, response message will
 *    indicate whether the platform supports IPMI
 * 2. send "Get Channel Authentication Capabilities" command
 *    with AUTHTYPE = none, response packet will contain information
 *    about supported challenge/response authentication types
 * 3. send "Get Session Challenge" command with AUTHTYPE = none
 *    and indicate the authentication type in the message, response
 *    packet will contain challenge string and temporary session ID.
 * 4. send "Activate Session" command, authenticated with AUTHTYPE
 *    sent in previous message.  Also sends the initial value for
 *    the outbound sequence number for BMC.
 * 5. BMC returns response confirming session activation and
 *    session ID for this session and initial inbound sequence.
 */
static int
ipmi_lan_activate_session(struct ipmi_intf * intf, int * error)
{
	int rc;

	/* don't fail on ping because its not always supported.
	 * Supermicro's IPMI LAN 1.5 cards don't tolerate pings.
	 */
	if (!ipmi_oem_active(intf, "supermicro"))
		ipmi_lan_ping(intf, error);

	/* Some particular Intel boards need special help
	 */
	if (ipmi_oem_active(intf, "intelwv2"))
		ipmi_lan_thump_first(intf, error);

	rc = ipmi_get_auth_capabilities_cmd(intf, error);
	if (rc < 0) {
#if 0
		sleep(1);
		rc = ipmi_get_auth_capabilities_cmd(intf, error);
		if (rc < 0)
#endif
			goto fail;
	}

	rc = ipmi_get_session_challenge_cmd(intf, error);
	if (rc < 0)
		goto fail;

	rc = ipmi_activate_session_cmd(intf, error);
	if (rc < 0)
		goto fail;

	intf->abort = 0;

	rc = ipmi_set_session_privlvl_cmd(intf, error);
	if (rc < 0)
		goto fail;

	return 0;

 fail:
	ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
	ipmi_printf("Error: Unable to establish LAN session\n");
	return -1;
}

void ipmi_lan_close(struct ipmi_intf * intf, int force, int * error)
{
	if (intf->abort == 0 && !force)
		ipmi_close_session_cmd(intf, error);

	if (intf->fd >= 0) {
		closesocket(intf->fd);
		intf->fd = -1;
	}

	ipmi_req_clear_entries();

	if (intf->session != NULL) {
		free(intf->session);
		intf->session = NULL;
	}

	intf->opened = 0;
	intf = NULL;
}

void ipmi_lan_abort(struct ipmi_intf * intf, int * error)
{
	if (!intf->abort)
		ipmi_close_session_cmd(intf, error);

	if (intf->fd >= 0) {
		closesocket(intf->fd);
		intf->fd = -1;
	}

	intf->cancel = 1;
}

int ipmi_lan_open(struct ipmi_intf * intf, int * error)
{
	int rc;
	struct ipmi_session *s;
#ifdef WIN32
	srand( (unsigned)time( NULL ) );
#endif // WIN32
	if (!intf || !intf->session) {
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		return -1;
	}
	s = intf->session;

	if (s->port == 0)
		s->port = IPMI_LAN_PORT;
	if (s->privlvl == 0)
		s->privlvl = IPMI_SESSION_PRIV_ADMIN;
	if (s->timeout == 0)
		s->timeout = IPMI_LAN_TIMEOUT;
	if (s->retry == 0)
		s->retry = IPMI_LAN_RETRY;

	if (!strlen((char *)s->hostname)) {
		ipmi_set_error(error, IPMI_ERROR_NO_HOSTNAME);
		ipmi_printf("No hostname specified!\n");
		return -1;
	}

	intf->abort = 1;

	/* open port to BMC */
	memset(&s->addr, 0, sizeof(struct sockaddr_in));
	s->addr.sin_family = AF_INET;
	s->addr.sin_port = htons((uint16_t)s->port);

	s->addr.sin_addr.s_addr = inet_addr((char *)s->hostname);
	if (s->addr.sin_addr.s_addr == INADDR_NONE) {
		struct hostent *host = gethostbyname((char *)s->hostname);
		if (host == NULL) {
			ipmi_set_error(error, IPMI_ERROR_LOOKUP_FAILED);
			ipmi_printf("Address lookup for %s failed\n",
				s->hostname);
			return -1;
		}
		s->addr.sin_family = host->h_addrtype;
		memcpy(&s->addr.sin_addr, host->h_addr, host->h_length);
	}

	if (verbose > 1)
		ipmi_printf("IPMI LAN host %s port %d\n",
		       s->hostname, ntohs(s->addr.sin_port));

	intf->fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (intf->fd < 0) {
		ipmi_set_error(error, IPMI_ERROR_NETWORK);
		ipmi_printf("socket failed");
		return -1;
	}

	/* connect to UDP socket so we get async errors */
	rc = connect(intf->fd, (struct sockaddr *)&s->addr, sizeof(struct sockaddr_in));
	if (rc < 0) {
		ipmi_set_error(error, IPMI_ERROR_NETWORK);
		ipmi_printf("connect failed");
		intf->close(intf, 0, error);
		return -1;
	}

	intf->opened = 1;

	/* try to open session */
	rc = ipmi_lan_activate_session(intf, error);
	if (rc < 0) {
		intf->close(intf, 0, error);
		intf->opened = 0;
		ipmi_set_error(error, IPMI_ERROR_NO_CONNECTION);
		return -1;
	}

	curr_seq = 0;

	return intf->fd;
}

static int ipmi_lan_setup(struct ipmi_intf * intf, int * error UNUSED)
{
	intf->cancel = 0;
	intf->session = malloc(sizeof(struct ipmi_session));
	if (intf->session == NULL) {
		ipmi_printf("%s malloc failure\n", __FUNCTION__);
		return -1;
	}
	memset(intf->session, 0, sizeof(struct ipmi_session));
	return 0;
}

