/*
 * Copyright (c) 2006 Raritan Computer Inc.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <fcntl.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <sys/ioctl.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>
#endif

#include <ipmitool/helper.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/bswap.h>

#include <config.h>
#include "ppscsi.h"
#include "scsi_api.h"

#if 0
static void dump(const unsigned char *buf, int len)
{
    int i;
    for (i = 0; i < len; i++) {
	printf(" %02x", buf[i]);
	if ((i % 16) == 15) printf("\n"); else if ((i % 16) == 7) printf(" -");
    }
    printf("\n");
}
#endif

static int
open_session(struct ipmi_intf *intf)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    ppscsi_scsi_cmd_t scsi_cmd;
    scsi_ioreq_t ioreq;

    memset(&scsi_cmd, 0, sizeof(scsi_cmd));
    scsi_cmd.opcode = SCSI_OPCODE_VENDOR_IPMI;
    scsi_cmd.command = SCSI_IPMI_OPEN_SESSION;
    scsi_cmd.timeout = 60;

    memset(&ioreq, 0, sizeof(ioreq));
    ioreq.cmd_buf = (unsigned char *)&scsi_cmd;
    ioreq.cmd_len = sizeof(scsi_cmd);
    ioreq.timeout = 1000;
    ioreq.datadir = SCSI_DATADIR_FROM_DEVICE;
    ioreq.data_buf = (unsigned char *)&ctx->session_secret;
    ioreq.data_len = 4;
    if (scsi_send_request(ctx->scsi_dev, &ioreq) != 0
	    || ioreq.actual_len != ioreq.data_len) {
	return -1;
    };

    return 0;
}

static int
close_session(struct ipmi_intf *intf)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    ppscsi_scsi_cmd_t scsi_cmd;
    scsi_ioreq_t ioreq;

    memset(&scsi_cmd, 0, sizeof(scsi_cmd));
    scsi_cmd.opcode = SCSI_OPCODE_VENDOR_IPMI;
    scsi_cmd.command = SCSI_IPMI_CLOSE_SESSION;
    scsi_cmd.session_secret = ctx->session_secret;

    memset(&ioreq, 0, sizeof(ioreq));
    ioreq.cmd_buf = (unsigned char *)&scsi_cmd;
    ioreq.cmd_len = sizeof(scsi_cmd);
    ioreq.timeout = 1000;
    ioreq.datadir = SCSI_DATADIR_NONE;
    if (scsi_send_request(ctx->scsi_dev, &ioreq) != 0) {
	return -1;
    };

    return 0;
}

static int
send_request(struct ipmi_intf *intf, ppscsi_imsg_t *req, int length)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    ppscsi_scsi_cmd_t scsi_cmd;
    scsi_ioreq_t ioreq;

//    printf("send_request():\n");
//    dump((unsigned char *)req, length);

    memset(&scsi_cmd, 0, sizeof(scsi_cmd));
    scsi_cmd.opcode = SCSI_OPCODE_VENDOR_IPMI;
    scsi_cmd.command = SCSI_IPMI_SEND_REQUEST;
    scsi_cmd.session_secret = ctx->session_secret;

    memset(&ioreq, 0, sizeof(ioreq));
    ioreq.cmd_buf = (unsigned char *)&scsi_cmd;
    ioreq.cmd_len = sizeof(scsi_cmd);
    ioreq.timeout = 1000;
    ioreq.datadir = SCSI_DATADIR_TO_DEVICE;
    ioreq.data_buf = (unsigned char *)req;
    ioreq.data_len = length;
    if (scsi_send_request(ctx->scsi_dev, &ioreq) != 0) {
	return -1;
    };

    return 0;
}

static int
get_response(struct ipmi_intf *intf, ppscsi_imsg_t *rsp, int *length)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    ppscsi_scsi_cmd_t scsi_cmd;
    scsi_ioreq_t ioreq;
    int ret = -1;

    unsigned char *data = malloc(*length + 2);

    memset(&scsi_cmd, 0, sizeof(scsi_cmd));
    scsi_cmd.opcode = SCSI_OPCODE_VENDOR_IPMI;
    scsi_cmd.command = SCSI_IPMI_GET_RESPONSE;
    scsi_cmd.session_secret = ctx->session_secret;

    memset(&ioreq, 0, sizeof(ioreq));
    ioreq.cmd_buf = (unsigned char *)&scsi_cmd;
    ioreq.cmd_len = sizeof(scsi_cmd);
    ioreq.timeout = 5000;
    ioreq.datadir = SCSI_DATADIR_FROM_DEVICE;
    ioreq.data_buf = data;
    ioreq.data_len = *length;
    if (scsi_send_request(ctx->scsi_dev, &ioreq) == 0 && ioreq.status == 0) {
	*length = le16_to_cpu(*(unsigned short *)data);
	if (*length + 2 <= ioreq.actual_len) {
	    memcpy(rsp, data + 2, *length);
//	    printf("get_response():\n");
//	    dump((unsigned char *)rsp, *length);
	    ret = 0;
	}
    }

    free(data);
    return ret;
}

/*
 * IPMI Get Session Challenge Command
 * returns a temporary session ID and 16 byte challenge string
 */
static int
ipmi_get_session_challenge_cmd(struct ipmi_intf * intf, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipmi_session * s = intf->session;
	uint8_t msg_data[17];

	s->authtype = IPMI_SESSION_AUTHTYPE_PASSWORD;
	memset(msg_data, 0, 17);
	msg_data[0] = s->authtype;
	memcpy(msg_data+1, s->username, 16);

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x39;
	req.msg.data		= msg_data;
	req.msg.data_len	= 17; /* 1 byte for authtype, 16 for user */

	rsp = intf->sendrecv(intf, &req, NULL);
	if (rsp == NULL) {
		ipmi_printf("Get Session Challenge command failed\n");
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "get_session_challenge");

	if (rsp->ccode > 0) {
		switch (rsp->ccode) {
		case 0x81:
			ipmi_printf("Invalid user name\n");
			ipmi_set_error(error, IPMI_ERROR_AUTH_FAILED);
			break;
		case 0x82:
			ipmi_printf("NULL user name not enabled\n");
			ipmi_set_error(error, IPMI_ERROR_AUTH_FAILED);
			break;
		default:
			ipmi_reset_error(error);
			ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
			ipmi_printf("Get Session Challenge command failed: %s\n",
				val2str(rsp->ccode, completion_code_vals));
		}
		return -1;
	}

	memcpy(&s->session_id, rsp->data, 4);
	memcpy(s->challenge, rsp->data + 4, 16);

	if (verbose > 1) {
	    ipmi_printf("Opening Session");
	    ipmi_printf("  Session ID      : %08lx", (long)s->session_id);
	}
	
	return 0;
}

/*
 * IPMI Activate Session Command
 */
static int
ipmi_activate_session_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	struct ipmi_session * s = intf->session;
	uint8_t msg_data[22];

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 0x3a;

	msg_data[0] = s->authtype;
	msg_data[1] = s->privlvl;
	memcpy(msg_data + 2, s->authcode, 16);

	/* initial outbound sequence number; unused, but must be non-zero */
	*(int *)(msg_data+18) = 1;

	req.msg.data = msg_data;
	req.msg.data_len = 22;

	s->active = 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		ipmi_printf("Activate Session command failed\n");
		s->active = 0;
		return -1;
	}
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "activate_session");

	if (rsp->ccode) {
		ipmi_printf("Activate Session error: ");
		switch (rsp->ccode) {
		case 0x81:
			ipmi_printf("No session slot available\n");
			break;
		case 0x82:
			ipmi_printf("No slot available for given user - "
				"limit reached\n");
			break;
		case 0x83:
			ipmi_printf("No slot available to support user "
				"due to maximum privilege capacity\n");
			break;
		case 0x84:
			ipmi_printf("Session sequence out of range\n");
			break;
		case 0x85:
			ipmi_printf("Invalid session ID in request\n");
			break;
		case 0x86:
			ipmi_set_error(error, IPMI_ERROR_NO_PRIVILEGE);
			ipmi_printf("Requested privilege level "
				"exceeds limit\n");
			break;
		case 0xd4:
			ipmi_set_error(error, IPMI_ERROR_NO_PRIVILEGE);
			ipmi_printf("Insufficient privilege level\n");
			break;
		default:
			ipmi_printf("%s\n", val2str(rsp->ccode, completion_code_vals));
		}
		ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
		return -1;
	}

	memcpy(&s->session_id, rsp->data + 1, 4);
	s->in_seq = rsp->data[8] << 24 | rsp->data[7] << 16 | rsp->data[6] << 8 | rsp->data[5];
	if (s->in_seq == 0)
		++s->in_seq;

	if (s->authstatus & IPMI_AUTHSTATUS_PER_MSG_DISABLED)
		s->authtype = IPMI_SESSION_AUTHTYPE_NONE;
	else if (s->authtype != (rsp->data[0] & 0xf)) {
		return -1;
	}

	return 0;
}

/*
 * IPMI Set Session Privilege Level Command
 */
static int
ipmi_set_session_privlvl_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t privlvl = intf->session->privlvl;

	if (privlvl <= IPMI_SESSION_PRIV_USER)
		return 0;	/* no need to set higher */

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x3b;
	req.msg.data		= &privlvl;
	req.msg.data_len	= 1;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL || rsp->ccode > 0) return -1;

	return 0;
}

static int
ipmi_close_session_cmd(struct ipmi_intf * intf, int *error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[4];
	uint32_t session_id = intf->session->session_id;

	if (intf->session->active == 0)
		return -1;

	intf->target_addr = IPMI_BMC_SLAVE_ADDR;

	memcpy(&msg_data, &session_id, 4);

	memset(&req, 0, sizeof(req));
	req.msg.netfn		= IPMI_NETFN_APP;
	req.msg.cmd		= 0x3c;
	req.msg.data		= msg_data;
	req.msg.data_len	= 4;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Close Session command failed\n");
		return -1;
	}
	intf->session->session_id = 0;
	if (verbose > 2)
		printbuf(rsp->data, rsp->data_len, "close_session");

	if (rsp->ccode == 0x87) {
		ipmi_printf("Failed to Close Session: invalid session ID %08lx\n",
			(long)session_id);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Close Session command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	if (verbose > 1)
		ipmi_printf("\nClosed Session %08lx\n\n", (long)session_id);

	return 0;
}

/*
 * ipmi_intf method implementations:
 */
static int
ipmi_ppscsi_setup(struct ipmi_intf * intf, int *error UNUSED)
{
    intf->session = malloc(sizeof(struct ipmi_session));
    memset(intf->session, 0, sizeof(struct ipmi_session));
    intf->intf_ctx = malloc(sizeof(ppscsi_ctx_t));
    memset(intf->intf_ctx, 0, sizeof(ppscsi_ctx_t));
    return 0;
}

static void
ipmi_ppscsi_close(struct ipmi_intf * intf, int force UNUSED, int *error)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    if (intf->abort == 0) {
	ipmi_close_session_cmd(intf, error);
    }
    if (ctx->session_secret != 0) {
	close_session(intf);
	ctx->session_secret = 0;
    }
    if (ctx->scsi_dev) {
	scsi_close(ctx->scsi_dev);
	ctx->scsi_dev = NULL;
    }
    if (intf->session != NULL) {
	free(intf->session);
	intf->session = NULL;
    }
    intf->opened = 0;
}

static int
ipmi_ppscsi_open(struct ipmi_intf * intf, int *error)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    struct ipmi_session *s;
    if (!intf || !intf->session) {
	ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
	goto bail;
    }
    s = intf->session;
    intf->abort = 1;

    if (s->device[0] != '\0') {
	ctx->scsi_dev = scsi_open(s->device);
    } else {
	scsi_drive_t *devs = scsi_scan_devices(intf->session->ppscsi_product);
	scsi_drive_t *dev = devs;
	while (dev->path) {
	    ctx->scsi_dev = scsi_open(dev->path);
	    if (ctx->scsi_dev) {
		ipmi_intf_session_set_device(intf, dev->path);
		break;
	    }
	    dev++;
	}
	scsi_free_device_list(devs);
    }

    if (!ctx->scsi_dev) {
	if (s->device) {
	    ipmi_printf("Error: Can't open device %s\n", s->device);
	    ipmi_set_error(error, IPMI_ERROR_NO_SUCH_DEVICE);
	} else {
	    ipmi_printf("Error: No device found\n");
	    ipmi_set_error(error, IPMI_ERROR_NO_SUCH_DEVICE);
	}
	goto bail;
    }
    ipmi_printf("Using device %s\n", s->device);

    if (open_session(intf) < 0 || ctx->session_secret == 0) {
	ipmi_printf("Error: Failed to open a ppscsi session\n", s->device);
	ipmi_set_error(error, IPMI_ERROR_ESTABLISH_SESSION);
	goto bail;
    }

    intf->opened = 1;

    if (ipmi_get_session_challenge_cmd(intf, error) != 0) goto bail;

    if (ipmi_activate_session_cmd(intf, error) != 0) goto bail;

    intf->abort = 0;

    if (ipmi_set_session_privlvl_cmd(intf, error) != 0) {
	ipmi_printf("Error: Failed to set privilege level\n");
	ipmi_set_error(error, IPMI_ERROR_NO_PRIVILEGE);
	goto bail;
    }

    return 0;

bail:
    if (intf->opened) {
	ipmi_ppscsi_close(intf, 0, error);
    }
    return -1;
}

static struct ipmi_rs *
ipmi_ppscsi_sendrecv(struct ipmi_intf *intf, struct ipmi_rq *req, int *error)
{
    ppscsi_ctx_t *ctx = intf->intf_ctx;
    static struct ipmi_rs rsp;
    ppscsi_imsg_t *imsg_req = NULL;
    int req_len;
    ppscsi_imsg_t *imsg_rsp = NULL;
    int rsp_len;

    if (!intf->opened && intf->open) {
	if (intf->open(intf, error) < 0) return NULL;
    }

    req_len = sizeof(ppscsi_imsg_t) + req->msg.data_len;
    imsg_req = malloc(req_len);
    memset(imsg_req, 0, req_len);
    imsg_req->session_secret = ctx->session_secret;
    imsg_req->session_id = intf->session->session_id;
    imsg_req->netfn = req->msg.netfn;
    imsg_req->cmd = req->msg.cmd;
    memcpy(imsg_req->data, req->msg.data, req->msg.data_len);
    if (send_request(intf, imsg_req, req_len) < 0) {
	ipmi_printf("Error: Failed to send message\n");
	ipmi_set_error(error, IPMI_ERROR_SEND_FAILED);
	goto bail;
    }

    rsp_len = IPMI_BUF_SIZE;
    imsg_rsp = malloc(rsp_len);
    if (get_response(intf, imsg_rsp, &rsp_len) < 0) {
	ipmi_printf("Error: Device did not respond\n");
	ipmi_set_error(error, IPMI_ERROR_NO_RESPONSE);
	goto bail;
    }

    rsp.ccode = imsg_rsp->data[0];
    rsp.data_len = rsp_len - sizeof(ppscsi_imsg_t) - 1;
    memcpy(rsp.data, imsg_rsp->data+1, rsp.data_len);

    if (rsp.ccode) {
	ipmi_set_error(error, rsp.ccode);
    }
bail:
    if (imsg_req) free(imsg_req);
    if (imsg_rsp) free(imsg_rsp);
    return &rsp;
}

#define PPSCSI_NAME	"ppscsi"
#define PPSCSI_DESC	"Peppercon IPMI-over-SCSI interface"

#ifdef WIN32

struct ipmi_intf ipmi_ppscsi_intf = {
    /* name:              */  PPSCSI_NAME,
    /* desc:              */  PPSCSI_DESC,

    /* fd:                */  0,
    /* opened:            */  0,
    /* abort:             */  0,
    /* noanswer:          */  0,
    /* cancel:            */  0,
    /* intf_ctx:          */  NULL,

    /* cmd_wfd:           */  0,
    /* rsp_rfd:           */  0,
    /* loopi_pipe_prefix: */  0,

    /* session:           */  NULL,
    /* oem:               */  NULL,
    /* my_addr:           */  0,
    /* bmc_addr:          */  IPMI_BMC_SLAVE_ADDR,
    /* target_addr:       */  IPMI_BMC_SLAVE_ADDR,

    /* setup:             */  ipmi_ppscsi_setup,
    /* open:              */  ipmi_ppscsi_open,
    /* close:             */  ipmi_ppscsi_close,
    /* do_abort:          */  NULL,
    /* sendrecv:          */  ipmi_ppscsi_sendrecv,
    /* sendrsp:           */  NULL,
    /* recv_sol:          */  NULL,
    /* send_sol:          */  NULL,
    /* keepalive:         */  NULL,
};

#else // !WIN32

struct ipmi_intf ipmi_ppscsi_intf = {
    name:	PPSCSI_NAME,
    desc:	PPSCSI_DESC,
    setup:	ipmi_ppscsi_setup,
    open:	ipmi_ppscsi_open,
    close:	ipmi_ppscsi_close,
    sendrecv:	ipmi_ppscsi_sendrecv,
};

#endif // WIN32
