/*
 * Copyright (c) 2006 Raritan Computer Inc.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <scsi/sg.h>
#include <scsi/scsi_ioctl.h>

#include "scsi_api.h"

#define MAX_DEVICES 5

static const char *sysfs_sg_dir = "/sys/class/scsi_generic";
static const char *procfs_sg_list = "/proc/scsi/sg/device_strs";
static const char *default_vfloppy_id = "Virtual Disc";

struct scsi_dev_s {
    int fd;
};

/* scan /sysfs/class/scsi_generic for virtual disc drives */
static scsi_drive_t *find_devices_sysfs(const char *productname)
{
    scsi_drive_t *ret;
    int i, n, count;
    struct dirent **namelist;

    n = scandir(sysfs_sg_dir, &namelist, NULL, alphasort);
    if (n < 0) return NULL;

    ret = malloc((MAX_DEVICES + 1) * sizeof(scsi_drive_t));
    memset(ret, 0, (MAX_DEVICES + 1) * sizeof(scsi_drive_t));

    count = 0;
    for (i = 0; i < n; i++) {
	struct dirent *ent = namelist[i];
	if (count < MAX_DEVICES) {
	    char *path;
	    FILE *f;
	    asprintf(&path, "%s/%s/device/model", sysfs_sg_dir, ent->d_name);
	    f = fopen(path, "r");
	    free(path);
	    if (f) {
		char line[20];
		fgets(line, 20, f);
		line[strlen(line) - 1] = '\0';
		if (!strncmp(line, productname, strlen(productname))) {
		    char *dev_path;
		    asprintf(&dev_path, "/dev/%s", ent->d_name);
		    ret[count].path = dev_path;
		    ret[count].name = strdup(line);
		    count++;
		}
		fclose(f);
	    }
	}
	free(namelist[i]);
    }
    free(namelist);

    return ret;
}

/* scan /proc/scsi_sg for virtual disc drives */
static scsi_drive_t *find_devices_procfs(const char *productname)
{
    scsi_drive_t *ret;
    int line = 0, count;
    char vendor[20], model[20], version[20];
    FILE *f = fopen(procfs_sg_list, "r");
    if (!f) return NULL;

    ret = malloc((MAX_DEVICES + 1) * sizeof(scsi_drive_t));
    memset(ret, 0, (MAX_DEVICES + 1) * sizeof(scsi_drive_t));

    count = 0;
    while (fscanf(f, "%20[^\t\n]  %20[^\t\n] %20[^\t\n]\n", vendor, model, version) == 3) {
	if (!strncmp(model, productname, strlen(productname))) {
	    char *dev_path;
	    asprintf(&dev_path, "/dev/sg%d", line);
	    ret[count].path = dev_path;
	    ret[count].name = strdup(model);
	    if (count++ >= MAX_DEVICES) break;
	}
	line++;
    }

    fclose(f);
    return ret;
}

static int is_dev_sg(const struct dirent *name)
{
    int idx;
    return (sscanf(name->d_name, "sg%d", &idx) == 1);
}

/* last resort: do an Inquiry on /dev/sg* to scan for virtual disc drives */
static scsi_drive_t *find_devices_legacy(const char *productname)
{
    scsi_drive_t *ret;
    struct dirent **namelist;
    int i, n, count;

    sg_io_hdr_t io_hdr;
    unsigned char inquiry_cmd[] = { 0x12, 0, 0, 0, 96, 0 };
    unsigned char inquiry_buf[96];
    memset(&io_hdr, 0, sizeof(io_hdr));
    io_hdr.interface_id = 'S';
    io_hdr.cmdp = inquiry_cmd;
    io_hdr.cmd_len = sizeof(inquiry_cmd);
    io_hdr.dxferp = inquiry_buf;
    io_hdr.dxfer_len = sizeof(inquiry_buf);
    io_hdr.dxfer_direction = SG_DXFER_FROM_DEV;
    io_hdr.timeout = 500;

    n = scandir("/dev", &namelist, is_dev_sg, alphasort);
    if (n < 0) return NULL;

    ret = malloc((MAX_DEVICES + 1) * sizeof(scsi_drive_t));
    memset(ret, 0, (MAX_DEVICES + 1) * sizeof(scsi_drive_t));

    count = 0;
    for (i = 0; i < n; i++) {
	struct dirent *ent = namelist[i];
	if (count < MAX_DEVICES) {
	    char *dev_path;
	    int fd;
	    asprintf(&dev_path, "/dev/%s", ent->d_name);
	    fd = open(dev_path, O_RDONLY);
	    if (fd) {
		if (ioctl(fd, SG_IO, &io_hdr) >= 0 && io_hdr.resid == 0) {
		    char *model = (char *)inquiry_buf + 16;
		    model[16] = '\0';
		    if (!strncmp(model, productname, strlen(productname))) {
			ret[count].path = strdup(dev_path);
			ret[count].name = strdup(model);
			count++;
		    }
		}
		close(fd);
	    }
	    free(dev_path);
	}
	free(namelist[i]);
    }
    free(namelist);

    return ret;
}

scsi_drive_t *scsi_scan_devices(const char * productname)
{
    scsi_drive_t *ret;
    if (productname == NULL) {
	productname = default_vfloppy_id;
    }
    ret = find_devices_sysfs(productname);
    if (!ret) ret = find_devices_procfs(productname);
    if (!ret) ret = find_devices_legacy(productname);
    return ret;
}

void scsi_free_device_list(scsi_drive_t *list)
{
    scsi_drive_t *elem = list;
    if (!list) return;
    while (elem->path) {
	free(elem->path);
	free(elem->name);
	elem++;
    }
    free(list);
}

scsi_dev_t *scsi_open(const char *path)
{
    scsi_dev_t *dev;
    int fd = open(path, O_RDWR);
    if (fd < 0) return NULL;

    dev = malloc(sizeof(scsi_dev_t));
    dev->fd = fd;
    return dev;
}

void scsi_close(scsi_dev_t *dev)
{
    close(dev->fd);
    free(dev);
}

int scsi_send_request(scsi_dev_t *dev, scsi_ioreq_t *req)
{
    sg_io_hdr_t io_hdr;

    memset(&io_hdr, 0, sizeof(io_hdr));
    io_hdr.interface_id = 'S';
    io_hdr.cmdp = req->cmd_buf;
    io_hdr.cmd_len = req->cmd_len;
    io_hdr.timeout = req->timeout;
    io_hdr.dxfer_direction
	= req->datadir == SCSI_DATADIR_FROM_DEVICE? SG_DXFER_FROM_DEV
	: req->datadir == SCSI_DATADIR_TO_DEVICE? SG_DXFER_TO_DEV
	: SG_DXFER_NONE;
    io_hdr.dxferp = req->data_buf;
    io_hdr.dxfer_len = req->data_len;

    if ((ioctl(dev->fd, SG_IO, &io_hdr)) < 0) return -1;

    req->status = (io_hdr.host_status != 0 || io_hdr.driver_status != 0 || io_hdr.masked_status != 0);
    req->actual_len = req->data_len - io_hdr.resid;

    return 0;
}

