/******************************************************************************\
* cat_block_matching.c                                                         *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Block matching algorithms for mouse cursor detection                         *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <sys/types.h>

#include <pp/vsc.h>

#include "iipptr_bitmap.h"
#include "cat_internal.h"
#include "cat_debug.h"
#include "iipptr_internal.h"

#if defined(CAT_FB_DIRECT)

/***** Santa's little helper **************************************************/

/*
#define RGB_R(__rgb__)          (((__rgb__) >> (fb_c_info.green_bits + \
                                                fb_c_info.blue_bits)) & \
                                 (0xff >> (8 - fb_c_info.red_bits)))
#define RGB_G(__rgb__)          (((__rgb__) >> fb_c_info.blue_bits) & \
                                 (0xff >> (8 - fb_c_info.red_bits)))
#define RGB_B(__rgb__)          ((__rgb__) & (0xff >> (8 - fb_c_info.red_bits)))
*/
#define RGB_R(__rgb__)          ((((__rgb__) >> (fb_c_info.blue_bits + \
                                                 fb_c_info.green_bits)) \
                                  << (8 - fb_c_info.red_bits)) & 0xFF)
#define RGB_G(__rgb__)          ((((__rgb__) >> fb_c_info.blue_bits) \
                                 << (8 - fb_c_info.green_bits)) & 0xFF)
#define RGB_B(__rgb__)          (((__rgb__) << \
                                  (8 - fb_c_info.blue_bits)) & 0xFF)
                                  
#define FB_OFFSET(_x_, _y_)     ((((_y_) / PP_FB_TILE_HEIGHT) * \
                                  PP_FB_TILE_HEIGHT * \
                                  fb_info.tiles_w + \
                                  (_y_) % PP_FB_TILE_HEIGHT) * \
                                 PP_FB_TILE_WIDTH + \
                                 ((_x_) / PP_FB_TILE_WIDTH) * \
                                 PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT + \
                                 (_x_) % PP_FB_TILE_WIDTH)

int cat_bmp2fb_rgb(u_int16_t **ret, const t_bitmap *bmp) {
    fb_color_info_t fb_c_info;
    u_int16_t *fb_rgb;
    u_int u, h, w;
    
    assert(ret);
    assert(bmp);
    assert(bmp->rgb);
    if(!bmp->width || !bmp->height) {
        /* no need to convert nothing */
        return PP_ERR;
    }
    
    pp_grab_get_color_info(0, &fb_c_info);
    assert(fb_c_info.bpp == 16);

    fb_rgb = (u_int16_t*)malloc(sizeof(u_int16_t) * bmp->width * bmp->height);
 
    for(h = 0, u = 0; h < bmp->height; ++h) {
        for(w = 0; w < bmp->width; ++w, ++u) {
            fb_rgb[u] = ((bmp->rgb[u].r >> (8 - fb_c_info.red_bits))
                         << (fb_c_info.green_bits + fb_c_info.blue_bits)) +
                        ((bmp->rgb[u].g >> (8 - fb_c_info.green_bits))
                         << fb_c_info.blue_bits) +
                        (bmp->rgb[u].b >> (8 - fb_c_info.blue_bits));
        }
    }
    
    if(*ret) {
        free(*ret);
    }
    *ret = fb_rgb;
    
    return PP_SUC;
}

#if 0
/* convert bitmap to hextile continous layout (or so ;-)) */
int cat_bmp2hcl(u_char **ret, t_bitmap *bmp) {
    u_int16_t *hcl;
    u_int hcl_size, u = 0, h, w;
    
    assert(ret);
    assert(bmp);
    assert(bmp->rgb);
    
    if(!bmp->width || !bmp->height) {
        /* no need to convert nothing */
        return PP_ERR;
    }
    
    hcl_size = OHDIV(bmp->width, PP_FB_TILE_WIDTH) * PP_FB_TILE_WIDTH *
               OHDIV(bmp->height, PP_FB_TILE_HEIGHT) * PP_FB_TILE_HEIGHT;
    hcl = (u_int16_t*)calloc(sizeof(u_int16_t), hcl_size);
    
    for(h = 0; h < bmp->height; ) {
        
        ++h;
        if(!(h % 
    }
    
    *ret = hcl;
    
    return PP_SUC;
}
#endif

/* obj function tries to find the mouse cursor on the screen in a
   horizontal stripe of cursor size at the top of the screen,
   provided for building the acceleration table */
int cat_find_cursor_for_table_fb(const driver_t *obj, 
                                 const u_int16_t *shape_fb_rgb, 
                                 const t_point *center, t_point *found) {
    fb_format_info_t fb_f_info;
    data_imouse_t* mouse_data;
    int top, height;
    
    assert(obj);
    assert(shape_fb_rgb);
    assert(center);
    assert(found);
    
    mouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->grab_client);
    
    if(mouse_data->current_cursor_mask.rgb == NULL) {
        return PP_ERR;
    }
    
    if(pp_grab_get_format_info(0, &fb_f_info) != 0) {
        return PP_ERR;
    }

    top = SET_WITHIN(center->pos_y - 5, 0, (int)fb_f_info.g_h -
                     (int)mouse_data->current_cursor_mask.height);
    height = SET_WITHIN(top + (int)mouse_data->current_cursor_mask.height + 5,
                        0, (int)fb_f_info.g_h);
    
    return cat_mse32_fb(mouse_data->grab_client, shape_fb_rgb, 
                        &mouse_data->current_cursor_mask,
                        0, top, fb_f_info.g_w, height, 
                        CAT_SSD_THRESHOLD, 1, found);
}
#if 0
/* original */
int cat_find_cursor_for_table_fb(const driver_t *obj, 
                                 const u_int16_t *shape_fb_rgb, 
                                 const t_point *center, t_point *found) {
    int diff_result;
    int search_diff = 10;
    int max_diff;
    fb_format_info_t fb_f_info;
    data_imouse_t* mouse_data;
    int left, top, width, height;
    
    assert(obj);
    assert(shape_fb_rgb);
    assert(center);
    assert(found);
    
    mouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->grab_client);
    
    if(mouse_data->current_cursor_mask.rgb == NULL) {
        return -1;
    }
    
    if(pp_grab_get_format_info(0, &fb_f_info) != 0) {
        return -1;
    }

    max_diff = fb_f_info.g_w;
    
    left = SET_WITHIN(center->pos_x - search_diff, 0, (int)fb_f_info.g_w);
    width = SET_WITHIN(center->pos_x + search_diff +
                       (int)mouse_data->current_cursor_mask.width,
                       0, (int)fb_f_info.g_w);
    top = center->pos_y;
    height = SET_WITHIN(top + (int)mouse_data->current_cursor_mask.height + 5,
                        0, (int)fb_f_info.g_h);
    
    while(((diff_result = cat_mse32_fb(mouse_data->grab_client, shape_fb_rgb, 
                                       &mouse_data->current_cursor_mask, 
                                       left, top, width, height, 
                                       32, found)) == PP_ERR)
          && (search_diff < max_diff)) {
        CDMSG(CAT_CDMSG_SYNC_INFO, 
              "cursor not found within (%4d; %3d) - (%4d; %3d)\n",
              left, top, left + width, top + height);
        left = SET_WITHIN(left - search_diff, 0, (int)fb_f_info.g_w);
        search_diff *=2;
        width = SET_WITHIN(width + search_diff, 0, (int)fb_f_info.g_w);
    }
    
    return diff_result;
}

/* left to right */
int cat_find_cursor_for_table_fb(const driver_t *obj, 
                                 const u_int16_t *shape_fb_rgb, 
                                 const t_point *upper_left, t_point *found) {
    int diff_result;
    int search_diff = 50;
    fb_format_info_t fb_f_info;
    data_imouse_t* mouse_data;
    int left, top, width, height;
    
    assert(obj);
    assert(shape_fb_rgb);
    assert(upper_left);
    assert(found);
    
    mouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->grab_client);
    
    if(mouse_data->current_cursor_mask.rgb == NULL) {
        return PP_ERR;
    }
    
    if(pp_grab_get_format_info(0, &fb_f_info) != 0) {
        return PP_ERR;
    }
    
    if(upper_left->pos_x >= (int)fb_f_info.g_w ||
       upper_left->pos_y >= (int)fb_f_info.g_h) {
        abort();
        return PP_ERR;
    }

    left = upper_left->pos_x;
    width = SET_WITHIN(search_diff + (int)mouse_data->current_cursor_mask.width,
                       0, (int)fb_f_info.g_w - left);
    top = upper_left->pos_y;
    height = SET_WITHIN(top + (int)mouse_data->current_cursor_mask.height + 5,
                        0, (int)fb_f_info.g_h - top);
    
    while(((diff_result = cat_mse32_fb(mouse_data->grab_client, shape_fb_rgb, 
                                       &mouse_data->current_cursor_mask, 
                                       left, top, width, height, 
                                       32, found)) == PP_ERR)
          && (width >= (int)mouse_data->current_cursor_mask.width)) {
        CDMSG(CAT_CDMSG_SYNC_INFO, 
              "cursor not found within (%4d; %3d) - (%4d; %3d)\n",
              left, top, left + width, top + height);
        left += width - mouse_data->current_cursor_mask.width;
        width = SET_WITHIN(left + search_diff, 0, (int)fb_f_info.g_w - left);
    }
    
    return diff_result;
}
#endif

/***** MSE ********************************************************************/

int cat_mse32_fb(pp_grab_client_t *grab_client, 
                 const u_int16_t *shape, const t_bitmap *mask, 
                 u_int32_t search_left, u_int32_t search_top,
                 u_int32_t search_width, u_int32_t search_height, 
                 u_int32_t threshold, int32_t iter, t_point *found) {
    t_point match;
    u_int32_t mse_min, mse_count, mse, shape_offset, found1;
    u_int32_t x, y, ox, oy, stopx, stopy, shape_tmp;
//    int32_t dr, dg, db;
    u_int16_t *screen = (u_int16_t*)grab_client->buf;
    fb_color_info_t fb_c_info;
    fb_format_info_t fb_info;
    u_int16_t diff;
    
    assert(grab_client);
    assert(grab_client->buf);
    assert(shape);
    assert(mask);
    assert(mask->rgb);
    assert(found);
    
    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO || CAT_CDMSG_WARNING, 
              "No framebuffer info yet\n");
        return PP_ERR;
    }
    
    CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO, 
          "searching for block (%d x %d) in area (%d x %d) @(%d; %d)\n",
          mask->width, mask->height, 
          search_width, search_height, search_left, search_top);
    
    if(mask->width > search_width || mask->height > search_height ||
       search_left + search_width > fb_info.g_w ||
       search_top + search_height > fb_info.g_h) {
        /* wrong dimensions */
        CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO,
              "wrong dimensions of shape or search area!\n");
//printf("fb_info.g_w = %d, fb_info.g_h = %d, mask->width = %d, mask->height = %d, search_left = %d, search_top = %d, search_width = %d, search_height = %d\n", fb_info.g_w, fb_info.g_h, mask->width, mask->height, search_left, search_top, search_width, search_height);abort();
        return PP_ERR;
    }
    
    pp_grab_get_color_info(0, &fb_c_info);
    assert(fb_c_info.bpp == 16);

    /* count unmasked shape pixels */
    for(oy = 0, mse_count = 0; oy < mask->height; ++oy) {
        for(ox = 0; ox < mask->width; ++ox) {
            /* all positions of shape */
            if(mask->rgb[oy * mask->width + ox].r) {
                /* all valid positions in mask */
                ++mse_count;
            }
        }
    }
    
    /* init */
    stopx = search_left + search_width - mask->width + 1;
    stopy = search_top + search_height - mask->height + 1;
    mse_min = mse_count * threshold * threshold * 3 * 4;
    found1 = 0;
    
    if(!iter) {
        /**
         * init coarse iterator...
         * the larger the unmasked area, the larger the iter... 
         * smaller side of shape is MIN(mask->width, mask->height)
         * iter should at least reach 3 times in min side
         * => iter = (MIN(mask->width, mask->height) / 3)
         * coverage = mse count / shape size
         * => iter = ? - (shape size / mse count)
         * we want iter to be at most CAT_BLOCK_MATCHING_MAX_ITER...
         * => iter = CAT_BLOCK_MATCHING_MAX_ITER
         */
        iter = MIN(MIN(mask->width, mask->height) / 3, 
                   (u_int)((CAT_BLOCK_MATCHING_MAX_ITER * mse_count) /
                           (float)(mask->width * mask->height) + 0.5));
    //    iter = MIN(MIN(mask->width, mask->height) / 5 + 1, 3);
    }
    
    /* coarse search */
    for(y = search_top; y < stopy; y += iter) {
        for(x = search_left; x < stopx; x += iter) {
            /* all possible positions (x; y) of shape in search space */
            
            mse = 0;
            for(oy = 0; oy < mask->height; oy += iter) {
                shape_tmp = oy * mask->width;
                for(ox = 0; ox < mask->width; ox += iter) {
                    /* all positions of shape */
                    
                    shape_offset = shape_tmp + ox;
                    if(mask->rgb[shape_offset].r) {
                        /* all valid position in mask */
                        
                        assert((u_int16_t)(x + ox) < fb_info.g_w);
                        assert((u_int16_t)(y + oy) < fb_info.g_h);

/*
int tmpr = RGB_R(screen[FB_OFFSET(x + ox, y + oy)]);
int tmpg = RGB_G(screen[FB_OFFSET(x + ox, y + oy)]);
int tmpb = RGB_B(screen[FB_OFFSET(x + ox, y + oy)]);
printf("tweb: rgb @(%4d, %3d): %5d => r %3d, g %3d, b %3d\n", x + ox, y + oy, screen[FB_OFFSET(x + ox, y + oy)], tmpr, tmpg, tmpb);
*/
/*
                        dr = RGB_R(screen[FB_OFFSET(x + ox, y + oy)]) -
                             RGB_R(shape[shape_offset]);
                        dg = RGB_G(screen[FB_OFFSET(x + ox, y + oy)]) -
                             RGB_G(shape[shape_offset]);
                        db = RGB_B(screen[FB_OFFSET(x + ox, y + oy)]) -
                             RGB_B(shape[shape_offset]);
                        mse += dr * dr + dg * dg + db * db;
*/
                        diff = screen[FB_OFFSET(x + ox, y + oy)] ^
                               shape[shape_offset];
                        mse += RGB_R(diff) * RGB_R(diff) + 
                               RGB_G(diff) * RGB_G(diff) + 
                               RGB_B(diff) * RGB_B(diff);
                        if(mse > mse_min) {
                            /* already worse than min, bail out */
                            goto bail_c;
                        }
                    }
                }
            }
            
            mse_min = mse;
            match.pos_x = x;
            match.pos_y = y;
            ++found1;
 bail_c:;
        }
    }

    if(!found1) {
        /* we didn't find pattern */
        CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO,
              "we didn't find pattern in coarse search!\n");
        return PP_ERR;
    } else {
        CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO, 
              "found pattern in coarse search @(%4d; %3d) with iter %d!\n",
              match.pos_x, match.pos_y, iter);
    }
    
    mse_min = mse_count * threshold * threshold * 3;
    found1 = 0;
   
    /* fine search */
    CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO,
          "fine searching area (%d; %d) -> (%d; %d)\n",
          MAX((int)search_left, match.pos_x - iter), 
          MAX((int)search_top, match.pos_y - iter), 
          MIN((int)stopx, match.pos_x + iter) - 1, 
          MIN((int)stopy, match.pos_y + iter) - 1);
    
    if(iter > 1) {
        /* if iter == 1 coarse search was fine enough ;-) */
        for(y = MAX((int)search_top, match.pos_y - iter); 
            (int)y < MIN((int)stopy, match.pos_y + iter); ++y) {
            for(x = MAX((int)search_left, match.pos_x - iter);
                (int)x < MIN((int)stopx, match.pos_x + iter); ++x) {
                /* all possible positions (x; y) of shape in search space */
                
                mse = 0;
                for(oy = 0; oy < mask->height; ++oy) {
                    shape_tmp = oy * mask->width;
                    for(ox = 0; ox < mask->width; ++ox) {
                        /* all positions of shape */
                        
                        shape_offset = shape_tmp + ox;
                        if(mask->rgb[shape_offset].r) {
                            /* all valid position in mask */
                            
                            assert((u_int16_t)(x + ox) < fb_info.g_w);
                            assert((u_int16_t)(y + oy) < fb_info.g_h);

/*
                            dr = RGB_R(screen[FB_OFFSET(x + ox, y + oy)]) -
                                 RGB_R(shape[shape_offset]);
                            dg = RGB_G(screen[FB_OFFSET(x + ox, y + oy)]) -
                                 RGB_G(shape[shape_offset]);
                            db = RGB_B(screen[FB_OFFSET(x + ox, y + oy)]) -
                                 RGB_B(shape[shape_offset]);
                            mse += dr * dr + dg * dg + db * db;
*/
                            diff = screen[FB_OFFSET(x + ox, y + oy)] ^
                                   shape[shape_offset];
                            mse += RGB_R(diff) * RGB_R(diff) + 
                                   RGB_G(diff) * RGB_G(diff) + 
                                   RGB_B(diff) * RGB_B(diff);
                            if(mse > mse_min) {
                                /* already worse than min, bail out */
                                goto bail_f;
                            }
                        }
                    }
                }
                
                mse_min = mse;
                match.pos_x = x;
                match.pos_y = y;
                ++found1;
     bail_f:;
            }
        }
        
        if(!found1) {
            /* we didn't find pattern */
            CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO,
                  "we didn't find pattern in fine search!\n");
            return PP_ERR;
        }
    }
        
    mse_min /= 3 * mse_count; /* normalize */
    found->pos_x = match.pos_x;
    found->pos_y = match.pos_y;
    
    CDMSG(CAT_CDMSG_BLOCK_MATCHING_INFO, "found shape @(%4d, %3d), mse = %d!\n",
          match.pos_x, match.pos_y, mse_min);

    return mse_min;
}

/***** little helpers *********************************************************/

#if 0
/**
 * detects mouse cursor shape at upper left corner of screen
 * - moves cursor away, to take first screenshot
 * - moves cursor into upper left corner to take second screenshot
 * - diffs and crops
 *
 * see iipptr_cursor.c:get_mousecursor_shape
 */
int cat_get_mousecursor_shape(driver_t* obj) {
    data_cat_t *mouse_data;
    t_mouse_data *md;
    u_int16_t *cursor1, *cursor2, *fb;
    int htrow_offset;
    fb_format_info_t fb_info;
    static const size_t ht_sz = PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT *
                                sizeof(u_int16_t);
    
    assert(obj);

    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->current_kvm_port);
    assert(mouse_data->grab_client);

    md = &mouse_data->current_kvm_port->mouse_data;
    assert(md);
    
    fb = mouse_data->grab_client->buf;
    assert(fb);
    
    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return PP_ERR;
    }
    
    htrow_offset = ht_sz * fb->tiles_w;

    md->cursor_state = ERIC_MOUSE_CURSOR_GETTING;

    CDMSG(CAT_CDMSG_SYNC_INFO, "Getting mouse cursor shape\n");
    move_mouse_direct(obj, 2000, 2000, 0, 0, 0);	
    usleep(CURSOR_SNAP_SLEEP);

    /* alloc memory for 4 hextiles */
    cursor1 = (u_int16_t)malloc(PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT * 4 *
                                sizeof(u_int16_t));
    cursor2 = (u_int16_t)malloc(PP_FB_TILE_WIDTH * PP_FB_TILE_HEIGHT * 4 *
                                sizeof(u_int16_t));
    
    memcpy(cursor1, fb, ht_sz * 2);
    memcpy(cursor1 + ht_sz * 2, fb + htrow_offset, ht_sz * 2);

    move_mouse_corner(obj, 0);
    usleep(CURSOR_SNAP_SLEEP);
    
    memcpy(cursor2, fb, ht_sz * 2);
    memcpy(cursor2 + ht_sz * 2, fb + htrow_offset, ht_sz * 2);
...
    /* create mask bitmap */
    mouse_data->current_cursor_mask.width  = mouse_data->current_cursor_shape.width;
    mouse_data->current_cursor_mask.height = mouse_data->current_cursor_shape.height;
    
    if (mouse_data->current_cursor_mask.rgb) {
	free(mouse_data->current_cursor_mask.rgb);
	mouse_data->current_cursor_mask.rgb = NULL;
    }
    mouse_data->current_cursor_mask.rgb    = (t_rgb *)malloc(mouse_data->current_cursor_mask.height *
							  mouse_data->current_cursor_mask.width  * sizeof(t_rgb));

    /* Diff both screenshots to get current cursor-mask */
    if (diff_mousecursor_shape(&mouse_data->current_cursor_shape,
			       &mouse_data->current_cursor_bg_shape,
			       &mouse_data->current_cursor_mask) != 0) {
	pp_log("liberic_km: Could not detect mouse cursor");
	md->cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
	return -1;
    }
    
    md->cursor_state = ERIC_MOUSE_CURSOR_IS_VALID;
    
    return 0;
}
#endif

#endif /* CAT_FB_DIRECT */

