/******************************************************************************\
* cat_clustering.c                                                             *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Clustering algorithms for mouse cursor tracking                              *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <pp/vsc.h>

#include "cat_internal.h"
#include "cat_debug.h"

static int cat_cluster_replace_ids(int sid, int rid, int *cluster_ids) {
    int i, found;

    for(i = 0, found = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        if(cluster_ids[i] == sid) {
            cluster_ids[i] = rid;
            ++found;
        }
    }
    
    return found ? found : PP_ERR;
}

int cat_cluster_simple_density(int *cluster_ids, const cat_sample32_t *samples,
                               u_int32_t max_diff_x, u_int32_t max_diff_y) {
    int clusters = 0;
    int i, j;
    
    /* initialize - clear all cluster IDs */
    memset(cluster_ids, -1, sizeof(int) * CAT_CONDENSATION_SAMPLE_COUNT);
    
    /* increment maxes */
    ++max_diff_x;
    ++max_diff_y;
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        for(j = 0; j < i; ++j) {
            /* now search for a cluster, the sample belongs to */
            /* TODO? faster than |xi - xj| < xmax && |yi - yj| < ymax? */
            
            if(abs(samples[i].x - samples[j].x) < max_diff_x &&
               abs(samples[i].y - samples[j].y) < max_diff_y) {
                if(cluster_ids[i] == -1) {
                    /* sample i belongs to cluster */
                    
                    cluster_ids[i] = cluster_ids[j];
                } else if(cluster_ids[i] != cluster_ids[j]) {
                    /* we got two clusters to merge together... */
                    
                    if(cluster_ids[i] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */
                        
                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);
                    } else if(cluster_ids[j] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */

                        cat_cluster_replace_ids(cluster_ids[j], cluster_ids[i], 
                                                cluster_ids);
                    } else {
                        /* reorganize ids in two steps */
                        
                        int tmp_id = cluster_ids[i];

                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);
                        cat_cluster_replace_ids(clusters - 1, tmp_id, cluster_ids);
                    }
                    
                    --clusters;
                } // else cluster_ids[i] == cluster_ids[j], do nothing
            }
        }
        
        if(cluster_ids[i] == -1) {
            /* sample does not belong to any cluster => it's one of its own */
            
            cluster_ids[i] = clusters++;
        }
    }    
    
    return clusters;
}

#if 0
int cat_cluster_simple_density(int *cluster_ids, const cat_sample32_t *samples,
                               u_int32_t max_diff_x, u_int32_t max_diff_y) {
    int clusters = 0;
    int i, j, k;
    u_int *x_min, *y_min, *x_max, *y_max;
        
    x_min = (u_int*)malloc(CAT_CONDENSATION_SAMPLE_COUNT * sizeof(u_int));
    x_max = (u_int*)malloc(CAT_CONDENSATION_SAMPLE_COUNT * sizeof(u_int));
    y_min = (u_int*)malloc(CAT_CONDENSATION_SAMPLE_COUNT * sizeof(u_int));
    y_max = (u_int*)malloc(CAT_CONDENSATION_SAMPLE_COUNT * sizeof(u_int));
    
    /* initialize - clear all cluster IDs */
    memset(cluster_ids, -1, sizeof(int) * CAT_CONDENSATION_SAMPLE_COUNT);
    
    /* increment maxes */
    ++max_diff_x;
    ++max_diff_y;
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        for(j = 0; j < i; ++j) {
            /* now search for a cluster, the sample belongs to */
            /* TODO? faster than |xi - xj| < xmax && |yi - yj| < ymax? */
            if(abs(samples[i].x - samples[j].x) < max_diff_x &&
               abs(samples[i].y - samples[j].y) < max_diff_y) {
                if(cluster_ids[i] == -1) {
                    /* sample i belongs to cluster */
                    cluster_ids[i] = cluster_ids[j];
                    
                    if(x_min[cluster_ids[j]] > samples[i].x) {
                        x_min[cluster_ids[j]] = samples[i].x;
                    }
                    if(x_max[cluster_ids[j]] < samples[i].x) {
                        x_max[cluster_ids[j]] = samples[i].x;
                    }
                    if(y_min[cluster_ids[j]] > samples[i].y) {
                        y_min[cluster_ids[j]] = samples[i].y;
                    }
                    if(y_max[cluster_ids[j]] < samples[i].y) {
                        y_max[cluster_ids[j]] = samples[i].y;
                    }
                } else if(cluster_ids[i] != cluster_ids[j]) {
                    /* we got two clusters to merge together... */
                    int sid, rid;
                    
                    if(cluster_ids[i] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */
                        
                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);

                        if(x_min[cluster_ids[i]] > x_min[cluster_ids[j]]) {
                            x_min[cluster_ids[i]] = x_min[cluster_ids[j]];
                        }
                        if(x_max[cluster_ids[i]] < x_max[cluster_ids[j]]) {
                            x_max[cluster_ids[i]] = x_max[cluster_ids[j]];
                        }
                        if(y_min[cluster_ids[i]] > y_min[cluster_ids[j]]) {
                            y_min[cluster_ids[i]] = y_min[cluster_ids[j]];
                        }
                        if(y_max[cluster_ids[i]] < y_max[cluster_ids[j]]) {
                            y_max[cluster_ids[i]] = y_max[cluster_ids[j]];
                        }
                    } else if(cluster_ids[j] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */

                        cat_cluster_replace_ids(cluster_ids[j], cluster_ids[i], 
                                                cluster_ids);

                        if(x_min[cluster_ids[j]] > x_min[cluster_ids[i]]) {
                            x_min[cluster_ids[j]] = x_min[cluster_ids[i]];
                        }
                        if(x_max[cluster_ids[j]] < x_max[cluster_ids[i]]) {
                            x_max[cluster_ids[j]] = x_max[cluster_ids[i]];
                        }
                        if(y_min[cluster_ids[j]] > y_min[cluster_ids[i]]) {
                            y_min[cluster_ids[j]] = y_min[cluster_ids[i]];
                        }
                        if(y_max[cluster_ids[j]] < y_max[cluster_ids[i]]) {
                            y_max[cluster_ids[j]] = y_max[cluster_ids[i]];
                        }
                    } else {
                        /* reorganize ids in two steps */
                        
                        int tmp_id = cluster_ids[i];

                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);
                        cat_cluster_replace_ids(clusters - 1, tmp_id, cluster_ids);

                        if(x_min[cluster_ids[j]] > x_min[tmp_id]) {
                            x_min[cluster_ids[j]] = x_min[tmp_id];
                        }
                        if(x_max[cluster_ids[j]] < x_max[tmp_id]) {
                            x_max[cluster_ids[j]] = x_max[tmp_id];
                        }
                        if(y_min[cluster_ids[j]] > y_min[tmp_id]) {
                            y_min[cluster_ids[j]] = y_min[tmp_id];
                        }
                        if(y_max[cluster_ids[j]] < y_max[tmp_id]) {
                            y_max[cluster_ids[j]] = y_max[tmp_id];
                        }

                        x_min[tmp_id] = x_min[clusters - 1];
                        x_max[tmp_id] = x_max[clusters - 1];
                        y_min[tmp_id] = y_min[clusters - 1];
                        y_max[tmp_id] = y_max[clusters - 1];
                    }
                    
                    --clusters;
                } // else cluster_ids[i] == cluster_ids[j], do nothing
            }
        }
        
        if(cluster_ids[i] == -1) {
            /* sample does not belong to any cluster => it's one of its own */
            cluster_ids[i] = clusters++;
            x_min[cluster_ids[i]] = samples[i].x;
            x_max[cluster_ids[i]] = samples[i].x;
            y_min[cluster_ids[i]] = samples[i].y;
            y_max[cluster_ids[i]] = samples[i].y;
        }
    }    
    

    {
        /* merge clusters */
        
        for(i = 0; i < clusters; ++i) {
            for(j = 0; j < clusters; ++j) {
printf("i = %d, j = %d, clusters = %d\n", i, j, clusters);
                if(i == j) {
                    continue;
                }

                if(x_min[i] >= x_min[j] && x_max[i] <= x_max[j] &&
                   y_min[i] >= y_min[j] && y_max[i] <= y_max[j]) {
                    /* cluster i in cluster j */

                    cat_cluster_replace_ids(i, j, cluster_ids);
                    cat_cluster_replace_ids(--clusters, i, cluster_ids);
                    
                    x_min[i] = x_min[clusters];
                    x_max[i] = x_max[clusters];
                    y_min[i] = y_min[clusters];
                    y_max[i] = y_max[clusters];
                } else if(x_min[i] <= x_min[j] && x_max[i] >= x_max[j] &&
                          y_min[i] <= y_min[j] && y_max[i] >= y_max[j]) {
                    /* cluster j in cluster i */

                    cat_cluster_replace_ids(j, i, cluster_ids);
                    cat_cluster_replace_ids(--clusters, j, cluster_ids);
                    
                    x_min[j] = x_min[clusters];
                    x_max[j] = x_max[clusters];
                    y_min[j] = y_min[clusters];
                    y_max[j] = y_max[clusters];
                }
            }
        }
    }

    return clusters;
}

int cat_cluster_simple_density_(int *cluster_ids, const cat_sample32_t *samples,
                               u_int32_t max_diff_x, u_int32_t max_diff_y) {
    int clusters = 0;
    int i, j, k;
    
    /* initialize - clear all cluster IDs */
    memset(cluster_ids, -1, sizeof(int) * CAT_CONDENSATION_SAMPLE_COUNT);
    
    /* increment maxes */
    ++max_diff_x;
    ++max_diff_y;
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        for(j = 0; j < i; ++j) {
            /* now search for a cluster, the sample belongs to */
            /* TODO? faster than |xi - xj| < xmax && |yi - yj| < ymax? */
            if(abs(samples[i].x - samples[j].x) < max_diff_x &&
               abs(samples[i].y - samples[j].y) < max_diff_y) {
                if(cluster_ids[i] == -1) {
                    /* sample i belongs to cluster */
                    cluster_ids[i] = cluster_ids[j];
                } else if(cluster_ids[i] != cluster_ids[j]) {
                    /* we got two clusters to merge together... */
                    int sid, rid;
                    
                    if(cluster_ids[i] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */
                        
                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);
                    } else if(cluster_ids[j] == (clusters - 1)) {
                        /* cluster_ids[i] is member of last added cluster */

                        cat_cluster_replace_ids(cluster_ids[j], cluster_ids[i], 
                                                cluster_ids);
                    } else {
                        /* reorganize ids in two steps */
                        
                        int tmp_id = cluster_ids[i];

                        cat_cluster_replace_ids(cluster_ids[i], cluster_ids[j], 
                                                cluster_ids);
                        cat_cluster_replace_ids(clusters - 1, tmp_id, cluster_ids);
                    }
                    
                    --clusters;
                } // else cluster_ids[i] == cluster_ids[j], do nothing
            }
        }
        
        if(cluster_ids[i] == -1) {
            /* sample does not belong to any cluster => it's one of its own */
            cluster_ids[i] = clusters++;
        }
    }    
    
//    return clusters;

#if 1 // look up table
    {
        /* merge clusters */
        
        int *x_min, *y_min, *x_max, *y_max;
        
        x_min = (int*)malloc(clusters * sizeof(int));
        x_max = (int*)malloc(clusters * sizeof(int));
        y_min = (int*)malloc(clusters * sizeof(int));
        y_max = (int*)malloc(clusters * sizeof(int));
        
        /* fill look up table */
        for(i = 0; i < clusters; ++i) {
            cat_cluster_get_dimensions(&x_min[i], &y_min[i], 
                                       &x_max[i], &y_max[i],
                                       i, cluster_ids, samples);
        }
        
        for(i = 0; i < clusters; ++i) {
            for(j = 0; j < clusters; ++j) {
printf("i = %d, j = %d, clusters = %d\n", i, j, clusters);
                if(i == j) {
                    continue;
                }

                if(x_min[i] >= x_min[j] && x_max[i] <= x_max[j] &&
                   y_min[i] >= y_min[j] && y_max[i] <= y_max[j]) {
                    /* cluster i in cluster j */

                    cat_cluster_replace_ids(i, j, cluster_ids);
                    cat_cluster_replace_ids(--clusters, i, cluster_ids);
                    
                    x_min[i] = x_min[clusters];
                    x_max[i] = x_max[clusters];
                    y_min[i] = y_min[clusters];
                    y_max[i] = y_max[clusters];
                } else if(x_min[i] <= x_min[j] && x_max[i] >= x_max[j] &&
                          y_min[i] <= y_min[j] && y_max[i] >= y_max[j]) {
                    /* cluster j in cluster i */

                    cat_cluster_replace_ids(j, i, cluster_ids);
                    cat_cluster_replace_ids(--clusters, j, cluster_ids);
                    
                    x_min[j] = x_min[clusters];
                    x_max[j] = x_max[clusters];
                    y_min[j] = y_min[clusters];
                    y_max[j] = y_max[clusters];
                }
            }
        }
    }
#else
    {
        /* merge clusters */

        int xi_min, yi_min, xi_max, yi_max, xj_min, yj_min, xj_max, yj_max;

        for(i = 0; i < clusters; ++i) {
            cat_cluster_get_dimensions(&xi_min, &yi_min, &xi_max, &yi_max,
                                       i, cluster_ids, samples);
            for(j = 0; j < clusters; ++j) {
printf("i = %d, j = %d, clusters = %d\n", i, j, clusters);
                if(i == j) {
                    continue;
                }

                cat_cluster_get_dimensions(&xj_min, &yj_min, &xj_max, &yj_max,
                                           j, cluster_ids, samples);

                if(xi_min >= xj_min && xi_max <= xj_max &&
                   yi_min >= yj_min && yi_max <= yj_max) {
                    /* cluster i in cluster j */
                    k = i;

                    cat_cluster_replace_ids(i, j, cluster_ids);
                    cat_cluster_replace_ids(--clusters, k, cluster_ids);
                } else if(xi_min <= xj_min && xi_max >= xj_max &&
                   yi_min <= yj_min && yi_max >= yj_max) {
                    /* cluster j in cluster i */
                    k = j;

                    cat_cluster_replace_ids(j, i, cluster_ids);
                    cat_cluster_replace_ids(--clusters, k, cluster_ids);
                }
            }
        }
    }
#endif

    return clusters;
}
#endif // 0

int cat_cluster_simple_density__(int *cluster_ids, const cat_sample32_t *samples,
                               u_int32_t max_diff_x, u_int32_t max_diff_y) {
    /* TODO! obj algo doesnt really work, since it doesnt merge clusters... */
    int clusters = 0;
    int i, j, new;
    
    /* initialize - clear all cluster IDs */
    memset(cluster_ids, 0, sizeof(int) * CAT_CONDENSATION_SAMPLE_COUNT);
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        if(cluster_ids[i]) {
            /* sample is already in cluster, skip */
            continue;
        }
        
        for(j = 0, new = 1; j < i; ++j) {
            /* now search for a cluster, the sample belongs to */
            /* TODO? faster than |xi - xj| < xmax && |yi - yj| < ymax? */
            if(!(abs(samples[i].x - samples[j].x) > max_diff_x ||
                 abs(samples[i].y - samples[j].y) > max_diff_y)) {
                cluster_ids[i] = cluster_ids[j];
                break;
            }
        }
        
        if(j == i) {
            /* sample does not belong to any cluster => it's one of its own */
            cluster_ids[i] = clusters++;
        }
    }    
    
    return clusters;
}

/**
 * get dimensions (x_min, y_min) - (x_max, y_max) of cluster
 * returns number of samples in cluster if cluster exists, PP_ERR otherwhise
 */
int cat_cluster_get_dimensions(int *x_min, int *y_min, int *x_max, int *y_max,
                               int cluster_id, const int *cluster_ids, 
                               const cat_sample32_t *samples) {
    u_int32_t lx_min = -1, ly_min = -1, lx_max = 0, ly_max = 0, i;
    int found;
#if !defined(NDEBUG)
    fb_format_info_t fb_info;

    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return PP_ERR;
    }
#endif /* NDEBUG */
    
    assert(cluster_ids);
    assert(samples);
    
    for(i = 0, found = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        if(cluster_ids[i] == cluster_id) {
            /* sample i belongs to cluster */
            if(samples[i].x > lx_max) {
                lx_max = samples[i].x;
//                assert(lx_max < fb_info.g_w);
            } 
            if(samples[i].x < lx_min) {
                lx_min = samples[i].x;
//                assert(lx_min < fb_info.g_w);
            }
            if(samples[i].y > ly_max) {
                ly_max = samples[i].y;
//                assert(ly_max < fb_info.g_h);
            }
            if(samples[i].y < ly_min) {
                ly_min = samples[i].y;
//                assert(ly_min < fb_info.g_h);
            }
            ++found;
        }
    }
    
    if(found) {
#if defined(CAT_DEBUG) && 0
        CDMSG(1, "found %d samples in cluster %d:\n", found, cluster_id);
        for(i = 0, found = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
            if(cluster_ids[i] == cluster_id) {
                CDMSG(1, "%4d: (%4d, %4d)\n", i, samples[i].x, samples[i].y);
            }
        }
        CDMSG(1, "cluster dimensions: (%4d, %4d) - (%4d, %4d)\n",
              lx_min, ly_min, lx_max, ly_max);
#endif /* CAT_DEBUG */

        *x_min = lx_min;
        *x_max = lx_max;
        *y_min = ly_min;
        *y_max = ly_max;
        
        assert(*x_min >= 0);
        assert(*x_max >= 0);
        assert(*y_min >= 0);
        assert(*y_max >= 0);
       
        return found;
    }
    
    return PP_ERR;
}

/**
 * get size of cluster
 */
int cat_cluster_get_size(int cluster_id, const int *cluster_ids) {
    int count, i;
    
    assert(cluster_ids);
    
    for(i = 0, count = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        if(cluster_ids[i] == cluster_id) {
            /* sample i belongs to cluster */
            ++count;
        }
    }
    
    return count;
}

/**
 * alligns cluster dimensions to given image size max_x, max_y
 */
int cat_cluster_align_dimensions(u_int32_t *left, u_int32_t *top, 
                                 u_int32_t *width, u_int32_t *height,
                                 u_int32_t x_min, u_int32_t y_min, 
                                 u_int32_t x_max, u_int32_t y_max,
                                 u_int32_t max_diff_x, u_int32_t max_diff_y,
                                 u_int32_t max_x, u_int32_t max_y) {
    int s_left, s_top, s_width, s_height;

    assert(left);
    assert(top);
    assert(width);
    assert(height);
    
    s_left = x_min - max_diff_x;
    s_top = y_min - max_diff_y;
    s_width = x_max - s_left + max_diff_x;
    s_height = y_max - s_top + max_diff_y;
//printf("tweb: s_left = %d, s_top = %d, s_width = %d, s_height = %d\n", s_left, s_top, s_width, s_height);usleep(11111);
    
    /* align to dimensions */
    *left = s_left > 0 ? s_left : 0;
    *top = s_top > 0 ? s_top : 0;
    *width = s_left + s_width < (int)max_x ? s_width : (int)max_x - s_left - 1;
    *height = s_top + s_height < (int)max_y ? s_height : (int)max_y - s_top - 1;
//printf("tweb: left = %d, top = %d, width = %d (%d), height = %d (%d)\n", *left, *top, *width, s_left + s_width, *height, s_top + s_height);usleep(11111);

//printf("tweb: max_diff_x = %d, max_diff_y = %d, max_x = %d, max_y = %d\n", max_diff_x, max_diff_y, max_x, max_y);usleep(11111);
    CDMSG(0, "aligned (%d, %d)-(%d, %d) to (%d, %d)-(%d, %d)\n",
          x_min, y_min, x_max, y_max,
          *left, *top, *left + *width, *top + *height);
              
    return PP_SUC;
}

int cat_cluster_get_search_space_for_cluster(u_int32_t *left, 
                                             u_int32_t *top, 
                                             u_int32_t *width, 
                                             u_int32_t *height,
                                             int cluster_id, 
                                             const int *cluster_ids, 
                                             const cat_sample32_t *samples,
                                             u_int32_t shape_width, 
                                             u_int32_t shape_height,
                                             u_int32_t grow_x, 
                                             u_int32_t grow_y,
                                             u_int32_t max_width, 
                                             u_int32_t max_height) {
    int x_min, y_min, x_max, y_max, count, tmp;

    /* get sample cluster dimensions */
    if(PP_ERR == (count = cat_cluster_get_dimensions(&x_min, &y_min, 
                                                     &x_max, &y_max,
                                                     cluster_id, cluster_ids,
                                                     samples))) {
        return PP_ERR;
    }

//printf("cat_cluster_get_search_space_for_cluster: cluster dim; x_min = %d, y_min = %d, x_max = %d, y_max = %d\n", x_min, y_min, x_max, y_max);usleep(11111);
    /* grow by mouse shape */
    tmp = max_width - shape_width;
    if(x_min > tmp) {
        x_min = tmp;
        x_max = max_width;
    } else {
        x_max += shape_width;
    }
    
    tmp = max_height - shape_height;
    if(y_min > tmp) {
        y_min = tmp;
        y_max = max_height;
    } else {
        y_max += shape_height;
    }
//printf("cat_cluster_get_search_space_for_cluster: left = %d, top = %d, right = %d, bottom = %d, width = %d, height = %d, shape_width = %d, shape_height = %d, max_width = %d, max_height = %d\n", x_min, y_min, x_max, y_max, x_max-x_min, y_max-y_min, shape_width, shape_height, max_width, max_height);usleep(11111);
    
    /* grow by one hex_tile and align to screen */
    if(PP_ERR == cat_cluster_align_dimensions(left, top, width, height,
                                              x_min, y_min, x_max, y_max,
                                              grow_x, grow_y,
                                              max_width, max_height)){
        return PP_ERR;
    }
//printf("cat_cluster_get_search_space_for_cluster: left = %d, top = %d, width = %d, height = %d\n", *left, *top, *width, *height);usleep(11111);

    return count;
}

