/******************************************************************************\
* cat_debug.c                                                                  *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* ... sometimes even rock-solid code needs debugging ;-)                       *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <pp/base.h>

#include <cat_internal.h>

#include <cat_debug.h>

#if defined(CAT_DEBUG)

void cat_debug_print_diffmap(u_int32_t *diffmap, size_t size, 
                             u_int line_len, u_int width, u_int height) {
    u_int x, y, i, offset, offset_width, line_len32, width32;

    assert(width);
    assert(height);
    assert(width <= line_len);
    assert(height * width <= size * 8);

    printf("\x1b[H\n"); // go to first line of console
    printf("0x%p\n", (u_char*)diffmap);
    printf("%d x %d tiles\n", width, height);
    line_len32 = OHDIV(line_len, 32);
    width32 = OHDIV(width, 32);
    printf("           01234567890123456789012345678901234567890123456789"
           "01234567890123");
    for (y = 0; y < height; y++) {
        printf("\x1b[K\n"); // clear lind from cursor right
        printf("line %04u: ", y);
        for (x = 0, offset_width = 32; x < width32; x++) {
            offset = y * line_len32 + x;
            if(x == (width32 - 1)) {
                /* last u_int32_t in line, crop */
                offset_width = width - x * 32;
//printf("tweb: offset_width is %d, width is %d\n", offset_width, width);
            }
//printf("tweb: offset = %d, y = %d, line_len = %d, x = %d, size = %d\n", offset, y, line_len, x, size);
            assert(offset < size * 4 - 3);
            for (i = 0; i < offset_width; ++i) {
                printf("%d", diffmap[offset] >> i & 0x1);
            }
        }
        usleep(10000);
    }
    printf("\n           01234567890123456789012345678901234567890123456789"
           "01234567890123\n");
}

void cat_debug_draw_diffmap(u_int32_t *diffmap, size_t size, 
                            u_int line_len, u_int width, u_int height,
                            t_bitmap *img, t_rgb color) {
    u_int x, y, i, offset, offset_width, line_len32, width32, px, py;

    assert(width);
    assert(height);
    assert(width <= line_len);
    assert(height * width <= size * 8);
    assert(img);
    assert(img->rgb);

    line_len32 = OHDIV(line_len, 32);
    width32 = OHDIV(width, 32);
    for (y = 0; y < height; y++) {
        py = y * PP_FB_TILE_HEIGHT;
        assert(py < img->height);
        for (x = 0, offset_width = 32; x < width32; x++) {
            offset = y * line_len32 + x;
            if(x == (width32 - 1)) {
                /* last u_int32_t in line, crop */
                offset_width = width - x * 32;
            }
            assert(offset < size * 4 - 3);
            for (i = 0; i < offset_width; ++i) {
                if(diffmap[offset] >> i & 0x1) {
                    int j, pxm, pym;
                    
                    px = (x * 32 + i) * PP_FB_TILE_WIDTH;
                    assert(px < img->width);
                    
                    pxm = SET_WITHIN(px + PP_FB_TILE_WIDTH, px, img->width);
                    pym = SET_WITHIN(py + PP_FB_TILE_HEIGHT, py, img->height);
                    
                    for(j = px; j < pxm; ++j) {
                        img->rgb[py * img->width + j] = color;
                    }
                    for(j = py; j < pym; ++j) {
                        img->rgb[j * img->width + px] = color;
                    }
                }
            }
        }
    }
}

int cat_debug_draw_rect(u_int x1, u_int y1, u_int x2, u_int y2,
                        t_bitmap *img, t_rgb color) {
    u_int i;
    
    if(x1 >= img->width || x2 >= img->width || 
       y1 >= img->height || y2 >= img->height) {
        return PP_ERR;
    }

    /* swap if needed */
    if(x1 > x2) {
        i = x1;
        x1 = x2;
        x2 = i;
    }
    if(y1 > y2) {
        i = y1;
        y1 = y2;
        y2 = i;
    }

    for(i = x1; i <= x2; ++i) {
        img->rgb[y1 * img->width + i] = color;
        img->rgb[y2 * img->width + i] = color;
    }
    
    for(i = y1; i <= y2; ++i) {
        img->rgb[i * img->width + x1] = color;
        img->rgb[i * img->width + x2] = color;
    }
    
    return PP_SUC;
}

int cat_debug_clustering_draw_clusters(u_int clusters, int *cluster_ids,
                                       cat_sample32_t *samples,
                                       t_bitmap *img, t_rgb color) {
    u_int i;
    
    assert(cluster_ids);
    assert(samples);
    assert(img);
    assert(img->rgb);
    
    for(i = 0; i < clusters; ++i) {
        /* draw cluster boundaries */ 
        
        int x_min, y_min, x_max, y_max;
        
        if(PP_ERR == cat_cluster_get_dimensions(&x_min, &y_min, &x_max, &y_max,
                                                i, cluster_ids, samples)) {
            break;
        }
        
        cat_debug_draw_rect(x_min, y_min, x_max, y_max, img, color);
    }
    
    return PP_SUC;
}

int cat_debug_draw_samples(cat_sample32_t *samples, 
                           t_bitmap *img, t_rgb color) {
    u_int i;
    
    assert(samples);
    assert(img);
    assert(img->rgb);
    
    for(i = 0; i < CAT_CONDENSATION_SAMPLE_COUNT; ++i) {
        if(samples[i].x >= img->width || samples[i].y >= img->height) {
            return PP_ERR;
        }
        img->rgb[samples[i].y * img->width + samples[i].x] = color;
    }
    
    return PP_SUC;
}

#endif /* CAT_DEBUG */
