/******************************************************************************\
* cat_diffmap.c                                                                *
*                                                                              *
* Some methods to deal with diffmaps on contest of mouse cursor tracking (CAT) *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include "cat_debug.h"
#include "cat_internal.h"

#if 1 // new
int cat_diffmap_initialize_sample(cat_sample32_t *sample,
                                  const cat_queue_diffmap_entry_t *cqde,
                                  const fb_format_info_t *fb_info) {
    u_int32_t count, idx, line_len32, width32, size32;
    static const u_int32_t max = 100;
    
    assert(sample);
    assert(cqde);
    assert(cqde->data);
    assert(fb_info);

    size32 = OHDIV(cqde->size, 4);
    line_len32 = OHDIV(cqde->line_len, 32);
    width32 = OHDIV(cqde->width, 32);
    
    for(count = 0; count < max; ++count) {
        idx = random() % size32; // get a random u_int32_t of diffmap
        if(cqde->data[idx]) {
            u_int32_t offset_width = 32;
            u_int32_t x = idx % line_len32;
            
            if(x == (width32 - 1)) {
                /* last u_int32_t in line, crop */
                offset_width = cqde->width - x * 32;
            }
            
            for(count = 0; count < max; ++count) {
                u_int32_t mask = 1 << random() % offset_width;
                
                if(cqde->data[idx] & mask) {
                    cat_diffmap_get_abs_coord(&sample->x, &sample->y,
                                                  idx, mask, cqde->line_len, 
                                                  fb_info);
                    sample->p = 0 /* p_min */;
                    
                    return PP_SUC;
                }
            }
            
            /* sample not set, return error code */
            return PP_ERR;
        }
    }
    
    /* sample not set, return error code */
    return PP_ERR;
}
#else // old!
int cat_diffmap_initialize_sample(cat_sample32_t *sample,
                                  const cat_queue_diffmap_entry_t *cqde,
                                  const fb_format_info_t *fb_info) {
    u_char count;
    u_int16_t idx;
    
    assert(sample);
    assert(cqde);
    assert(cqde->data);
    assert(fb_info);

    for(count = 0; count < 100; ++count) {
        idx = random() % cqde->size; // get a random _byte_!!! of diffmap
        if(((u_char*)cqde->data)[idx]) {
            u_char offset;
            u_int32_t mask;
            
            /* byte idx -> u_int32_t idx mapping, thats simple: */
            offset = idx / 4;
            
            /* get least significant set bit and convert to u_int32_t mask:
             * - calc offset of idx byte in offset u_int32_t:
             *   idx - (offset * 4)
             * - note, we are on PPC, care about byte ordering
             *   3 - (idx - (offset * 4))
             * - multiply with 8 bits in byte:
             *   (3 - (idx - (offset * 4))) * 8
             * - add offset of least significant bit set it in idx byte:
             *   (3 - (idx - (offset * 4))) * 8 +
             *   ffs(((u_char*)cqde->data)[idx]) - 1
             * - shift by that... thats the magic...
             *   1 << ((3 - (idx - (offset * 4))) * 8 +
             *         ffs(((u_char*)cqde->data)[idx]) - 1)
             */
            mask = 1 << ((3 - (idx - (offset * 4))) * 8 +
                         ffs(((u_char*)cqde->data)[idx]) - 1);
            
            assert(cqde->data[offset] & mask); // assert mask is correct ;-)
            
            cat_diffmap_get_abs_coord(&sample->x, &sample->y,
                                          offset, mask, cqde->line_len, 
                                          fb_info);
            sample->p = 0 /* p_min */;
            
            return PP_SUC;
        }
    }
    
    /* sample not set, return error code */
    return PP_ERR;
}
#endif

static int initialize_sample_biased_core(cat_sample32_t *sample,
                                         int bias_x, int bias_y,
                                         const cat_queue_diffmap_entry_t *cqde,
                                         const fb_format_info_t *fb_info) {
    u_char idx;
    u_int32_t mask;
    
    assert(sample);
    assert(cqde);
    assert(cqde->data);
    assert(fb_info);

    if(bias_x >= 0 && bias_y >= 0 &&
       bias_x < (int)fb_info->g_w && bias_y < (int)fb_info->g_h &&
       cat_diffmap_get_offset_from_abs_coord(&idx, &mask, cqde->line_len,
                                             bias_x, bias_y, 
                                             fb_info) == PP_SUC &&
       (cqde->data[idx] & mask)) {
        sample->x = bias_x;
        sample->y = bias_y;
        return PP_SUC;
    }
        
    return PP_ERR;
}

int cat_diffmap_initialize_sample_biased(cat_sample32_t *sample,
                                         int bias_x, int bias_y,
                                         const cat_queue_diffmap_entry_t *cqde,
                                         const fb_format_info_t *fb_info) {
    u_int32_t u;
    int ret;
    
    assert(sample);
    assert(cqde);
    assert(cqde->data);
    assert(fb_info);
    
    for(u = 0; u < cqde->size; ++u) {
        if(((u_char*)cqde->data)[u]) {
            /* at least one bit in diffmap is set */
            break;
        }
    }
    
    if(u >= cqde->size || bias_x < 0 || bias_y < 0 ||
       bias_x >= (int)fb_info->g_w || bias_y >= (int)fb_info->g_h) {
        return PP_ERR;
    }
    
    if((ret = initialize_sample_biased_core(sample, bias_x, bias_y, 
                                            cqde, fb_info)) == PP_ERR) {
        /**
         * no changes at biased position... seach for nearest diff around
         *
         * four neighbouring spiral search:
         *           17
         *        16  8 18                       
         *     15  7  3  9 19                    
         *  14  6  2  1  4 10 20              
         *    ... 13  5 11 21                    
         *        24 12 22
         *           23
         */
        
        int start_x, x, y, inc_x, inc_y;
        
        start_x = x = bias_x;
        y = bias_y;
        inc_x = -PP_FB_TILE_WIDTH;
        inc_y = -PP_FB_TILE_HEIGHT;
        
        while(ret == PP_ERR) {
            if(x == start_x) {
                /* we are at start position */
                start_x = x = x + inc_x;
                assert((bias_x - x) < 5000);
            }

            ret = initialize_sample_biased_core(sample, x, y, cqde, fb_info);
            
            if(x == bias_x) {
                inc_y = -inc_y;
            } else if(y == bias_y) {
                inc_x = -inc_x;
            }
            x += inc_x;
            y += inc_y;
        }
    }
    
    CDMSG(CAT_CDMSG_DIFFMAP_INFO, 
          "bias (%4d; %3d) => init (%4d, %3d), diff (%4d, %3d)\n",
          bias_x, bias_y, sample->x, sample->y, 
          sample->x - bias_x, sample->y - bias_y);

    return ret;
}

int cat_diffmap_get_abs_coord(u_int32_t *x, u_int32_t *y,
                              u_char idx, u_int32_t mask, u_char line_len,
                              const fb_format_info_t *fb_info) {
    u_char line, column, len;
    
    assert(x);
    assert(y);
    assert(mask); // at least one bit has to be set!
    assert(fb_info);
    
    len = OHDIV(line_len, 32);
    line = idx / len;
    column = (idx % len) * 32 + ffs(mask) - 1;
    
    /* use upper left corner of hextile instead of center
       does not work for all resolutions (e.g. 800x600!) */
    *x = column * PP_FB_TILE_WIDTH /* + PP_FB_TILE_WIDTH / 2 */;
    *y = line * PP_FB_TILE_HEIGHT /* + PP_FB_TILE_HEIGHT / 2 */;

//printf("tweb: line_len = %u, len = %u, idx = %u, line = %u, column = %u, x = %d(%d), y = %d(%d)\n", line_len, len, idx, line, column, *x, fb_info->g_w, *y, fb_info->g_h); usleep(22222);
    
    return PP_SUC;
}

int cat_diffmap_get_coord_from_abs_coord(u_char *dmx, u_char *dmy,
                                         u_int32_t fbx, u_int32_t fby,
                                         const fb_format_info_t *fb_info) {
    /* calculate diffmap coordinates from absolute cordinates */
    assert(dmx);
    assert(dmy);
    assert(fb_info);
    
    if(fbx < fb_info->g_w && fby < fb_info->g_h) {
        *dmy = fby / PP_FB_TILE_HEIGHT;
        *dmx = fbx / PP_FB_TILE_WIDTH;
        return PP_SUC;
    }
    
    return PP_ERR;
}

int cat_diffmap_get_offset_from_diffmap_coord(u_char *idx, u_int32_t *mask,
                                              u_char line_len, 
                                              u_char dmx, u_char dmy,
                                              const fb_format_info_t *fb_info) {
    /* calculate diffmap offset from absolute cordinates */
    u_int32_t offset;
    
    assert(idx);
    assert(mask);
    assert(fb_info);

    if(dmx < fb_info->tiles_w && dmy < fb_info->tiles_h) {
        offset = dmy * line_len + dmx;
        
        *idx = offset / 32;
        *mask = 1 << (offset % 32);
        
        return PP_SUC;
    }
    return PP_ERR;
}

int cat_diffmap_get_offset_from_abs_coord(u_char *idx, u_int32_t *mask,
                                          u_char line_len,
                                          u_int32_t x, u_int32_t y,
                                          const fb_format_info_t *fb_info) {
    /* calculate diffmap offset from absolute cordinates */
    u_char dmx, dmy;
    
    cat_diffmap_get_coord_from_abs_coord(&dmx, &dmy, x,  y, fb_info);
    cat_diffmap_get_offset_from_diffmap_coord(idx, mask, 
                                              line_len, dmx, dmy, fb_info);
    
    return PP_SUC;
}

int cat_get_merged_diffmap(cat_queue_diffmap_entry_t *cqde,
                           cat_queue_diffmap_entry_t *cqdle,
                           cat_queue_t *diffmap_queue) {
    u_char i;
    cat_queue_entry_t *cqe;
    u_int32_t cqe_size, cqe_line_len, cqe_width, cqe_height, max_x, y_offs;
    u_int32_t *diffmap, *last_diffmap = NULL;
    
    assert(cqde);
    assert(diffmap_queue);
    
    cat_queue_lock(diffmap_queue);
    if(NULL == (cqe = cat_queue_dequeue(diffmap_queue))) {
        /* there are no entries in diffmap queue, return */
        CDMSG(1, "nothing to dequeue!\n");
        goto error;
    }
    
    assert(cqe->type == CAT_QUEUE_DIFFMAP_ENTRY);
    assert(cqe->data.diffmap.data);
//    DEBUG_CQDE(cqe->data.diffmap);
   
    cqe_size = cqe->data.diffmap.size;
    cqe_line_len = cqe->data.diffmap.line_len;
    cqe_width = cqe->data.diffmap.width;
    cqe_height = cqe->data.diffmap.height;
    
    /* init diffmap to first entry dequed */
    diffmap = (u_int32_t*)malloc(cqe_size);
    memcpy(diffmap, cqe->data.diffmap.data, cqe_size);

    if(cqdle) {
        /* init last_diffmap to first entry dequed */
        last_diffmap = (u_int32_t*)malloc(cqe_size);
        memcpy(last_diffmap, cqe->data.diffmap.data, cqe_size);
    }
    
    cat_queue_destroy_entry(cqe);

    max_x = OHDIV(cqe_width, 32);
    y_offs = OHDIV(cqe_line_len, 32);

    for(i = 1; NULL != (cqe = cat_queue_dequeue(diffmap_queue)); ++i) {
        u_int32_t x, y, offset;
        
        /* paranoia... */
        assert(cqe->type == CAT_QUEUE_DIFFMAP_ENTRY);
        assert(cqe->data.diffmap.data);
        
        /* TODO! auto cleanup on resolution change should solve obj... */
        if(cqe->data.diffmap.size != cqe_size ||
           cqe->data.diffmap.line_len != cqe_line_len ||
           cqe->data.diffmap.width != cqe_width ||
           cqe->data.diffmap.height != cqe_height) {
            CDMSG(1, "resolution changed, clearing queue\n");
            cat_queue_clear(diffmap_queue);
            cat_queue_destroy_entry(cqe);
            goto error;
        }
        
//        DEBUG_CQDE(cqe->data.diffmap);
        
        for(y = 0; y < cqe_height; ++y) {
            for(x = 0; x < max_x; ++x) {
                offset = y * y_offs + x;
                diffmap[offset] |= cqe->data.diffmap.data[offset];
            }
        }

        if(cqdle && cat_queue_is_empty(diffmap_queue)) {
            /* set last_diffmap */
            memcpy(last_diffmap, cqe->data.diffmap.data, cqe_size);
        }
        cat_queue_destroy_entry(cqe);
    }
    cat_queue_unlock(diffmap_queue);
    
//    CDMSG(1, "read %u entries from diffmap queue\n", i);
    
    cqde->data = diffmap;
    cqde->size = cqe_size;
    cqde->line_len = cqe_line_len;
    cqde->width = cqe_width;
    cqde->height = cqe_height;
//    DEBUG_CQDE(*cqde);
    
    if(cqdle) {
        cqdle->data = last_diffmap;
        cqdle->size = cqe_size;
        cqdle->line_len = cqe_line_len;
        cqdle->width = cqe_width;
        cqdle->height = cqe_height;
    }
    
    return PP_SUC;
 error:
    cat_queue_unlock(diffmap_queue);
    
    return PP_ERR;
}



