/* layer: comm_proto_pppp
   data path: send_pdu() -> FD
*/

/* system includes */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <malloc.h>
#include <sched.h>

/* firmware includes */
#include <pp_gpio.h>
#include <gpio_hilevel_functions.h>
#include <pp/base.h>
#include <liberic_misc.h>
#include <liberic_pthread.h>

/* local includes */
#include "driver.h"
#include "kmebase_int.h"
#include "comm_proto_pppp.h"

/* object functions */
static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int send_pdu(driver_t*, const void* pdu, const int length, const input_t input);
static int suspend(driver_t*);
static int resume(driver_t*);
static int ping(driver_t*, char** info);
static int rw_pin (driver_t* obj, int is_write, int pin, int *val);
static int is_equal(driver_t* obj, driver_t* d);
static int reset_driver(driver_t* obj);
static int send_cmd_ack(driver_t* obj);

/* local object functions */
static void* kme_thread_func(void * arg);
static int kme_send(driver_t* obj, const char* s, int len, const unsigned long timeout);
static int get_kme_version(driver_t* obj, char* version, int len);
static int examine_response(char* line);

int init_comm_proto_pppp(driver_t* obj, int reset_pin) {
    /* create the driver */
    comm_proto_data_t* context;
    pthread_mutexattr_t attr;
    int j,r;
    obj->comm_proto.is_equal = is_equal;
    obj->comm_proto.cleanup = cleanup;
    obj->comm_proto.reconfigure = reconfigure;
    obj->comm_proto.send_pdu = send_pdu;
    obj->comm_proto.suspend = suspend;
    obj->comm_proto.resume = resume;
    obj->comm_proto.ping = ping;
    obj->comm_proto.rw_pin = rw_pin;
    obj->comm_proto.reset_driver = reset_driver;
    obj->comm_proto.send_cmd_ack = send_cmd_ack;
    
    /* initializes our private data */
    obj->comm_proto.data = malloc(sizeof(comm_proto_data_t));
    if (obj->comm_proto.data == NULL) {
	exit(1);
    }
    context = (comm_proto_data_t*)obj->comm_proto.data;
    /* TODO TIMEOUT */
    context->timeout = 500000L;
    context->reset_pin = reset_pin;
    
    /* init kme receive threads and contexts for all ports */
    pthread_cond_init(&context->kme_wait_cond, NULL);
    pthread_cond_init(&context->kme_prompt_cond, NULL);
    pthread_cond_init(&context->response_buffer_cond, NULL);
    pthread_mutex_init(&context->kme_prompt_cond_mtx, NULL);
    pthread_mutex_init(&context->response_buffer_list_mtx, NULL);
    pthread_mutex_init(&context->response_buffer_cond_mtx, NULL);

    if ((r = pthread_mutexattr_init(&attr)) != 0) {
	pp_log("Cannot initialize thread attributes (%s).\n",
		 strerror(r));
	return -1;
    }
    if ((r = pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK_NP)) != 0) {
	pp_log("Cannot initialize thread attributes (%s).\n",
	       strerror(r));
	return -1;
    }
    pthread_mutex_init(&context->kme_suspend_cond_mtx, &attr);
    pthread_mutex_init(&context->kme_resume_cond_mtx, &attr);
    pthread_mutexattr_destroy(&attr);
    pthread_cond_init(&context->kme_suspend_cond, NULL);
    pthread_cond_init(&context->kme_resume_cond, NULL);
    
    context->kme_prompt_wait = KME_PROMPT_RECEIVED;
    for (j = 0; j < KME_RESPONSE_MAX; j++) {
	context->buffer_list[j].buf = NULL;
    }
    context->kmefd = -1;
    
    context->kme_suspend_flag = KME_STOPPED;
    return 0;
}

static int cleanup(driver_t* obj ) {

    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    /* only if suspended - so the fd is already closed */
    MUTEX_LOCK(&context->kme_suspend_cond_mtx);
    assert(context->kme_suspend_flag == KME_SUSPENDED ||
	   context->kme_suspend_flag == KME_STOPPED);
    if (context->kme_suspend_flag != KME_STOPPED) {
	/* thread is running - stop it */
	context->kme_suspend_flag = KME_EXIT;
	pthread_cond_signal(&context->kme_suspend_cond);
	MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
	
	pthread_join(context->kme_thread, NULL);
	MUTEX_LOCK(&context->kme_suspend_cond_mtx);
	context->kme_suspend_flag = KME_STOPPED;
	MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
    } else {
	MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
    }
    KD("cleanup done\n");
    return 0;
}

	   

static int reconfigure(driver_t * /* obj */) {
    KD("comm_proto_reconfigure\n");
    return 0;
}
    
/**
 * checks whether kme is alive and responding 
 * returns true if yes, false otherwise
 * if info contains a valid pointer some descriptive
 * information will be written there
 */
static int ping(driver_t* obj, char** info) {
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
#define KMPBS 256
    int ret;
    const char* alive = "\n";
    const char* knal  = "no response";
    char v[KMPBS];

    ret = kme_send(obj, alive, KME_SEND_STR, context->timeout);
    if(ret) {
	/* kme_send returned != 0 */
	if(info) *info = strdup(knal);
    } else {
	/* kme_send successful */
	get_kme_version(obj, v, KMPBS);
	if(info) *info = strdup(v);
    }

    return !ret;
}


static int send_pdu(driver_t* obj, const void* pdu, const int length, const input_t input) {
    char buf[100], inpchar;
    int i, retry_cnt, retry_barrier, ret = 0;
    unsigned long retry_delay;
    const char* logfmt = "kme_send: %c-cmd failed, error %d%s\n";
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    
    if (input == PP_KM_INPUT_KEYBOARD) inpchar = 'K';
    else if (input == PP_KM_INPUT_MOUSE) inpchar = 'M';
    else return -1;

    buf[0] = inpchar;
    for (i = 0; i < length; i++) {
	sprintf(&buf[i*2+1], "%02x", ((const char *)pdu)[i]);
    }
    sprintf(&buf[i*2+1], "\r\n");

    /*
     * there are some types of kme errors after which we'll try to redo
     * the thing right here, for some time, as there is justified hope i
     * that we might succeed those next times:
     *
     * KME_MDE_NOT_INITIALIZED:
     * KME_KDE_NOT_INITIALIZED: kme dev emu is initializing (BAT)
     * 
     * KME_MDE_HOSTPROCESSING:  
     * KME_KDE_HOSTPROCESSING:  kme dev emu is bussy with host cmds
     * 
     * KME_CDP_UKNOWN_CMD:
     * KME_CDP_KBD_CMD_WRONG:
     * KME_CDP_MOUSE_CMD_WRONG: wrong cmds can't actually happen,
     *    however, I've seen it, once or twice. Communication errors
     *    might be a possible explanation, so we will try again, one time,
     *    in order to handle obj possibility.
     */
    retry_cnt = 0;
    while ((ret = kme_send(obj, buf, KME_SEND_STR, context->timeout)) != 0) {

	if (ret == KME_KDE_NOT_INITIALIZED || ret == KME_MDE_NOT_INITIALIZED) {
	    retry_delay = 200000;
	    retry_barrier = 3;
	} else if (ret == KME_KDE_HOSTPROCESSING ||
		   ret == KME_MDE_HOSTPROCESSING) {
	    retry_delay = 10000;
	    retry_barrier = 5;
	} else if (ret == KME_CDP_UKNOWN_CMD || ret == KME_CDP_KBD_CMD_WRONG ||
		   ret == KME_CDP_MOUSE_CMD_WRONG) {
	    retry_delay = 1000;
	    retry_barrier = 1;
	} else {
	    pp_log(logfmt, inpchar, ret, "");
	    break;
	}
	if (++retry_cnt > retry_barrier) {
	    pp_log(logfmt, inpchar, ret, ", giving up!");
	    break;
	} else {
	    pp_log(logfmt, inpchar, ret, ", retrying!");
	}
	usleep(retry_delay); // sleep a sensible little bit and try again
    }
    return ret;
}

/**
 * kme_send tries to write the given string to the kme.
 * this is done by writing it to the kme
 * descriptor and waiting for a prompt. Other characters 
 * from the kme are allowed, however they will be ignored
 * unless a prompt is seen.
 * In case the prompt wont be seen within a specified amount
 * of time (given in kme_init) the function returns with a 
 * timeout.
 * on success 0 will be returned
 * on system error -1 will be returned and an errno will be set
 * on kme error the return code will be the kme error code
 *     which is > 0
 */
int kme_send(driver_t* obj, 
	     const char* s, int len,
	     const unsigned long timeout) {
    int ret = 0, slen, waitret;
    struct timeval now;
    struct timespec to;
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;

    if (s == NULL) return 0;

    if (len < 0) {
	slen = strlen(s);
    } else {
	slen = len;
    }
    
    MUTEX_LOCK(&context->kme_prompt_cond_mtx);

    if (0 != (ret = context->send_data(context, slen, s)))
	goto bailout;

    /* use the condition variable to wait for the prompt */
    context->kme_prompt_error = KME_NO_ERROR;
    context->kme_prompt_wait = KME_PROMPT_WAITING;
    
    while (context->kme_prompt_wait == KME_PROMPT_WAITING) {
	unsigned long tmpto_usecs;
	gettimeofday(&now, NULL);
	to.tv_sec = now.tv_sec;
	if((tmpto_usecs = now.tv_usec + timeout) > 1000000) {
	    to.tv_sec++;
	    tmpto_usecs -= 1000000;
	}
	to.tv_nsec = tmpto_usecs * 1000;

	waitret = pthread_cond_timedwait(&context->kme_prompt_cond,
					 &context->kme_prompt_cond_mtx,
					 &to);
	//	pp_hrtime_stop(&ppsw);
	//	printf(" waited for respond %llu ms\n", pp_hrtime_read(&ppsw) / (1000*1000) );
	if (waitret == ETIMEDOUT) {
	    printf("signal timeout in kme_send\n");
	    errno = ETIMEDOUT;
	    ret = -1;
	    context->kme_prompt_wait = KME_PROMPT_RECEIVED;
	    goto bailout;
	}
    }
    ret = context->kme_prompt_error; // this is either KME_NO_ERROR, if left
                                     // untouched, or the prompt error
                                     // seen by the kme thread
    
 bailout:
    MUTEX_UNLOCK(&context->kme_prompt_cond_mtx);
    return ret;
}

/**
 * stops kme read thread temporalily
 *
 * ATTENTION: don't use obj function simultaneously
 *            by mutliple threads! It's not thread safe, so to say!
 */

static int suspend (driver_t* obj) {
    /* stop the receive thread */
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    KD("comm_proto_suspend\n");


    MUTEX_LOCK(&context->kme_suspend_cond_mtx);
    context->kme_suspend_flag = KME_SUSPEND_REQ;
    while (context->kme_suspend_flag != KME_SUSPENDED) {
	pthread_cond_wait(&context->kme_suspend_cond, &context->kme_suspend_cond_mtx);	
    }
    MUTEX_UNLOCK(&context->kme_suspend_cond_mtx); 

    obj->target_helper.close_fd(obj);
    context->kmefd = -1;
	
    return 0;
}

static int resume (driver_t* obj) {
    /* recheck, if the received thread isn't already running */
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    int i;
    
    i = obj->target_helper.get_open_fd(obj);
    context->kmefd = i;
    if (context->kmefd < 0) {
	return -1;
    }

    MUTEX_LOCK(&context->kme_suspend_cond_mtx);
    if (context->kme_suspend_flag == KME_STOPPED) {
	/* start the thread if it is not already running */
	if (eric_pthread_create(&context->kme_thread, 0, 65536,
				kme_thread_func, obj)) {
	    pp_log("eric_kmebase_init(): pthread_create() failed.");
	    MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
	    return -1;
	}
	context->kme_suspend_flag = KME_RUNNING;
	MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
    } else {
	context->kme_suspend_flag = KME_RESUME;
	pthread_cond_signal(&context->kme_suspend_cond);
	MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
    }
    
    kme_send(obj, "L\r\n", KME_SEND_STR, ((comm_proto_data_t*)obj->comm_proto.data)->timeout);
    
    return 0;
}


static int is_equal(driver_t* obj, driver_t* d) {
    if (!strcmp(obj->comm_proto.id, d->comm_proto.id)) {
	return 1;
    } else {
	return 0;
    }
}


/**
 * kme_thread_func reads data from kme and
 * does some appropriate stuff
 */
static void*
kme_thread_func(void * arg) {
    /* we define, that the protocol is line oriented,
       so we have to read as much as we can, and then
       supply the data line by line - if there is no
       complete line, just wait for more data */

    driver_t * obj = (driver_t*) arg;
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    
    unsigned char ringbuf[250];
    unsigned char buf[MAX_BUF_LEN];
    int maxlen = 250;
    int startptr, endptr, lineptr, eolptr;
    unsigned int kme_error = KME_NO_ERROR;
    int nn, kme_resp;
    //    struct timeval to;
    //int ret;
    //    pp_stopwatch_t ppsw2;

    KD("start kme thread func\n");
    startptr = 0;
    endptr = 0;
    lineptr = -1;
    eolptr = -1;

    while (1) {
	    int r;
	    MUTEX_LOCK(&context->kme_suspend_cond_mtx);
	    if (context->kme_suspend_flag == KME_EXIT) {
		MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
		break;
	    }
	    if (context->kme_suspend_flag == KME_SUSPEND_REQ) {
		context->kme_suspend_flag = KME_SUSPENDED;
		pthread_cond_signal(&context->kme_suspend_cond);
		while (context->kme_suspend_flag == KME_SUSPENDED) {
		    pthread_cond_wait(&context->kme_suspend_cond,
				      &context->kme_suspend_cond_mtx);
		}
		MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
		continue;
	    }
	    MUTEX_UNLOCK(&context->kme_suspend_cond_mtx);
	    
	    r = context->read_data(context, &nn, (char*)buf);
	    if ((r == 0)||(r==1)) continue;

	    buf[nn] = '\0';   // terminate buf (just for the debug)

	    /* Uncomment for kme debug */
#ifdef PP_KM_KME_DEBUG
	    printf("%s", buf);
#endif /* PP_KM_KME_DEBUG */

	    if ( ((startptr <= endptr) && (endptr-startptr+nn > maxlen)) ||
		 ((endptr < startptr) && (startptr-endptr < nn)) ) {
		/* read to much for the ring buffer */
		/* discard the old stuff */
		//printf("ringbuffer overflow\n");
		startptr = endptr = 0;
		continue;
	    }

	    /* copy readbuf into ring buffer */
	    //printf("debug: ep: %d, nn: %d, maxlen: %d\n", endptr, nn, maxlen);
	    if (endptr+nn >= maxlen) {
		memcpy(&ringbuf[endptr], buf, maxlen-endptr);
		memcpy(&ringbuf[0], &buf[maxlen-endptr], nn-(maxlen-endptr));
		endptr = nn-(maxlen-endptr);
	    } else {
		memcpy(&ringbuf[endptr], buf, nn);
		endptr += nn;
	    }

	    lineptr = startptr;
	    eolptr = startptr;

	    while(((startptr <= endptr) && startptr <= eolptr && eolptr < endptr) ||
		  ((endptr < startptr) && (startptr <= eolptr || eolptr < endptr))) {

		//printf("THREAD: start while1: startpt: %d, endptr: %d, lineptr: %d, eolptr: %d\n", startptr, endptr, lineptr, eolptr);
		if (ringbuf[eolptr++] != '\n') {
		    if (eolptr >= maxlen) {
			eolptr = 0;
		    }
		    continue;
		}
		if (eolptr >= maxlen) {
		    eolptr = 0;
		}
		
		/* found a newline, copy out string, reuse buf */
		//printf("found newline from %d to %d\n", lineptr, eolptr);
		if (lineptr <= eolptr) {
		    // ----S------L*****O------E---- , Lineptr <= eOlptr 
		    if (eolptr-lineptr < 3) {
			// there is only the newline char, so we got an empty string
			//printf("empty string\n");
			buf[0] = '\0';
		    } else {
			int len;
			// we got more than zero characters - copy them
			len = eolptr-lineptr-2;
			//printf("copy @%d %d bytes\n", lineptr, len);
			memcpy(&buf[0], &ringbuf[lineptr], len);

			// eolptr-lineptr >= 3, so index is >= 0
			buf[len] = '\0';
		    }
		} else {
		    // *****O-----E-----S-----L*****M
		    if (eolptr > 1) {
			// the newline characters are wrap around
			assert(lineptr < maxlen); // because lineptr starts from 0
			//printf("will copy from %d %d bytes\n", lineptr, maxlen-lineptr);
			// so copy the rest of the ringbuffer
			memcpy(&buf[0], &ringbuf[lineptr], maxlen-lineptr);

			if (eolptr > 2) {
			    // there is more than one character (incl. newline)
			    // printf("will2 copy from %d %d bytes\n", 0, eolptr-1);
			    memcpy(&buf[maxlen-lineptr], &ringbuf[0], eolptr-1);

			    buf[(maxlen-lineptr-1)+(eolptr-1)] = '\0';
			} else {
			    buf[(maxlen-lineptr)] = '\0';
			}
		    } else if (eolptr == 1) {
			// eol == 1, so \r is at the last position and \n is wrap around
			if (lineptr < maxlen-1) {
			    // there is more than one character on the remaining line
			    // printf("will3 copy from %d %d bytes\n", lineptr, maxlen-lineptr-2);
			    memcpy(&buf[0], &ringbuf[lineptr], maxlen-lineptr-1);
			    buf[maxlen-lineptr-1] = '\0';
			} else {
			    buf[0] = '\0';
			}
			
		    } else {
			// eol == 0, so the newline is at the last position in the ringbfufer
			if (lineptr < maxlen-2) {
			    // there is more than one character on the remaining line
			    //printf("will4 copy from %d %d bytes\n", lineptr, maxlen-lineptr-2);
			    memcpy(&buf[0], &ringbuf[lineptr], maxlen-lineptr-2);
			    buf[maxlen-lineptr-2] = '\0';
			} else {
			    buf[0] = '\0';
			}
		    }
		
		}
		//printf("copy done\n");
		//		lineptr = eolptr;
		assert(startptr < maxlen);
		startptr = eolptr;
		lineptr = eolptr;
		//printf("line: *%s*\n", buf);
		/* examine response */
		kme_resp = examine_response((char*)buf);
		//		printf("before mutex in thread\n");
		MUTEX_LOCK(&context->response_buffer_cond_mtx);
		//		printf("write data %d into buffer @ %p\n", kme_resp, buffer_list[kme_resp].buf);
		MUTEX_LOCK(&context->response_buffer_list_mtx);
		if (context->buffer_list[kme_resp].buf != NULL) {
		    
		    memcpy(context->buffer_list[kme_resp].buf, buf, context->buffer_list[kme_resp].buf_len);
		    context->buffer_list[kme_resp].data_ok = 1;
		    MUTEX_UNLOCK(&context->response_buffer_list_mtx);
		    /* set a variable ... */
		    pthread_cond_signal(&context->response_buffer_cond);
		} else {
		    MUTEX_UNLOCK(&context->response_buffer_list_mtx);
		}
		MUTEX_UNLOCK(&context->response_buffer_cond_mtx);

		switch(kme_resp) {
		    int len;
		  case KME_RESPONSE_HOST_COMMAND_KEYBOARD:
		  case KME_RESPONSE_HOST_COMMAND_MOUSE:
		      if ((len = strlen((char*)buf)) > 2) {
		          int l = (len - 2) / 2, i, err = 0;
		          unsigned char op[l];
		          char *_buf = (char*)buf + 2;
		          
		          for (i = 0; i < l; i++) {
		              int x;
		              if (sscanf(_buf, "%02x", &x) == 1) {
		                  op[i] = x & 0xff;
		                  _buf += 2;
		              } else {
		                  err = 1;
		              }
		          }
		          
		          if (err) break;

			  if (kme_resp == KME_RESPONSE_HOST_COMMAND_KEYBOARD && obj->data_unit_kbd.handle_host_cmd) {
			      obj->data_unit_kbd.handle_host_cmd(obj, op, l);
			  } else if (obj->data_unit_mouse.handle_host_cmd) {
			      obj->data_unit_mouse.handle_host_cmd(obj, op, l);
			  }
		      }
		      break;
		  case KME_RESPONSE_ERROR:
		      sscanf((char*)buf, "F%02x", &kme_error);
		      // Even KME_CDP_IS_BUSSY is ignored here, it may happen
                       // in case the kmethread timed out but the kme is alive
                       // indeed and still working on the previous cmd.
                       // However it could also be a faulty program, so watch
                       // out for F2 kme errors!!
                       //
                       // other asynchronous errors are ignored for the
                       // moment (should rarely happen anyway...)
		      pp_log("kme_thread_func: kme async error: %d\n",
			     kme_error);
		      kme_error = KME_NO_ERROR;
		      break;

		  case KME_RESPONSE_ERROR_PROMPT:
		      sscanf((char*)buf, "F%02x", &kme_error);
		      // fall through
		      
		  case KME_RESPONSE_PROMPT:
		      MUTEX_LOCK(&context->kme_prompt_cond_mtx);

		      // set error into context and reset own error state
		      if (kme_error != KME_NO_ERROR) {
			  context->kme_prompt_error = kme_error;
			  kme_error = KME_NO_ERROR;
		      }
		      
		      if (context->kme_prompt_wait == KME_PROMPT_WAITING) {
			  context->kme_prompt_wait = KME_PROMPT_RECEIVED;
			  pthread_cond_signal(&context->kme_prompt_cond);
		      }

		      MUTEX_UNLOCK(&context->kme_prompt_cond_mtx);

		  case KME_RESPONSE_DEBUG:
		  default:
		      KD("kme-dbg: [%s]\n", buf);
		      break;
		      
		}
	    }
    }
    pthread_exit(NULL);
}

static int
examine_response(char* line) {
    if(!strncmp("F", line, 1)) {
	if (strstr(line, "=>")) {
	    return KME_RESPONSE_ERROR_PROMPT;
	} else {
	    KD("KME_RESPONSE_ERROR: %s\n", line);
	    return KME_RESPONSE_ERROR;
	}
    } else if (!strncmp("=>", line, 2)) {
	return KME_RESPONSE_PROMPT;
    } else if (!strncmp("V", line, 1) || !strncmp("0", line, 1)) {
	return KME_RESPONSE_VERSION;
    } else if (!strncmp("ER", line, 2)) {
	return KME_RESPONSE_HOST_ECHO;
    } else if (!strncmp("CK", line, 2)) {
	return KME_RESPONSE_HOST_COMMAND_KEYBOARD;
    } else if (!strncmp("CM", line, 2)) {
	return KME_RESPONSE_HOST_COMMAND_MOUSE;
    } else if (!strncmp("SC", line, 2)) {
	return KME_RESPONSE_SCSI_COMMAND;
    } else if (!strncmp("SD", line, 2)) {
	return KME_RESPONSE_SCSI_DATA;
    } else {
	return KME_RESPONSE_DEBUG;
    }
}

static void register_buffer(driver_t* obj, int response_code,
			    char* buffer, int len) {
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    
    MUTEX_LOCK(&context->response_buffer_list_mtx);
    context->buffer_list[response_code].buf = buffer;
    //    printf("register buffer %d, buf: %p, len: %d\n",
    //            response_code, buffer_list[response_code].buf , len);
    context->buffer_list[response_code].buf_len = len;
    context->buffer_list[response_code].data_ok = 0;
    MUTEX_UNLOCK(&context->response_buffer_list_mtx);
}

static int unregister_buffer_wait(driver_t* obj, int response_code,
				  unsigned long timeout /* in ms */) {
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    int ret, waitret;
    unsigned long to_secs, to_usecs, tmpto_usecs;
    struct timeval now;
    struct timespec to;
    //    printf("unregister buffer at %d, buf: %p\n",
    //           response_code, buffer_list[response_code].buf);

    to_secs = timeout / 1000;
    to_usecs = (timeout % 1000) * 1000;
    
    MUTEX_LOCK(&context->response_buffer_cond_mtx);
    while (!context->buffer_list[response_code].data_ok) {
	//	printf("while s: %d\n", buffer_list[response_code].data_ok);

	gettimeofday(&now, NULL);
	to.tv_sec = now.tv_sec + to_secs;
	if((tmpto_usecs = now.tv_usec + to_usecs) > 1000000) {
	    to.tv_sec++;
	    tmpto_usecs -= 1000000;
	}
	to.tv_nsec = tmpto_usecs * 1000;
	
	waitret = pthread_cond_timedwait(&context->response_buffer_cond,
					 &context->response_buffer_cond_mtx,
					 &to);
	if (waitret == ETIMEDOUT) {
	    //	printf("while e3: %d\n", buffer_list[response_code].data_ok);
	    break;
	}
	//  printf("while e2: %d\n", buffer_list[response_code].data_ok);
    }
    MUTEX_UNLOCK(&context->response_buffer_cond_mtx);

    MUTEX_LOCK(&context->response_buffer_list_mtx);
    ret = context->buffer_list[response_code].data_ok;
    //    printf("set buffer to NULL\n");
    context->buffer_list[response_code].buf = NULL;
    MUTEX_UNLOCK(&context->response_buffer_list_mtx);
    //    printf("unregister for %d end\n", response_code);
    return ret;
}

/**
 * tries to get the version string
 */
static int get_kme_version(driver_t* obj, char* version, int len)
{
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    const char* unknown_str = "unknown";
    const char* rv = "I\n";
    char* retstr;
    int ret, ret2;
    unsigned long to = 50; /* ms */

    if ((retstr = (char*)malloc(len)) == NULL) {
	strncpy(version, unknown_str, len);
	version[len - 1] = '\0';
	return -1;
    }
    
    register_buffer(obj, KME_RESPONSE_VERSION, retstr, len);
    if(0 != (ret = kme_send(obj, rv, KME_SEND_STR, context->timeout))) {
	to = 10;
    }
    ret2 = unregister_buffer_wait(obj, KME_RESPONSE_VERSION, to);

    if (ret || !ret2) {
        strncpy(version, "unknown", len);
        version[len - 1] = '\0';
    } else {	
	strncpy(version, &retstr[1], len);
    }
    version[len - 1] = '\0';
    
    free(retstr);
    return ret;
}


/**
 * reset the KME (on LARA)
 */
static int reset_driver(driver_t* obj)
{
    comm_proto_data_t* context;
    assert(obj);
    context = (comm_proto_data_t*)obj->comm_proto.data;
    assert(context);

    if (context->reset_pin == -1) {
	/* we cannot reset the KME via GPIOs, use the RESET command of KME */
	/* note: actually we cannot determine, whether the KME has reset or just answered with a prompt */
	int ret;
	const char* reset = "R\r\n";
	
	ret = kme_send(obj, reset, KME_SEND_STR, context->timeout);
	if(ret) {
	    /* no response */
	    return -1;
	}
    } else {
	pp_gpio_bit_set(PP_GPIO_ATMEL_DEV, context->reset_pin, 1);
	usleep(100);
	pp_gpio_bit_set(PP_GPIO_ATMEL_DEV, context->reset_pin, 0);
    }
    
    /* reset internal led status */
    usleep(100);
    kme_send(obj, "L\r\n", KME_SEND_STR, context->timeout);
    
    return 0;
}

/*
  Is called for the f6 "set to defaults" and f0 "change scancode set"
  commands from the KME.  
  Signals that the virtual keyboard did accept the command and is in
  the new state.
*/
static int
send_cmd_ack(driver_t* obj)
{
    int err;
    comm_proto_data_t* context;
    
    assert(obj);
    context = (comm_proto_data_t*) obj->comm_proto.data;
    assert(context);

#define ACK "A\r\n"
    if (0 != (err = context->send_data(context, sizeof(ACK) -1, ACK))) {
	pp_log("in send_cmd_ack()");
    }
    return err;
}


#if 0 
/**
 * checks if host reacts
 */
static int echo(eric_km_ddops* obj, km_context_t *context) {
    const char* rv_reset  = "Er\n";
    const char* rv_check  = "Ec\n";
    const unsigned char key_press[] = {0x9C, 0x1C};
    const unsigned char key_press_shift[] = {0xA9, 0x29};
    int ret = 0, state;
    char response[7];

    /* reset echo state */
    if (0 != kme_send(obj, context, rv_reset, KME_SEND_STR, timeout)) {
	return -1;
    }

    /* press caps key */
    if (0 != obj->send_vkeycodes(obj, context, key_press, 2)) {
	return -1;
    }
 
    /* check LED state */
    register_buffer(context, KME_RESPONSE_HOST_ECHO, response,
		    sizeof(response));
    if(0 != kme_send(obj, context, rv_check, KME_SEND_STR, timeout)) {
	unregister_buffer_wait(context, KME_RESPONSE_HOST_ECHO, 10);
	ret = -1;
    } else {
	if (!unregister_buffer_wait(context, KME_RESPONSE_HOST_ECHO, 200)) {
	    ret = -1;
	} else {
	    if (sscanf(response, "ER%02x", &state) == 1) {
		ret = state;
	    } else {
		ret = -1;
	    }
	}
    }

    /* press caps key again */
    if (0 != obj->send_vkeycodes(obj, context, key_press, 2)) {
	return ret;
    }

    /* also press shift key (to release caps for some os) */
    if (0 != obj->send_vkeycodes(obj, context, key_press_shift, 2)) {
	return ret;
    }

    return ret;
}
#endif 

/* Hack for kimtester:
 * Reads/sets PS2 pin state of the KME (control-cmds go via I2C).
 */
static int
rw_pin (driver_t* obj, int is_write, int pin, int *val)
{
    comm_proto_data_t* context = (comm_proto_data_t*)obj->comm_proto.data;
    int send_err;
    int wait_ok;
#define RW_PIN_SET_FMT "S%d\n"
#define RW_PIN_CLR_FMT "C%d\n"
#define RW_PIN_GET_FMT "G%d\n"
    char cmd[20];
    char retstr[20];
    int len = sizeof(retstr);
    unsigned long to = 50; /* ms */

    if (is_write) {
	if (*val) {
	    snprintf(cmd, sizeof(cmd), RW_PIN_SET_FMT, pin);
	} else {
	    snprintf(cmd, sizeof(cmd), RW_PIN_CLR_FMT, pin);
	}

	if (kme_send(obj, cmd, KME_SEND_STR, context->timeout)) {
	    goto fail;
	}
    } else {
	/* note: Using KME_RESPONSE_VERSION for returning the pin value is a hack! */
	register_buffer(obj, KME_RESPONSE_VERSION, retstr, len);
	snprintf(cmd, sizeof(cmd), RW_PIN_GET_FMT, pin);

	if(0 != (send_err = kme_send(obj, cmd, KME_SEND_STR, context->timeout))) {
	    to = 10;
	}
	wait_ok = unregister_buffer_wait(obj, KME_RESPONSE_VERSION, to);
	if (!send_err && wait_ok) {
	    if (retstr[1] == '0') {
		*val = 0;
	    } else if (retstr[1] == '1'){
		*val = 1;
	    } else {
		pp_log_err("%s: KME returned invalidly formatted string '%s'", ___F, retstr);
		goto fail;
	    }
	} else {
	    pp_log_err("%s: could not read return value from KME", ___F);
	    goto fail;
	}
    }

    return 0;
fail:
    return 1;
}
