/* layer: data conv
   data path: data_conv_kbd->send_keycodes()  ->  data_unit_kbd->send_kbd_data()
*/

/* system includes */
#include <stdlib.h>

/* firmware includes */
#include <liberic_misc.h>
#include <pp/kvm.h>
#include <pp/propchange.h>
#include <pp_kernel_common.h>

#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
# include <pp/um.h>
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */

/* local includes */
#include "data_conv_kbd.h"

/* object functions */
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int cleanup(driver_t* obj);
static int reset_driver(driver_t* obj);
static int reconfigure(driver_t* obj);
static int send_keycodes (driver_t* obj, const u_char keys[], int length);
static int send_keycodes_timed(driver_t* obj, const u_char keys[], int length);
#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
static int check_timeout(driver_t* obj, const u_char keys[], int length);
static int timed_release(void *ctx);
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
static int is_equal(driver_t* obj, driver_t* d);

/* local object functions */
static int update_kbd_states(driver_t* obj, const u_char keys[], int length,
			     keyboard_pstate_t* keyboard_state);
static int alloc_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);
static int reset_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);
static int free_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);

/* property change handler */
static void propchange_handler(pp_propchange_listener_t * listener, 
			       u_short prop, u_short param);

#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
typedef struct {
    driver_t* driver;
    u_char key;
} kbd_timerq_elem_t;
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */

/* function definitions */
int
init_data_conv_kbd(driver_t* obj)
{
    kbd_data_t *kbd_data = NULL;
    u_char kvm_unit;

    assert(obj);
    
    /* create the driver */
    obj->data_conv_kbd.id = __FUNCTION__;
    obj->data_conv_kbd.is_equal = is_equal;
    obj->data_conv_kbd.cleanup = cleanup;
    obj->data_conv_kbd.reconfigure = reconfigure;
    obj->data_conv_kbd.reset_driver = reset_driver;
    obj->data_conv_kbd.send_keycodes = send_keycodes;
    obj->data_conv_kbd.suspend = suspend;
    obj->data_conv_kbd.resume = resume;
    
    /* initializes our private data */
    obj->data_conv_kbd.data = calloc(sizeof(kbd_data_t), 1);
    if (obj->data_conv_kbd.data == NULL) exit(1);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;

    /* init mutex */
    pthread_mutex_init(&kbd_data->kbd_data_mtx, NULL);

    /* register listener for kvm unit insertion/removal events */
    pp_propchange_init_listener(
	    (pp_propchange_listener_t *)&kbd_data->propchange_listener,
	    propchange_handler);
    kbd_data->propchange_listener.obj = obj;
    pp_propchange_add(
	    (pp_propchange_listener_t *)&kbd_data->propchange_listener,
	    PP_PROP_KVM_UNIT_ADD_OR_REMOVE);
    
    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
	if (pp_kvm_is_unit_present(kvm_unit)) {
	    alloc_kbd_state_for_kvm_unit(obj, kvm_unit);
	}
    }

    /* resets the module */
    obj->data_conv_kbd.reset_driver(obj);

#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
    {
	kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
	pp_cfg_is_enabled(&kbd_data->timeout_enabled, "unit.port.kbd.break_timeout_enabled");
	pp_cfg_get_uint(&kbd_data->timeout, "unit.port.kbd.break_timeout_msec");
    }
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
    
    return 0;
}

static int resume(driver_t * /* obj */ ) {
    return 0;
}

static int suspend(driver_t * /* obj */ ) {
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_conv_kbd.id, d->data_conv_kbd.id)) {
	return 1;
    } else {
	return 0;
    }
}

static int
cleanup(driver_t* obj)
{
    kbd_data_t *kbd_data;
    u_char kvm_unit;
    
    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);

    pp_propchange_remove_all((pp_propchange_listener_t *)&kbd_data->propchange_listener);

    /* free kbd state mem for all units */
    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
        {
            u_short kvm_port;

            for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; ++kvm_port) {
                if (kbd_data->timerq[kvm_unit][kvm_port]) {
                    pp_timerq_destroy(kbd_data->timerq[kvm_unit][kvm_port]);
		}
	    }
        }
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
	free_kbd_state_for_kvm_unit(obj, kvm_unit);
    }

    free(obj->data_conv_kbd.data);

    return 0;
}

static int
alloc_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    kbd_data_t *kbd_data;
    keyboard_pstate_t *p_kbdstat = NULL;
    
    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);

    if (kbd_data->keyboard_pstate_list_units[kvm_unit] == NULL) {
	/* unit present -> allocate kbd state mem for max nr of ports */
	p_kbdstat = kbd_data->keyboard_pstate_list_units[kvm_unit] =
	    (keyboard_pstate_t *) calloc(sizeof(keyboard_pstate_t),
					 PP_KVM_MAX_UNIT_PORT_COUNT);
	KD("%s: %s(): alloc, pointer %hhu: %p\n", __FILE__, ___F, kvm_unit,
	   p_kbdstat);
	if (!p_kbdstat) {
	    pp_log_err("%s: %s: allocating memory failed", __FILE__, ___F);
	    exit(1);
	}
    } else {
	KD("%s: %s(): unit %hhu was already allocated: %p\n", __FILE__, ___F,
	   kvm_unit, p_kbdstat);
    }

    return 0;
}

/* resets the keyboard states of all ports in KVM_UNIT (if allocated) */
static int
reset_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    int key;
    u_short kvm_port;
    kbd_data_t *kbd_data;
    keyboard_pstate_t *p_kbdstat;
    
    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);

    if (kbd_data->keyboard_pstate_list_units[kvm_unit]) {
	KD("%s: %s(): resetting port state for unit %hhu\n", __FILE__, ___F, kvm_unit);
	for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; kvm_port++) {
	    for (key = 0; key < MAX_KEYDEFS; key++) {
		p_kbdstat = &kbd_data->keyboard_pstate_list_units[kvm_unit][kvm_port];
		p_kbdstat->keys[key] = KEY_STATE_NONE;
	    }
#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
            kbd_data->timerq[kvm_unit][kvm_port] = NULL; /* init only if really used */
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
	}
    }

    return 0;
}

static int
free_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    kbd_data_t *kbd_data;
    keyboard_pstate_t *p_kbdstat;
    
    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);

    /* unit not present -> release mem */
    p_kbdstat = kbd_data->keyboard_pstate_list_units[kvm_unit];
    KD("%s: %s: %hhu: freeing, pointer %p\n", __FILE__, ___F, kvm_unit, p_kbdstat);

    free(kbd_data->keyboard_pstate_list_units[kvm_unit]);
    kbd_data->keyboard_pstate_list_units[kvm_unit] = NULL;

    return 0;
}


static int
reset_driver(driver_t* obj)
{
    u_char kvm_unit;

    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; kvm_unit++) {
	reset_kbd_state_for_kvm_unit(obj, kvm_unit);
    }

    return 0;
}

static int
reconfigure(driver_t* obj)
{
    kbd_data_t *kbd_data;
    const char* update;

    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);
    
    update = obj->data_unit_kbd.get_info(obj, "update_state_before");

    if (update && !(strcmp(update, "yes"))) {
	kbd_data->update_before = 1;
    } else {
	kbd_data->update_before = 0;
    }
    update = obj->data_unit_kbd.get_info(obj, "bulk_keys");
    
    if (update && !(strcmp(update, "yes"))) {
	kbd_data->bulk_keys = 1;
    } else {
	kbd_data->bulk_keys = 0;
    }
    return 0;
}

static int
send_keycodes(driver_t* obj, const u_char keys[], int length)
{
#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
    kbd_data_t *kbd_data;

    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);
    if(kbd_data->timeout_enabled) {
        if (check_timeout(obj, keys, length)) {
            return send_keycodes_timed(obj, keys, length);
	} else {
            return 0;
	}
    }
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */
    
    return send_keycodes_timed(obj, keys, length);
}

static int
send_keycodes_timed(driver_t* obj, const u_char keys[], int length)
{
    kbd_data_t *kbd_data;
    u_char kvm_unit;
    u_short kvm_port;
    keyboard_pstate_t *p_keyboard_pstate;

    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);

    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port) != 0) {
	pp_log("%s(): pp_kvm_get_unit_port_for_data_link() failed\n", ___F);
	return -1;
    }
    
    if (kvm_unit >= PP_KVM_MAX_UNIT_COUNT) {
        pp_log_err("data_conv_kbd: %s: current kvm unit (%hhu) is out of range 0-%u", ___F, kvm_unit, PP_KVM_MAX_UNIT_COUNT);
        return -1;
    }

    if (kvm_port >= PP_KVM_MAX_UNIT_PORT_COUNT) {
        pp_log_err("data_conv_kbd: %s: current kvm port (%hu) is out of range 0-%u", ___F, kvm_port, PP_KVM_MAX_UNIT_PORT_COUNT);
        return -1;
    }

    if (kbd_data->keyboard_pstate_list_units[kvm_unit] == NULL) {
        pp_log_err("data_conv_kbd: %s: no memory has been allocated for kvm unit %hhu", ___F, kvm_unit);
        return -1;
    }

    KD("send_keycodes port: %hu, before: %d, bulk_keys: %d\n", kvm_port, kbd_data->bulk_keys, kbd_data->update_before);

    p_keyboard_pstate = &kbd_data->keyboard_pstate_list_units[kvm_unit][kvm_port];

    if (kbd_data->update_before) {
	if (kbd_data->bulk_keys) {
	    update_kbd_states(obj, keys, length, p_keyboard_pstate);
	    obj->data_unit_kbd.send_kbd_data(obj, keys, length, p_keyboard_pstate);
	} else {
	    int i;
	    for(i = 0; i < length; i++) {
		update_kbd_states(obj, &keys[i], 1, p_keyboard_pstate);
		obj->data_unit_kbd.send_kbd_data(obj, &keys[i], 1, p_keyboard_pstate);
	    }
	}
    } else {
	if (kbd_data->bulk_keys) {
	    obj->data_unit_kbd.send_kbd_data(obj, keys, length, p_keyboard_pstate);
	    update_kbd_states(obj, keys, length, p_keyboard_pstate);
	} else {
	    int i;
	    for(i = 0; i < length; i++) {
		obj->data_unit_kbd.send_kbd_data(obj, &keys[i], 1, p_keyboard_pstate);
		update_kbd_states(obj, &keys[i], 1, p_keyboard_pstate);
	    }
	}
    }
    return 0;
}

#ifdef PP_FEAT_KBD_BREAK_TIMEOUT
/**
 * handle timeout magic for given key(-list)
 *
 * check if make code 
 *     case control-key:  set control_sequence = 1
 *     case modifier-key: do nothing
 *     case normal-key:   check if control_sequence == 1 then do nothing,
 *                        else enqueue key to timer queue
 * else 
 *     case control-key:  do nothing
 *     case modifier-key: do nothing
 *     case normal-key:   check if control_sequence == 1 then do nothing,
 *                        else dequeue key from timer queue
 *     if timer queue is empty, set control_sequence = 0
 *
 * control-keys are STRG, ALT, ...
 * modifier-keys are SHIFT, ...
 */
static int
check_timeout(driver_t* obj, const u_char keys[], int length)
{
    kbd_data_t *kbd_data;
    u_char kvm_unit;
    u_short kvm_port;
    int i, ret = 1;
    pp_timerq_t *tq;
    u_char event, key;
    kbd_timerq_elem_t *tqe;

    assert(obj);
    kbd_data = (kbd_data_t*)obj->data_conv_kbd.data;
    assert(kbd_data);
    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port) != 0) {
	pp_log("%s(): pp_kvm_get_unit_port_for_data_link() failed\n", ___F);
	return -1;
    }
    
    if (kvm_unit >= PP_KVM_MAX_UNIT_COUNT) {
        pp_log_err("data_conv_kbd: %s: current kvm unit (%hhu) is out of range 0-%u", 
                   ___F, kvm_unit, PP_KVM_MAX_UNIT_COUNT);
        return -1;
    }

    if (kvm_port >= PP_KVM_MAX_UNIT_PORT_COUNT) {
        pp_log_err("data_conv_kbd: %s: current kvm port (%hu) is out of range 0-%u", 
                   ___F, kvm_port, PP_KVM_MAX_UNIT_PORT_COUNT);
        return -1;
    }

    if(kbd_data->timerq[kvm_unit][kvm_port] == NULL) { /* not yet initialized */
        kbd_data->timerq[kvm_unit][kvm_port] = pp_timerq_create();
        pp_timerq_start(kbd_data->timerq[kvm_unit][kvm_port]);
    }
    tq = kbd_data->timerq[kvm_unit][kvm_port];

    for(i = 0; i < length; ++i) {
	event = (keys[i] & 0x80) ?  KEY_EVENT_PRESSED : KEY_EVENT_RELEASED;
	key   = keys[i] & 0x7F;
        if(event == KEY_EVENT_PRESSED) {
            switch(key) {
                /* control keys */
                case 54: /* LCTRL */
                case 55: /* LALT */
                case 57: /* RALT */
                case 58: /* RCTRL */
                case 105: /* LWIN */
                case 106: /* RWIN */
                    kbd_data->control_sequence = 1;
                    break;

                /* modifier keys */
                case 41: /* LSHIFT */
                case 53: /* RSHIFT */
                    /* do nothing... */
                    break;
                
                default: /* anything else is normal ;-) */
                    if(!kbd_data->control_sequence) {
                        tqe = (kbd_timerq_elem_t*)
                              malloc(sizeof(kbd_timerq_elem_t));
                        tqe->driver = obj;
                        tqe->key = key;
                        pp_timerq_add(tq, kbd_data->timeout, 
                                      timed_release, tqe);
                    }
                    break;
            }
        } else {
            /* dequeue key release from timer queue */
            /* I guess we'd have to walk over entire queue, looking for key in
               timer queue entry context as we do not know the id of the timer
               entry...
               obj would be quite slow, therefore we simply ignore break codes
               for keys, which are not pressed */
            keyboard_pstate_t* keyboard_state =
                &kbd_data->keyboard_pstate_list_units[kvm_unit][kvm_port];
            if((keyboard_state->keys[key] & KEY_STATE_PRESSED) == 0)
                ret = 0;

            if(pp_timerq_empty(tq))
                kbd_data->control_sequence = 0;
        }
    }
    
    return ret;
}

static int
timed_release(void *ctx)
{
    kbd_timerq_elem_t *tqe;
    int ret;
    
    assert(ctx);
    
    tqe = (kbd_timerq_elem_t*)ctx;
    assert((tqe->key & 0x80) == 0);
    ret = send_keycodes(tqe->driver, &tqe->key, 1);

    free(tqe);
    return ret;
}
#endif /* PP_FEAT_KBD_BREAK_TIMEOUT */

static int
update_kbd_states(driver_t * /* obj */, const u_char keys[], int length,
		  keyboard_pstate_t* keyboard_state)
{
    int i;
    
    for (i = 0; i < length; i++) {
	u_char event = (keys[i] & 0x80) ?  KEY_EVENT_PRESSED : KEY_EVENT_RELEASED;
	u_char key   = keys[i] & 0x7F;
	
	if (key >= MAX_KEYDEFS) return -1;
	
	if (event == KEY_EVENT_PRESSED) {
	    keyboard_state->keys[key] |= KEY_STATE_PRESSED;
	} else {
	    keyboard_state->keys[key] &= ~KEY_STATE_PRESSED ;
	}
    }
    return 0;
}

static void
propchange_handler(pp_propchange_listener_t * listener, u_short prop, u_short param)
{
    data_conv_kbd_propchange_listener_t *mylistener;
    driver_t * obj;
    u_char kvm_unit;

    assert(listener);

    mylistener = (data_conv_kbd_propchange_listener_t *)listener;
    obj = mylistener->obj;

    if (prop == PP_PROP_KVM_UNIT_ADD_OR_REMOVE) {
	assert(param < 256);
	kvm_unit = (u_char)param;
	if (pp_kvm_is_unit_present(kvm_unit)) {
	    // unit was added
	    alloc_kbd_state_for_kvm_unit(obj, kvm_unit);
	    reset_kbd_state_for_kvm_unit(obj, kvm_unit);
	} else {
	    // unit was removed
	    free_kbd_state_for_kvm_unit(obj, kvm_unit);
	}
    } else {
	pp_log("data_conv_kbd: %s(): Unhandled property %hu received.\n", ___F, prop);
    }
}

