/******************************************************************************\
* data_conv_mouse_cat.c                                                        *
*                                                                              *
* CURSOR ACTIVITY TRACKING                                                     *
*                                                                              *
* Intelligent online mouse cursor detection and tracking                       *
*                                                                              *
* Copyright 2005 Peppercon AG                                                  *
* Thomas Weber tweb@peppercon.de                                               *
\******************************************************************************/

#include <math.h> // sqrt

/* firmware includes */
#include <pp/cfg.h>
#include <pp/kvm.h>
#include <pp/vsc.h>

/* local includes */
#include "driver.h"
#include "cat_debug.h"
#include "cat_internal.h"

/* driver api */
static int is_equal(driver_t *obj, driver_t *d);
static int resume(driver_t *obj);
static int suspend(driver_t *obj);
static int cleanup(driver_t *obj);
static int reconfigure(driver_t *obj);
static int send_ptrmove(driver_t *obj,
                        const int x, const int y, const int z, 
                        const unsigned char buttonmask,
                        const unsigned char abs_rel);
static int sync_mouse(driver_t *obj, const int sync_type);

/* little helpers */
static int alloc_port_state_for_kvm_unit(driver_t *obj, int kvm_unit);
static int reset_port_state_for_kvm_unit(driver_t *obj, int kvm_unit);
static int free_port_state_for_kvm_unit(driver_t *obj, int kvm_unit);

static void set_current_kvm_port(driver_t *obj);
static void propchange_handler(pp_propchange_listener_t * listener, 
			       unsigned short prop, unsigned short prop_flags);

#if defined(PP_KM_CAT_INTELLI)
/* iipptr externals */
int move_mouse_diff(driver_t* obj, long diff_x, long diff_y, 
                    const unsigned char buttonmask, int* moved);
#endif /* PP_KM_CAT_INTELLI */

///////////////// old stuff ... //////////////////////

/* static vars */
static const char* mouse_mode_key = "unit[%u].port[%u].mouse.mode";
static const char* mouse_direct_sc_key = "unit[%u].port[%u].mouse.direct_scaling";


int init_data_conv_cat(driver_t *obj) {
    data_cat_t *mouse_data;
#if defined(PP_KM_CAT_INTELLI)
    data_imouse_t *imouse_data;
#endif /* PP_KM_CAT_INTELLI */
    int kvm_unit;

    assert(obj);
#ifdef PP_FEAT_VSC_HW_ENCODING
    abort(); // cat initially is vsc1 stuff!
#endif
    
    /* sets the object functions */
    obj->data_conv_mouse.id = __FUNCTION__;
    obj->data_conv_mouse.is_equal = is_equal;

    obj->data_conv_mouse.resume = resume;
    obj->data_conv_mouse.suspend = suspend;
    obj->data_conv_mouse.cleanup = cleanup;
    obj->data_conv_mouse.reconfigure = reconfigure;
    obj->data_conv_mouse.send_ptrmove = send_ptrmove;
    obj->data_conv_mouse.sync_mouse = sync_mouse;

    /* initializes our private data */
    obj->data_conv_mouse.data = calloc(sizeof(data_cat_t), 1);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    
    /* create grabber client */
    mouse_data->grab_client = pp_grab_new_client(0, NULL);
    
//    pthread_mutex_init(&mouse_data->move_mouse_corner_flag_mtx, NULL);
    /* we don't need to initialize to NULL due to calloc */
//    mouse_data->current_cursor_shape.rgb = NULL;
//    mouse_data->current_cursor_bg_shape.rgb = NULL;
//    mouse_data->current_cursor_mask.rgb = NULL;
    
//    MUTEX_LOCK(&mouse_data->move_mouse_corner_flag_mtx);
//    mouse_data->move_mouse_corner_flag = 0;
//    MUTEX_UNLOCK(&mouse_data->move_mouse_corner_flag_mtx);
  
    /* init for all ports */
    //    cnt = eric_misc_kvm_get_nr_ports();
    pthread_mutex_init(&mouse_data->kvm_port_mtx, NULL);    
    for (kvm_unit=0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; kvm_unit++) {
	if (pp_kvm_is_unit_present(kvm_unit)) {
	    alloc_port_state_for_kvm_unit(obj, kvm_unit);
	    reset_port_state_for_kvm_unit(obj, kvm_unit);
	}
    }

    set_current_kvm_port(obj);

#if defined(PP_KM_CAT_INTELLI)
    mouse_data->state = ERIC_MOUSE_NOT_VALID;
    imouse_data = (data_imouse_t*)mouse_data;
    imouse_data->current_kvm_port = mouse_data->current_kvm_port;
    imouse_data->grab_client = mouse_data->grab_client;
#endif /* PP_KM_CAT_INTELLI */

    /* initialize tracking */
    /* use simple tracking */
//    cat_condensation_simple_init(obj);
    /* use image understanding tracking */
    cat_condensation_init(obj);
    
    /* initialize lookup table */
    cat_lookup_table_init(obj);

    pp_propchange_init_listener((pp_propchange_listener_t *)
                                &mouse_data->propchange_listener,
                                propchange_handler);
    mouse_data->propchange_listener.obj = obj;
    pp_propchange_add((pp_propchange_listener_t *)
                      &mouse_data->propchange_listener,
                      PP_PROP_KVM_PORT_SWITCHED);
    pp_propchange_add((pp_propchange_listener_t *)
                      &mouse_data->propchange_listener,
                      PP_PROP_VIDEO_MODE_CHANGED);
    pp_propchange_add((pp_propchange_listener_t *)
                      &mouse_data->propchange_listener,
                      PP_PROP_KVM_UNIT_ADD_OR_REMOVE);

    if(pp_cat_init(obj) == PP_ERR) {
	pp_log("Initializing CAT failed.\n");
        cleanup(obj);
	return -1;
    }
    
    return 0;
}

static int is_equal(driver_t *obj , driver_t *d ) {
    assert(obj);
    assert(d);
    
    return !strcmp(obj->data_conv_mouse.id, d->data_conv_mouse.id);
}

static int resume(driver_t *obj ) {
    return 0;
}

static int suspend(driver_t *obj ) {
    return 0;
}

static int cleanup(driver_t *obj) {
    int kvm_unit;
    data_cat_t *mouse_data;

    assert(obj);
    cat_cleanup(obj);

    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    
    pp_propchange_remove_all((pp_propchange_listener_t *)
                             &mouse_data->propchange_listener);
    
    /* free kbd state mem for all units */
    for (kvm_unit=0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; kvm_unit++) {
	free_port_state_for_kvm_unit(obj, kvm_unit);
    }

    /* free lookup table */
    free(mouse_data->lt);
    free(mouse_data->ltc);

    /* remove grabber client */
    pp_grab_remove_client(mouse_data->grab_client);

    /* free all data structures */
    free(mouse_data);

    return 0;
}

static int reconfigure(driver_t *obj ) {
    km_kvm_entry_t *port, *unit;
    u_char kvm_unit;
    u_short kvm_port;
    data_cat_t* mouse_data;
    char *var_val;

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);

    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {	
	if (pp_kvm_is_unit_present(kvm_unit)) {
	    unit = mouse_data->kvm_port_state_list_units[kvm_unit];
	    if (unit) {
	        for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT;
                     ++kvm_port) {
		    port = &unit[kvm_port];
		    /* get mouse mode for obj port */
		    pp_cfg_get(&var_val, mouse_mode_key, kvm_unit, kvm_port);
		    if (!strcmp(var_val, "auto")) {
//			CDMSG(1, "set port %d to auto\n", kvm_port);
			port->mouse_data.mode = ERIC_MOUSE_ON;
			port->mouse_data.cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
		    } else {
//			CDMSG(1, "set port %d to 1:n mode\n", kvm_port);
			port->mouse_data.mode = ERIC_MOUSE_OFF;
			port->mouse_data.cursor_state = ERIC_MOUSE_CURSOR_NOT_VALID;
		    }
		    free(var_val);
		    /* set direct mouse scaling for obj port */
		    pp_cfg_get_float(&port->mouse_data.scaling, mouse_direct_sc_key, kvm_unit, kvm_port);
		}
	    }
	}
    }

    set_current_kvm_port(obj);

    return 0;
}

#if !defined(CAT_APPROACH_ON_THE_FLY)
static int cat_ptr_approach(driver_t *obj,
                            u_int16_t estim_x, u_int16_t estim_y,
                            const int x, const int y, const int z , 
                            const unsigned char buttonmask) {
    int dx, dy, ret = PP_SUC;
    data_cat_t* mouse_data;
#if defined(CAT_FB_DIRECT)
    u_int16_t *shape_fb_rgb = NULL;
#else /* CAT_FB_DIRECT */
    t_bitmap screen = {0, 0, NULL};
#endif /* CAT_FB_DIRECT */
    fb_format_info_t fb_info;
    u_int32_t shape_width, shape_height;
    int moved = 1;
#if defined(PP_KM_CAT_INTELLI)
    t_mouse_data *md;
    cat_queue_ll_move_entry_t ctx = { step: 1 };
#endif /* PP_KM_CAT_INTELLI */
    int diff_x = 0, diff_y = 0;
    t_point last = {pos_x: -1, pos_y: -1};
    u_int32_t d = -1;
#if defined(CAT_PROFILE_APPROACHING)
    pp_stopwatch_t watch;
    int cycle = 0;
    
    CDMSG(1, "called\n");
    pp_hrtime_start(&watch);
#endif /* CAT_PROFILE_APPROACHING */
    
    CDMSG(1, "position (%4d, %3d) is reliable, approaching (%4d; %3d)\n", 
          estim_x, estim_y, x, y);

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->shape);
    assert(mouse_data->mask);
    assert(mouse_data->grab_client);

#if defined(PP_KM_CAT_INTELLI)
    md = &mouse_data->current_kvm_port->mouse_data;
    assert(md);
#endif /* PP_KM_CAT_INTELLI */

    shape_width = mouse_data->shape->width;
    shape_height = mouse_data->shape->height;

    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return PP_ERR;
    }
    
    if(x < 0 || x >= (int)fb_info.g_w || y < 0 || y >= (int)fb_info.g_h) {
        CDMSG(CAT_CDMSG_ERROR, "target positon (%4d; %3d) invalid!\n", x, y);
        abort();
        return PP_ERR;
    }
    
    if(estim_x >= (int)fb_info.g_w || estim_y >= (int)fb_info.g_h) {
        CDMSG(CAT_CDMSG_ERROR, "estimated positon (%4d; %3d) invalid!\n",
              estim_x, estim_y);
        abort();
        return PP_ERR;
    }
    
#if defined(CAT_FB_DIRECT)
#warning initialize obj only once!!!
    if(((!mouse_data->shape->rgb || !mouse_data->mask->rgb) &&
        PP_ERR == (ret = cat_intelli_get_mousecursor_shape(obj))) ||
       PP_ERR == (ret = cat_bmp2fb_rgb(&shape_fb_rgb, mouse_data->shape))) {
        CDMSG(CAT_CDMSG_ERROR,
              "could not convert shape to frame buffer RGB!\n");
        abort();
    }
#endif /* CAT_FB_DIRECT */

#if defined(CAT_PROFILE_APPROACHING)
    pp_hrtime_stop(&watch);
    CDMSG(1, "init finished, took %llu usec\n", pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

    dx = x - estim_x;
    dy = y - estim_y;
    while(ret != PP_ERR && moved &&
          (abs(dx) + abs(dy)) >= MIN_PTR_DEV_4_APPROACH) {
        int _left, _top, _width, _height;
        t_point found;
        u_short left, top, width, height;
        u_int32_t d_current;

#if defined(CAT_PROFILE_APPROACHING)
        pp_hrtime_start(&watch);
#endif /* CAT_PROFILE_APPROACHING */

/*        
        tmp = estim_x - (abs(diff_x) + 1) * shape_width;
        left = tmp > 0 ? tmp : 0;
        tmp = estim_y - (abs(diff_y) + 1) * shape_height;
        top = tmp > 0 ? tmp : 0;
        tmp = estim_x + (abs(diff_x) + 2) * shape_width;
        width = tmp > (int)fb_info.g_w ? (int)fb_info.g_w - left : tmp - left;
        tmp = estim_y + (abs(diff_x) + 2) * shape_height;
        height = tmp > (int)fb_info.g_h ? (int)fb_info.g_h - top : tmp - top;
*/
/*
        tmp = estim_x + diff_x - shape_width;
        left = tmp > 0 ? tmp : 0;
        tmp = estim_y + diff_y - shape_height;
        top = tmp > 0 ? tmp : 0;
        tmp = 3 * shape_width;
        width = left + tmp > (int)fb_info.g_w ? (int)fb_info.g_w - left : tmp;
        tmp = 3 * shape_height;
        height = top + tmp > (int)fb_info.g_h ? (int)fb_info.g_h - top : tmp;
*/
        if(diff_x > 0) {
            _left = estim_x - shape_width;
        } else {
            _left = estim_x + 2 * diff_x - shape_width;
        }
        if(diff_y > 0) {
            _top = estim_y - shape_height;
        } else {
            _top = estim_y + 2 * diff_y - shape_height;
        }
        _width = 2 * abs(diff_x) + 3 * shape_width;
        _height = 2 * abs(diff_y) + 3 * shape_height;
        
        left = _left > 0 ? _left : 0;
        top = _top > 0 ? _top : 0;
        width = left + _width > (int)fb_info.g_w ? 
                (int)fb_info.g_w - left : _width;
        height = top + _height > (int)fb_info.g_h ? 
                 (int)fb_info.g_h - top : _height;

        CDMSG(1, "searching for pointer @(%4d; %3d) - (%4d; %3d)\n", 
              left, top, left + width, top + height);
        
#if !defined(CAT_FB_DIRECT)
        if(width != screen.width || height != screen.height) {
            /* (re)init screen */
            free(screen.rgb);
            screen.rgb = (t_rgb*)malloc(width * height * sizeof(t_rgb));
            screen.width = width;
            screen.height = height;
        }
        
        /* grab screenshot and verify estimation */
        if(!screen.rgb ||
           PP_ERR == (ret = fb_to_rgb_area(mouse_data->grab_client, &screen, 
                                           left, top, width, height)) ||
           PP_ERR == (ret = cat_mse32(&screen, mouse_data->shape,
                                      mouse_data->mask, 0, 0, width, height,
                                      32, &found))) {
#else /* CAT_FB_DIRECT */
        if(PP_ERR == (ret = cat_mse32_fb(mouse_data->grab_client,
                                         shape_fb_rgb, mouse_data->mask,
                                         left, top, width, height, 
                                         32, 0, &found))) {
#endif /* CAT_FB_DIRECT */
            CAT_INVALIDATE_POSITION_ESTIMATE;

#if defined(PP_KM_CAT_INTELLI)
            /* this is NOT! reliable but supposed */
            md->cur_x = (estim_x + diff_x) * MOUSE_RASTER;
            md->cur_y = (estim_y + diff_y) * MOUSE_RASTER;
#endif /* PP_KM_CAT_INTELLI */
            
            CDMSG(1, "position (%4d, %3d) is no more reliable, exiting\n", 
                  estim_x, estim_y);
            break;
#if defined(PP_KM_CAT_INTELLI)
        } else if(ctx.ticks_x || ctx.ticks_y) {
            /* seems we moved... */
            int moved_x = found.pos_x - last.pos_x;
            int moved_y = found.pos_y - last.pos_y;
            
            CDMSG(1, "UPDATING LOOKUP TABLE x: %d => %d, y: %d => %d\n",
                     ctx.ticks_x, moved_x, ctx.ticks_y, moved_y);
            cat_lookup_table_add(obj, ctx.ticks_x, moved_x);
            cat_lookup_table_add(obj, ctx.ticks_y, moved_y);
#endif /* PP_KM_CAT_INTELLI */
        }
        
#if defined(CAT_PROFILE_APPROACHING)
        pp_hrtime_stop(&watch);
        CDMSG(1, "cycle %2d block matching finished, took %llu usec\n",
              ++cycle, pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

        if(found.pos_x == last.pos_x && found.pos_y == last.pos_y) {
            CDMSG(1, "we did not move, exiting\n");
            break;
        } else {
            last = found;
        }
        
#if defined(CAT_FB_DIRECT)
        estim_x = found.pos_x;
        estim_y = found.pos_y;
#else /* CAT_FB_DIRECT */
        estim_x = left + found.pos_x;
        estim_y = top + found.pos_y;
#endif /* CAT_FB_DIRECT */
        cat_set_position_estimate(obj, 1, estim_x, estim_y, pp_hrtime()?);
        CDMSG(1, "tracked new reliable position (%4d; %3d), still approaching "
                 "(%4d; %3d)\n", estim_x, estim_y, x, y);
        
        dx = x - estim_x;
        dy = y - estim_y;
        d_current = abs(dx) + abs(dy);
        if(d_current > d) {
            CDMSG(1, "distance to target increading, we better exit here\n");
            break;
        }
       
#define APPROACHING_APPROACH 5

#if APPROACHING_APPROACH == 5 // completely new approach with limits;-)
        {
            int __max_direct__TODO__ = 50;
            int __max_abs__TODO__ = 100;
            int adx = abs(dx), ady = abs(dy);
            int cityblock = adx + ady;
            
            assert(__max_abs__TODO__ > __max_direct__TODO__);
      
            if(adx > __max_abs__TODO__ || ady > __max_abs__TODO__) {
                int f = (MAX(adx, ady) / __max_abs__TODO__) + 1;
                diff_x = dx / f;
                diff_y = dy / f;
            } else if(cityblock > __max_direct__TODO__) {
                diff_x = (dx * 4) / 3;
                diff_y = (dy * 4) / 3;
            } else {
                diff_x = dx;
                diff_y = dy;
            }
        }
#endif

#if APPROACHING_APPROACH == 4 // completely new approach ;-)
        {
            int __max_direct__TODO__ = 50;
            int cityblock = abs(dx) + abs(dy);
      
            if(cityblock > __max_direct__TODO__) {
                diff_x = (dx * 4) / 3;
                diff_y = (dy * 4) / 3;
            } else {
                diff_x = dx;
                diff_y = dy;
            }
        }
#endif

#if APPROACHING_APPROACH == 3 // new approach ;-)
        {
            float f, f_max, __max__TODO__ = 75.0;
            
            f_max = __max__TODO__ / MAX(shape_width, shape_height);
            f = MAX(fabs((float)dx / shape_width), 
                    fabs((float)dy / shape_height));
//printf("tweb: f = %f, f_max = %f, dx = %d, shape_width = %d, dy = %d, shape_height = %d\n", f, f_max, dx, shape_width, dy, shape_height);           
            if(f < f_max) {
                diff_x = dx;
                diff_y = dy;
            } else {
                diff_x = floor(dx * f_max / f);
                diff_y = floor(dy * f_max / f);
            }
//printf("tweb: diff_x = %d, diff_y = %d\n", diff_x, diff_y);           
        }
#endif

#if APPROACHING_APPROACH == 2 // max movement limited by shape size
        diff_x = dx < 0 ? -MIN(OHDIV(-dx, 4), (int)shape_width)
                        : MIN(OHDIV(dx, 4), (int)shape_width);
        diff_y = dy < 0 ? -MIN(OHDIV(-dy, 4), (int)shape_height)
                        : MIN(OHDIV(dy, 4), (int)shape_height);
#endif

#if APPROACHING_APPROACH == 1 // max movement is unlimited
        diff_x = dx < 0 ? -OHDIV(-dx, 4) : OHDIV(dx, 4);
        diff_y = dy < 0 ? -OHDIV(-dy, 4) : OHDIV(dy, 4);
#endif

#if defined(CAT_PROFILE_APPROACHING)
        pp_hrtime_stop(&watch);
        CDMSG(1, "enqueueing pointer move, took %llu usec\n",
              pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

        pp_km_cat_enqueue_ptr_move(diff_x, diff_y);

#if defined(CAT_PROFILE_APPROACHING)
        pp_hrtime_stop(&watch);
        CDMSG(1, "pointer move enqueued, took %llu usec\n",
              pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

        CDMSG(1, "moving dx = %d (%d), dy = %d (%d)\n", diff_x, dx, diff_y, dy);
#if defined(PP_KM_CAT_INTELLI)
        {
            long target_diff_x, target_diff_y;

/*            
            target_diff_x = (x - diff_x * MOUSE_RASTER) - md->cur_x;
            target_diff_y = (y - diff_y * MOUSE_RASTER) - md->cur_y;
*/
            target_diff_x = diff_x * MOUSE_RASTER;
            target_diff_y = diff_y * MOUSE_RASTER;
            
            md->cur_x = estim_x * MOUSE_RASTER;
            md->cur_y = estim_y * MOUSE_RASTER;
            
#if defined(CAT_PROFILE_APPROACHING)
            pp_hrtime_stop(&watch);
            CDMSG(1, "calculated movement diffs, took %llu usec\n",
                  pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

            ret = cat_intelli_move_mouse_diff_core(obj, 
                                                   target_diff_x, target_diff_y,
			   	                   buttonmask, &moved, &ctx);
            CDMSG(1, "moved mouse by diff (%ld; %ld): ret = %d, moved = %d\n", 
                  target_diff_x, target_diff_y, ret, moved);
		  
            if(ctx.step) {
	    	diff_x = ctx.moved_x / MOUSE_RASTER;
		diff_y = ctx.moved_y / MOUSE_RASTER;
	    }
        }
#else /* PP_KM_CAT_INTELLI */
        ret = obj->data_unit_mouse.send_mouse_data(obj, diff_x, diff_y * (-1), 
                                                    z, buttonmask);
#endif /* PP_KM_CAT_INTELLI */

#if defined(CAT_PROFILE_APPROACHING)
        pp_hrtime_stop(&watch);
        CDMSG(1, "cycle %2d finished, took %llu usec\n",
              cycle, pp_hrtime_read(&watch) / 1000);
#endif /* CAT_PROFILE_APPROACHING */

        usleep(11111);
    }
    
    CDMSG(1, "returning, ret = %d, dx = %d, dy = %d, moved = %d\n", 
          ret, dx, dy, moved);

#if defined(CAT_FB_DIRECT)
    free(shape_fb_rgb);
#else /* CAT_FB_DIRECT */
    free(screen.rgb);
#endif /* CAT_FB_DIRECT */
    
    return ret;
}
#endif /* CAT_APPROACH_ON_THE_FLY */

#if !defined(PP_KM_CAT_INTELLI)
#if 1
#if 0
static int cat_ptr_approach_bresenham(driver_t *obj,
                                      u_int16_t estim_x, u_int16_t estim_y,
                                      const int x, const int y, const int z, 
                                      const unsigned char buttonmask) {
    int dx, dy, incx, incy, dist, xerr, yerr, cx, cy, ret = PP_SUC;
    data_cat_t* mouse_data;
    t_bitmap screen = {0, 0, NULL};
    fb_format_info_t fb_info;
    u_int32_t shape_width, shape_height;
    
    CDMSG(1, "position (%4d, %3d) is reliable, approaching\n", 
          estim_x, estim_y);

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    assert(mouse_data->shape);
    assert(mouse_data->mask);
    assert(mouse_data->grab_client);

    if(pp_vsc_get_fb_format(0, &fb_info) == PP_ERR) {
        CDMSG(CAT_CDMSG_WARNING, "No framebuffer info yet\n");
        return PP_ERR;
    }

    dx = x - estim_x;
    dy = y - estim_y;
    
    if(dx == 0 && dy == 0) {
        return PP_SUC;
    }
    
    shape_width = mouse_data->shape->width;
    shape_height = mouse_data->shape->height;

    /* compute the sign of the increment in both directions */
    if (dx < 0) {
        incx = -1;
        dx = -dx;
    } else {
        incx = dx ? 1 : 0;
    }
     
    if (dy < 0) {
        incy = -1;
        dy = -dy;
    } else {
        incy = dy ? 1 : 0;
    }
 
    /* determine which distance is larger */
    dist = (dx > dy) ? dx : dy;
    
    xerr = dx;
    yerr = dy;
    cx = x;
    cy = y;
    
    while(ret != PP_ERR && abs(dx) > 1 && abs(dy) > 1) {
        int diff_x, diff_y, tmp;
        t_point found;
        u_short left, top, width, height;
        
        blabla
        
        tmp = estim_x - shape_width;
        left = tmp > 0 ? tmp : 0;
        tmp = estim_y - shape_height;
        top = tmp > 0 ? tmp : 0;
        tmp = estim_x + 2 * shape_width;
        width = tmp > (int)fb_info.g_w ? (int)fb_info.g_w - left : tmp - left;
        tmp = estim_y + 2 * shape_height;
        height = tmp > (int)fb_info.g_h ? (int)fb_info.g_h - top : tmp - top;
        
        if(width != screen.width || height != screen.height) {
            /* (re)init screen */
            free(screen.rgb);
            screen.rgb = (t_rgb*)malloc(width * height * sizeof(t_rgb));
            screen.width = width;
            screen.height = height;
        }
            
        assert(screen.rgb);

        /* grab screenshot and verify estimation */
        if(!screen.rgb ||
           PP_ERR == (ret = fb_to_rgb_area(mouse_data->grab_client, &screen, 
                                           left, top, 
                                           screen.width, screen.height)) ||
           PP_ERR == (ret = cat_mse32(&screen, mouse_data->shape,
                                      mouse_data->mask, 0, 0, 
                                      screen.width, screen.height, 
                                      32, &found))) {
            CDMSG(1, "position (%4d, %3d) is no more reliable, exiting\n", 
                  estim_x, estim_y);
            continue;
        }
        
        estim_x = left + found.pos_x;
        estim_y = top + found.pos_y;
        cat_set_position_estimate(obj, 1, estim_x, estim_y, pp_hrtime()?);
        CDMSG(1, "tracked new reliable position (%4d, %3d)\n", 
              estim_x, estim_y);
        
        dx = x - estim_x;
        dy = y - estim_y;
/*
        diff_x = dx < 0 ? -MIN(OHDIV(-dx, 4), (int)shape_width)
                        : MIN(OHDIV(dx, 4), (int)shape_width);
        diff_y = dy < 0 ? -MIN(OHDIV(-dy, 4), (int)shape_height)
                        : MIN(OHDIV(dy, 4), (int)shape_height);
*/
        diff_x = dx < 0 ? -OHDIV(-dx, 4) : OHDIV(dx, 4);
        diff_y = dy < 0 ? -OHDIV(-dy, 4) : OHDIV(dy, 4);

        pp_km_cat_enqueue_ptr_move(diff_x, diff_y);

        CDMSG(1, "moving dx = %d (%d), dy = %d (%d)\n", diff_x, dx, diff_y, dy);
        obj->data_unit_mouse.send_mouse_data(obj, diff_x, diff_y * (-1), 
                                              z, buttonmask);
        usleep(11111);
    };
    
    CDMSG(1, "returning, ret = %d, dx = %d, dy = %d\n", ret, dx, dy);
    free(screen.rgb);
    
    return ret;
}
#endif

static int send_ptrmove(driver_t *obj,
                        const int x, const int y, const int z, 
                        const unsigned char buttonmask,
                        const unsigned char abs_rel ) {
    int16_t diff_x, diff_y, tdiff_x, tdiff_y;
    data_cat_t* mouse_data;
    u_int16_t estim_x, estim_y;

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    
    CDMSG(0, "x: %d, y: %d, z: %d, buttonmask %u, abs_rel %u\n",
          x, y, z , buttonmask, abs_rel);
    
    pp_km_cat_enqueue_ptr_pos(x, y);

    if(PP_ERR == cat_get_position_estimate(obj, &estim_x, &estim_y) ||
       PP_ERR == cat_ptr_approach(obj, estim_x, estim_y, 
                                  x, y, z, buttonmask)) {
        /* traditional style */
        
        /* movement since the last known absolute position */
        diff_x = x - mouse_data->rc_abs.x;
        diff_y = y - mouse_data->rc_abs.y;
    
        mouse_data->rc_abs.x = x;
        mouse_data->rc_abs.y = y;
//printf("tweb: >>>>>>>>>>>>> set rc_abs (%4d; %3d) <<<<<<<<<<<<<<<<<<<<<\n", mouse_data->rc_abs.x, mouse_data->rc_abs.y);
        
        pp_km_cat_enqueue_ptr_move(diff_x, diff_y);
        
        /* translate */
        tdiff_x = cat_translation_lookup(obj, diff_x);
        tdiff_y = cat_translation_lookup(obj, diff_y); 
        CDMSG(0, "translated ptr move (%4d, %3d) -> (%4d, %3d)\n",
              diff_x, diff_y, tdiff_x, tdiff_y);
        
        obj->data_unit_mouse.send_mouse_data(obj, tdiff_x, tdiff_y * (-1), 
                                              z, buttonmask);
    }
    
    return 0;
}
#else
static int send_ptrmove(driver_t *obj,
                        const int x, const int y, const int z, 
                        const unsigned char buttonmask,
                        const unsigned char abs_rel ) {
    int16_t diff_x, diff_y, tdiff_x, tdiff_y, pdiff_x, pdiff_y;
    data_cat_t* mouse_data;
//    u_int16_t estim_x, estim_y;

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    
//    CDMSG(1, "x: %d, y: %d, z: %d, buttonmask %u, abs_rel %u\n",
//          x, y, z , buttonmask, abs_rel);
    
    pp_km_cat_enqueue_ptr_pos(x, y);

    /* movement since the last known absolute position */
    diff_x = x - mouse_data->rc_abs.x;
    diff_y = y - mouse_data->rc_abs.y;

#if 0
    if(PP_ERR != cat_get_position_estimate(obj, &estim_x, &estim_y)) {
        /* we got a (more or less ;-)) reliable position estimate,
           merge it with the movement diff */
           
        CDMSG(1, "position is reliable, diff is %d, %d\n",
              mouse_data->rc_abs.x - estim_x, mouse_data->rc_abs.y - estim_y);
        diff_x += mouse_data->rc_abs.x - estim_x; // fast approach
        diff_y += mouse_data->rc_abs.y - estim_y; // fast approach
//        diff_x += (mouse_data->rc_abs.x - estim_x) >> 1; // medium approach
//        diff_y += (mouse_data->rc_abs.y - estim_y) >> 1; // medium approach
//        diff_x += (mouse_data->rc_abs.x - estim_x) >> 2; // slow approach
//        diff_y += (mouse_data->rc_abs.y - estim_y) >> 2; // slow approach
//printf("estimate is reliable, applying to diff => (%d; %d)\n", diff_x, diff_y);
        CAT_INVALIDATE_POSITION_ESTIMATE;
    }
#else
    if(PP_SUC == cat_get_position_diff(obj, &pdiff_x, &pdiff_y)) {
        /* we got a (more or less ;-)) reliable position estimate,
           merge it with the movement diff */
           
        CDMSG(1, "position is reliable, diff is %d, %d\n", pdiff_x, pdiff_y);
        diff_x += pdiff_x; // fast approach
        diff_y += pdiff_y; // fast approach
        CAT_INVALIDATE_POSITION_ESTIMATE;
    }
#endif
    mouse_data->rc_abs.x = x;
    mouse_data->rc_abs.y = y;
//printf("tweb: >>>>>>>>>>>>> set rc_abs (%4d; %3d) <<<<<<<<<<<<<<<<<<<<<\n", mouse_data->rc_abs.x, mouse_data->rc_abs.y);
    
    pp_km_cat_enqueue_ptr_move(diff_x, diff_y);
    
    /* translate */
    tdiff_x = cat_translation_lookup(obj, diff_x);
    tdiff_y = cat_translation_lookup(obj, diff_y); 
    CDMSG(0, "translated ptr move (%4d, %3d) -> (%4d, %3d)\n",
          diff_x, diff_y, tdiff_x, tdiff_y);
    
    obj->data_unit_mouse.send_mouse_data(obj, tdiff_x, tdiff_y * (-1), 
                                          z, buttonmask);
    
    return 0;
}
#endif
#else /* PP_KM_CAT_INTELLI */
static int send_ptrmove(driver_t *obj, int x, int y, int z, 
                        unsigned char buttonmask, 
                        unsigned char abs_rel) {
    int16_t diff_x, diff_y;
    data_cat_t* mouse_data;
    data_imouse_t* imouse_data;
    t_mouse_data* md;
    int ret;
#if !defined(CAT_APPROACH_ON_THE_FLY)
    u_int16_t estim_x, estim_y;
    int __approach__tolerance__TODO__ = 20;
#endif /* !CAT_APPROACH_ON_THE_FLY */

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);
    imouse_data = (data_imouse_t*)obj->data_conv_mouse.data;
    assert(imouse_data);
    assert(mouse_data->current_kvm_port);
    md = &mouse_data->current_kvm_port->mouse_data;
    assert(md);
    
    CDMSG(CAT_CDMSG_BLABLA, "x: %d, y: %d, z: %d, buttonmask %u, abs_rel %u\n",
          x, y, z , buttonmask, abs_rel);
    
    /* we don't know about the mouse, so let's get knowledge */
    if (mouse_data->state == ERIC_MOUSE_NOT_VALID) {
	CDMSG(CAT_CDMSG_SYNC_INFO, "Syncing mouse, please wait ...\n");
	ret = cat_intelli_get_mouse_universe(obj);
	switch(ret) {
	  case ERIC_MOUSE_UNIVERSE_OK:
	      CDMSG(CAT_CDMSG_SYNC_INFO, "Mouse sync successful.\n");
	      break;
	  case ERIC_MOUSE_UNIVERSE_FAILED:
	      CDMSG(CAT_CDMSG_SYNC_INFO, "Could not sync mouse.\n");
	      //MUTEX_UNLOCK(&context->kvm_port->mtx);
	      return 0;   	      
	  case ERIC_MOUSE_UNIVERSE_WARN_GOOD:
	      CDMSG(CAT_CDMSG_SYNC_INFO,
                    "Mouse sync ambiguous, might not work exactly.\n");
	      break;
	  case ERIC_MOUSE_UNIVERSE_WARN_BAD:
	      CDMSG(CAT_CDMSG_SYNC_INFO, 
                    "Mouse sync ambiguous, please try to change mouse "
                    "settings.\n");
	      break;
	}
#warning cat / imouse conflict
mouse_data->state = ERIC_MOUSE_IS_VALID;
	if (save_mouse_universe(obj) != PP_SUC) {
	    CDMSG(CAT_CDMSG_SYNC_INFO, "could not save mouse universe\n");
	}
    }

    pp_km_cat_enqueue_ptr_pos(x, y);

    if(abs_rel == PP_KM_PTR_MOVE_RELATIVE) {
        diff_x = x;
        diff_y = y;

        mouse_data->rc_abs.x += x;
        mouse_data->rc_abs.y += y;
//printf("tweb: >>>>>>>>>>>>> set rc_abs (%4d; %3d) <<<<<<<<<<<<<<<<<<<<<\n", mouse_data->rc_abs.x, mouse_data->rc_abs.y);
#warning DEBUG!
abort();
    } else {
        /* movement since the last known absolute position */
        diff_x = x - mouse_data->rc_abs.x;
        diff_y = y - mouse_data->rc_abs.y;

        mouse_data->rc_abs.x = x;
        mouse_data->rc_abs.y = y;
//printf("tweb: >>>>>>>>>>>>> set rc_abs (%4d; %3d) <<<<<<<<<<<<<<<<<<<<<\n", mouse_data->rc_abs.x, mouse_data->rc_abs.y);
    }

    pp_km_cat_enqueue_ptr_move(diff_x, diff_y);

#if !defined(CAT_APPROACH_ON_THE_FLY)
    if(PP_ERR == cat_get_position_estimate(obj, &estim_x, &estim_y) ||
       (abs((md->cur_x / MOUSE_RASTER ) - estim_x) < __approach__tolerance__TODO__ &&
        abs((md->cur_x / MOUSE_RASTER ) - estim_y) < __approach__tolerance__TODO__) ||
       PP_ERR == (ret = cat_ptr_approach(obj, estim_x, estim_y, 
                                         x, y, z, buttonmask)))
#endif /* !CAT_APPROACH_ON_THE_FLY */
    {
        /* traditional style */
        
        ret = cat_intelli_move_mouse(obj, x, y, z, buttonmask);
    }

    return ret;
}
#endif /* PP_KM_CAT_INTELLI */

static int sync_mouse(driver_t *obj , const int sync_type ) {
    /* we should be syncronous all the time... hopefully ;-) */

#if defined(PP_KM_CAT_INTELLI)
    data_cat_t* mouse_data;

    assert(obj);
    mouse_data = (data_cat_t*)obj->data_conv_mouse.data;
    assert(mouse_data);

    CDMSG(1, "PP_KM_CAT_INTELLI enabled, invalidating mouse state\n");
    mouse_data->state = ERIC_MOUSE_NOT_VALID;
#else /* PP_KM_CAT_INTELLI */
    CDMSG(1, "PP_KM_CAT_INTELLI disabled, do nothing\n");
#endif /* PP_KM_CAT_INTELLI */

    return 0;
}

/******************************************************************************/

/* TODO merge obj with data_conv_mouse_intelli.c */
static int
alloc_port_state_for_kvm_unit(driver_t *obj, int kvm_unit)
{
    data_cat_t *mouse_data;
    
    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    if (mouse_data->kvm_port_state_list_units[kvm_unit] == NULL) {
	/* allocate mem for mouse state for max nr of ports in obj unit */
	mouse_data->kvm_port_state_list_units[kvm_unit] =
	    (km_kvm_entry_t *)calloc(sizeof(km_kvm_entry_t),
                                     PP_KVM_MAX_UNIT_PORT_COUNT);
    }

    return 0;
}

/* resets the states of all ports in KVM_UNIT (if allocated) */
/* TODO merge obj with data_conv_mouse_intelli.c */
static int
reset_port_state_for_kvm_unit(driver_t *obj, int kvm_unit)
{
    int i, j;
    km_kvm_entry_t *unit, *port; /* mouse data entries for ports of one unit */
    data_cat_t *mouse_data;
    
    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    if (mouse_data->kvm_port_state_list_units[kvm_unit]) {
	CDMSG(1, "resetting port state for unit %d\n", kvm_unit);

	/* initialize all ports in obj unit */
	unit = mouse_data->kvm_port_state_list_units[kvm_unit];
	if (unit) {
	    for (i = 0; i < PP_KVM_MAX_UNIT_PORT_COUNT; i++) {
		port = &unit[i];
		port->index = i;
		port->u_index = kvm_unit;

		/* initialize mouse values */
		port->mouse_data.cur_x = 500;
		port->mouse_data.cur_y = -500;
		port->mouse_data.accel_steps = 1;
		for (j = 0; j < MAX_ACCEL_STEPS; ++j) {
		    port->mouse_data.m[j] = 1.0 * MOUSE_RASTER;
		    port->mouse_data.ymax[j] = 0;
		}
		port->mouse_data.fm_corr = non_mouse_corr; 
                // 1:1 correction fkt;

		/* load mouse settings for obj port */	
		port->mouse_data.state = ERIC_MOUSE_IS_VALID;
		if (load_mouse_universe(obj, port) != PP_SUC) {
		    port->mouse_data.state = ERIC_MOUSE_NOT_YET;
		}
	    }
	}
    }

    return 0;
}

/* TODO merge obj with data_conv_mouse_intelli.c */
static int
free_port_state_for_kvm_unit(driver_t *obj, int kvm_unit)
{
    data_cat_t * mouse_data;
    void * p;
    
    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);

    p = mouse_data->kvm_port_state_list_units[kvm_unit];
    /* unit not present -> release mem */
    CDMSG(1, "freeing unit %d, pointer %p\n", kvm_unit, p);
    free(p);
    mouse_data->kvm_port_state_list_units[kvm_unit] = NULL;

    return 0;
}

/* TODO merge obj with data_conv_mouse_intelli.c */
static void
set_current_kvm_port(driver_t *obj)
{
    data_cat_t* mouse_data;
    u_char kvm_unit;
    u_short kvm_port;

    assert(obj);
    mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
    assert(mouse_data);
    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, 
                                      &kvm_unit, &kvm_port) == 0) {
	CDMSG(1, "set current_kvm_port for data_link %hhu: to %hu (unit %hhu)\n",
              obj->data_link, kvm_port, kvm_unit);
	MUTEX_LOCK(&mouse_data->kvm_port_mtx);
	mouse_data->current_kvm_port = &mouse_data->kvm_port_state_list_units[kvm_unit][kvm_port];
    	MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    } else {
	CDMSG(1, "pp_kvm_get_unit_port_for_data_link() failed\n");
	MUTEX_LOCK(&mouse_data->kvm_port_mtx);
	mouse_data->current_kvm_port = NULL;
    	MUTEX_UNLOCK(&mouse_data->kvm_port_mtx);
    }
}

/* TODO merge obj with data_conv_mouse_intelli.c */
static void
propchange_handler(pp_propchange_listener_t * listener, 
		   unsigned short prop, unsigned short prop_flags )
{
    local_propchange_listener_t *mylistener;
    driver_t *obj;
    data_cat_t* mouse_data;
    int kvm_unit;
    
    mylistener = (local_propchange_listener_t *)listener;
    obj = mylistener->obj;
    assert(obj);
    mouse_data = (data_cat_t *)obj->data_conv_mouse.data;
    assert(mouse_data);

    switch (prop) {
      case PP_PROP_KVM_PORT_SWITCHED:
	set_current_kvm_port(obj);
	break;
      case PP_PROP_VIDEO_MODE_CHANGED:
//	MUTEX_LOCK(&mouse_data->move_mouse_corner_flag_mtx);
//	mouse_data->move_mouse_corner_flag = 1;
//	MUTEX_UNLOCK(&mouse_data->move_mouse_corner_flag_mtx);
	break;
      case PP_PROP_KVM_UNIT_ADD_OR_REMOVE:
	assert(obj);
	mouse_data = (data_cat_t*) obj->data_conv_mouse.data;
	assert(mouse_data);
	for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
	    if (mouse_data->kvm_port_state_list_units[kvm_unit] == NULL &&
                pp_kvm_is_unit_present(kvm_unit)) {
		// unit was added
		alloc_port_state_for_kvm_unit(obj, kvm_unit);
		reset_port_state_for_kvm_unit(obj, kvm_unit);
	    } else if (mouse_data->kvm_port_state_list_units[kvm_unit] != NULL
                       && !pp_kvm_is_unit_present(kvm_unit)) {
		// unit was removed
		free_port_state_for_kvm_unit(obj, kvm_unit);
	    }
	}
	break;
      default:
	pp_log("%s: %s(): Unhandled property %hu received.\n",
               __FILE__, ___F, prop);
	break;
    }
}

