/* layer: data_unit
   data path: data_unit_kbd->send_kbd_data()  ->  comm_proto->send_pdu()
*/

/* system includes */
#include <stdlib.h>
#include <unistd.h>

/* firmware includes */
#include <pp/base.h>
#include <pp/kvm.h>
#include <pp/rfb.h>
#include <pp/cfg.h>
#include <pp/propchange.h>
#include <pp/hal_icpmmd.h>

/* local includes */
#include "driver.h"
#include "data_unit_kbd_ps2.h"
#include "data_conv_kbd.h"
#include "data_kbd.h"
// #include "km_dd.h"
#include "sc1to2conv.h"
#include "kbd.h"

#define KM_KEYPRESS_DELAY 30000

/* object functions */
static int resume(driver_t* obj);
static int suspend(driver_t* obj);
static int cleanup(driver_t* obj);
static int reconfigure(driver_t* obj);
static int send_kbd_data (driver_t* obj, const u_char key[], int length, void* status);
static const char * get_info (driver_t* obj, const char* parameter);
static int is_equal(driver_t* obj, driver_t* d);
static int reset_driver(driver_t* obj);
static int get_led_status(driver_t* obj);

/* local object functions */
static void configure_keylayouts_on_unit(driver_t* obj, u_char kvm_unit);
static void configure_keylayout_on_port(driver_t* obj, u_char kvm_unit, u_short kvm_port);
static int handle_host_cmd (driver_t* obj, const u_char* opt, size_t len);
static int handle_keyevent_prefix(driver_t* obj, u_char sc[],
				  keydef_t* keydef, u_char event,
				  keyboard_state_t* state);
static int handle_keyevent_special(driver_t* obj, u_char sc[],
				   keydef_t* keydef, u_char event,
				   keyboard_state_t* state);
static int handle_keyevent_normal(driver_t* obj, u_char sc[],
				  keydef_t* keydef, u_char event);
static int handle_keyevent_normal_sc3(driver_t* obj, u_char sc[],
				      keydef_t* keydef,
				      u_char event,
				      keyboard_state_t* state);
static keyspecial_t* get_keyspecial(keyboard_state_t *state, keydef_t* keydef);
static int update_active_state(driver_t* obj, keydef_t* keydef,
			       u_char event,
			       keyboard_state_t* keyboard_state);
static keylayout_t* get_keylayout(driver_t* obj, int layoutid);
static int send_scancodes(driver_t* obj, const u_char sc[],
			  int length, u_char scset);
static int send_scancodes_intern(driver_t* obj, const u_char * scs, int length,
				 u_char scset);
static int packetize_scanode(const u_char* sc_buf, int sc_buf_len, u_char scset);
static int stretch_scancode_pktlen_by_prefix(int len, u_char prefix,
					     u_char scset);
static keydef_t* lookup_keydef_by_sc3(keylayout_t* layout, u_char sc3);
static int lookup_keyidx_by_sc(keylayout_t* layout, u_char scset, const u_char *sc,
			       size_t len);
static void make_break_mask_set_for_all(driver_t* obj, u_char mbmask);
static void make_break_mask_set_scs3_defaults(driver_t* obj);
static int alloc_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);
static int reset_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);
static int free_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit);
    
static void handle_local_key(keylayout_t* layout, u_char scset,
			     const u_char* scancode, size_t len);

/* property change handler */
static void propchange_handler(pp_propchange_listener_t * listener, 
                              u_short prop, u_short param);

typedef struct {
    pp_propchange_listener_t propchange_listener;
    driver_t* obj;
} local_propchange_listener_t;

/* this is our private data structure */
typedef struct {
  /* here are the "real" layouts */
    keylayout_t *layouts[MAX_KBD_LAYOUTS];

    /*
     * array of keyboard_state_list pointers, one for each kvm unit
     *   - a ptr is NULL if no mem is allocated
     *   - each ptr points to an array, indexed by kvm port, holding
     *     a keyboard_state_t (with a link to the layout) for each port
     */
    keyboard_state_t *keyboard_state_list_units [PP_KVM_MAX_UNIT_COUNT];

    /* listens e.g. for kvm unit add/remove events */
    local_propchange_listener_t propchange_listener;

    /*
     * this is channel specific, because there are only two possibilities:
     *   - there is no kvm-switch -> no kvm-ports - so the host tells us
     *     always the charset
     *   - kvm-switch which emulates the different charsets - lara<->switch
     *     use always the same cs
     *   - kvm-switch sends itself the cs back to us when the port is switched
     */
    int current_scset; 
    int default_scset;
    pp_stopwatch_t t_wait;
    u_char make_break_mask[MAX_KEYDEFS];

    /* the following is for thread safe and synchronized changing
     * of kbd-state by the kme-thread, sending cmd from kme
     * the magic is as follows:
     * kbd_mtx locks the complete structure and send_state
     * send_state will be set to UNIT_KBD_SEND_STATE_SENDING when the
     * kbd-thread is down in the comm-proto and waiting for the kme
     * to return its prompt. during that time kbd_mtx won't be locked
     * and the kme-thread may change the kbd-state. It then needs
     * to send an ack to the kme. If the send-state is UNIT_KBD_SEND_STATE_IDLE
     * then the kme-thread will send the ack, else, the kdb-thread will do
     * it once it resurfaces from send_scancode fkt.
     */
#define UNIT_KBD_SEND_STATE_SENDING 1
#define UNIT_KBD_SEND_STATE_IDLE    2
    pthread_mutex_t kbd_mtx; /* locks sc_set and make-break-mask */
    int send_state;          /* determines whether send_scancodes active */
    int ack_pending;         /* ack needs to be send to kme */
    
    int led_status;          /* status of keyboard leds */
} unit_kbd_data_t;

int
init_data_unit_kbd_ps2(driver_t* obj, int default_scset)
{
    unit_kbd_data_t *unit_kbd_data;
    int i;
    u_char kvm_unit;

    assert(obj);
    /* create the driver */    
    obj->data_unit_kbd.id = __FUNCTION__;
    obj->data_unit_kbd.is_equal = is_equal;
    obj->data_unit_kbd.cleanup = cleanup;
    obj->data_unit_kbd.reconfigure = reconfigure;
    obj->data_unit_kbd.send_kbd_data = send_kbd_data;
    obj->data_unit_kbd.get_info = get_info;
    obj->data_unit_kbd.suspend = suspend;
    obj->data_unit_kbd.resume = resume;
    obj->data_unit_kbd.handle_host_cmd = handle_host_cmd;
    obj->data_unit_kbd.reset_driver = reset_driver;
    obj->data_unit_kbd.get_led_status = get_led_status;
    /* initializes our private data */
    obj->data_unit_kbd.data = calloc(1, sizeof(unit_kbd_data_t));
    assert(obj->data_unit_kbd.data);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    pthread_mutex_init(&(unit_kbd_data->kbd_mtx), NULL);

    /* no layouts loaded at startup - not touched on reset */
    for (i = 0; i < MAX_KBD_LAYOUTS; i++) {
	unit_kbd_data->layouts[i] = NULL;
    }

    /* default scancode set - not touched on reset */
    unit_kbd_data->default_scset = default_scset;

    unit_kbd_data->t_wait.t_start = 0;
    unit_kbd_data->t_wait.t_stop = 0;

    /* register listener for kvm unit insertion/removal events */
    pp_propchange_init_listener(
           (pp_propchange_listener_t *)&unit_kbd_data->propchange_listener,
           propchange_handler);
    unit_kbd_data->propchange_listener.obj = obj;
    pp_propchange_add(
           (pp_propchange_listener_t *)&unit_kbd_data->propchange_listener,
           PP_PROP_KVM_UNIT_ADD_OR_REMOVE);
    
    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
       if (pp_kvm_is_unit_present(kvm_unit)) {
           alloc_kbd_state_for_kvm_unit(obj, kvm_unit);
       }
    }
    
    /* sets the data to standard values */
    reset_driver(obj);
    
    unit_kbd_data->send_state = UNIT_KBD_SEND_STATE_IDLE;
    unit_kbd_data->ack_pending = 0;
    return 0;
}

static int
alloc_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    unit_kbd_data_t *unit_kbd_data;
    keyboard_state_t *p_kbdstat = NULL;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*)obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    if (unit_kbd_data->keyboard_state_list_units[kvm_unit] == NULL) {
	/* unit present -> allocate kbd state mem for max nr of ports */
	p_kbdstat = unit_kbd_data->keyboard_state_list_units[kvm_unit] =
	    (keyboard_state_t *) calloc(sizeof(keyboard_state_t),
					PP_KVM_MAX_UNIT_PORT_COUNT);
	KD("%s: %s(): alloc, pointer %hhu: %p\n", __FILE__, ___F, kvm_unit,
	   p_kbdstat);
	if (!p_kbdstat) {
	    pp_log_err("%s: %s: allocating memory failed", __FILE__, ___F);
	    exit(1);
	}
	configure_keylayouts_on_unit(obj, kvm_unit);
    } else {
	KD("%s: %s(): unit %hhu was already allocated: %p\n", __FILE__, ___F,
	   kvm_unit, p_kbdstat);
    }

    return 0;
}

/* resets the keyboard states of all ports in KVM_UNIT (if allocated) */
static int
reset_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    int key;
    u_short kvm_port;
    unit_kbd_data_t *unit_kbd_data;
    keyboard_state_t *p_kbdstat;
    
    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*)obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    if (unit_kbd_data->keyboard_state_list_units[kvm_unit]) {
	// TODO remove
	KD("%s: %s(): resetting port state for unit %hhu\n", __FILE__, ___F, kvm_unit);

	/* reset the keyboard states, but not the links to the layouts */
	for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; ++kvm_port) {
	    for (key = 0; key < MAX_KEYDEFS; key++) {
		p_kbdstat = &unit_kbd_data->keyboard_state_list_units[kvm_unit][kvm_port];
		p_kbdstat->keys[key].state = KEY_STATE_NONE;
	    }
	}
    }

    return 0;
}

static int
free_kbd_state_for_kvm_unit(driver_t* obj, u_char kvm_unit)
{
    unit_kbd_data_t *unit_kbd_data;
    keyboard_state_t *p_kbdstat;
    
    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*)obj->data_unit_kbd.data;
    assert(unit_kbd_data);


    /* unit not present -> release mem */
    p_kbdstat = unit_kbd_data->keyboard_state_list_units[kvm_unit];
    KD("%s: %s: %hhu: freeing, pointer %p\n", __FILE__, ___F, kvm_unit, p_kbdstat);

    free(unit_kbd_data->keyboard_state_list_units[kvm_unit]);
    unit_kbd_data->keyboard_state_list_units[kvm_unit] = NULL;

    return 0;
}

static int
reset_driver(driver_t* obj)
{
    unit_kbd_data_t *unit_kbd_data;
    u_char kvm_unit;
    
    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    MUTEX_LOCK(&unit_kbd_data->kbd_mtx);
    
    /* set default make/break mask */
    make_break_mask_set_scs3_defaults(obj);

    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
	reset_kbd_state_for_kvm_unit(obj, kvm_unit);
    }
    /* this is a little bit ugly, because the kbd layouts (which are stored in the keyboard_state_list)
       are cleared too - so just do a reconfigure to fill them again

       note: the 100% correct solution would be to separate the data structures... */
    reconfigure(obj);
    
    unit_kbd_data->current_scset = unit_kbd_data->default_scset;
    
    MUTEX_UNLOCK(&unit_kbd_data->kbd_mtx);
    return 0;
}

static int get_led_status(driver_t* obj) {
    /* note: PS/2 and USB led status differ, convert to USB before returning */
    u_char led;
    u_char scrl = 0;
    led = ((unit_kbd_data_t*)(obj->data_unit_kbd.data))->led_status;
    scrl = led & 0x01;
    led = led >> 1;
    if (scrl) { led = led | 0x04; }
    return led;
}

static int resume(driver_t * /* obj */) {
    return 0;
}

static int suspend(driver_t * /* obj */) {
    return 0;
}

static int is_equal(driver_t* obj, driver_t* d) {
    unit_kbd_data_t* unit_kbd_data;
    unit_kbd_data_t* unit_kbd_data_d;
    
    assert(obj);
    assert(d);
    
    if (!strcmp(obj->data_unit_kbd.id, d->data_unit_kbd.id)) {
	unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
	assert(unit_kbd_data);
	
	unit_kbd_data_d = (unit_kbd_data_t*) d->data_unit_kbd.data;
	assert(unit_kbd_data_d);

	if (unit_kbd_data->default_scset == unit_kbd_data_d->default_scset) {
	    return 1;
	} else {
	    return 0;
	}
    } else {
	return 0;
    }
}

static int
cleanup(driver_t* obj)
{
    int i;
    unit_kbd_data_t* unit_kbd_data;
    u_char kvm_unit;
    
    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    pp_propchange_remove_all(
           (pp_propchange_listener_t *)&unit_kbd_data->propchange_listener);

    /* cleanup possible allocated layouts */
    for (i = 0; i < MAX_KBD_LAYOUTS; i++) {
	/* no layouts loaded at startup */
	keylayout_t *layout = unit_kbd_data->layouts[i];
	if (layout != NULL) {
	    layout->cleanup(layout);
	}
	unit_kbd_data->layouts[i] = NULL;
    }
    
    /* free kbd state mem for all units */
    for (kvm_unit = 0; kvm_unit < PP_KVM_MAX_UNIT_COUNT; ++kvm_unit) {
       free_kbd_state_for_kvm_unit(obj, kvm_unit);
    }
    
    return 0;
}

static int
reconfigure(driver_t* obj)
{
    u_char kvm_unit;
    
    assert(obj);
    
    KD("data_unit_kbd_ps2: reconfigure...\n");
    
    /* FIXME kvm_unit: currently only support for unit 0 implemented */
    kvm_unit = 0;

    /* set the apropriate layout */
    configure_keylayouts_on_unit(obj, kvm_unit);

    return 0;
}

static void
configure_keylayouts_on_unit(driver_t* obj, u_char kvm_unit)
{
    u_short kvm_port;

    for (kvm_port = 0; kvm_port < PP_KVM_MAX_UNIT_PORT_COUNT; ++kvm_port) {
	/* set the apropriate layout */
	configure_keylayout_on_port(obj, kvm_unit, kvm_port);
    }
}

static void
configure_keylayout_on_port(driver_t* obj, u_char kvm_unit, u_short kvm_port)
{
    unit_kbd_data_t *unit_kbd_data;
    int kbdmodel;
    char *val;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*)obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    pp_cfg_get(&val, "unit[%u].port[%u].kbd.model", kvm_unit, kvm_port);
    if (!strcmp(val, KM_KBD_PC104_STR)) {
	kbdmodel = KBD_LAYOUT_PC104;
    } else if (!strcmp(val, KM_KBD_PC109_STR)) {
	kbdmodel = KBD_LAYOUT_PC109;
    } else if (!strcmp(val, KM_KBD_MAC_STR)) {
	kbdmodel = KBD_LAYOUT_MAC;
    } else {
	/* fallback in final - assert in devel */
	kbdmodel = KBD_LAYOUT_PC104;
	assert(0);
    }
    free(val);

    if (unit_kbd_data->keyboard_state_list_units[kvm_unit] != NULL) {
	unit_kbd_data->keyboard_state_list_units[kvm_unit][kvm_port].layout = get_keylayout(obj, kbdmodel);
    }
}

static keylayout_t* get_keylayout(driver_t* obj, int layoutid) {
    unit_kbd_data_t *unit_kbd_data;
    
    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    /* check that the layout id is in the correct range */
    if (!(0 <= layoutid && layoutid < MAX_KBD_LAYOUTS)) return NULL;
    
    if (unit_kbd_data->layouts[layoutid] == NULL) {
	/* our layout is not created yet */
	switch (layoutid) {
	  case KBD_LAYOUT_PC104:
	      unit_kbd_data->layouts[layoutid] = keylayout_104pc_init();
	      break;
	  case KBD_LAYOUT_PC109:
	      unit_kbd_data->layouts[layoutid] = keylayout_109pc_init();
	      break;
	  case KBD_LAYOUT_MAC:
	      unit_kbd_data->layouts[layoutid] = keylayout_mac_init();
	      break;
	}
    }

    return unit_kbd_data->layouts[layoutid];
}

static const char* get_info (driver_t * /* obj */ , const char* parameter) {
    if (parameter == NULL) return NULL;
    
    if (!strcmp(parameter, "update_state_before")) {
	return "no";
    } else if (!strcmp(parameter, "bulk_keys")) {
	return "no";
    } else {
	return NULL;
    }
};

static int
send_kbd_data(driver_t* obj, const u_char keys[], int length, void * /* pstate */)
{
    u_char kvm_unit;
    u_short kvm_port;
    unit_kbd_data_t* unit_kbd_data;
    keyboard_state_t *state;
    u_char event, key, scset;
    keydef_t* keydef;
    int ret = 0;
    u_char sc[MAX_SCANCODES];
    int sc_length;
    

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port)
	!= 0) {
	pp_log("%s(): pp_kvm_get_unit_port_for_data_link() failed\n", ___F);
	return ret;
    }
	 
    state = &unit_kbd_data->keyboard_state_list_units[kvm_unit][kvm_port];
    assert(state);
    
    /* nobody should call us without a key or with more than one key! */
    if (length != 1) return ret;

    /* we need a layout */
    if (state->layout == NULL) {
	pp_log("no layout defined\n");
	return ret;
    }
    
    event = (keys[0] & 0x80) ? KEY_EVENT_PRESSED : KEY_EVENT_RELEASED;
    key   = keys[0] & 0x7F;
    if (key >= MAX_KEYDEFS) return ret;

    /* Mutual exclusion with handle-host-command */
    MUTEX_LOCK(&unit_kbd_data->kbd_mtx);

    KD("KeyEvent %d for %d using layout %d\n", event, key, state->layout->id);

    scset = unit_kbd_data->current_scset;
    keydef = &state->layout->keys[key];

    if (scset == 3) {
	 sc_length = handle_keyevent_normal_sc3(obj, sc, keydef, event,
						state);
    } else switch (keydef->type) {
      case KEY_TYPE_NORMAL:
	  sc_length = handle_keyevent_normal(obj, sc, keydef, event);
	  break;
      case KEY_TYPE_SPECIAL:
	  sc_length = handle_keyevent_special(obj, sc, keydef, event, state);
	  break;
      case KEY_TYPE_PREFIX:
	  sc_length = handle_keyevent_prefix(obj, sc, keydef, event, state);
	  break;
      case KEY_TYPE_NONE:
	  sc_length = 0;
	  break;
      default:
	  sc_length = 0;
	  assert(0);
    }

    if (sc_length > 0) {
	unit_kbd_data->send_state = UNIT_KBD_SEND_STATE_SENDING;
	MUTEX_UNLOCK(&unit_kbd_data->kbd_mtx);
	
	ret = send_scancodes(obj, sc, sc_length, scset);

	MUTEX_LOCK(&unit_kbd_data->kbd_mtx);
	if (ret == 0 && keydef->type == KEY_TYPE_NORMAL) {
	    /* update modifier states */
	    update_active_state(obj, keydef, event, state);
	}
	if (unit_kbd_data->ack_pending) {
	    obj->comm_proto.send_cmd_ack(obj);
	    unit_kbd_data->ack_pending = 0;
	}
	unit_kbd_data->send_state = UNIT_KBD_SEND_STATE_IDLE;
    }

    MUTEX_UNLOCK(&unit_kbd_data->kbd_mtx);
    return ret;
}

static int
handle_keyevent_normal(driver_t *obj, u_char sc[],
		       keydef_t* keydef, u_char event)
{
    int i = 0;
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    
    KD("KeyNr %d (normal), scset %d\n", keydef->nr,
       unit_kbd_data->current_scset);
    while ((sc[i] = keydef->sc[i]) != 0 && i < MAX_SCANCODES_NORMAL) {
	if (event == KEY_EVENT_RELEASED) {
	    sc[i] |= 0x80;
	}
	
	i++;
    }
    return i;
}

static int
handle_keyevent_normal_sc3(driver_t* obj, u_char sc[],
			   keydef_t* keydef, u_char event,
			   keyboard_state_t* /* state */)
{
    u_char i=0;
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    
    KD("KeyNr %d (normal), scset 3\n", keydef->nr);

    if (!keydef->sc3) return 0;
	
    if (event == KEY_EVENT_RELEASED) {
	if (!(unit_kbd_data->make_break_mask[keydef->nr]
	      & KEY_MASK_SEND_BREAK)) return 0;
	sc[i++] = 0xF0;
    }

    sc[i++] = keydef->sc3;
    return i;
}
    
static int
handle_keyevent_special(driver_t* obj, u_char sc[],
			keydef_t* keydef, u_char event,
			keyboard_state_t* state)
{
    int i = 0;
    u_char* sclist;
    keyspecial_t *ks;
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    
    KD("KeyNr %d (special), scset %d\n", keydef->nr,
       unit_kbd_data->current_scset);

    if (!(ks = get_keyspecial(state, keydef))) return -1;

    if (event == KEY_EVENT_PRESSED) {
	sclist = ks->mk;
    } else {
	sclist = ks->brk;
    }
    
    while ((sc[i] = sclist[i]) != 0 && i < MAX_SCANCODES_SPECIAL)
	i++;

    return i;
}

static int
handle_keyevent_prefix(driver_t* obj, u_char sc[], keydef_t* keydef,
		       u_char event, keyboard_state_t* state)
{
    int i = 0, scidx = 0;
    u_char* sclist;
    keyspecial_t *ks;
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    KD("KeyNr %d (prefixed), scset %d\n", keydef->nr,
       unit_kbd_data->current_scset);

    ks = get_keyspecial(state, keydef);

    if (ks && event == KEY_EVENT_PRESSED) {
	/* send prefix scancodes */
	sclist = ks->mk;

	while ((sc[scidx] = sclist[i]) != 0 && i++ < MAX_SCANCODES_SPECIAL)
	    scidx++;
    }

    /* send real scancodes */
    sclist = keydef->sc;
    i = 0;
    while ((sc[scidx] = sclist[i]) != 0 && i < MAX_SCANCODES_NORMAL) {
	if (event == KEY_EVENT_RELEASED) {
	    sc[i] |= 0x80;
	}
	scidx++; i++;
    }
        
    i = 0;
    if (ks && event == KEY_EVENT_RELEASED) {
	/* send postfix scancodes */
	sclist = ks->brk;

	while ((sc[scidx] = sclist[i]) != 0 && i++ < MAX_SCANCODES_SPECIAL)
	    scidx++;
    }
    return scidx;
}

static keyspecial_t*
get_keyspecial(keyboard_state_t *state, keydef_t* keydef)
{
    u_char i, j, kreq, found=0;
    keydef_t *kreqdef;
    keyspecial_t* ks = NULL;
    
    if (!keydef || !(keydef->type & (KEY_TYPE_SPECIAL | KEY_TYPE_PREFIX))
	|| !keydef->special) return NULL;

    /* look if one of the key combinations is pressed */
    for (i = 0; i < MAX_KEYSPECIALS; i++) {
	ks = &keydef->special[i];

	/* nothing left */
	if (ks->keys[0] == 0xFF) {
	    KD("ks: nothing left\n");
	    ks = NULL;
	    break;
	}

	/* final key itself for special keys*/
	if (ks->keys[0] == keydef->nr) {
	    KD("ks: found myself\n");
	    found = 1;
	    break;
	}

	/* look if combination fits */
	j = 0; found = 1;
	while ((kreq = ks->keys[j]) != 0xFF &&
	       j < MAX_SPECIAL_KEYREQS) {
	    kreqdef = &state->layout->keys[kreq];
	    KD("ks%d: looking for key %d...", i, kreqdef->nr);
	    if ((kreqdef->mod == KEY_MOD_NONE)||
		(kreqdef->mod == KEY_MOD_MODI
		 && state->keys[kreq].state != KEY_STATE_PRESSED) ||
		(kreqdef->mod == KEY_MOD_PERM
		 && state->keys[kreq].state != KEY_STATE_ACTIVE)) {
		KD("not found\n");
		found = 0;
		break;
	    } else {
		KD("ok\n");
	    }
	    j++;
	}

	if (found) {
	    break;
	}
    }

    if (!found) ks = NULL;
	
    return ks;
}

static int
send_scancodes(driver_t* obj, const u_char sc[], int length, u_char scset)
{
    int sc_conv_length, ret, send_sc_length = 0;
    u_char* sc_conv = NULL;
    const u_char* send_scs;
    
    assert(obj);
    
#ifdef KBD_DEBUG
    {
	int j;
	printf("Sending %d scancodes: ", length);
	for (j=0;j<length;j++) printf("%02x ", sc[j]);
	printf("\n");
    }
#endif   

    switch (scset) {
      case 2:
	  conv_sc1_sc2(sc, length, &sc_conv, &sc_conv_length);
	  send_scs = sc_conv;
	  send_sc_length = sc_conv_length;
	  break;	  
      case 1:
      case 3:
	  send_scs = sc;
	  send_sc_length = length;
	  break;	  
      default:
	  send_scs = NULL;
	  send_sc_length = 0;
	  assert(0);
    }

    ret = send_scancodes_intern(obj, send_scs, send_sc_length, scset);

    free(sc_conv);
    return ret;
}



static int send_scancodes_intern(driver_t* obj, const u_char* scs, int scsno,
				 u_char scset) {
    /* now we have some scancodes to send  -  decide now, if we should
       packetize them for the the kme design */
    unit_kbd_data_t* unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    const u_char* scptr;
    int scpos, sclen;
    u_long stop_time;
    int ret = 0;
    
    /*   TODO:  if(!kme_is_dead) {*/
    assert(obj);
    
    pp_hrtime_stop(&unit_kbd_data->t_wait);
    stop_time = (u_long)pp_hrtime_read(&unit_kbd_data->t_wait) / 1000;
    if (stop_time < KM_KEYPRESS_DELAY) {
	usleep(KM_KEYPRESS_DELAY - stop_time);
    }
    
    /* we need to packetize the scancode buf into single scancode *
     * packets in the same way the kme is expecting them.         */
    scpos = 0;
    while (scpos < scsno) {
	scptr = &scs[scpos];
	sclen = packetize_scanode(scptr, scsno - scpos, scset);
	scpos += sclen;

	ret = obj->comm_proto.send_pdu(obj, scptr, sclen, PP_KM_INPUT_KEYBOARD);
    }
    pp_hrtime_start(&unit_kbd_data->t_wait);
    return 0;
}

/*
 * returns the length (the end index) of a complete scandcode
 * in a continous buffer of scancodes
 */
static int
packetize_scanode(const u_char* sc_buf, int sc_buf_len, u_char scset)
{
    int sc_len = 1, i = 0;
    do {
	sc_len = stretch_scancode_pktlen_by_prefix(sc_len, sc_buf[i], scset);
	++i;
	// if obj assertion fails it were because of a
	// scancode generation bug in the vkeyboard
	(void)sc_buf_len;
	assert(i <= sc_buf_len);
    } while (i < sc_len);
    return sc_len;
}


/*
 * calculates the packet length of a complete scancode
 * depending on certain scancode prefixes
 */
static int
stretch_scancode_pktlen_by_prefix(int len, u_char prefix, u_char scset)
{
    int new_len = len;
    switch (prefix) {
      case 0xF0:
	  if (scset == 1) break; // 0xF0 is regular break code in sc-set 1
	  // else: fall through
      case 0xE0:
	  new_len += 1;
	  break;
      case 0xE1:
	  new_len += 2;
	  break;
      //default: // no stretching needed for normal scancodes
    }
    // never return more then 5: if obj assertion fails it
    // were because of a scancode generation bug in the vkeyboard
    assert(new_len <= 5);
    return new_len;
}



static int update_active_state(driver_t * /* obj */, keydef_t* keydef,
			       u_char event,
			       keyboard_state_t* state) {
	/* some comments about the state machine:
	   - press and release results in STATE_PRESSED and STATE_RELEASE

	   - to get the "active" state of NumLock, Caps and ScrollLock
	     (pressing caps actives CapsLock until the key is pressed again),
	     the STATE_ACTIVE acts in the same way as the original
	     state machine:

	   - N: (p) -> A
	        (r) -> N
	   - A: (p) -> P
	        (r) -> A
	   - P: (p) -> P
	        (r) -> N


	*/
    switch (keydef->mod) {
      case KEY_MOD_MODI:
	  state->keys[keydef->nr].state = (event == KEY_EVENT_PRESSED) ?
	      KEY_STATE_PRESSED : KEY_STATE_NONE;
	  KD("Modi State %d\n", state->keys[keydef->nr].state);
	  break;
      case KEY_MOD_PERM:
	  switch (state->keys[keydef->nr].state) {
	    case KEY_STATE_NONE:
		state->keys[keydef->nr].state = (event == KEY_EVENT_PRESSED) ?
		    KEY_STATE_ACTIVE : KEY_STATE_NONE;
		break;
	    case KEY_STATE_PRESSED:
		state->keys[keydef->nr].state = (event == KEY_EVENT_PRESSED) ?
		    KEY_STATE_PRESSED : KEY_STATE_NONE;
		break;
	    case KEY_STATE_ACTIVE:
		state->keys[keydef->nr].state = (event == KEY_EVENT_PRESSED) ?
		    KEY_STATE_PRESSED : KEY_STATE_ACTIVE;
		break;
	  }
	  KD("Perm State %d\n", state->keys[keydef->nr].state);
	  break;
      default:
	  ; /* do nothing */
    }
    return 0;
}

static void
handle_host_cmd_ack(driver_t* obj)
{
    unit_kbd_data_t *unit_kbd_data =(unit_kbd_data_t*)obj->data_unit_kbd.data;
    if (unit_kbd_data->send_state == UNIT_KBD_SEND_STATE_SENDING) {
	unit_kbd_data->ack_pending = 1;
    } else {
	obj->comm_proto.send_cmd_ack(obj);
    }
}

static int handle_host_cmd (driver_t* obj, const u_char* opt, size_t len) {
    keyboard_state_t *state;
    unit_kbd_data_t *unit_kbd_data;
    u_char kvm_unit;
    u_short kvm_port;
    keydef_t* kdef;
    
    assert(obj);
    if (len < 2) return 0;
    if (pp_kvm_get_unit_port_for_data_link(obj->data_link, &kvm_unit, &kvm_port) != 0) {
	pp_log("%s(): pp_kvm_get_unit_port_for_data_link() failed\n", ___F);
	return -1;
    }
    
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);

    // Mutual exclusion with send_kbd_data()
    MUTEX_LOCK(&unit_kbd_data->kbd_mtx);

    state = &unit_kbd_data->keyboard_state_list_units[kvm_unit][kvm_port];
    assert(state);
  
    KD("Got KME options : %2.2x, %2.2x\n", opt[0], opt[1]);
    switch (opt[0]) {
      case KEY_CMD_SET_DEFAULTS:
	  make_break_mask_set_scs3_defaults(obj);
	  if (opt[1] == KEY_CMD_SET_DEFAULTS_RST) {
	      unit_kbd_data->current_scset = unit_kbd_data->default_scset;
	      handle_host_cmd_ack(obj);
	      pp_propchange_enqueue(PP_PROP_KEYBOARD_RESET, 0);
	  }
	  break;

      case KEY_CMD_SCANCODE:
	  if (0 < opt[1] && opt[1] <= 3) {
	      unit_kbd_data->current_scset = opt[1];
	      handle_host_cmd_ack(obj);
	  }
	  break;

      case KEY_CMD_MAKE_BREAK:
	  switch (opt[1]) {
	    case KEY_CMD_MB_MAKE_BREAK:
		KD("KEY_CMD_MB_MAKE_BREAK: all\n");
		make_break_mask_set_for_all(obj, KEY_MASK_SEND_MAKE |
					    KEY_MASK_SEND_BREAK);
		break;
	    case KEY_CMD_MB_MAKE_ONLY:
		KD("KEY_CMD_MB_MAKE_ONLY: all\n");
		make_break_mask_set_for_all(obj, KEY_MASK_SEND_MAKE);
		break;
	    default:
		pp_log("handle_host_cmd: KEY_CMD_MAKE_BREAK:%d ?\n", opt[1]);
	  }
	  break;
	  
      case KEY_CMD_MB_MAKE_BREAK_KEY:
	  if (NULL != (kdef = lookup_keydef_by_sc3(state->layout, opt[1]))) {
	      KD("KEY_CMD_MB_MAKE_BREAK_KEY: key %d (sc3: 0x%x)\n",
		 kdef->nr, opt[1]);
	      unit_kbd_data->make_break_mask[kdef->nr] =
		  KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK;
	  } else {
	      KD("KEY_CMD_MB_MAKE_BREAK_KEY: key sc3: %d not "
		 "found\n", opt[1]);
	  }
	  break;
	  
      case KEY_CMD_MB_MAKE_ONLY_KEY:
	  if (NULL != (kdef = lookup_keydef_by_sc3(state->layout, opt[1]))) {
	      KD("KEY_CMD_MB_MAKE_ONLY_KEY: key %d (sc3: 0x%x)\n",
		 kdef->nr, opt[1]);
	      unit_kbd_data->make_break_mask[kdef->nr] = KEY_MASK_SEND_MAKE;
	  } else {
	      KD("KEY_CMD_MB_MAKE_ONLY_KEY: key sc3: 0x%x not found\n", opt[1]);
	  }
	  break;

#if defined(PRODUCT_ICPMMD)
      case KEY_CMD_HOTKEY: {
	  int kidx, port;
	  /* port switching port with key 1...9 a...c */
	  kidx = lookup_keyidx_by_sc(state->layout,
				     unit_kbd_data->current_scset, &opt[1], len - 1);
	  /* make codes have an additional bit set - kill it */
	  kidx &= ~0x80;
	  if (kidx == 10) {                     /* 0    */
	      port = 0;                         
	  } else if (kidx >= 1 && kidx <= 9) {	/* 1..9 */
	      port = kidx;
	  } else if (kidx == 29) {              /* a    */
	      port = 10;
	  } else if (kidx == 47) {              /* b    */
	      port = 11;
	  } else {
	      KD("KEY_CMD_HOTKEY: key %d (scset=%d); unknown kidx %d\n",
		 opt[1], unit_kbd_data->current_scset, kidx);
	      break;
	  }
	  
	  KD("KEY_CMD_HOTKEY: key %d (scset=%d) -> port %d\n",
	     opt[1], unit_kbd_data->current_scset, port);
	  if (PP_ERR == pp_hal_icpmmd_switch_kvm(port)) {
	      pp_log_err("pp_hal_icpmmd_switch_kvm(%d)", port);
	  }
	  break;
      }
#endif /* PRODUCT_ICPMMD */
      
      case KEY_CMD_LOCALKEY:
          handle_local_key(state->layout, unit_kbd_data->current_scset,
          		   &opt[1], len - 1);
          break;
	  
      case KEY_CMD_LED:
          ((unit_kbd_data_t*)(obj->data_unit_kbd.data))->led_status = opt[1];
          KD("KEY_CMD_LED: leds = %d\n", opt[1]);
          pp_propchange_enqueue(PP_PROP_KBD_LED_STATE_CHANGED, 0);
          break;
          
      default:	  
	  break;
	  /* unknown command */
    }

    MUTEX_UNLOCK(&unit_kbd_data->kbd_mtx);
    return 0;
}

#if defined(KBD_DEBUG)

static void print_scancode(const char *s, const u_char *sc, size_t len) {
    u_int i;
    printf("%s: ", s);
    for (i = 0; i < len; i++) printf(" %02x", sc[i]);
    printf("\n");
}

#else /* KBD_DEBUG */

#define print_scancode(s, sc, len)

#endif /* KBD_DEBUG */

/* convert scancodes to MAKE codes */
static u_int convert_sc(u_char scset, const u_char* sc_in,
			size_t len, u_char* sc_out, int *released) {
    u_int i, j;
    
    *released = 0;
    switch (scset) {
    	case 1:
    	    for (i = 0, j = 0; i < len; i++, j++) {
    	    	if (sc_in[i] != 0xE0 && sc_in[i] != 0xE1 && (sc_in[i] & 0x80)) {
    	    	    sc_out[j] = sc_in[i] & 0x7f;
    	    	    (*released)++;
    	    	} else {
    	    	    sc_out[j] = sc_in[i];
    	    	}
    	    }
    	    break;
    	case 2:
    	case 3:
    	    for (i = 0, j = 0; i < len; i++) {
    	    	if (sc_in[i] == 0xF0) {
    	    	    (*released)++;
    	    	    continue;
    	    	}
    	    	sc_out[j++] = sc_in[i];
    	    }
    	    break;
    	default:
    	    return 0;
    }
    
    print_scancode("old scancode", sc_in, len);
    print_scancode("new scancode", sc_out, j);

    return j;
}

static int match_sc_internal(u_char scset, u_char* sc1, u_char* sc2, size_t len) {
    u_int i;
    u_char kbuf[2*len];
    size_t klen = sizeof(kbuf);

    /* special handling for the PrintScreen key: it sends
       to make codes and 2 break codes; igonre one of each */
    if (len == 2 && sc1[0] == 0xE0 && sc1[1] == 0x2A) {
    	return 0;
    }

#if 0
    print_scancode("pressed scancode", sc2, len);
#endif // KBD_DEBUG

    if (scset == 2) {
    	conv_sc1_sc2_core(sc1, len, kbuf, (int*)&klen);

#if 0
    	print_scancode("sc1 scancode", sc1, len);
    	print_scancode("sc2 scancode", kbuf, klen);
#endif // KBD_DEBUG
    	
    	sc1 = kbuf;
    	if (klen != len) return 0;
    }
    
    for (i = 0; i < len; i++) {
    	if (sc1[i] != sc2[i]) return 0;
    }
    
    return 1;
}

static int match_sc(u_char scset, keydef_t* key, u_char* sc, size_t len) {
    if (key->type == KEY_TYPE_SPECIAL) {
    	u_int i;
    	
    	// FIXME!!!
    	if (len > MAX_SCANCODES_SPECIAL || key->special == NULL) return 0;
    	
    	for (i = 0; i < MAX_KEYSPECIALS; i++) {
    	    if (match_sc_internal(scset, key->special[i].mk, sc, len)) {
    	    	return 1;
    	    }
    	}
    	
    	return 0;
    } else if (len > MAX_SCANCODES_NORMAL) {
    	// no special key, but length > MAX_SCANCODES_NORMAL -> must be an error
    	return 0;
    }
    
    return match_sc_internal(scset, key->sc, sc, len);
}

/* -1 returned on error */
static int
lookup_keyidx_by_sc(keylayout_t* layout, u_char scset, const u_char *sc,
		    size_t len)
{
    int ret = -1, i;
    keydef_t* keys = layout->keys;
    u_char sc_make[len];
    int released;
    size_t len_make = convert_sc(scset, sc, len, sc_make, &released);
    
    if (len_make == 0) {
    	return -1;
    }
    
    switch(scset) {
      case 1:
      case 2:
	  for (i = 0; i < layout->count; ++i) {
	      if (match_sc(scset, &keys[i], sc_make, len_make)) {
		  ret = i;
		  goto finally;
	      }
	  }
	  break;
      case 3:
	  for (i = 0; i < layout->count; ++i) {
	      if (keys[i].sc3 == sc_make[0]) {
		  ret = i;
		  goto finally;
	      }
	  }
	  break;
    }
 finally:
    if (!released && ret != -1) {
    	ret |= 0x80;
    }
    return ret;
}

static keydef_t* lookup_keydef_by_sc3(keylayout_t* layout, u_char sc3) {
    int i;
    keydef_t* ret = NULL;
    keydef_t* keys = layout->keys;
    for (i = 0; i < layout->count; ++i) {
	if (keys[i].sc3 == sc3) {
	    ret = &keys[i];
	    break;
	}
    }
    return ret;
}

static void make_break_mask_set_for_all(driver_t* obj,
					u_char mbmask) {
    int i;
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    
    for (i = 0; i < MAX_KEYDEFS; ++i) {
	unit_kbd_data->make_break_mask[i] = mbmask;
    }
}

u_char make_break_mask_scs3_defaults[] = {
    KEY_MASK_SEND_MAKE,				/* ~		*/
    KEY_MASK_SEND_MAKE,				/* 1		*/
    KEY_MASK_SEND_MAKE,				/* 2		*/
    KEY_MASK_SEND_MAKE,				/* 3		*/
    KEY_MASK_SEND_MAKE,				/* 4		*/
    KEY_MASK_SEND_MAKE,				/* 5		*/
    KEY_MASK_SEND_MAKE,				/* 6		*/
    KEY_MASK_SEND_MAKE,				/* 7		*/
    KEY_MASK_SEND_MAKE,				/* 8		*/
    KEY_MASK_SEND_MAKE,				/* 9		*/
    KEY_MASK_SEND_MAKE,				/* 0		*/
    KEY_MASK_SEND_MAKE,				/* -		*/
    KEY_MASK_SEND_MAKE,				/* =		*/
    KEY_MASK_SEND_MAKE,				/* <-		*/
    KEY_MASK_SEND_MAKE,				/* TAB		*/
    KEY_MASK_SEND_MAKE,				/* Q		*/
    KEY_MASK_SEND_MAKE,				/* W		*/
    KEY_MASK_SEND_MAKE,				/* E		*/
    KEY_MASK_SEND_MAKE,				/* R		*/
    KEY_MASK_SEND_MAKE,				/* T		*/
    KEY_MASK_SEND_MAKE,				/* Y		*/
    KEY_MASK_SEND_MAKE,				/* U		*/
    KEY_MASK_SEND_MAKE,				/* I		*/
    KEY_MASK_SEND_MAKE,				/* O		*/
    KEY_MASK_SEND_MAKE,				/* P		*/
    KEY_MASK_SEND_MAKE,				/* {,[		*/
    KEY_MASK_SEND_MAKE,				/* },]		*/
    KEY_MASK_SEND_MAKE,				/* ENTER	*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* CAPS		*/
    KEY_MASK_SEND_MAKE,				/* A		*/
    KEY_MASK_SEND_MAKE,				/* S		*/
    KEY_MASK_SEND_MAKE,				/* D		*/
    KEY_MASK_SEND_MAKE,				/* F		*/
    KEY_MASK_SEND_MAKE,				/* G		*/
    KEY_MASK_SEND_MAKE,				/* H		*/
    KEY_MASK_SEND_MAKE,				/* J		*/
    KEY_MASK_SEND_MAKE,				/* K		*/
    KEY_MASK_SEND_MAKE,				/* L		*/
    KEY_MASK_SEND_MAKE,				/* ;		*/
    KEY_MASK_SEND_MAKE,				/* '		*/
    KEY_MASK_SEND_MAKE,				/* \		*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* LSHIFT	*/
    KEY_MASK_SEND_MAKE,				/* \		*/
    KEY_MASK_SEND_MAKE,				/* Z		*/
    KEY_MASK_SEND_MAKE,				/* X		*/
    KEY_MASK_SEND_MAKE,				/* C		*/
    KEY_MASK_SEND_MAKE,				/* V		*/
    KEY_MASK_SEND_MAKE,				/* B		*/
    KEY_MASK_SEND_MAKE,				/* N		*/
    KEY_MASK_SEND_MAKE,				/* M		*/
    KEY_MASK_SEND_MAKE,				/* ,		*/
    KEY_MASK_SEND_MAKE,				/* .		*/
    KEY_MASK_SEND_MAKE,				/* /		*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* RSHIFT	*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* LCTRL	*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* ALT		*/
    KEY_MASK_SEND_MAKE,		                /* SPACE	*/
    KEY_MASK_SEND_MAKE,				/* ALTGR	*/
    KEY_MASK_SEND_MAKE,				/* RCTRL	*/
    KEY_MASK_SEND_MAKE,				/* ESC		*/
    KEY_MASK_SEND_MAKE,				/* F1		*/
    KEY_MASK_SEND_MAKE,				/* F2		*/
    KEY_MASK_SEND_MAKE,				/* F3		*/
    KEY_MASK_SEND_MAKE,				/* F4		*/
    KEY_MASK_SEND_MAKE,				/* F5		*/
    KEY_MASK_SEND_MAKE,				/* F6		*/
    KEY_MASK_SEND_MAKE,				/* F7		*/
    KEY_MASK_SEND_MAKE,				/* F8		*/
    KEY_MASK_SEND_MAKE,				/* F9		*/
    KEY_MASK_SEND_MAKE,				/* F10		*/
    KEY_MASK_SEND_MAKE,				/* F11		*/
    KEY_MASK_SEND_MAKE,				/* F12		*/
    KEY_MASK_SEND_MAKE,				/* PRINTSCREEN	*/
    KEY_MASK_SEND_MAKE,				/* SCROLL LOCK	*/
    KEY_MASK_SEND_MAKE,				/* PAUSE	*/
    KEY_MASK_SEND_MAKE,				/* INSERT	*/
    KEY_MASK_SEND_MAKE,				/* HOME		*/
    KEY_MASK_SEND_MAKE,				/* PAGE UP	*/
    KEY_MASK_SEND_MAKE,				/* DELETE	*/
    KEY_MASK_SEND_MAKE,				/* END		*/
    KEY_MASK_SEND_MAKE,				/* PAGE DOWN	*/

    KEY_MASK_SEND_MAKE,				/* UP		*/
    KEY_MASK_SEND_MAKE,				/* LEFT		*/
    KEY_MASK_SEND_MAKE,				/* DOWN		*/
    KEY_MASK_SEND_MAKE,				/* RIGHT	*/

    KEY_MASK_SEND_MAKE,				/* NUMLOCK	*/
    KEY_MASK_SEND_MAKE,				/* NUM 7	*/
    KEY_MASK_SEND_MAKE,				/* NUM 8	*/
    KEY_MASK_SEND_MAKE,				/* NUM 9	*/
    KEY_MASK_SEND_MAKE,				/* NUM +	*/
    KEY_MASK_SEND_MAKE,				/* NUM /	*/
    KEY_MASK_SEND_MAKE,				/* NUM 4	*/
    KEY_MASK_SEND_MAKE,				/* NUM 5	*/
    KEY_MASK_SEND_MAKE,				/* NUM 6	*/
    KEY_MASK_SEND_MAKE,				/* NUM *	*/
    KEY_MASK_SEND_MAKE,				/* NUM 1	*/
    KEY_MASK_SEND_MAKE,				/* NUM 2	*/
    KEY_MASK_SEND_MAKE,				/* NUM 3	*/
    KEY_MASK_SEND_MAKE,				/* NUM RET	*/
    KEY_MASK_SEND_MAKE,				/* NUM -	*/
    KEY_MASK_SEND_MAKE,				/* NUM 0	*/
    KEY_MASK_SEND_MAKE,				/* NUM .	*/

    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* Dummy	*//* NO SPEC!*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* Dummy	*//* NO SPEC!*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* Dummy	*//* NO SPEC!*/

    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* LWIN		*//* NO SPEC!*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK,	/* MENU		*//* NO SPEC!*/
    KEY_MASK_SEND_MAKE | KEY_MASK_SEND_BREAK	/* RWIN		*//* NO SPEC!*/
};

static void
make_break_mask_set_scs3_defaults(driver_t* obj)
{
    unit_kbd_data_t* unit_kbd_data;

    assert(obj);
    unit_kbd_data = (unit_kbd_data_t*) obj->data_unit_kbd.data;
    assert(unit_kbd_data);
    
    memcpy(unit_kbd_data->make_break_mask, make_break_mask_scs3_defaults,
           MAX_KEYDEFS < sizeof(make_break_mask_scs3_defaults) ?
           MAX_KEYDEFS : sizeof(make_break_mask_scs3_defaults));
}

static void
propchange_handler(pp_propchange_listener_t * listener, u_short prop, u_short param)
{
    local_propchange_listener_t *mylistener;
    driver_t* obj;
    u_char kvm_unit;

    mylistener = (local_propchange_listener_t *)listener;
    obj = mylistener->obj;

    if (prop == PP_PROP_KVM_UNIT_ADD_OR_REMOVE) {
	assert(param < 256);
	kvm_unit = (u_char)param;

	if (pp_kvm_is_unit_present(kvm_unit)) {
	    // unit was added
	    alloc_kbd_state_for_kvm_unit(obj, kvm_unit);
	    reset_kbd_state_for_kvm_unit(obj, kvm_unit);
	} else {
	    // unit was removed
	    free_kbd_state_for_kvm_unit(obj, kvm_unit);
	}
    } else {
	pp_log("data_unit_kbd_ps2: %s(): Unhandled property %hu received.\n", ___F, prop);
    }
}

static void handle_local_key(keylayout_t* layout, u_char scset,
			     const u_char* scancode, size_t len) {
    int keycode;
    struct list_head *ptr;
    km_encoder_desc_data_t *enc_data;
    
    print_scancode("local key from KME", scancode, len);

    keycode = lookup_keyidx_by_sc(layout, scset, scancode, len);
    if (keycode == -1) return;
    
    KD("keycode: %d (%s)\n", keycode & 0x7f, keycode & 0x80 ? "pressed" : "released");

    MUTEX_LOCK(&km_encoder_hook_mtx);
    list_for_each(ptr, &km_encoder_hook_list) {
        enc_data = list_entry(ptr, km_encoder_desc_data_t, listnode);
        if (enc_data->encoder_hook && enc_data->encoder_hook->encoder_send_local_key_event) {
            enc_data->encoder_hook->encoder_send_local_key_event(enc_data->encoder_hook, keycode & 0xff);
        }
    }
    MUTEX_UNLOCK(&km_encoder_hook_mtx);
}
